<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Quiz;

use \WP_REST_Server;
use \WP_REST_Request;
use \WP_REST_Response;
use \WP_Error;
use \WpProQuiz_Model_Quiz;
use \WpProQuiz_Model_QuizMapper;
use \WpProQuiz_Model_Toplist;
use \WpProQuiz_Model_ToplistMapper;
use \WpProQuiz_Helper_Until;

class LeaderboardRest extends QuizRest {

	protected static $instance;

	/**
	 * singleton instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		parent::__construct();
	}

	/**
	 * Check if a given request has access to Quiz Leaderboard items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the Quiz Leaderboard `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_leaderboard_items_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to Quiz Leaderboard item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the Quiz Leaderboard `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_leaderboard_item_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/leaderboard', array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/leaderboard', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'save_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );
	}


	/**
	 * Retrieve quiz Leaderboard items.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/quiz/:id/leaderboard Get quiz Leaderboard items
	 * @apiName        GetLDQuizLeaderboard
	 * @apiGroup       QuizLeaderboard
	 * @apiDescription Retrieve quiz Leaderboard items
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function get_items( $request ) {

		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/learndash/v1/quiz/{$quiz_id}/leaderboard" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Save leaderboard items.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return boolean
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/quiz/:id/leaderboard Save leaderboard item
	 * @apiName        SaveLDQuizLeaderboard
	 * @apiGroup       QuizLeaderboard
	 * @apiDescription Save leaderboard item
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function save_item( $request ) {
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/quiz/{$quiz_id}/leaderboard" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'ld_leaderboard',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id'   => array(
					'description' => __( 'Unique identifier for the object.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'name' => array(
					'description' => __( 'The Name of user.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date' => array(
					'description' => __( "The date the object was created, in the site's timezone.", 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
			),
		);

		$schema['properties']['point'] = array(
			'description' => __( 'Point for object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['result']        = array(
			'description' => __( 'Result for object.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);
		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

}