<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Lesson;

use BuddyBossApp\Migration\LearnDash\V1\Course\CoursesError;
use BuddyBossApp\Migration\LearnDash\V1\Topic\TopicsActionRest;
use \WP_REST_Server;
use \WP_REST_Request;

class LessonsActionRest extends LessonsRest {

	protected static $instance;

	/**
	 * singleton instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/complete', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'complete' ),
					'permission_callback' => array( $this, 'get_complete_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );


		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/download', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'download_item' ),
					'permission_callback' => array( $this, 'get_download_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			) );
	}

	/**
	 * Check if a given request has access to lesson item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_complete_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = LessonsError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the lesson `complete` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_lesson_complete_permissions_check', $retval, $request );
	}


	/**
	 * Check if a given request has access to lesson item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_download_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = LessonsError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the lesson `download` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_lesson_download_permissions_check', $retval, $request );
	}

	/**
	 * Process Lesson complete.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/lessons/:id/complete Process Lesson complete
	 * @apiName        PostLDLessonComplete
	 * @apiGroup       Lessons
	 * @apiDescription Process Lesson Complete
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the lesson.
	 */
	public function complete( $request ) {
		$lesson_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/lessons/{$lesson_id}/complete" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Process Lesson Download.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/lessons/:id/download Process lesson Download
	 * @apiName        PostLDLessonDownload
	 * @apiGroup       Lessons
	 * @apiDescription Process Lesson Download
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the lesson.
	 */
	public function download_item( $request ) {

		$lesson_id = is_numeric( $request ) ? $request : (int) $request['id'];
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/lessons/{$lesson_id}/download" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}
}