<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Course;

use BuddyBossApp\Admin\Settings;
use BuddyBossApp\Migration\LearnDash\V1\Certificate\CertificateRest;
use BuddyBossApp\Migration\LearnDash\V1\Core\LDRestController;
use BuddyBossApp\Migration\LearnDash\V1\Lesson\LessonsRest;
use BuddyBossApp\Migration\LearnDash\V1\Topic\TopicsRest;
use BuddyBossApp\Migration\LearnDash\V1\Quiz\QuizRest;
use BuddyBossApp\Helpers\PostCover;
use \WP_REST_Server;
use \WP_REST_Request;
use \WP_REST_Response;
use \WP_Error;
use \WP_Query;
use \LearnDash_Settings_Section;

class CoursesRest extends LDRestController {

	protected static $instance;

	protected $post_type = 'sfwd-courses';
	protected $_my_course_progress;

	/**
	 * singleton instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'courses';
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route( $this->namespace,
			'/' . $this->rest_base, array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)', array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'single' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/details', array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item_details' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

	}

	/**
	 * Retrieve Courses.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/courses Get Courses
	 * @apiName        GetLDCourses
	 * @apiGroup       Courses
	 * @apiDescription Retrieve Courses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String} [after]  Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to excludes those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [categories] Limit results to those assigned to specific categories.
	 * @apiParam {Number=0,1} [recommended] Limit results to those recommended.
	 * @apiParam {Number=0,1} [mycourses] Limit results to current user courses.
	 */
	public function get_items( $request ) {
		global $wp_rest_server;
		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/learndash/v1/courses' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check if a given request has access to course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the course `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_courses_permissions_check', $retval, $request );
	}


	/**
	 * Retrieve Course.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/courses/:id Get Course
	 * @apiName        GetLDCourse
	 * @apiGroup       Courses
	 * @apiDescription Retrieve single Course
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function get_item( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/learndash/v1/courses/{$course_id}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check if a given request has access to course item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the course `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_course_permissions_check', $retval, $request );
	}


	/**
	 * Retrieve Course details like lesson/topics and quiz.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/courses/:id/details Get course details
	 * @apiName        GetLDCourse
	 * @apiGroup       Courses
	 * @apiDescription Retrieve single Course
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function get_item_details( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/learndash/v1/courses/{$course_id}/details" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
				'my_progress',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.' ),
		);

		$params['categories'] = array(
			'description' => sprintf( __( 'Limit result set to all items that have the specified term assigned in the %s.' ), 'category' ),
			'type'        => 'array',
			'items'       => array( 'type' => 'integer' ),
		);

		$params['categories_exclude'] = array(
			'description' => sprintf( __( 'Limit result set to all items that have the specified term assigned in the %s.' ), 'category' ),
			'type'        => 'array',
			'items'       => array( 'type' => 'integer' ),
		);

		$params['tags'] = array(
			'description' => sprintf( __( 'Limit result set to all items that have the specified term assigned in the %s.' ), 'tag' ),
			'type'        => 'array',
			'items'       => array( 'type' => 'integer' ),
		);

		$params['tags_exclude'] = array(
			'description' => sprintf( __( 'Limit result set to all items except those that have the specified term assigned in the %s.' ), 'tag' ),
			'type'        => 'array',
			'items'       => array( 'type' => 'integer' ),
		);

		$params['recommended'] = array(
			'description'       => __( 'Limit result set to posts that are recommended.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['mycourses'] = array(
			'description'       => __( 'Limit response to resources which is taken by current user.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['group_id'] = array(
			'description'       => __( 'Limit response to resources that are connected with a group.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'course',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'single', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'single', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'single', 'edit' ),
						),
						'short' => array(
							'description' => __( 'Short content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'single', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'modified'       => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'single', 'edit' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'single', 'edit' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'             => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.','buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'single', 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The ID for the author of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'single', 'edit' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'single', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full URL of the image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'cover_media' => array(
					'description' => __( 'Cover media object containing thumb and full URL of the image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'single', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.','buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'single', 'edit' ),
				),
			),
		);

		$schema['properties']['categories'] = array(
			'description' => sprintf( __( 'The terms assigned to the object in the %s taxonomy.', 'buddyboss-app' ), 'categories' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['enrolled_members'] = array(
			'description' => __( 'Total count of enrolled member in course.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'single', 'edit' ),
		);


		$schema['properties']['certificate'] = array(
			'description' => __( 'Certificate Details', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'view', 'single', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'available' => array(
						'description' => __( 'Tells if a certificate is available with this course.', 'buddyboss-app' ),
						'type'        => 'boolean',
						'context'     => array( 'view', 'single', 'edit' ),
					),
					'download_link' => array(
						'description' => __( 'Download link to the certificate if available.', 'buddyboss-app' ),
						'type'        => 'string',
						'context'     => array( 'view', 'single', 'edit' ),
					)
				),
			),
		);

		$schema['properties']['tags'] = array(
			'description' => sprintf( __( 'The terms assigned to the object in the %s taxonomy.', 'buddyboss-app' ), 'tags' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['module'] = array(
			'description' => sprintf( __( 'The terms assigned to the object in the %s taxonomy.', 'buddyboss-app' ), 'module' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['featured'] = array(
			'description' => __( 'Whether or not the object should be treated as featured.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['materials'] = array(
			'description' => __( 'Materials for the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['purchasable'] = array(
			'description' => __( 'Whether or not object is purchasable', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['offline_disabled'] = array(
			'description' => __( 'Disable offline support', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['price'] = array(
			'description' => __( 'The price of the object', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'single', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'value'    => array(
						'description' => __( 'value for the object, as it exists in the database.', 'buddyboss-app' ),
						'type'        => 'integer',
						'context'     => array( 'single', 'edit' ),
					),
					'rendered' => array(
						'description' => __( 'Rendered value for object.', 'buddyboss-app' ),
						'type'        => 'string',
						'context'     => array( 'single', 'edit' ),
					),
					'code'     => array(
						'description' => __( 'currency code for object.', 'buddyboss-app' ),
						'type'        => 'string',
						'context'     => array( 'view' ),
					),
				),
			),
		);

		$schema['properties']['video'] = array(
			'description' => __( 'video for the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['group'] = array(
			'description' => __( 'buddypress group for the object.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['forum'] = array(
			'description' => __( 'bbpress forum for the object.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['is_closed'] = array(
			'description' => __( 'Whether or not object is closed ot not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['has_course_access'] = array(
			'description' => __( 'Whether or not user have the object access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['has_content_access'] = array(
			'description' => __( 'Whether or not user have the object content access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['hide_content_table'] = array(
			'description' => __( 'Whether or not user have the access of object content access.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['progression'] = array(
			'description' => __( 'The progression % of user for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['points'] = array(
			'description' => __( 'Point for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['duration'] = array(
			'description' => __( 'The time duration for the object', 'buddyboss-app' ),
			'type'        => 'object',
			'context'     => array( 'single', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'min' => array(
						'description' => __( 'Min required time for this object.', 'buddyboss-app' ),
						'type'        => 'integer',
						'context'     => array( 'single', 'edit' ),
					),
				),
			),
		);

		$schema['properties']['quizzes'] = array(
			'description' => __( 'The quizes list which has access of the object','buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['lessons'] = array(
			'description' => __( 'The lessons list which has access of the object', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'single', 'edit' ),
		);

		$schema['properties']['completed'] = array(
			'description' => __( 'The object is completed by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['quiz_completed'] = array(
			'description' => __( 'The object Quizes is completed by current user or not.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'single', 'edit' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'single', 'edit', 'embed' ),
		);

		$schema['properties']['show_start'] = array(
			'description' => __( 'Whether or not user has started process..', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'single', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}
}