<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Course;

use BuddyBossApp\Migration\LearnDash\V1\Core\LDUserController;
use \WP_REST_Server;

class CoursesMembersRest extends LDUserController {

	protected static $instance;
	protected $post_type = 'sfwd-courses';

	/**
	 * singleton instance.
	 *
	 * @return CoursesMembersRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		parent::__construct();
		$this->rest_base = 'courses';
	}

	/**
	 * Check if a given request has access to course item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the course `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_course_memebers_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route( $this->namespace,
			$this->rest_base . '/(?P<id>\d+)/members',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_course_members' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );
	}

	/**
	 * Retrieve Course Members.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/courses/:id/members Get Course Members
	 * @apiName        GetLDCourseMembers
	 * @apiGroup       Courses
	 * @apiDescription Retrieve single Courses Members
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=3] Maximum number of items to be returned in result set.
	 */
	public function get_course_members( $request ) {

		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/learndash/v1/courses/{$course_id}/members" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		return  array(
			'context'  => $this->get_context_param( array( 'default' => 'view' ) ),
			'per_page' => array(
				'description'       => __( 'Maximum number of items to be returned in result set.' ),
				'type'              => 'integer',
				'default'           => 10,
				'minimum'           => 1,
				'maximum'           => 100,
				'sanitize_callback' => 'absint',
				'validate_callback' => 'rest_validate_request_arg',
			),
			'page' => array(
				'description'       => __( 'Number of p[age you want to retrieve items for.' ),
				'type'              => 'integer',
				'default'           => 1,
				'minimum'           => 1,
				'maximum'           => 100000,
				'sanitize_callback' => 'absint',
				'validate_callback' => 'rest_validate_request_arg',
			),
		);
	}

}