<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Course;

use BuddyBossApp\Migration\LearnDash\V1\Lesson\LessonsError;
use \WP_REST_Server;
use \WP_REST_Request;

class CoursesActionRest extends CoursesRest {

	protected static $instance;

	/**
	 * singleton instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>\d+)/enroll', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'enroll' ),
					'permission_callback' => array( $this, 'get_enroll_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/download', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'download_item' ),
					'permission_callback' => array( $this, 'get_download_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/upload', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'upload_progress' ),
					'permission_callback' => array( $this, 'get_upload_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			) );
	}

	/**
	 * Process Course enrollment.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/courses/:id/enroll Process Course enrollment
	 * @apiName        PostLDCourseEnroll
	 * @apiGroup       Courses
	 * @apiDescription Process Course enrollment
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function enroll( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/courses/{$course_id}/enroll" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check if a given request has access to enroll course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_enroll_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = CoursesError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the course `enroll` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_courses_enroll_permissions_check', $retval, $request );
	}

	/**
	 * Process Course Download.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/courses/:id/download Process Course Download
	 * @apiName        PostLDCourseDownload
	 * @apiGroup       Courses
	 * @apiDescription Process Course Download
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 */
	public function download_item( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/courses/{$course_id}/download" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check if a given request has access to download course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_download_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = CoursesError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the course `download`  permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_courses_download_permissions_check', $retval, $request );
	}

	/**
	 * Process Course Statements.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/courses/:id/upload Process Course Statements.
	 * @apiName        PostLDCourseStatements
	 * @apiGroup       Courses
	 * @apiDescription Process Course Statements.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course.
	 * @apiParam {Array} [statements] statements data to perform action. action,object,item_id are required.  .
	 */
	public function upload_progress( $request ) {
		$course_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/courses/{$course_id}/upload" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check if a given request has access to upload course items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_upload_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = CoursesError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the course `upload`  permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_courses_upload_permissions_check', $retval, $request );
	}

}