<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Course\Category;

use BuddyBossApp\Migration\LearnDash\V1\Core\LDRestController;
use BuddyBossApp\Migration\LearnDash\V1\Course\CoursesError;
use \WP_REST_Server;
use \WP_REST_Request;
use \WP_REST_Response;
use \WP_Error;
use \WP_Query;
use \WP_REST_Term_Meta_Fields;

class CategoriesRest extends LDRestController {

	protected static $instance;

	public $post_type = 'sfwd-courses';
	protected $taxonomy = 'ld_course_category';

	/**
	 * singleton instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'course-categories';
		$this->meta = new \WP_REST_Term_Meta_Fields($this->taxonomy);
		parent::__construct();
	}

	/**
	 * Check if a given request has access to course category items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the course category `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_course_categories_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to course category item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the course category `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_course_category_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route( $this->namespace,
			'/' . $this->rest_base, array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)', array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );
	}

	/**
	 * Retrieve Course Categories.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/course-categories Get Course Categories
	 * @apiName        GetLDCourseCategories
	 * @apiGroup       Courses
	 * @apiDescription Retrieve Course Categories
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String=asc,desc} [order=asc] Sort result set by given order.
	 * @apiParam {String=date,term_id,name,slug} [orderby=name] Sort result set by given field.
	 * @apiParam {String=asc,desc} [course_order=desc] Sort courses's of result set  by given order.
	 * @apiParam {String=date,id,title,menu_order} [course_orderby=date] Sort courses's of result set by given field.
	 * @apiParam {Number} [courses_limit=5] Maximum number of courses's of item to be returned in result set.
	 */
	public function get_items( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/learndash/v1/course-categories' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Retrieve Course Category.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/course-categories/:id Get Course category
	 * @apiName        GetLDCourseCategory
	 * @apiGroup       Courses
	 * @apiDescription Retrieve single Course category
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the course category.
	 */
	public function get_item( $request ) {

		$category_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/learndash/v1/course-categories/{$category_id}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $item, $request ) {

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();

		// Base fields for every post.
		$data = array(
			'id'          => $item->id,
			'count'       => $item->count,
			'description' => $item->description,
			'link'        => $item->link,
			'name'        => $item->name,
			'slug'        => $item->slug,
		);

		if ( ! empty( $schema['properties']['parent'] ) ) {
			$data['parent'] = (int) $item->parent;
		}

		if ( ! empty( $schema['properties']['meta'] ) ) {
			$data['meta'] = $item->meta;
		}

		if ( ! empty( $schema['properties']['courses'] ) && isset( $item->courses ) ) {
			$data['courses'] = $item->courses;
		}

		/**
		 * icon
		 */
		$data['icon']          = array();
		$data['icon']["small"] = ( is_array( $item->icon ) && isset( $item->icon["small"] ) ) ? $item->icon["small"] : null;
		$data['icon']["large"] = ( is_array( $item->icon ) && isset( $item->icon["large"] ) ) ? $item->icon["large"] : null;

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		return apply_filters( 'bbapp_ld_rest_prepare_course_category', $response, $item, $request );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['order'] = array(
			'description'       => __( 'Order sort attribute ascending or descending.' ),
			'type'              => 'string',
			'default'           => 'asc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['orderby'] = array(
			'description'       => __( 'Sort collection by object attribute.' ),
			'type'              => 'string',
			'default'           => 'name',
			'enum'              => array(
				'date',
				'name',
				'id',
				'slug',
				'term_group',
				'term_id',
				'description',
				'parent',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['course_order']   = array(
			'description'       => __( 'Order sort attribute ascending or descending of courses.' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['course_orderby'] = array(
			'description'       => __( 'Sort courses collection by object attribute.' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
				'menu_order',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['hide_empty'] = array(
			'description' => __( 'Whether to hide terms not assigned to any items.' ),
			'type'        => 'boolean',
			'default'     => false,
		);

		$params['slug'] = array(
			'description' => __( 'Limit result set to terms with one or more specific slugs.' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'string',
			),
		);

		$params['courses_limit'] = array(
			'description'       => __( 'Limit result set to courses populated on each category.' ),
			'type'              => 'integer',
			'default'           => 5,
			'sanitize_callback' => 'absint',
		);

		$params['parent'] = array(
			'description'       => __( 'Limit result set by courses category parent.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
		);

		$params['courses_exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific course IDs.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['mycourses'] = array(
			'description'       => __( 'Limit response to resources which are taken by the current user.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/schema#',
			'title'      => 'lms-categories',
			'type'       => 'object',
			'properties' => array(
				'id'          => array(
					'description' => __( 'Unique identifier for the term.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'embed', 'edit' ),
					'readonly'    => true,
				),
				'count'       => array(
					'description' => __( 'Number of published posts for the term.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'description' => array(
					'description' => __( 'HTML description of the term.' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'link'        => array(
					'description' => __( 'URL of the term.' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'embed', 'edit' ),
					'readonly'    => true,
				),
				'name'        => array(
					'description' => __( 'HTML title for the term.' ),
					'type'        => 'string',
					'context'     => array( 'view', 'embed', 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_text_field',
					),
					'required'    => true,
				),
				'slug'        => array(
					'description' => __( 'An alphanumeric identifier for the term unique to its type.' ),
					'type'        => 'string',
					'context'     => array( 'view', 'embed', 'edit' ),
					'arg_options' => array(
						'sanitize_callback' => array( $this, 'sanitize_slug' ),
					),
				),
				'icon'        => array(
					'description' => __( 'icon object containing thumb and full url of image .' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
			),
		);

		$schema['properties']['parent'] = array(
			'description' => __( 'The parent term ID.' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['courses'] = array(
			'description' => __( 'Courses for the term..' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['meta'] = array(
			'description' => __( 'Meta for the term..' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * @param int $term_id
	 *
	 * @return array
	 */
	public function get_icon_media( $term_id = 0 ) {
		$return = array(
			"large" => null,
			"small" => null,
		);

		if ( ! empty( $term_id ) && term_exists( $term_id ) ) {
			$large = bbapp_lms_taxonomy_image_url( $term_id, 'full', true );
			$small = bbapp_lms_taxonomy_image_url( $term_id, 'thumbnail', true );

			if ( isset( $large ) ) {
				$return["large"] = $large;
			}
			if ( isset( $small ) ) {
				$return["small"] = $small;
			}
		}

		return $return;

	}

	/**
	 * @return object
	 */
	private function get_uncategorised_category() {
		return (object) array(
			'term_id'          => 0,
			'id'               => 0,
			'link'             => '',
			'name'             => 'Uncategorised',
			'slug'             => 'uncategorised',
			'term_group'       => 0,
			'term_taxonomy_id' => 0,
			'taxonomy'         => $this->taxonomy,
			'description'      => '',
			'parent'           => 0,
			'count'            => 0,
			'filter'           => 'raw',
			'meta'             => array(),
			'icon'             => $this->get_icon_media(),
		);
	}

	private function get_courses( $term, $args ) {

		$user_id = get_current_user_id();

		$query_args = array(
			'post_type'              => $this->post_type,
			'fields'                 => 'ids',
			'post__not_in'           => isset( $args["courses_exclude"] ) ? $args["courses_exclude"] : null,
			'order'                  => $args["course_order"],
			'orderby'                => $args["course_orderby"],
			'no_found_rows'          => true,
			'update_post_meta_cache' => false,
			'update_post_term_cache' => false,
		);

		if ( isset( $args['mycourses'] ) && $args['mycourses'] ) {
			$mycourse_ids = ld_get_mycourses( $user_id, array() );

			if ( ! empty( $mycourse_ids ) && ! is_wp_error( $mycourse_ids ) ) {
				$query_args['post__in'] = ! empty( $query_args['post__in'] ) ? array_intersect( $mycourse_ids, $query_args['post__in'] ) : $mycourse_ids;
				/*
					 * If we intersected, but there are no post ids in common,
					 * WP_Query won't return "no posts" for post__in = array()
					 * so we have to fake it a bit.
				*/
				if ( ! $query_args['post__in'] ) {
					$query_args['post__in'] = array( 0 );
				}
			} else {
				$query_args['post__in'] = array( 0 );
			}
		}

		$query_args['posts_per_page'] = $args['courses_limit'];

		if ( isset( $term->term_id ) && empty( $term->term_id ) ) {
			$query_args['tax_query'][] = array(
				'taxonomy' => 'ld_course_category',
				'operator' => 'NOT EXISTS',
			);
			unset( $query_args['no_found_rows'] );
		} else {
			$query_args['tax_query'][] = array(
				'taxonomy'         => 'ld_course_category',
				'field'            => 'term_id',
				'terms'            => $term->term_id,
				'include_children' => false,
			);
		}

		$posts_query = new WP_Query();
		$posts       = $posts_query->query( $query_args );

		// Return count for uncategorised so we no need to do extra query to count course without category
		if ( isset( $term->term_id ) && empty( $term->term_id ) ) {
			$posts = array( 'posts' => $posts, 'count' => $posts_query->found_posts );
		}

		return $posts;
	}

}