<?php

namespace BuddyBossApp\Migration\Integrations\GamiPress;

use \WP_Error as WP_Error;

class RestAPIAchievement extends \WP_REST_Controller {


	protected $namespace = 'appboss/gamipress/v1';
	protected $rest_base = 'achievements';
	protected static $instance;

	/**
	 * RestAPIAchievement constructor.
	 */
	public function __construct() {
		 /** Nothing here */
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}
		return self::$instance;
	}

	/**
	 *
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'registerRoutes' ), 99 );
	}

	/**
	 *
	 */
	public function registerRoutes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * @param $request
	 * @return array
     * @apiPrivate
	 * @api {GET} /wp-json/appboss/gamipress/v1/achievements Learner achievements
	 * @apiName GetGamipressAchievements
	 * @apiGroup Gamipress
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get Gamipress achievements
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiParam {Number} user_id Limit result set to specific user.
	 */
	public function get_items( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/gamipress/v1/achievements" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 *
	 * @return mixed
	 */
	public function get_items_permissions_check( $request ) {
		return apply_filters( 'bbapp_gamipress_achievements_items_permissions', true, $request );
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'gamipress_achievement',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id'      => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'   => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content' => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'link'    => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'image'   => array(
					'description' => __( 'URL to the image object.', 'buddyboss-app' ),
					'type'        => 'image',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
			),
		);

		$schema['properties']['points'] = array(
			'description' => __( 'Point for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['points_type'] = array(
			'description' => __( 'Point type for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		$params['user_id'] = array(
			'description'       => __( 'Limit result set to specific user.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['order']   = array(
			'description'       => __( 'Order sort attribute ascending or descending.' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby'] = array(
			'description'       => __( 'Sort collection by object attribute.' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param $request
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()["_embed"] ) ) {
			$query_params["_embed"] = $request->get_query_params()["_embed"];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}

}
