<?php

namespace BuddyBossApp\Migration\InAppPurchases;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\Controller;
use WP_Error as WP_Error;
use WP_REST_Server;

final class Products {

	private static $instance = null;
	public $namespace        = 'appboss/iap/v1';

	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return object
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Define all actions and filters here
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register endpoints related to this class
	 */
	public function register_routes() {

		$iap_integrations = Controller::instance()->integrations;
		$integrations_types = wp_list_pluck( $iap_integrations, 'type' );

		// BuddyBossApp-IAP Products.
		register_rest_route(
			$this->namespace,
			'/products',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_bbapp_products' ),
					'permission_callback' => '__return_true',
					'args'                => array(),
				),
			)
		);

		// BuddyBossApp-IAP Product.
		register_rest_route(
			$this->namespace,
			'/products/(?P<id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'getBuddyBossAppProduct' ),
					'permission_callback' => '__return_true',
					'args'                => array(),
				),
			)
		);

		// BuddyBossApp Product by item which is connected directly/indirectly with BuddyBossApp Product for all Integration.
		register_rest_route(
			$this->namespace,
			'/products/integration/(?P<item_id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_integrated_products_by_item' ),
					'permission_callback' => '__return_true',
					'args'                => array(),
				),
			)
		);

		// BuddyBossApp Product by items which is connected directly/indirectly with BuddyBossApp Product for given Integration.
		register_rest_route(
			$this->namespace,
			'/products/(?P<integration_type>\S+)/(?P<item_id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_integrated_products_by_integration_items' ),
					'args'                => array(
						'integration_type' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'Integration Type' ),
							'enum'        => $integrations_types,
						),
					),
					'permission_callback' => '__return_true',
				),
			)
		);

		// BuddyBossApp Product by Integration for InApp-Purchase.
		register_rest_route(
			$this->namespace,
			'/products/(?P<integration_type>\S+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_bbapp_products_by_integration' ),
					'args'                => array(
						'integration_type' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'Integration Type' ),
							'enum'        => $integrations_types,
						),
					),
					'permission_callback' => '__return_true',
				),
			)
		);
	}

	/**
	 * @param $request
	 * @apiPrivate
	 * @return WP_Error
	 * @api            {GET} /wp-json/appboss/iap/v1/products Products
	 * @apiName        GetBuddyBossAppIAPProducts
	 * @apiGroup       InAppPurchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription BuddyBossApp product(s) for InAppPurchase purpose
	 * @apiUse         apidocForGetIapProducts
	 */
	public function get_bbapp_products( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/iap/v1/products' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 * @apiPrivate
	 * @return WP_Error
	 * @api            {GET} /wp-json/appboss/iap/v1/products/:id Product
	 * @apiName        GetBuddyBossAppIAPProduct
	 * @apiGroup       InAppPurchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription BuddyBossApp product for InAppPurchase purpose
	 * @apiUse         apidocForGetIapProduct
	 */
	public function getBuddyBossAppProduct( $request ) {

		$product_id = $request->get_param( 'id' );

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/iap/v1/products/{$product_id}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 * @apiPrivate
	 * @return WP_Error
	 * @api            {GET} /wp-json/appboss/iap/v1/products/:integration_type Products by integration
	 * @apiName        GetIAPProductsByIntegration
	 * @apiGroup       InAppPurchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get BuddyBossApp product(s) by integration type for InAppPurchase
	 * @apiUse         apidocForGetIapProductBYIntegration
	 */
	public function get_bbapp_products_by_integration( $request ) {

		$integration_type = $request->get_param( 'integration_type' );

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/iap/v1/products/{$integration_type}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 * @apiPrivate
	 * @return WP_Error
	 * @api            {GET} /wp-json/appboss/iap/v1/products/:integration_type/:item_id Integrated products
	 * @apiName        GetIAPProductsByIntegrationItems
	 * @apiGroup       InAppPurchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get BuddyBossApp product(s) where particular integration item of given integration_type is attached/linked for InAppPurchase.
	 * <br/>Related to:<a href="#api-InAppPurchases-CreateIAPOrder">CreateIAPOrder</a>
	 * @apiUse         apidocForGetIAPProductsByIntegrationItems
	 */
	public function get_integrated_products_by_integration_items( $request ) {
		$integration_type = $request->get_param( 'integration_type' );
		$item_id          = $request->get_param( 'item_id' );

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/iap/v1/products/{$integration_type}/{$item_id}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 * @apiPrivate
	 * @return WP_Error
	 * @api            {GET} /wp-json/appboss/iap/v1/products/integration/:item_id Integrated products
	 * @apiName        GetIAPProductsByItem
	 * @apiGroup       InAppPurchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get BuddyBossApp product(s) where particular item is attached/linked for InAppPurchase.
	 * <br/>Related to:<a href="#api-InAppPurchases-CreateIAPOrder">CreateIAPOrder</a>
	 * @apiUse         apidocForGetIAPProductsByItem
	 */
	public function get_integrated_products_by_item( $request ) {
		$item_id = $request->get_param( 'item_id' );

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/iap/v1/products/integration/{$item_id}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param $request
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}

}
