<?php

namespace BuddyBossApp\Migration\InAppPurchases;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Admin\InAppPurchases\ProductHelper;
use WP_Error as WP_Error;
use WP_REST_Server;

final class Orders {

	private static $instance = null;

	public $namespace = 'appboss/iap/v1';

	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return Orders
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * All hooks here
	 *
	 * @return void
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register all api endpoints for this class here
	 *
	 * @return void
	 */
	public function register_routes() {

		// Create Order Endpoint.
		register_rest_route(
			$this->namespace,
			'/order',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'rest_get_orders' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'device_platform' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'InAppPurchase Platform(also known as IAP Type)', 'buddyboss-app' ),
							'enum'        => array( 'ios', 'android' ),

						),
						'status'          => array(
							'required'    => true,
							'type'        => 'string',
							'default'     => 'all',
							'description' => __( 'Order status', 'buddyboss-app' ),
							'enum'        => array( 'all', 'active' ),
						),
					),
				),
				array(
					'methods'  => WP_REST_Server::EDITABLE,
					'callback' => array( $this, 'rest_create_order' ),
					'permission_callback' => '__return_true',
					'args'     => array(
						'iap_receipt_token' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'InAppPurchase Token (receipt token)' ),
						),
						'device_platform'   => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'InAppPurchase Platform(also known as IAP Type)', 'buddyboss-app' ),
							'enum'        => array( 'ios', 'android' ),

						),
						'bbapp_product_id'  => array(
							'required'    => true,
							'type'        => 'integer',
							'description' => __( 'Integration Id(also known as iap_product_id)', 'buddyboss-app' ),
						),
					),
				),
			)
		);

		// Refresh Status Order
		register_rest_route(
			$this->namespace,
			'/refresh_order/(?P<order_id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'rest_refresh_order' ),
					'args'                => array(
						'order_id' => array(
							'required'    => true,
							'type'        => 'integer',
							'description' => __( 'In-App Purchase - Purchase ID.' ),
						),
					),
					'permission_callback' => '__return_true',
				),
			)
		);
	}

	/**
	 * @param $request
	 * @apiPrivate
	 * @return array|WP_Error
	 * @api            {PUT | PATCH} /wp-json/appboss/iap/v1/refresh_order/:order_id Refresh order
	 * @apiName        RefreshIAPOrder
	 * @apiGroup       InAppPurchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Fix the Order Permission & Status.(To provide user access if not)
	 * <br/>Related to:<a href="#api-InAppPurchases-CreateIAPOrder">CreateIAPOrder</a>
	 * @apiParam {String} order_id InApp Purchase Order ID
	 */
	public function rest_refresh_order( $request ) {
		$order_id = (int) $request->get_param( 'order_id' );

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/iap/v1/refresh_order/{$order_id}" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );

	}

	/**
	 * @param $request
	 * @apiPrivate
	 * @return array|WP_Error
	 * @api            {POST} /wp-json/appboss/iap/v1/order Create order
	 * @apiName        CreateIAPOrder
	 * @apiGroup       InAppPurchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Create order for InApp Purchases
	 * <br/>Related to:<a href="#api-InAppPurchases-GetIAPProducts">GetIAPProducts</a>
	 * @apiParam {String} iap_receipt_token InAppPurchase Token (receipt token)
	 * @apiParam {String=ios,android} device_platform InAppPurchase Platform (also known as InAppPurchase type)
	 * @apiParam {Number} bbapp_product_id BuddyBossApp InAppPurchase Product Id (also known as Integration Id, nothing related to store_product_id)
	 * <a href="#api-InAppPurchases-GetIAPProducts">Gettable at endpoint</a>
	 */
	public function rest_create_order( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/iap/v1/order' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 * @apiPrivate
	 * @return WP_Error
	 * @api            {GET} /wp-json/appboss/iap/v1/orders Orders
	 * @apiName        GetIAPOrders
	 * @apiGroup       InAppPurchases
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get InAppPurchase orders
	 * <br/>Related to:<a href="#api-InAppPurchases-CreateIAPOrder">CreateIAPOrder</a>
	 * @apiUse         apidocForGetOrders
	 */
	public function rest_get_orders( $request ) {

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/iap/v1/orders' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param $request
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}
}
