<?php

namespace BuddyBossApp\Api\Learner;

use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

// NOTE : Old classname was class.bbapp_learner_topics_rest. By Ketan, Oct-2019
//(v1 Standard) Contain functionality for required additional rest api endpoints for learndash.
class TopicsRest extends Rest {

	protected static $instance;
	protected $topic_helper;

	/**
	 * TopicsRest constructor.
	 */
	public function __construct() {
		$this->rest_base = 'topics';
		parent::__construct();
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * @return void|WP_Error
	 */
	public function register_routes() {

		$this->topic_helper = bbapp_learner_learndash()->c->bbapp_learner_learndash_topics_rest;

		register_rest_route( $this->namespace, '/' . $this->rest_base, array(
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_items' ),
				'permission_callback' => array( $this, 'get_items_permissions_check' ),
				'args'                => $this->get_collection_params(),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)', array(
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_item' ),
				'permission_callback' => array( $this, 'get_item_permissions_check' ),
				'args'                => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/action/(?P<id>\d+)', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'callback' => array( $this, 'do_action' ),
				'permission_callback' => '__return_true',
				'args'     => array(
					'action' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
				),
			),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/complete/(?P<id>\d+)', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'permission_callback' => '__return_true',
				'callback' => array( $this, 'complete' ),
			),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/download', array(
			array(
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => array( $this, 'download_item' ),
				'permission_callback' => array( $this, 'get_item_permissions_check' ),
				'args'                => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
		) );
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {

		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'learner_topic',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( "The date the object was published, in the site's timezone." ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit' ),
				),
				'modified'       => array(
					'description' => __( "The date the object was last modified, in the site's timezone." ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'           => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The id for the author of the object.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full URL of image.' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),

			),
		);

		$schema['properties']['category'] = array(
			'description' => sprintf( __( 'The terms assigned to the object in the %s taxonomy.' ), 'category' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['tag'] = array(
			'description' => sprintf( __( 'The terms assigned to the object in the %s taxonomy.' ), 'tag' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['module'] = array(
			'description' => sprintf( __( 'The terms assigned to the object in the %s taxonomy.' ), 'module' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['course'] = array(
			'description' => __( 'The Course id for the object' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['lesson'] = array(
			'description' => __( 'The Lesson id for the object' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['next_topic'] = array(
			'description' => __( 'The next topic id for the object' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['duration'] = array(
			'description' => __( 'The time duration for the object' ),
			'type'        => 'object',
			'context'     => array( 'view', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'min' => array(
						'description' => __( 'Minimum required time for this object.' ),
						'type'        => 'integer',
						'context'     => array( 'view', 'edit' ),
					),
				),
			),
		);

		$schema['properties']['assignment_upload'] = array(
			'description' => __( 'Assignment upload enable for the object' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['video'] = array(
			'description' => __( 'video for the object' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['has_course_access'] = array(
			'description' => __( 'Whether or not user have the object access.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['has_content_access'] = array(
			'description' => __( 'Whether or not user have the object content access.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['quizzes'] = array(
			'description' => __( 'The quizzes list which has access of the object' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['completed'] = array(
			'description' => __( 'The object is completed by current user or not.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['quiz_completed'] = array(
			'description' => __( 'The object Quizzes is completed by current user or not.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author']         = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.' ),
			'type'              => 'array',
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.' ),
			'type'              => 'array',
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before']  = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.' ),
			'type'              => 'array',
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.' ),
			'type'              => 'array',
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['menu_order'] = array(
			'description'       => __( 'Limit result set to resources with a specific menu_order value.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent ids.' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default'           => array(),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent id.' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default'           => array(),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.' ),
		);

		return $params;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param WP_Post $data Post object.
	 *
	 * @return array Links for the given post.
	 */
	public function prepare_links( $data ) {
		$links = parent::prepare_links( $data );

		if ( isset( $data["quizzes"] ) ) {
			$links['quizzes'] = array();
			foreach ( $data["quizzes"] as $quiz ) {
				$links['quizzes'][] = array(
					'href'       => rest_url( $this->namespace . '/quiz/' . $quiz['id'] ),
					'embeddable' => true
				);
			}
		}

		if ( isset( $data["course"] ) ) {
			$links['course'] = array(
				'href'       => rest_url( $this->namespace . '/courses/' . $data["course"] ),
				'embeddable' => true
			);
		}

		if ( isset( $data["lesson"] ) ) {
			$links['lesson'] = array(
				'href'       => rest_url( $this->namespace . '/lessons/' . $data["lesson"] ),
				'embeddable' => true
			);
		}

		if ( isset( $data["next_topic"] ) ) {
			$links['next_topic'] = array(
				'href'       => rest_url( $this->namespace . '/topics/' . $data["next_topic"] ),
				'embeddable' => true
			);
		}

		$links = apply_filters( 'bbapp_learner_topic_prepare_links', $links, $data );

		return $links;
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learner/v1/topics Learner topics
	 * @apiName        GetLearnerTopics
	 * @apiGroup       LearnerTopics
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get all topics of learner component
	 * @apiUse         apidocForLearnerAllTopicsV1
	 * @apiParam {Number} [course_id=0,false] Topics attached to particular course
	 * @apiParam {Number} [lesson_id=0,false] Topics attached to particular lesson
	 * @apiDeprecated  Retrieve Topics. Check (#Topics:GetLDTopics)
	 */
	public function get_items( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		//NOTE : Applying topics filter if applicable/available/hooked
		$request = apply_filters_deprecated( 'bbapp_learner_get_topics_request', array( $request ), '1.0.0', 'bbapp_ld_get_topics_request' );
		$request = apply_filters( 'bbapp_ld_get_topics_request', $request );

		// Retrieve the list of registered collection query parameters.
		$registered = $this->get_collection_params();

		$args = array();

		/*
			 * This array defines mappings between public API query parameters whose
			 * values are accepted as-passed, and their internal WP_Query parameter
			 * name equivalents (some are the same). Only values which are also
			 * present in $registered will be set.
		*/
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'menu_order'     => 'menu_order',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
		);

		/*
			 * For each known parameter which is both registered and present in the request,
			 * set the parameter's value on the query $args.
		*/
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			}
		}

		// Check for & assign any parameters which require special handling or setting.
		$args['date_query'] = array();

		// Set before into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['before'], $request['before'] ) ) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['after'], $request['after'] ) ) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		// Ensure our per_page parameter overrides any provided posts_per_page filter.
		if ( isset( $registered['per_page'] ) ) {
			$args['posts_per_page'] = $request['per_page'];
		}

		if ( isset( $registered['sticky'], $request['sticky'] ) ) {
			$sticky_posts = get_option( 'sticky_posts', array() );
			if ( ! is_array( $sticky_posts ) ) {
				$sticky_posts = array();
			}
			if ( $request['sticky'] ) {
				/*
					 * As post__in will be used to only get sticky posts,
					 * we have to support the case where post__in was already
					 * specified.
				*/
				$args['post__in'] = $args['post__in'] ? array_intersect( $sticky_posts, $args['post__in'] ) : $sticky_posts;

				/*
					 * If we intersected, but there are no post ids in common,
					 * WP_Query won't return "no posts" for post__in = array()
					 * so we have to fake it a bit.
				*/
				if ( ! $args['post__in'] ) {
					$args['post__in'] = array( 0 );
				}
			} elseif ( $sticky_posts ) {
				/*
					 * As post___not_in will be used to only get posts that
					 * are not sticky, we have to support the case where post__not_in
					 * was already specified.
				*/
				$args['post__not_in'] = array_merge( $args['post__not_in'], $sticky_posts );
			}
		}

		$query_args = $this->prepare_items_query( $args, $request );
		//NOTE : Applying topics filter if applicable/available/hooked
		$query_result = $this->topic_helper->get_topics( array(), $query_args, $request );
		$query_result = apply_filters_deprecated( 'bbapp_learner_get_topics', array(
			$query_result,
			$query_args,
			$request
		), '1.0.0', 'bbapp_ld_get_topics' );
		$query_result = apply_filters( 'bbapp_ld_get_topics', $query_result, $query_args, $request );

		$posts = array();
		foreach ( $query_result['posts'] as $post ) {
			if ( ! $this->check_read_permission( $post ) ) {
				continue;
			}

			$data    = $this->prepare_item_for_response( $post, $request );
			$posts[] = $this->prepare_response_for_collection( $data );
		}
		$total_posts = $query_result['total_posts'];

		$max_pages = ceil( $total_posts / (int) $query_args['posts_per_page'] );

		$page = (int) $query_args['paged'];

		if ( $page > $max_pages && $total_posts > 0 ) {
			return new WP_Error( 'rest_post_invalid_page_number', __( 'The page number requested is larger than the number of pages available.' ), array( 'status' => 400 ) );
		}

		$response = rest_ensure_response( $posts );
		$response->header( 'X-WP-Total', (int) $total_posts );
		$response->header( 'X-WP-TotalPages', (int) $max_pages );

		$request_params = $request->get_query_params();
		$base           = add_query_arg( $request_params, rest_url( sprintf( '/%s/%s', $this->namespace, $this->rest_base ) ) );

		if ( $page > 1 ) {
			$prev_page = $page - 1;
			if ( $prev_page > $max_pages ) {
				$prev_page = $max_pages;
			}
			$prev_link = add_query_arg( 'page', $prev_page, $base );
			$response->link_header( 'prev', $prev_link );
		}
		if ( $max_pages > $page ) {
			$next_page = $page + 1;
			$next_link = add_query_arg( 'page', $next_page, $base );
			$response->link_header( 'next', $next_link );
		}

		return $response;
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learner/v1/topics/:id Learner topic
	 * @apiName        GetLearnerTopic
	 * @apiGroup       LearnerTopics
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get single topic of/from learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Topic ID
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiDeprecated  Retrieve single Topic. Check (#Topic:GetLDTopic)
	 */
	public function get_item( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$id = (int) $request['id'];

		$post = $this->topic_helper->get_topic( array(), $id );
		$post = apply_filters_deprecated( 'bbapp_learner_get_topic', array(
			$post,
			$post
		), '1.0.0', 'bbapp_ld_get_topic' );
		$post = apply_filters( 'bbapp_ld_get_topic', $post, $id );

		if ( empty( $id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}

		$data     = $this->prepare_item_for_response( $post, $request );
		$response = rest_ensure_response( $data );

		return $response;
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/topics/action/:id Update topic action
	 * @apiName        UpdateLearnerTopicAction
	 * @apiGroup       LearnerTopics
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Update topic of/from learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Topic Id
	 * @apiParam {String=complete} [action=''] What action to perform
	 * @apiDeprecated  Topic actions.
	 */
	public function do_action( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$topic_id = (int) $request['id'];
		$action   = $request['action'];

		// @todo : @ayush or @dipesh, why is possible_actions empty? By Ketan, May, 2019
		// Setup possible get actions
		$possible_actions = array();

		// Bail if actions aren't meant for this function
		if ( ! in_array( $action, $possible_actions ) ) {
			return new WP_Error( 'learndash_json_course_invalid_action', __( 'Action is invalid.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$user_id = get_current_user_id();

		if ( empty( $topic_id ) ) {
			return new WP_Error( 'learndash_json_topic_id', __( 'No lesson found! On which lesson are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$result = $this->topic_helper->topic_action( false, $action, $topic_id );
		$result = apply_filters( 'bbapp_learner_topic_action', $result, $action, $topic_id );

		return rest_ensure_response( $result );

	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/topics/complete/:id Complete topic
	 * @apiName        CompleteTopic
	 * @apiGroup       LearnerTopics
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Complete topic of learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Topic Id
	 * @apiParam {Number} lesson_id Lesson Id
	 * @apiParam {Number} course_id Course Id
	 * @apiDeprecated  Process Topic enrollment. Check (#Topics:PostLDTopicComplete)
	 */
	public function complete( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$topic_id = (int) $request['id'];
		$user_id  = get_current_user_id();

		if ( empty( $topic_id ) ) {
			return new WP_Error( 'learndash_json_topic_id', __( 'No topic found! On which topic are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( ! isset( $request['lesson_id'] ) ) {
			return new WP_Error( 'learndash_json_lesson_id', __( 'No lesson found! On which lesson are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( ! isset( $request['course_id'] ) ) {
			return new WP_Error( 'learndash_json_course_id', __( 'No Course found! On which Course are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$result = $this->topic_helper->complete_topic( false, $topic_id, $request );
		$result = apply_filters( 'bbapp_learner_complete_topic', $result, $topic_id, $request );

		if ( is_wp_error( $result ) ) {
			return $result;
		} elseif ( true === $result ) {
			return $this->get_item( $request );
		}

		return new WP_Error( 'learndash_json_lesson_id', __( 'Something is wrong with complete process.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post;
		setup_postdata( $post );

		$data = array();

		if ( ! $post->has_content_access ) {
			$post->post_content = '';
			$post->post_excerpt = '';
			$post->video        = '';
		}

		if ( isset( $post->ID ) ) {
			$data['id'] = $post->ID;
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "ID" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( isset( $post->post_date_gmt ) && isset( $post->post_date ) ) {
			$data['date']     = $this->prepare_date_response( $post->post_date_gmt, $post->post_date );
			$data['date_gmt'] = $this->prepare_date_response( $post->post_date_gmt );
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Date or GMT Date" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( isset( $post->guid ) ) {
			$data['guid'] = array(
				/** This filter is documented in wp-includes/post-template.php */
				'rendered' => apply_filters( 'get_the_guid', $post->guid ),
				'raw'      => $post->guid,
			);
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Guid" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( isset( $post->post_modified_gmt ) && isset( $post->post_modified ) ) {
			$data['modified']     = $this->prepare_date_response( $post->post_modified_gmt, $post->post_modified );
			$data['modified_gmt'] = $this->prepare_date_response( $post->post_modified_gmt );
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Modified Date or GMT Date" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$data['password'] = $post->post_password;

		if ( isset( $post->post_name ) ) {
			$data['slug'] = $post->post_name;
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Slug" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$data['link'] = $post->link;

		$schema = $this->get_item_schema();

		if ( ! empty( $schema['properties']['title'] ) ) {
			if ( isset( $post->post_title ) ) {
				$data['title'] = array(
					'raw'      => $post->post_title,
					'rendered' => get_the_title( $post->ID ),
				);
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Title" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['content'] ) ) {

			if ( isset( $post->post_content ) ) {
				$data['content'] = array(
					'raw'      => bbapp_learners_fix_relative_urls_protocol( $post->post_content ),
					/** This filter is documented in wp-includes/post-template.php */
					'rendered' => bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ),
				);
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Content" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}

		}

		if ( ! empty( $schema['properties']['excerpt'] ) ) {
			$data['excerpt'] = array(
				'raw'      => $post->post_excerpt,
				'rendered' => $this->prepare_excerpt_response( $post->post_excerpt ),
			);
		}

		if ( ! empty( $schema['properties']['author'] ) ) {
			if ( isset( $post->post_author ) ) {
				$data['author'] = (int) $post->post_author;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Author" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		/**
		 * Feature Media
		 */
		$data['featured_media']          = array();
		$data['featured_media']["small"] = ( is_array( $post->featured_media ) && isset( $post->featured_media["small"] ) ) ? $post->featured_media["small"] : null;
		$data['featured_media']["large"] = ( is_array( $post->featured_media ) && isset( $post->featured_media["large"] ) ) ? $post->featured_media["large"] : null;

		if ( ! empty( $schema['properties']['course'] ) ) {
			if ( isset( $post->course ) ) {
				$data['course'] = (int) $post->course;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Course ID" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['lesson'] ) ) {
			if ( isset( $post->lesson ) ) {
				$data['lesson'] = (int) $post->lesson;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Lessonn ID" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['next_topic'] ) ) {
			$data['next_topic'] = (int) $post->next_topic;
		}

		if ( ! empty( $schema['properties']['duration'] ) ) {
			$data['duration']        = array();
			$data['duration']["min"] = (int) ( is_array( $post->duration ) && isset( $post->duration["min"] ) ) ? $post->duration["min"] : 0;
		}

		if ( ! empty( $schema['properties']['assignment_upload'] ) ) {
			$data['assignment_upload'] = (boolean) $post->assignment_upload;
		}

		if ( ! empty( $schema['properties']['video'] ) ) {
			$data['video'] = $post->video;
		}

		if ( ! empty( $schema['properties']['has_course_access'] ) ) {
			if ( isset( $post->has_course_access ) ) {
				$data['has_course_access'] = (boolean) $post->has_course_access;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Has course access" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['has_content_access'] ) ) {
			if ( isset( $post->has_content_access ) ) {
				$data['has_content_access'] = (boolean) $post->has_content_access;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Has content access" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['category'] ) ) {
			if ( isset( $post->category ) ) {
				$data['category'] = $post->category;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Category" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['tag'] ) ) {
			if ( isset( $post->tag ) ) {
				$data['tag'] = $post->tag;
			} else {
				$data['tag'] = array();
			}
		}

		if ( ! empty( $schema['properties']['module'] ) ) {
			if ( isset( $post->module ) ) {
				$data['module'] = $post->module;
			} else {
				$data['module'] = array();
			}
		}

		if ( ! empty( $schema['properties']['quizzes'] ) ) {
			if ( isset( $post->quizzes ) ) {
				$data['quizzes'] = $post->quizzes;
			} else {
				$data['quizzes'] = array();
			}
		}

		if ( ! empty( $schema['properties']['completed'] ) ) {
			$data['completed'] = (boolean) $post->completed;
		}

		if ( ! empty( $schema['properties']['quiz_completed'] ) ) {
			$data['quiz_completed'] = (boolean) $post->quiz_completed;
		}

		if ( ! empty( $schema['properties']['error_message'] ) ) {
			if ( ! empty( $post->error_message ) ) {
				$data['error_message'] = $post->error_message;
			} else {
				$data['error_message'] = array();
			}
		}

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filter the post data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
		 * prepared for the response.
		 *
		 * @param WP_REST_Response $response The response object.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		$response = apply_filters_deprecated( 'rest_prepare_bbapp_learner_topic', array(
			$response,
			$post,
			$request
		), '1.0.0', 'bbapp_ld_rest_prepare_topic' );

		return apply_filters( 'bbapp_ld_rest_prepare_topic', $response, $post, $request );
	}

	/**
	 * @param        $request
	 * @param string $parent_path
	 *
	 * @return bool|WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/topics/:id/download Download topic
	 * @apiName        DownloadTopic
	 * @apiGroup       LearnerTopics
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Download topic of/for learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Topic ID (Valid post id)
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiDeprecated  Process Topic Download. Check (#Topics:PostLDTopicDownload)
	 */
	public function download_item( $request, $parent_path = '' ) {

		$topic_id = (int) $request['id'];

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		if ( empty( $topic_id ) ) {
			return new WP_Error( 'learndash_json_topic_id', __( 'No Topic was found! On which topic are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		//Set topic folder path
		$folder_name = 'topic_' . $topic_id . '/';
		if ( ! empty( $parent_path ) ) {
			$folder_name = trailingslashit( $parent_path ) . $folder_name;
		}

		//Delete course directory if exist
		$folder_path = bbapp_learners_getpath( $folder_name );
		bbapp_learners_delete_directory( $folder_path );

		//Create topic directory
		$folder_path = bbapp_learners_setpath( $folder_name );

		//Create Assests directory for topic
		$assests_path = bbapp_learners_setpath( $folder_name ) . 'topics/';
		if ( ! file_exists( $assests_path ) ) {
			wp_mkdir_p( $assests_path );
		}

		//Create lesson data
		$post = $this->topic_helper->get_topic( array(), $topic_id );
		$post = apply_filters( 'bbapp_learner_get_topic', $post, $topic_id );
		if ( empty( $topic_id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}
		$data = $this->prepare_item_for_response( $post, $request );

		//Download lesson html and assests
		$html_attrs = apply_filters( 'bbapp_learner_topic_html_attrs', array( 'content' ) );
		foreach ( $html_attrs as $attr ) {
			if ( isset( $data->data[ $attr ]['rendered'] ) ) {
				$value = $data->data[ $attr ]['rendered'] = bbapp_learners_download_html( $assests_path, $data->data[ $attr ]['rendered'] );
			} else {
				$value = $data->data[ $attr ] = bbapp_learners_download_html( $assests_path, $data->data[ $attr ] );
			}
			$file_name = 'topic_' . $attr . '.html';
			bbapp_learners_write_file( $assests_path . $file_name, $value );
		}

		//create lesson.json
		$file_data = json_encode( $data );
		$file_name = 'topic_' . $topic_id . '.json';
		bbapp_learners_write_file( $folder_path . $file_name, $file_data );

		//create info.json
		$info_data = json_encode( array(
			'id'       => $topic_id,
			'hash_key' => sha1( $post->post_modified ),
			'modified' => $this->prepare_date_response( $post->post_modified_gmt, $post->post_modified ),
		) );
		$file_name = 'info.json';
		bbapp_learners_write_file( $folder_path . $file_name, $info_data );

		if ( ! empty( $parent_path ) ) {
			//Zip will create with parent object
			return true;
		} else {
			//Create zip file
			$zip_url = bbapp_learners_zip_create( $folder_path, 'topic_' . $topic_id );
			if ( empty( $zip_url ) ) {
				return new WP_Error( 'rest_post_invalid_zip', __( 'Zip file not created.' ), array( 'status' => 500 ) );
			}

			return rest_ensure_response( array( 'zip_url' => $zip_url ) );
		}
	}
}