<?php

namespace BuddyBossApp\Api\Learner;

use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

// NOTE : Old classname was class.bbapp_learner_quiz_questions_rest. By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for learndash.
class QuizQuestionsRest extends Rest {

	protected static $instance;
	protected $quiz_helper;

	/**
	 * QuizQuestionsRest constructor.
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		parent::__construct();
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function hooks() {
	}

	/**
	 * @return void|WP_Error
	 */
	public function register_routes() {

		$this->quiz_helper = bbapp_learner_learndash()->c->bbapp_learner_learndash_quiz_rest;

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/start', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'callback' => array( $this, 'start' ),
				'permission_callback' => '__return_true',
				'args'     => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/next', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'callback' => array( $this, 'next' ),
				'permission_callback' => '__return_true',
				'args'     => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/prev', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'callback' => array( $this, 'prev' ),
				'permission_callback' => '__return_true',
				'args'     => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/upload', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'callback' => array( $this, 'upload' ),
				'permission_callback' => '__return_true',
				'args'     => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/check', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'callback' => array( $this, 'check' ),
				'permission_callback' => '__return_true',
				'args'     => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/save', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'callback' => array( $this, 'save' ),
				'permission_callback' => '__return_true',
				'args'     => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => $this->rest_base . '-question',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id'    => array(
					'description' => __( 'Unique identifier for the object.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title' => array(
					'description' => __( 'The title for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
			),
		);

		$schema['properties']['index'] = array(
			'description' => __( 'Index for the object' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['points'] = array(
			'description' => __( 'Point for the object' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['category'] = array(
			'description' => __( 'Category for the object' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['question_type'] = array(
			'description' => __( 'Question type for the object' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['hint'] = array(
			'description' => __( 'Hint for the object' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['options'] = array(
			'description' => __( 'Answers Option for the object' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['options_1'] = array(
			'description' => __( 'Answers additional Option for the object' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['answer'] = array(
			'description' => __( 'Answers for the object' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['gradedType'] = array(
			'description' => __( 'Type of essay Answers for the object' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['is_last'] = array(
			'description' => __( 'Whether it is a last object or not' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * @param $user_id
	 * @param $quiz_id
	 *
	 * @return array
	 */
	protected function get_quiz_lock( $user_id, $quiz_id ) {
		$quiz_locks = get_user_meta( $user_id, 'bbapp_learner_quiz', true );
		if ( isset( $quiz_locks[ $quiz_id ] ) ) {
			return $quiz_locks[ $quiz_id ];
		}

		return array();
	}

	/**
	 * @param $user_id
	 * @param $quiz_id
	 * @param $quiz_meta
	 *
	 * @return mixed
	 */
	protected function set_quiz_lock( $user_id, $quiz_id, $quiz_meta ) {
		$quiz_locks = get_user_meta( $user_id, 'bbapp_learner_quiz', true );
		if ( empty( $quiz_locks ) ) {
			$quiz_locks = array();
		}
		$quiz_meta              = apply_filters( 'bbapp_learner_quiz_lock', $quiz_meta, $user_id, $quiz_id );
		$quiz_locks[ $quiz_id ] = $quiz_meta;

		return update_user_meta( $user_id, 'bbapp_learner_quiz', $quiz_locks );
	}

	/**
	 * @param $user_id
	 * @param $quiz_id
	 *
	 * @return mixed
	 */
	protected function delete_quiz_lock( $user_id, $quiz_id ) {
		$quiz_locks = get_user_meta( $user_id, 'bbapp_learner_quiz', true );
		unset( $quiz_locks[ $quiz_id ] );
		if ( empty( $quiz_locks ) ) {
			return delete_user_meta( $user_id, 'bbapp_learner_quiz' );
		} else {
			return update_user_meta( $user_id, 'bbapp_learner_quiz', $quiz_locks );
		}
	}

	/**
	 * @param $request
	 *
	 * @return array|WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/quiz/:id/start Start quiz
	 * @apiName        StartLearnerQuiz
	 * @apiGroup       LearnerQuizQuestions
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Start Quiz of learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Quiz Id (Valid post id)
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiDeprecated  Start Quiz. Check (#Quiz:StartLDQuiz)
	 */
	public function start( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		global $wp_rest_server;

		$id      = (int) $request['id'];
		$user_id = get_current_user_id();

		$post = $this->quiz_helper->get_quiz( array(), $id );
		$post    = apply_filters( 'bbapp_learner_get_quiz', $post, $id );

		if ( empty( $id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}

		if ( ! $post->has_content_access ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Quiz not accessible.' ), array( 'status' => 404 ) );
		}

		$response = $this->quiz_helper->check_before_start_quiz( true, $id );
		$response = apply_filters( "bbapp_learner_start_quiz", $response, $id );

		if ( ! is_wp_error( $response ) && $response ) {

			$response = array();

			$questions = $this->quiz_helper->get_quiz_questions( array(), $id );
			$questions = apply_filters( "bbapp_learner_get_quiz_questions", $questions, $id );

			$arr_questions = array();
			$index         = 1;
			if ( ! empty( $questions ) ) {
				foreach ( $questions as $question ) {
					$arr_questions[ $index ] = array(
						'id'         => $question->ID,
						'start_time' => 0,
						'end_time'   => 0,
						'answer'     => array(),
						'responce'   => array(),
						'point'      => ( isset( $post->points ) ) ? $question->points : 0,
					);
					$question->index         = $index;
					$question->answer        = array();
					$temp_res                = $this->prepare_item_for_response( $question, $request );
					$response[]              = $wp_rest_server->response_to_data( rest_ensure_response( $temp_res ), $request['_embed'] );
					$index ++;
				}
			}

			$quiz_meta = array(
				'start_time' => current_time( 'timestamp' ),
				'end_time'   => 0,
				'questions'  => $arr_questions,
				'form'       => array(),
				'next'       => ( ! empty( $arr_questions ) ) ? 1 : - 1,
				'prev'       => - 1,
			);

			$this->set_quiz_lock( $user_id, $id, $quiz_meta );

			//Store answer Question in Meta
			if ( ! empty( $request['form'] ) ) {
				$this->save_form_meta( $user_id, $id, $request['form'] );
			}

			return $response;
			//return $this->next( $request );
		} else {
			return $response;
		}
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/quiz/:id/next Next quiz
	 * @apiName        NextLearnerQuiz
	 * @apiGroup       LearnerQuizQuestions
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Next Quiz of learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Quiz ID
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiParam {String} [answer='']
	 * @apiDeprecated  Next Quiz Question. Check (#Quiz:LDQuizQuestionNext)
	 */
	public function next( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$id      = (int) $request['id'];
		$user_id = get_current_user_id();

		$post = $this->quiz_helper->get_quiz( array(), $id );
		$post    = apply_filters( 'bbapp_learner_get_quiz', $post, $id );

		if ( empty( $id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}

		if ( ! $post->has_content_access ) {
			return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Quiz not accessible.' ), array( 'status' => 404 ) );
		}

		//Store answer Question in Meta
		if ( ! empty( $request['answer'] ) ) {
			$this->save_answer_meta( $user_id, $id, $request['answer'] );
		}

		$quiz_meta = $this->get_quiz_lock( $user_id, $id );
		if ( empty( $quiz_meta ) ) {
			return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Please start the quiz.' ), array( 'status' => 404 ) );
		}

		$question_index = (int) $quiz_meta['next'];
		if ( - 1 === $question_index ) {
			return new WP_Error( 'learndash_json_quiz_invalid_question', __( 'Next Question not found.' ), array( 'status' => 404 ) );
		}

		if ( empty( $quiz_meta['questions'][ $question_index ]['start_time'] ) ) {
			$quiz_meta['questions'][ $question_index ]['start_time'] = current_time( 'timestamp' );
			$this->set_quiz_lock( $user_id, $id, $quiz_meta );
		}

		$questions = $this->quiz_helper->get_quiz_question( array(), $quiz_meta['questions'][ $question_index ]['id'] );
		$next_Question         = apply_filters( "bbapp_learner_get_quiz_question", $questions, $quiz_meta['questions'][ $question_index ]['id'] );
		$next_Question->index  = $question_index;
		$next_Question->answer = $quiz_meta['questions'][ $question_index ]['answer'];

		if ( ! isset( $quiz_meta['questions'][ $question_index + 1 ] ) ) {
			$next_Question->is_last = true;
		}
		$data     = $this->prepare_item_for_response( $next_Question, $request );
		$response = rest_ensure_response( $data );

		return $response;
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/quiz/:id/prev Previous quiz
	 * @apiName        PreviousLearnerQuiz
	 * @apiGroup       LearnerQuizQuestions
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Previous Quiz of learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Quiz id (Valid post id)
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiDeprecated  Previous Quiz Question. Check (#Quiz:LDQuizQuestionPrevious)
	 */
	public function prev( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$id      = (int) $request['id'];
		$user_id = get_current_user_id();

		$post = $this->quiz_helper->get_quiz( array(), $id );
		$post    = apply_filters( 'bbapp_learner_get_quiz', $post, $id );

		if ( empty( $id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}

		if ( ! $post->has_content_access ) {
			return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Quiz not accessible.' ), array( 'status' => 404 ) );
		}

		$quiz_meta = $this->get_quiz_lock( $user_id, $id );
		if ( empty( $quiz_meta ) ) {
			return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Please start the quiz.' ), array( 'status' => 404 ) );
		}

		$question_index = $quiz_meta['prev'];

		if ( - 1 === $question_index ) {
			return new WP_Error( 'learndash_json_quiz_invalid_question', __( 'Prev Question not found.' ), array( 'status' => 404 ) );
		}

		if ( empty( $quiz_meta['questions'][ $question_index ]['start_time'] ) ) {
			$quiz_meta['questions'][ $question_index ]['start_time'] = current_time( 'timestamp' );
		}

		$quiz_meta['next'] = $question_index;
		$quiz_meta['prev'] = $question_index - 1;

		if ( ! isset( $quiz_meta['questions'][ $quiz_meta['prev'] ] ) ) {
			$quiz_meta['prev'] = - 1;
		}

		if ( ! isset( $quiz_meta['questions'][ $quiz_meta['next'] ] ) ) {
			$quiz_meta['next'] = - 1;
		}

		$this->set_quiz_lock( $user_id, $id, $quiz_meta );

		$questions = $this->quiz_helper->get_quiz_question( array(), $quiz_meta['questions'][ $question_index ]['id'] );
		$prev_Question         = apply_filters( "bbapp_learner_get_quiz_question", $questions, $quiz_meta['questions'][ $question_index ]['id'] );
		$prev_Question->index  = $question_index;
		$prev_Question->answer = $quiz_meta['questions'][ $question_index ]['answer'];

		$data     = $this->prepare_item_for_response( $prev_Question, $request );
		$response = rest_ensure_response( $data );

		return $response;
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/quiz/:id/upload Upload quiz
	 * @apiName        UploadLearnerQuiz
	 * @apiGroup       LearnerQuizQuestions
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Upload Quiz of learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Quiz ID (Valid post id)
	 * @apiParam {Number} question_id Question ID
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiDeprecated  Upload Quiz Question answers file. Check (#Quiz:LDQuizQuestionUpload)
	 */
	public function upload( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$id          = (int) $request['id'];
		$user_id     = get_current_user_id();
		$question_id = (int) $request['question_id'];

		$post = $this->quiz_helper->get_quiz( array(), $id );
		$post        = apply_filters( 'bbapp_learner_get_quiz', $post, $id );

		if ( empty( $id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}

		if ( ! $post->has_content_access ) {
			return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Quiz not accessible.' ), array( 'status' => 404 ) );
		}

		if ( empty( $question_id ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid Question id.' ), array( 'status' => 404 ) );
		}

		if ( empty( $_FILES ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid Files.' ), array( 'status' => 404 ) );
		}

		$data = $this->quiz_helper->upload_quiz_question_answer( array(), $id, $question_id, $_FILES['attachment'] );
		$data = apply_filters( "bbapp_learner_upload_quiz_question_answer", $data, $id, $question_id, $_FILES['attachment'] );

		return rest_ensure_response( $data );
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/quiz/:id/check Check quiz
	 * @apiName        CheckLearnerQuiz
	 * @apiGroup       LearnerQuizQuestions
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Check Quiz of learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Quiz ID (Valid post id)
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiParam {String} answer Note : Quiz question answer shouldn't be empty
	 * @apiDeprecated  Check Quiz Question is correct or not. Check (#Quiz:LDQuizQuestionCheck)
	 */
	public function check( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$id      = (int) $request['id'];
		$user_id = get_current_user_id();

		$post = $this->quiz_helper->get_quiz( array(), $id );
		$post    = apply_filters( 'bbapp_learner_get_quiz', $post, $id );

		if ( empty( $id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}

		if ( ! $post->has_content_access ) {
			return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Quiz not accessible.' ), array( 'status' => 404 ) );
		}

		$quiz_meta = $this->get_quiz_lock( $user_id, $id );
		if ( empty( $quiz_meta ) ) {
			return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Please start the quiz.' ), array( 'status' => 404 ) );
		}

		//$question_index = (int) $quiz_meta['next'];
		$answers = $request['answer'];

		if ( empty( $answers ) ) {
			return new WP_Error( 'learndash_json_quiz_invalid_answer', __( "Quiz question answer shouldn't be empty." ), array( 'status' => 404 ) );
		}

		$answer_data = array();
		foreach ( $answers as $index => $answer ) {
			/*if ( -1 === $question_index || $index != $question_index ){
				return new WP_Error( 'learndash_json_quiz_invalid_question', __( 'Invalid request.' ), array( 'status' => 404 ) );
			}*/
			$question_index                                                  = (int) $index;
			$answer_data[ $quiz_meta['questions'][ $question_index ]['id'] ] = $answer;
		}

		if ( $answer_data[ $quiz_meta['questions'][ $question_index ]['id'] ] == $quiz_meta['questions'][ $question_index ]['answer'] && ! empty( $quiz_meta['questions'][ $question_index ]['responce'] ) ) {
			$data = $quiz_meta['questions'][ $question_index ]['responce'];
		} else {

			$data = $this->quiz_helper->check_quiz_question_answer( array(
				'is_correct' => false,
				'message'    => ''
			), $id, $answer_data );
			$data = apply_filters( "bbapp_learner_check_quiz_question_answer", $data, $id, $answer_data );

			if ( ! empty( $data['data'] ) ) {
				$quiz_meta['questions'][ $question_index ]['responce'] = $data;
				$quiz_meta['questions'][ $question_index ]['answer']   = $answer_data[ $quiz_meta['questions'][ $question_index ]['id'] ];
				$this->set_quiz_lock( $user_id, $id, $quiz_meta );
			}
		}

		return rest_ensure_response( $data['data'] );
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/quiz/:id/save Save quiz
	 * @apiName        SaveLearnerQuiz
	 * @apiGroup       LearnerQuizQuestions
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Save Quiz of learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {String} id Quiz ID (Valid post id)
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiParam {String} [answer=""] Answer to be stored in Quiz Meta
	 * @apiParam {String} [form=""] Form(Answer) to be stored in Quiz Meta
	 * @apiDeprecated  Save quiz question answers. Check (#Quiz:LDQuizQuestionSave)
	 */
	public function save( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$id      = (int) $request['id'];
		$user_id = get_current_user_id();

		$post = $this->quiz_helper->get_quiz( array(), $id );
		$post    = apply_filters( 'bbapp_learner_get_quiz', $post, $id );

		if ( empty( $id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}

		if ( ! $post->has_content_access ) {
			return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Quiz not accessible.' ), array( 'status' => 404 ) );
		}

		//Store answer Question in Meta
		if ( ! empty( $request['answer'] ) ) {
			$this->save_answer_meta( $user_id, $id, $request['answer'], true );
		}

		//Store answer Question in Meta
		if ( ! empty( $request['form'] ) ) {
			$this->save_form_meta( $user_id, $id, $request['form'] );
		}

		$quiz_meta = $this->get_quiz_lock( $user_id, $id );

		$quiz_meta['end_time'] = current_time( 'timestamp' );
		$this->set_quiz_lock( $user_id, $id, $quiz_meta );

		if ( empty( $quiz_meta ) ) {
			return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Please start the quiz.' ), array( 'status' => 404 ) );
		}

		$data = $this->quiz_helper->save_quiz( array(
			'is_correct' => false,
			'statistic'  => ''
		), $quiz_meta, $id );
		$data = apply_filters( "bbapp_learner_save_quiz", $data, $quiz_meta, $id );

		if ( true === $data['finished'] ) {
			$this->delete_quiz_lock( $user_id, $id );
		}
		$response = rest_ensure_response( $data );

		return $response;
	}

	protected function save_form_meta( $user_id, $id, $form ) {
		$quiz_meta = $this->get_quiz_lock( $user_id, $id );
		if ( empty( $quiz_meta ) ) {
			return false;
		}
		$quiz_meta['form'] = $form;

		return $this->set_quiz_lock( $user_id, $id, $quiz_meta );
	}

	protected function save_answer_meta( $user_id, $id, $answers, $is_all = false ) {

		$quiz_meta = $this->get_quiz_lock( $user_id, $id );
		if ( empty( $quiz_meta ) ) {
			return false;
		}

		foreach ( $answers as $index => $answer ) {

			$question_index = (int) $quiz_meta['next'];

			if ( $index == $question_index || $is_all ) {

				if ( $answer != $quiz_meta['questions'][ $index ]['answer'] ) {
					$quiz_meta['questions'][ $index ]['answer']   = $answer;
					$quiz_meta['questions'][ $index ]['responce'] = array();
				}
				if ( empty( $quiz_meta['questions'][ $index ]['end_time'] ) ) {
					$quiz_meta['questions'][ $index ]['end_time'] = current_time( 'timestamp' );
				}

				$quiz_meta['prev'] = $index;
				$quiz_meta['next'] = $index + 1;

				if ( ! isset( $quiz_meta['questions'][ $quiz_meta['prev'] ] ) ) {
					$quiz_meta['prev'] = - 1;
				}

				if ( ! isset( $quiz_meta['questions'][ $quiz_meta['next'] ] ) ) {
					$quiz_meta['next'] = - 1;
				}
			}
		}

		return $this->set_quiz_lock( $user_id, $id, $quiz_meta );
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post;
		setup_postdata( $post );

		// Base fields for every post.
		$data = array();

		if ( isset( $post->ID ) ) {
			$data['id'] = $post->ID;
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "ID" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$schema = $this->get_item_schema();

		if ( ! empty( $schema['properties']['title'] ) ) {
			if ( isset( $post->title ) ) {
				$data['title'] = array(
					'raw'      => $post->title,
					'rendered' => apply_filters( 'the_content', $post->title ), // Question content is a title.
				);
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Title" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['index'] ) ) {
			$data['index'] = (int) $post->index;
		}

		if ( ! empty( $schema['properties']['points'] ) && isset( $post->points ) ) {
			$data['points'] = (int) $post->points;
		}

		if ( ! empty( $schema['properties']['category'] ) ) {
			$data['category'] = $post->category;
		}

		if ( ! empty( $schema['properties']['question_type'] ) ) {
			$data['question_type'] = $post->question_type;
		}

		if ( ! empty( $schema['properties']['hint'] ) ) {
			$data['hint'] = $post->hint;
		}

		if ( ! empty( $schema['properties']['options'] ) ) {
			$data['options'] = $post->options;
		}

		if ( ! empty( $schema['properties']['options_1'] ) && isset( $post->options_1 ) ) {
			$data['options_1'] = $post->options_1;
		}

		if ( ! empty( $schema['properties']['answer'] ) ) {
			$data['answer'] = $post->answer;
		}

		if ( ! empty( $schema['properties']['gradedType'] ) && isset( $post->gradedType ) ) {
			$data['gradedType'] = $post->gradedType;
		}

		if ( ! empty( $schema['properties']['is_last'] ) && isset( $post->is_last ) ) {
			$data['is_last'] = $post->is_last;
		}

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filter the post data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
		 * prepared for the response.
		 *
		 * @param WP_REST_Response $response The response object.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		$response = apply_filters_deprecated( 'rest_prepare_learner_quiz_question', array(
			$response,
			$post,
			$request
		), '1.0.0', 'bbapp_ld_rest_prepare_quiz_question' );

		return apply_filters( "bbapp_ld_rest_prepare_quiz_question", $response, $post, $request );
	}

}