<?php
namespace BuddyBossApp\Api\Learner;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

// NOTE : Old classname was class.bbapp_learner_quiz_leaderboard_rest. By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for learndash.
class QuizLeaderboardRest extends Rest {

	protected static $instance;
	protected $quiz_helper;
	protected $leaderboard_helper;

	/**
	 * QuizLeaderboardRest constructor.
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		parent::__construct();
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if (!isset(self::$instance)) {
			$class = __CLASS__;
			self::$instance = new $class;
		}
		return self::$instance;
	}

	/**
	 *
	 */
	public function hooks() {}

	/**
	 * @return void|WP_Error
	 */
	public function register_routes() {

		$this->quiz_helper = bbapp_learner_learndash()->c->bbapp_learner_learndash_quiz_rest;
		$this->leaderboard_helper = bbapp_learner_learndash()->c->bbapp_learner_learndash_quiz_leaderboard_rest;

		register_rest_route($this->namespace, '/' . $this->rest_base . '/leaderboard/(?P<id>[\d]+)', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_items'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => $this->get_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/leaderboard/(?P<id>[\d]+)/', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'save'),
				'permission_callback' => '__return_true',
				'args' => $this->get_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema' => 'http://json-schema.org/draft-04/schema#',
			'title' => $this->rest_base . '-leaderboard',
			'type' => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id' => array(
					'description' => __('Unique identifier for the object.', 'buddyboss-app'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'name' => array(
					'description' => __('The Name of user.', 'buddyboss-app'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'date' => array(
					'description' => __("The date the object was created, in the site's timezone.", 'buddyboss-app'),
					'type' => 'string',
					'format' => 'date-time',
					'context' => array('view', 'edit', 'embed'),
				),
			),
		);
		$schema['properties']['point'] = array(
			'description' => __('Point for object.', 'buddyboss-app'),
			'type' => 'string',
			'context' => array('view', 'edit', 'embed'),
		);

		$schema['properties']['result'] = array(
			'description' => __('Result for object.', 'buddyboss-app'),
			'type' => 'boolean',
			'context' => array('view', 'edit', 'embed'),
		);
		$schema['properties']['error_message'] = array(
			'description' => __('Error message for this object.', 'buddyboss-app'),
			'type' => 'array',
			'context' => array('view', 'edit', 'embed'),
		);

		return $this->add_additional_fields_schema($schema);
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		return parent::get_collection_params();
	}

	/**
	 * @param $request
	 * @return WP_Error
     * @apiPrivate
	 * @api {GET} /wp-json/appboss/learner/v1/quiz/leaderboard/:id Leaderboard
	 * @apiName GetLearnerQuizLeaderboard
	 * @apiGroup LearnerQuizLeaderboard
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get Quiz's leaderboard of learner's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Unique Identifier (Valid post id)
	 * @apiDeprecated  Retrieve quiz Leaderboard items. Check (#QuizLeaderboard:GetLDQuizLeaderboard)
	 */
	public function get_items($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		global $wp_rest_server;

		$id = (int) $request['id'];

		$post = $this->quiz_helper->get_quiz( array(), $id );
		$post = apply_filters('bbapp_learner_get_quiz', $post, $id);

		if (empty($id) || empty($post->ID)) {
			return new WP_Error('rest_post_invalid_id', __('Invalid post ID.'), array('status' => 404));
		}

		// Retrieve the list of registered collection query parameters.
		$registered = $this->get_collection_params();

		$args = array();

		/*
			 * This array defines mappings between public API query parameters whose
			 * values are accepted as-passed, and their internal WP_Query parameter
			 * name equivalents (some are the same). Only values which are also
			 * present in $registered will be set.
		*/
		$parameter_mappings = array(
			'page' => 'paged',
		);

		/*
			 * For each known parameter which is both registered and present in the request,
			 * set the parameter's value on the query $args.
		*/
		foreach ($parameter_mappings as $api_param => $wp_param) {
			if (isset($registered[$api_param], $request[$api_param])) {
				$args[$wp_param] = $request[$api_param];
			}
		}

		$query_args = $this->prepare_items_query($args, $request);

		if (isset($registered['per_page'])) {
			$query_args['per_page'] = $request['per_page'];
		}

		$query_args['quiz_id'] = $id;

		$query_result = $this->leaderboard_helper->get_lists( array(), $query_args, $request);
		$query_result = apply_filters("bbapp_learner_get_quiz_leaderboard", $query_result, $query_args, $request);

		$posts = array();
		foreach ($query_result['posts'] as $list) {
			$data = $this->prepare_item_for_response($list, $request);
			$posts[] = $this->prepare_response_for_collection($data);
		}

		$total_posts = $query_result['total_posts'];

		$max_pages = ceil($total_posts / (int) $query_args['per_page']);

		$page = (int) $query_args['paged'];

		if ($page > $max_pages && $total_posts > 0) {
			return new WP_Error('rest_post_invalid_page_number', __('The page number requested is larger than the number of pages available.', 'buddyboss-app'), array('status' => 400));
		}

		$response = rest_ensure_response($posts);
		$response->header('X-WP-Total', (int) $total_posts);
		$response->header('X-WP-TotalPages', (int) $max_pages);
		return $response;
	}

	/**
	 * @param $request
	 * @return WP_Error
     * @apiPrivate
	 * @api {PATCH} /wp-json/appboss/learner/v1/quiz/leaderboard/:id Update leaderboard
	 * @apiName UpdateLearnerQuizLeaderboard
	 * @apiGroup LearnerQuizLeaderboard
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update Quiz's leaderboard under learner component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Unique Identifier (Valid post id)
	 * @apiDeprecated  Save leaderboard item. Check (#QuizLeaderboard:SaveLDQuizLeaderboard)
	 */
	public function save($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$id = (int) $request['id'];
		$user_id = get_current_user_id();

		$post = $this->quiz_helper->get_quiz( array(), $id );
		$post = apply_filters('bbapp_learner_get_quiz', $post, $id);

		if (empty($id) || empty($post->ID)) {
			return new WP_Error('rest_post_invalid_id', __('Invalid post ID.'), array('status' => 404));
		}

		$data = $this->leaderboard_helper->save_item( $id, $request);
		$data = apply_filters("bbapp_learner_save_quiz_leaderboard_item", $id, $request);

		$response = rest_ensure_response($data);
		return $response;
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post $post Post object.
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response($post, $request) {

		// Base fields for every post.
		$data = array();

		if (isset($post->ID)) {
			$data['id'] = $post->ID;
		} else {
			return new WP_Error('learndash_json_internal_error', __('Required field "ID" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
		}

		$schema = $this->get_item_schema();

		if (isset($post->date) && isset($post->date)) {
			$data['date'] = $post->date;
		} else {
			return new WP_Error('learndash_json_internal_error', __('Required field "Date or GMT Date" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
		}

		if (!empty($schema['properties']['name'])) {
			if (isset($post->name)) {
				$data['name'] = $post->name;
			} else {
				return new WP_Error('learndash_json_internal_error', __('Required field "Title" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
			}
		}

		if (!empty($schema['properties']['point'])) {
			$data['point'] = (int) $post->point;
		}

		if (!empty($schema['properties']['result'])) {
			$data['result'] = $post->result;
		}

		if (!empty($schema['properties']['error_message'])) {
			if (!empty($post->error_message)) {
				$data['error_message'] = $post->error_message;
			} else {
				$data['error_message'] = array();
			}
		}

		$context = !empty($request['context']) ? $request['context'] : 'view';
		$data = $this->add_additional_fields_to_object($data, $request);
		$data = $this->filter_response_by_context($data, $context);

		// Wrap the data in a response object.
		$response = rest_ensure_response($data);

		$response->add_links($this->prepare_links($data));

		/**
		 * Filter the post data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
		 * prepared for the response.
		 *
		 * @param WP_REST_Response   $response   The response object.
		 * @param WP_Post            $post       Post object.
		 * @param WP_REST_Request    $request    Request object.
		 */
		return apply_filters("rest_prepare_learner_quiz_leaderboard", $response, $post, $request);
	}

}