<?php

namespace BuddyBossApp\Api\Learner\LearnDash;

use BuddyBossApp\Api\Learner\LearnDash\LdQuizpro;
use LD_QuizPro;
use LearnDash_Settings_Section;
use stdClass;
use WpProQuiz_Controller_Quiz;
use WpProQuiz_Controller_Statistics;
use WpProQuiz_Model_CategoryMapper;
use WpProQuiz_Model_FormMapper;
use WpProQuiz_Model_QuestionMapper;
use WpProQuiz_Model_Quiz;
use WpProQuiz_Model_QuizMapper;
use WP_Error;
use WP_Query;
use WpProQuiz_Model_Form;
use LearnDash_Custom_Label;
use LDLMS_Factory_Post;

// NOTE : Old classname was class.bbapp_learner_learndash_quiz_rest. By Ketan, Oct-2019
// Contain functionality for required additional rest api endpoints for learndash - v1 Standard
class QuizRest {

	public $post_type = 'sfwd-quiz';
	private $ld_quiz_pro = null;

	/**
	 * QuizRest constructor.
	 */
	public function __construct() {
		$this->hooks();
	}

	/**
	 *
	 */
	public function hooks() {

		/**
		 * Load LD_QuizPro extented class
		 */
		$this->ld_quiz_pro = new LdQuizpro();

		add_filter( 'bbapp_learner_get_quiz_items_permissions', array( $this, "get_items_permissions" ), 10, 2 );
		add_filter( 'bbapp_learner_get_quiz_item_permissions', array( $this, "get_item_permissions" ), 10, 2 );
		add_filter( 'bbapp_learner_get_quiz_read_permissions', array( $this, "get_read_permissions" ), 10, 2 );

		//add_filter( 'bbapp_learner_get_quizes', array( $this, "get_quizes" ), 10, 3 );
		//add_filter( 'bbapp_learner_get_quiz', array( $this, "get_quiz" ), 10, 2 );
		//add_filter( 'bbapp_learner_get_user_quizdata', array( $this, "get_user_quizdata" ), 10, 2 );

		//add_filter( 'bbapp_learner_start_quiz', array( $this, 'check_before_start_quiz' ), 10, 2 );

		//add_filter( 'bbapp_learner_get_quiz_questions', array( $this, "get_quiz_questions" ), 10, 2 );
		//add_filter( 'bbapp_learner_get_quiz_question', array( $this, "get_quiz_question" ), 10, 2 );
		//add_filter( 'bbapp_learner_check_quiz_question_answer', array( $this, "check_quiz_question_answer" ), 10, 3 );
		//add_filter( 'bbapp_learner_upload_quiz_question_answer', array( $this, "upload_quiz_question_answer" ), 10, 4 );
		//add_filter( 'bbapp_learner_save_quiz', array( $this, "save_quiz" ), 10, 3 );

	}

	/**
	 * @param $flag
	 * @param $request
	 *
	 * @return bool|WP_Error
	 */
	public function get_items_permissions( $flag, $request ) {

		$post_type = get_post_type_object( $this->post_type );

		if ( 'edit' === $request['context'] && ! current_user_can( $post_type->cap->edit_posts ) ) {
			return new WP_Error( 'rest_forbidden_context', __( 'Sorry, you are not allowed to edit these posts in this post type' ), array( 'status' => rest_authorization_required_code() ) );
		}

		return true;
	}

	/**
	 * @param $flag
	 * @param $request
	 *
	 * @return bool|WP_Error
	 */
	public function get_item_permissions( $flag, $request ) {

		$post = get_post( (int) $request['id'] );

		if ( 'edit' === $request['context'] ) {
			return new WP_Error( 'rest_forbidden_context', __( 'Sorry, you are not allowed to edit this post' ), array( 'status' => rest_authorization_required_code() ) );
		}

		if ( $post ) {
			return $this->get_read_permissions( false, $post );
		}

		return true;
	}

	/**
	 * @param $flag
	 * @param $post
	 *
	 * @return bool
	 */
	public function get_read_permissions( $flag, $post ) {

		$post_type = get_post_type_object( $post->post_type );
		if ( empty( $post_type ) ) {
			return false;
		}

		// Can we read the post?
		if ( 'publish' === $post->post_status || current_user_can( $post_type->cap->read_post, $post->ID ) ) {
			return true;
		}

		$post_status_obj = get_post_status_object( $post->post_status );
		if ( $post_status_obj && $post_status_obj->public ) {
			return true;
		}

		// Can we read the parent if we're inheriting?
		if ( 'inherit' === $post->post_status && $post->post_parent > 0 ) {
			$parent = get_post( $post->post_parent );

			return $this->get_read_permissions( false, $parent );
		}

		// If we don't have a parent, but the status is set to inherit, assume
		// it's published (as per get_post_status()).
		if ( 'inherit' === $post->post_status ) {
			return true;
		}

		return false;
	}

	/**
	 * @param $flag
	 * @param $id
	 *
	 * @return bool|WP_Error
	 */
	public function check_before_start_quiz( $flag, $id ) {
		$quiz_settings = learndash_get_setting( $id );

		$attempts_count = 0;
		$repeats        = trim( @$quiz_settings['repeats'] );

		$user_id  = get_current_user_id();
		$usermeta = get_user_meta( $user_id, '_sfwd-quizzes', true );
		$usermeta = maybe_unserialize( $usermeta );

		if ( ! is_array( $usermeta ) ) {
			$usermeta = array();
		}

		if ( ! empty( $usermeta ) ) {
			foreach ( $usermeta as $k => $v ) {
				if ( $v['quiz'] == $id ) {
					$attempts_count ++;
				}
			}
		}

		$attempts_left = ( $repeats == '' || $repeats >= $attempts_count );

		if ( $attempts_left ) {
			return true;
		}

		return new WP_Error( 'learndash_json_quiz_invalid_access', __( 'Not allowed to take a quiz.' ), array( 'status' => 404 ) );
	}

	/**
	 * @param $result
	 * @param $query_args
	 * @param $request
	 *
	 * @return array
	 */
	public function get_quizes( $result, $query_args, $request ) {
		$query_args['post_type'] = $this->post_type;

		$parameter_mappings = array(
			'category' => 'ld_lesson_category',
			'tag'      => 'ld_lesson_tag',
		);

		/*
			 * For each known parameter which is both registered and present in the request,
			 * set the parameter's value on the query $args.
		*/
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $request[ $api_param ] ) ) {
				$request[ $wp_param ] = $request[ $api_param ];
			}
		}

		//filter by course
		if ( isset( $request['course_id'] ) ) {
			// Order by course setting
			$course_id    = (int) ! empty( $request['course_id'] ) ? $request['course_id'] : 0;
			$default_args = sfwd_lms_get_post_options( $this->post_type );

			$order                 = bbapp_learndash_get_course_meta_setting( $course_id, 'course_lesson_order' );
			$orderby               = bbapp_learndash_get_course_meta_setting( $course_id, 'course_lesson_orderby' );
			$query_args['order']   = ! empty( $order ) ? $order : $default_args['order'];
			$query_args['orderby'] = ! empty( $orderby ) ? $orderby : $default_args['orderby'];

			if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'enabled' ) == 'yes' ) {
				$ld_course_steps_object = LDLMS_Factory_Post::course_steps( $course_id );
				$ld_course_steps_object->load_steps();
				$steps    = $ld_course_steps_object->get_steps();
				$quiz_ids = array();
				if ( isset( $steps['sfwd-lessons'] ) ) {
					foreach ( $steps['sfwd-lessons'] as $lesson ) {
						if ( isset( $lesson['sfwd-topic'] ) ) {
							foreach ( $lesson['sfwd-topic'] as $topic ) {
								if ( isset( $topic['sfwd-quiz'] ) ) {
									$quiz_ids = array_merge( $quiz_ids, array_keys( $topic['sfwd-quiz'] ) );
								}
							}
						}
						if ( isset( $lesson['sfwd-quiz'] ) ) {
							$quiz_ids = array_merge( $quiz_ids, array_keys( $lesson['sfwd-quiz'] ) );
						}
					}
				}
				if ( isset( $steps['sfwd-quiz'] ) ) {
					$quiz_ids = array_merge( $quiz_ids, array_keys( $steps['sfwd-quiz'] ) );
				}

				if ( ! empty( $quiz_ids ) ) {
					$query_args['post__in'] = $quiz_ids;
				} else {
					$query_args['post__in'] = array( - 1 );
				}
			} else {
				$query_args['meta_query'][] = array(
					'key'     => 'course_id',
					'value'   => $course_id,
					'compare' => '=',
				);
			}
		} else {

			if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'enabled' ) == 'yes' ) {
				//logic not clear
				/*$query_args['meta_query'][] = array(
					'relation' => 'OR',
				);
				$query_args['meta_query'][] = array(
					'key' => 'ld_course_%',
					'value' => 0,
					'compare' => '!='
				);
				$query_args['meta_query'][] = array(
					'key' => 'course_id',
					'value' => 0,
					'compare' => '!='
				);*/
			} else {
				$query_args['meta_query'][] = array(
					'key'     => 'course_id',
					'value'   => 0,
					'compare' => '!=',
				);
			}
		}

		$taxonomies = wp_list_filter( get_object_taxonomies( $this->post_type, 'objects' ), array( 'show_in_rest' => true ) );
		foreach ( $taxonomies as $taxonomy ) {
			$base = ! empty( $taxonomy->rest_base ) ? $taxonomy->rest_base : $taxonomy->name;

			if ( ! empty( $request[ $base ] ) ) {
				$query_args['tax_query'][] = array(
					'taxonomy'         => $taxonomy->name,
					'field'            => 'term_id',
					'terms'            => $request[ $base ],
					'include_children' => false,
				);
			}
		}

		$posts_query = new WP_Query();
		$posts       = $posts_query->query( $query_args );

		foreach ( $posts as $post ) {
			$post = $this->get_additional_data( $post );
		}

		$total_posts = $posts_query->found_posts;

		if ( $total_posts < 1 ) {
			// Out-of-bounds, run the query again without LIMIT for total count
			unset( $query_args['paged'] );
			$count_query = new WP_Query();
			$count_query->query( $query_args );
			$total_posts = $count_query->found_posts;
		}

		return array( 'posts' => $posts, 'total_posts' => $total_posts );
	}

	/**
	 * @param $result
	 * @param $id
	 *
	 * @return mixed|void
	 */
	public function get_quiz( $result, $id ) {
		$post = get_post( $id );

		if ( $this->post_type !== $post->post_type ) {
			return;
		}

		$post = $this->get_additional_data( $post );

		return $post;
	}

	/**
	 * @param $result
	 * @param $request
	 *
	 * @return array
	 */
	public function get_user_quizdata( $result, $request ) {

		$offset  = ( $request['page'] - 1 ) * $request['per_page'];
		$user_id = get_current_user_id();
		$quizzes = get_user_meta( $user_id, '_sfwd-quizzes', true );

		$quiz_ids              = wp_list_pluck( $quizzes, 'quiz' );
		$quiz_total_query_args = array(
			'post_type' => 'sfwd-quiz',
			'fields'    => 'ids',
			'orderby'   => 'title', //$atts['quiz_orderby'],
			'order'     => 'ASC', //$atts['quiz_order'],
			'nopaging'  => true,
			'post__in'  => $quiz_ids,
		);
		$quiz_query            = new WP_Query( $quiz_total_query_args );
		$quizzes_tmp           = array();
		if ( ! empty( $quizzes ) && ! empty( $quiz_query->posts ) ) {
			foreach ( $quiz_query->posts as $post_idx => $quiz_id ) {
				foreach ( $quizzes as $quiz_idx => $quiz_attempt ) {
					if ( $quiz_attempt['quiz'] == $quiz_id ) {
						$quiz_key                 = $quiz_attempt['time'] . '-' . $quiz_attempt['quiz'];
						$quizzes_tmp[ $quiz_key ] = $quiz_attempt;
						unset( $quizzes[ $quiz_idx ] );
					}
				}
			}
		}
		$quizzes = $quizzes_tmp;

		if ( isset( $request['order'] ) && $request['order'] == 'ASC' ) {
			ksort( $quizzes );
		} else {
			krsort( $quizzes );
		}

		$total_posts = count( $quizzes );
		$quizzes     = array_slice( $quizzes, $offset, $request['per_page'] );

		foreach ( $quizzes as $quizdata ) {
			$quiz_settings = learndash_get_setting( $quizdata['quiz'] );
			$quiz_post     = get_post( $quizdata['quiz'] );
			$date          = $quizdata['time'] + get_option( 'gmt_offset' ) * 3600;
			$date          = date( "Y-m-d H:i:s", $date );

			$quiz_statistic = array(
				'quiz'            => $quiz_post,
				//'quiz'            => $this->get_additional_data( $quiz_post ),
				'result'          => $quizdata['percentage'],
				'score'           => $quizdata['score'],
				'count'           => ( isset( $quizdata['question_show_count'] ) && ! empty( $quizdata['question_show_count'] ) ) ? $quizdata['question_show_count'] : $quizdata['count'],
				'awarded_points'  => $quizdata['points'],
				'possible_points' => $quizdata['total_points'],
				'Date'            => mysql_to_rfc3339( $date ),
			);

			if ( true === $quizdata['has_graded'] && ! empty( $quizdata['graded'] ) ) {
				foreach ( $quizdata['graded'] as $quiz_question_id => $graded ) {
					$graded_post = get_post( $graded['post_id'] );
					if ( $graded_post instanceof WP_Post ) {
						$quiz_statistic['graded'][ $quiz_question_id ] = array(
							'title'          => get_the_title( $graded['post_id'] ),
							'status'         => $graded['status'],
							'awarded_points' => $graded['points_awarded'],
						);
					}
				}
			}

			$certificate_post_id = intval( $quiz_settings['certificate'] );
			if ( ! empty( $certificate_post_id ) ) {
				$certificate_post = get_post( $certificate_post_id );
				$certificate      = learndash_certificate_details( $quizdata['quiz'], $user_id );

				if ( ( isset( $certificate['certificateLink'] ) ) && ( ! empty( $certificate['certificateLink'] ) ) ) {
					$quiz_statistic['certificate'] = array(
						'title'    => get_the_title( $quizdata['quiz'] ),
						'filename' => sanitize_file_name( $quiz_post->post_title ) . "-" . sanitize_file_name( $certificate_post->post_title ) . ".pdf",
						'link'     => $certificate['certificateLink'],
					);
				}
			}

			$result[] = $quiz_statistic;
		}

		return array( 'posts' => $result, 'total_posts' => $total_posts );
	}

	/**
	 * @param $result
	 * @param $quiz_id
	 *
	 * @return array
	 */
	public function get_quiz_questions( $result, $quiz_id ) {

		$quiz_settings = learndash_get_setting( $quiz_id );
		$quiz_pro      = $quiz_settings['quiz_pro'];

		$quizMapper     = new WpProQuiz_Model_QuizMapper();
		$questionMapper = new WpProQuiz_Model_QuestionMapper();

		$quiz = $quizMapper->fetch( $quiz_pro );

		if ( $quiz->isShowMaxQuestion() && $quiz->getShowMaxQuestionValue() > 0 ) {

			$value = $quiz->getShowMaxQuestionValue();

			if ( $quiz->isShowMaxQuestionPercent() ) {
				$count = $questionMapper->count( $quiz_pro );

				$value = ceil( $count * $value / 100 );
			}

			$question = $questionMapper->fetchAll( $quiz, true, $value );

		} else {
			if ( $quiz->isQuestionRandom() ) {
				$question = $questionMapper->fetchAll( $quiz, true );
			} else {
				$question = $questionMapper->fetchAll( $quiz );
			}
		}

		$arr_question = array();
		foreach ( $question as $que_post ) {
			$que_data = $this->prepare_question_data( $que_post );
			if ( $quiz->isAnswerRandom() ) {
				if ( ! empty( $que_data->options_1 ) ) {
					shuffle( $que_data->options_1 );
				} else {
					shuffle( $que_data->options );
				}
			}
			$arr_question[] = $que_data;
		}

		if ( $quiz->isSortCategories() ) {
			usort( $arr_question, array( $this, "SortCategories" ) );
		}

		return $arr_question;
	}

	/**
	 * @param $a
	 * @param $b
	 *
	 * @return int|\lt
	 */
	public function SortCategories( $a, $b ) {
		return strcmp( $b->category, $a->category );
	}

	/**
	 * @param $result
	 * @param $ques_id
	 *
	 * @return mixed
	 */
	public function get_quiz_question( $result, $ques_id ) {
		$questionMapper = new WpProQuiz_Model_QuestionMapper();
		$que_post       = $questionMapper->fetchById( $ques_id );

		return $this->prepare_question_data( $que_post );
	}

	/**
	 * @param $result
	 * @param $id
	 * @param $question_id
	 * @param $files
	 *
	 * @return mixed
	 */
	public function upload_quiz_question_answer( $result, $id, $question_id, $files ) {
		return learndash_essay_fileupload_process( $files, $question_id );
	}

	/**
	 * @param $result
	 * @param $quiz_id
	 * @param $answers
	 *
	 * @return mixed
	 */
	public function check_quiz_question_answer( $result, $quiz_id, $answers ) {

		/**
		 * Ref_Files: sfwd-lms/includes/quiz/ld-quiz-pro.php
		 * Ref_Funcation: checkAnswers -> Ajax Request
		 */

		$user_id       = get_current_user_id();
		$quiz_settings = learndash_get_setting( $quiz_id );
		$quiz_pro      = $quiz_settings['quiz_pro'];

		$questionMapper = new WpProQuiz_Model_QuestionMapper();

		foreach ( $answers as $ques_id => $answer ) {
			$que_post    = $questionMapper->fetchById( $ques_id );
			$que_post    = $this->prepare_question_data( $que_post );
			$answer_data = array();
			if ( ! empty( $que_post->options ) && 'assessment_answer' !== $que_post->question_type ) {
				$answer_index = 0;
				foreach ( $que_post->options as $option ) {
					if ( $que_post->question_type === 'single' ) {
						$answer_data[ $answer_index ] = in_array( $option->value, $answer );
					} else if ( $que_post->question_type === 'multiple' ) {
						$answer_data[ $answer_index ] = $answer[ $answer_index ] == 'true';
					} else if ( $que_post->question_type === 'cloze_answer' ) {
						$answer      = array_map( array( $this, 'cleanupCurlyQuotes' ), $answer );
						$answer_data = $answer;
					} else {
						$answer_data[ $answer_index ] = $answer[ $answer_index ];
					}
					$answer_index ++;
				}
			} elseif ( 'assessment_answer' == $que_post->question_type ) {
				$answer_data = 0;
				foreach ( $answer as $ans ) {
					$answer_data += $ans;
				}
			} else {
				$answer_data = $answer[0];
			}

			$data = array(
				'quizId'     => $quiz_pro,
				'quiz'       => $quiz_id,
				'quiz_nonce' => wp_create_nonce( 'sfwd-quiz-nonce-' . $quiz_id . '-' . $quiz_pro . '-' . $user_id ),
				'responses'  => array(
					$ques_id => array( 'response' => $answer_data ),
				),
			);

			if ( ! isset( $this->ld_quiz_pro ) ) {
				$this->ld_quiz_pro = new LdQuizpro();
			}
			$responces = $this->ld_quiz_pro->checkAnswers( $data );
			break;
		}

		if ( ! empty( $responces ) ) {
			$data      = array();
			$responces = (array) json_decode( $responces );
			foreach ( $responces as $q_id => $responce ) {
				$answer['id']          = $q_id;
				$data[ $q_id ]['data'] = array(
					'id'               => $q_id,
					'question_type'    => $responce->e->type,
					'isCorrect'        => $responce->c,
					'message'          => $responce->e->AnswerMessage,
					'point'            => $responce->p,
					'possiblePoints'   => $responce->e->possiblePoints,
					'sentItems'        => $answers[ $ques_id ],
					'sentItemsCorrect' => $this->get_Item_correct( $responce ),
					'possibleItems'    => $responce->e->c,
				);

				if ( 'essay' === $responce->e->type ) {
					$data[ $q_id ]['data']['graded_id']     = $responce->e->graded_id;
					$data[ $q_id ]['data']['graded_status'] = $responce->e->graded_status;
				}
				$data[ $q_id ]['responce'] = $responce;
			}

			/**
			 * Update Quiz Question data.
			 *
			 * @param object Quiz Question data.
			 * @param int $ques_id
			 *
			 * @return object Question data.
			 */
			return apply_filters( 'bbapp_check_quiz_question_data', $data[ $ques_id ], $ques_id );
		}

		return $result;
	}

	/**
	 * @param $responce
	 *
	 * @return array
	 */
	private function get_Item_correct( $responce ) {
		switch ( $responce->e->type ) {
			case 'cloze_answer':
				$sentItemsCorrect = array_values( (array) $responce->s );
				break;
			case 'multiple':
				$sentItemsCorrect = array();
				foreach ( $responce->e->c as $index => $value ) {
					$sentItemsCorrect[] = ( $responce->e->r[ $index ] === (bool) $value ) ? true : false;
				}
				break;
			case 'sort_answer':
			case 'matrix_sort_answer':
				$sentItemsCorrect = array();
				foreach ( $responce->e->c as $index => $value ) {
					$sentItemsCorrect[] = ( $responce->e->r[ $index ] === $value ) ? true : false;
				}
				break;
			default:
				$sentItemsCorrect = $responce->c;
		}

		return $sentItemsCorrect;
	}

	/**
	 * @param $result
	 * @param $quiz_meta
	 * @param $quiz_id
	 *
	 * @return array|bool|WP_Error
	 */
	public function save_quiz( $result, $quiz_meta, $quiz_id ) {

		/**
		 * Ref_Files: sfwd-lms/includes/quiz/ld-quiz-pro.php
		 * Ref_Funcation: checkAnswers -> Ajax Request
		 */

		$user_id       = get_current_user_id();
		$quiz_settings = learndash_get_setting( $quiz_id );
		$quiz_pro      = $quiz_settings['quiz_pro'];

		$quizMapper = new WpProQuiz_Model_QuizMapper();
		$quiz       = $quizMapper->fetch( $quiz_pro );

		$questionMapper = new WpProQuiz_Model_QuestionMapper();
		$categoryMapper = new WpProQuiz_Model_CategoryMapper();

		$statisticController = new WpProQuiz_Controller_Statistics();

		//Prepare quiz result for learndash
		$quiz_result          = array();
		$total_awarded_points = $total_possible_points = $total_correct = 0;

		$data = array(
			'quizId'     => $quiz_pro,
			'quiz'       => $quiz_id,
			'quiz_nonce' => wp_create_nonce( 'sfwd-quiz-nonce-' . $quiz_id . '-' . $quiz_pro . '-' . $user_id ),
			'responses'  => array(),
		);

		$responces = array();
		foreach ( $quiz_meta['questions'] as $question ) {
			$ques_id = $question['id'];

			if ( ! empty( $question['responce'] ) ) {
				$responces[ $ques_id ] = (object) $question['responce']['responce'];
			} else {
				$answer = $question['answer'];

				$que_post = $questionMapper->fetchById( $ques_id );
				$que_post = $this->prepare_question_data( $que_post );

				$answer_data = array();
				if ( ! empty( $que_post->options ) && 'assessment_answer' !== $que_post->question_type ) {
					$answer_index = 0;
					foreach ( $que_post->options as $option ) {
						if ( $que_post->question_type === 'single' ) {
							$answer_data[ $answer_index ] = in_array( $option->value, $answer );
						} else if ( $que_post->question_type === 'multiple' ) {
							$answer_data[ $answer_index ] = $answer[ $answer_index ] == 'true';
						} else if ( $que_post->question_type === 'cloze_answer' ) {
							$answer      = array_map( array( $this, 'cleanupCurlyQuotes' ), $answer );
							$answer_data = $answer;
						} else {
							$answer_data[ $answer_index ] = $answer[ $answer_index ];
						}
						$answer_index ++;
					}
				} elseif ( 'assessment_answer' == $que_post->question_type ) {
					$answer_data = 0;
					foreach ( $answer as $ans ) {
						$answer_data += $ans;
					}
				} else {
					$answer_data = $answer[0];
				}

				$data['responses'][ $ques_id ] = array( 'response' => $answer_data );
			}
		}

		if ( empty( $data['responses'] ) && empty( $responces ) ) {
			return false;
		}

		if ( ! empty( $data['responses'] ) ) {
			if ( ! isset( $this->ld_quiz_pro ) ) {
				$this->ld_quiz_pro = new LdQuizpro();
			}
			$ste_responces = $this->ld_quiz_pro->checkAnswers( $data );
			if ( empty( $ste_responces ) ) {
				return false;
			}
			$arr_responces = json_decode( $ste_responces );
			foreach ( $data['responses'] as $ques_id => $ans ) {
				$responces[ $ques_id ] = $arr_responces->$ques_id;
			}
		}

		//set cat point
		$cats       = array();
		$catPoints  = array();
		$categories = $categoryMapper->fetchByQuiz( $quiz_pro );
		foreach ( $categories as $cat ) {
			/* @var $cat WpProQuiz_Model_Category */

			if ( ! $cat->getCategoryId() ) {
				$cats[0] = array(
					'name'   => __( 'Not categorized', 'wp-pro-quiz' ),
					'result' => 0,
				);
				continue;
			}
			$cats[ $cat->getCategoryId() ] = array(
				'name'   => $cat->getCategoryName(),
				'result' => 0,
			);
		}

		$quiz_answers = array();

		foreach ( $quiz_meta['questions'] as $index => $question ) {
			$ques_id = $question['id'];

			//Category question total points
			$que_post = $questionMapper->fetchById( $ques_id );
			if ( ! isset( $catPoints[ $que_post->getCategoryId() ] ) ) {
				$catPoints[ $que_post->getCategoryId() ] = 0;
			}
			$catPoints[ $que_post->getCategoryId() ] += $que_post->getPoints();

			$responce = $responces[ $ques_id ];

			if ( $responce->c ) {
				$total_correct += 1;
			}
			$total_awarded_points += $responce->p;

			//Category correct question total points
			$cats[ $que_post->getCategoryId() ]['result'] += $responce->p;

			$total_possible_points += $responce->e->possiblePoints;

			$userResponse = is_array( $responce->e->r ) ? $responce->e->r : array( $responce->e->r );
			if ( ( is_array( $userResponse ) ) && ( ! empty( $userResponse ) ) ) {
				foreach ( $userResponse as $key => $value ) {
					if ( $value === true ) {
						$userResponse[ $key ] = 1;
					} else if ( $value === false ) {
						$userResponse[ $key ] = 0;
					}
				}
			}

			$quiz_result[ $question['id'] ] = array(
				"time"           => $question['end_time'] - $question['start_time'],
				"points"         => $responce->p,
				"correct"        => $responce->c,
				"data"           => $userResponse,
				"possiblePoints" => $responce->e->possiblePoints,
				"a_nonce"        => $responce->a_nonce,
				"p_nonce"        => $responce->p_nonce,
			);

			/**
			 * Update Quiz Question answers data.
			 */
			$quiz_answers[ $index ] = apply_filters( 'bbapp_quiz_question_answers', array(
				'id'               => $ques_id,
				'question_type'    => $responce->e->type,
				'isCorrect'        => $responce->c,
				'message'          => $responce->e->AnswerMessage,
				'point'            => $responce->p,
				'possiblePoints'   => $responce->e->possiblePoints,
				'sentItems'        => $question['answer'],
				'sentItemsCorrect' => $this->get_Item_correct( $responce ),
				'possibleItems'    => isset( $responce->e->c ) ? $responce->e->c : '',
			), $ques_id );

			if ( 'essay' === $responce->e->type ) {
				$quiz_result[ $question['id'] ]['graded_id']     = $responce->e->graded_id;
				$quiz_result[ $question['id'] ]['graded_status'] = $responce->e->graded_status;
				$quiz_result[ $question['id'] ]['data']          = array( $responce->e->graded_id );

				$quiz_answers[ $index ]['graded_id']     = $responce->e->graded_id;
				$quiz_answers[ $index ]['graded_status'] = $responce->e->graded_status;
			}
		}

		//Calculate persentage for cate result
		foreach ( $cats as $id => $value ) {
			if ( ! empty( $catPoints[ $id ] ) ) {
				$cats[ $id ]['result'] = round( $cats[ $id ]['result'] / $catPoints[ $id ] * 100 * 100 ) / 100;
			} else {
				$cats[ $id ]['result'] = 0;
			}
		}

		//$total_possible_points is empty if quiz have only quest which support answer point
		$result = null;
		if ( ! empty( $total_possible_points ) ) {
			$result = ( $total_awarded_points / $total_possible_points * 100 * 100 ) / 100;
		}

		//timelimited fixed
		$timelimit = $quiz->getTimeLimit();
		if ( ! empty( $timelimit ) ) {
			$diff = $quiz_meta['end_time'] - $quiz_meta['start_time'];
			if ( $diff > $timelimit ) {
				$quiz_meta['end_time'] = $quiz_meta['start_time'] + $timelimit;
			}
		}

		$quiz_result['comp'] = array(
			'points'             => $total_awarded_points,
			'correctQuestions'   => $total_correct,
			'quizTime'           => $quiz_meta['end_time'] - $quiz_meta['start_time'],
			'quizEndTimestamp'   => $quiz_meta['end_time'] * 1000,
			'quizStartTimestamp' => $quiz_meta['start_time'] * 1000,
			'result'             => $result,
			'cats'               => $cats,
		);

		/**
		 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/controller/WpProQuiz_Controller_Admin.php
		 * Ref_Funcation: completedQuiz -> Ajax Request
		 */

		$_temp_REQUEST = $_REQUEST;
		$_temp_POST    = $_POST;
		$_temp_GET     = $_GET;

		if ( empty( $_POST['results'] ) ) {

			$_REQUEST = $_POST = $_GET = array();

			$_POST['quizId']    = $quiz_pro;
			$_POST['quiz']      = $quiz_id;
			$_POST['results']   = $quiz_result;
			$_POST['timespent'] = $quiz_result['comp']['quizTime'];
			$_POST['forms']     = isset( $quiz_meta['form'] ) ? $quiz_meta['form'] : array();
			$_POST['course_id'] = isset( $_temp_REQUEST['course_id'] ) ? intval( $_temp_REQUEST['course_id'] ) : bbapp_learndash_get_course_id( $quiz_id );
			$_POST['lesson_id'] = isset( $_temp_REQUEST['lesson_id'] ) ? intval( $_temp_REQUEST['lesson_id'] ) : 0;
			$_POST['topic_id']  = isset( $_temp_REQUEST['topic_id'] ) ? intval( $_temp_REQUEST['topic_id'] ) : 0;

			$_REQUEST['quiz'] = $_POST['quiz'];
		}

		if ( empty( $_POST['quiz_nonce'] ) ) {
			$_POST['quiz_nonce'] = wp_create_nonce( 'sfwd-quiz-nonce-' . intval( $_POST['quiz'] ) . '-' . intval( $_POST['quizId'] ) . '-' . $user_id );
		}

		if ( ! isset( $_POST['results'] ) ) {
			return new WP_Error( 'learndash_json_quiz_invalid_data', __( 'An error has occurred.' ), array( 'status' => 400 ) );
		}

		//To stop "Uncanny LearnDash Toolkit" Redirection after complete event for rest endpoint
		$_REQUEST['doing_rest'] = 1;

		$is_submitted = bbapp_lms_completedQuiz();

		//store leaderboard data

		if ( $quiz->isToplistActivated() && $quiz->isToplistDataAddAutomatic() ) {
			bbapp_learner_learndash()->c->bbapp_learner_learndash_quiz_leaderboard_rest->createitem( $quiz, array(
				'point'  => $total_awarded_points,
				'result' => $quiz_result['comp']['result'],
				'ip'     => ''
			) );
		}

		$_REQUEST = $_temp_REQUEST;
		$_POST    = $_temp_POST;
		$_GET     = $_temp_GET;

		$quiz_post = get_post( $quiz_id );

		$passingpercentage = intVal( $quiz_settings['passingpercentage'] );
		$pass              = ( $quiz_result['comp']['result'] >= $passingpercentage ) ? true : false;

		$reponce = array(
			'quiz'               => $this->get_additional_data( $quiz_post ),
			'finished'           => $is_submitted,
			'is_pass'            => $pass,
			'result'             => $quiz_result['comp']['result'],
			'answers'            => $quiz_answers,
			'passing_percentage' => $passingpercentage,
		);

		//Average Result
		if ( $quiz->isShowAverageResult() ) {
			$reponce['average_result'] = $statisticController->getAverageResult( $quiz_pro );
		}

		$certificate_post_id = intval( $quiz_settings['certificate'] );
		if ( ! empty( $certificate_post_id ) ) {
			$certificate_post = get_post( $certificate_post_id );
			if ( ! empty( $certificate_post ) ) {
				$certificate = learndash_certificate_details( $quiz_id );

				if ( ( $reponce['result'] / 100 >= $certificate['certificate_threshold'] ) ) {
					$reponce['certificate']['title']    = get_the_title( $quiz_id );
					$reponce['certificate']['filename'] = sanitize_file_name( $quiz_post->post_title ) . "-" . sanitize_file_name( $certificate_post->post_title ) . ".pdf";
					$reponce['certificate']['link']     = $certificate['certificateLink'];
				}
			}
		}

		//category score
		if ( $quiz->isShowCategoryScore() ) {
			$reponce['cats'] = array_values( $quiz_result['comp']['cats'] );
		}

		if ( ! $quiz->isHideResultQuizTime() ) {
			$reponce['time'] = $quiz_result['comp']['quizTime'];
		}

		if ( ! $quiz->isHideResultPoints() ) {
			$reponce['awarded_points']  = $total_awarded_points;
			$reponce['possible_points'] = $total_possible_points;
		}

		//quiz result handle
		if ( $quiz->isResultGradeEnabled() ) {
			$_result     = $reponce['result'];
			$_resultText = $quiz->getResultText();
			$diff        = 99999;
			foreach ( $_resultText['prozent'] as $i => $value ) {
				if ( ( $_result >= $value ) && ( ( $_result - $value ) < $diff ) ) {
					$diff               = $_result - $value;
					$reponce['message'] = $_resultText['text'][ $i ];
				}
			}
			$reponce['message'] = apply_filters( 'comment_text', $reponce['message'], null, null );
		} else {
			$reponce['message'] = apply_filters( 'comment_text', $quiz->getResultText(), null, null );
		}

		return $reponce;
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function get_additional_data( $post ) {
		$post->mode               = $this->get_mode( $post );
		$post->has_course_access  = $this->get_has_course_access( $post );
		$post->has_content_access = $this->get_has_content_access( $post );
		$post->can_take_again     = $this->get_can_take_again( $post );
		$post->completed          = $this->is_completed( $post );
		$post->featured_media     = $this->get_feature_media( $post );
		$post->course_id          = $this->get_course_id( $post );
		$post->lesson_id          = $this->get_lesson_id( $post );
		$post->topic_id           = $this->get_topic_id( $post );
		$post->link               = get_permalink( $post->ID );
		$post->form               = $this->get_from( $post );
		$post->settings           = $this->get_settings( $post );
		$post->error_message      = $this->get_error_message( $post );

		return $post;
	}

	/**
	 * @param $que_post
	 *
	 * @return mixed
	 */
	public function prepare_question_data( $que_post ) {

		/**
		 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/view/WpProQuiz_View_FrontQuiz.php
		 * Ref_Funcation: showQuizBox
		 */

		$question           = new stdClass();
		$question->ID       = $que_post->getId();
		$question->title    = $que_post->getQuestion();
		$question->category = $que_post->getCategoryName();

		if ( ! $que_post->isAnswerPointsActivated() ) {
			$question->points = $que_post->getPoints();
		}

		//diffMode
		$question->question_type = $que_post->getAnswerType();
		$question->hint          = ( $que_post->isTipEnabled() ) ? do_shortcode( apply_filters( 'comment_text', $que_post->getTipMsg(), null, null ) ) : '';
		$question->options       = array();

		if ( $question->question_type === 'matrix_sort_answer' ) {
			$question->options_1 = array();
		}

		if ( ! in_array( $question->question_type, array(
			'free_answer',
			'cloze_answer',
			'assessment_answer',
			'essay',
		) ) ) {
			$answers      = $que_post->getAnswerData();
			$answer_index = 0;
			foreach ( $answers as $answer_post ) {
				if ( $que_post->isAnswerPointsActivated() ) {
					$question->points[ $answer_index ] = $answer_post->getPoints();
				}
				$answer        = new stdClass();
				$answer->title = $answer_post->getAnswer();
				$answer->value = $answer_post->getAnswer();

				if ( $question->question_type === 'sort_answer' ) {
					$answer->value = LD_QuizPro::datapos( $question->ID, $answer_index );
				}

				if ( $question->question_type !== 'sort_answer' ) {
					$answer->pos = $answer_index;
				}

				if ( $question->question_type === 'matrix_sort_answer' ) {
					$answer_1              = new stdClass();
					$answer_1->title       = $answer_post->getSortString();
					$answer_1->value       = LD_QuizPro::datapos( $question->ID, $answer_index );
					$question->options_1[] = $answer_1;
				}

				$question->options[] = $answer;

				if ( $question->question_type === 'sort_answer' || $question->question_type === 'matrix_sort_answer' ) {
					if ( ! empty( $question->options_1 ) ) {
						shuffle( $question->options_1 );
					} else {
						shuffle( $question->options );
					}
				}

				$answer_index ++;
			}
		} else if ( 'cloze_answer' === $question->question_type ) {
			$answers = $que_post->getAnswerData();
			foreach ( $answers as $answer_post ) {
				$clozeData = bbapp_lms_fetch_cloze( $answer_post->getAnswer() );
				if ( $que_post->isAnswerPointsActivated() ) {
					$question->points = $clozeData['points'];
				}
				$answer              = new stdClass();
				$answer->title       = str_replace( '@@wpProQuizCloze@@', '{___}', $clozeData['replace'] );
				$answer->title       = nl2br( $answer->title ); // Convert nl to <br />
				$question->options[] = $answer;
			}
		} else if ( 'assessment_answer' === $question->question_type ) {

			$answers = $que_post->getAnswerData();
			foreach ( $answers as $answer_post ) {
				$getAnswer = $answer_post->getAnswer();
//				$getAnswer = str_replace( '[', '', $getAnswer );
				//				$getAnswer = explode( ']', $getAnswer );
				//				$getAnswer = implode( ',', $getAnswer );
				//				$answer        = new stdClass();
				//				$answer->title = $getAnswer;
				$question->options[] = $this->format_assessment_answer( $getAnswer );
			}

		} else if ( 'essay' === $question->question_type ) {
			$answers = $que_post->getAnswerData();
			foreach ( $answers as $answer_post ) {
				$question->gradedType = $answer_post->getGradedType();
			}
		}

		/**
		 * Use filter to update Question data.
		 *
		 * @param Object $question
		 * @param Object $que_post
		 *
		 * @return Object $question
		 */
		$question = apply_filters_deprecated( 'bbapp_prepare_question_data', array(
			$question,
			$que_post
		), '1.0.0', 'bbapp_ld_prepare_quiz_question_answer' );

		return apply_filters( 'bbapp_ld_prepare_quiz_question_answer', $question, $que_post );
	}

	/**
	 * Formats the assessment answer according to learner format.
	 *
	 * @param $question
	 *
	 * @return array
	 */
	public function format_assessment_answer( $question ) {

		$field_text = $question;

		# Split all html from answer.
		$get_html = preg_split( "/{.*?}/", $field_text );

		# split all fields
		preg_match_all( "/{.*?}/", $field_text, $matched_field );
		if ( isset( $matched_field[0] ) ) {
			$matched_field = $matched_field[0];
		}

		$data = array();
		$i    = 0;

		foreach ( $get_html as $get_html ) {

			$data[] = array(
				"type"  => "html",
				"value" => bbapp_learndash_fix_html( $get_html ),
			);

			if ( isset( $matched_field[ $i ] ) ) {

				# format the values from fields into array.
				preg_match_all( "/\[(.*?)\]/", $matched_field[ $i ], $formated );
				if ( isset( $formated[1] ) ) {
					$formated = $formated[1];
				} else {
					$formated = array();
				}

				$data[] = array(
					"type"  => "radio",
					"value" => $formated,
				);

			}

			$i ++;
		}

		return $data;

	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_feature_media( $post ) {
		$return = array(
			"large" => null,
			"small" => null,
		);
		$large  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
		$small  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'medium' );

		if ( isset( $large[0] ) ) {
			$return["large"] = $large[0];
		}
		if ( isset( $small[0] ) ) {
			$return["small"] = $small[0];
		}

		return $return;

	}

	/**
	 * @param $post
	 *
	 * @return int
	 */
	public function get_course_id( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );
		if ( ! isset( $course_id ) ) {
			$course_id = 0;
		}

		return $course_id;
	}

	/**
	 * @param $post
	 *
	 * @return int
	 */
	public function get_lesson_id( $post ) {
		$lesson_id = bbapp_learndash_get_lesson_id( $post->ID );
		if ( 'sfwd-topic' == get_post_type( $lesson_id ) ) {
			$lesson_id = bbapp_learndash_get_lesson_id( $lesson_id );
		}
		if ( ! isset( $lesson_id ) ) {
			$lesson_id = 0;
		}

		return $lesson_id;
	}

	/**
	 * @param $post
	 *
	 * @return int
	 */
	public function get_topic_id( $post ) {
		$topic_id = bbapp_learndash_get_lesson_id( $post->ID );
		if ( 'sfwd-lessons' == get_post_type( $topic_id ) ) {
			unset( $topic_id );
		}
		if ( ! isset( $topic_id ) ) {
			$topic_id = 0;
		}

		return $topic_id;
	}

	/**
	 * @param $post
	 *
	 * @return string
	 */
	public function get_mode( $post ) {
		$quiz_settings = learndash_get_setting( $post->ID );
		$quiz_pro      = $quiz_settings['quiz_pro'];
		$quizMapper    = new WpProQuiz_Model_QuizMapper();
		$quiz          = $quizMapper->fetch( $quiz_pro );
		$quiz_mode_id  = (int) $quiz->getQuizModus();
		$quiz_mode     = '';
		switch ( $quiz_mode_id ) {
			case WpProQuiz_Model_Quiz::QUIZ_MODUS_NORMAL:
				$quiz_mode = 'Normal';
				break;
			case WpProQuiz_Model_Quiz::QUIZ_MODUS_BACK_BUTTON:
				$quiz_mode = 'Back Button';
				break;
			case WpProQuiz_Model_Quiz::QUIZ_MODUS_CHECK:
				$quiz_mode = 'Check';
				break;
			case WpProQuiz_Model_Quiz::QUIZ_MODUS_SINGLE:
				$quiz_mode = 'Single';
				break;
		}

		return $quiz_mode;
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function get_has_course_access( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );

		return sfwd_lms_has_access( $course_id );
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function get_has_content_access( $post ) {
		return bbapp_lms_is_content_access( $post, 'prerequities_completed' )
		       && bbapp_lms_is_content_access( $post, 'points_access' )
		       && $this->get_quiz_accessable( $post )
		       && $this->check_quiz_lock( $post, 'prerequisite' )
		       && sfwd_lms_has_access( $post->ID );
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function get_can_take_again( $post ) {
		$user_id = get_current_user_id();

		$quiz_settings  = learndash_get_setting( $post->ID );
		$repeats        = trim( @$quiz_settings['repeats'] );
		$attempts_count = 0;
		if ( $repeats != '' ) {
			$usermeta = get_user_meta( $user_id, '_sfwd-quizzes', true );
			$usermeta = maybe_unserialize( $usermeta );
			if ( ! is_array( $usermeta ) ) {
				$usermeta = array();
			}
			if ( ! empty( $usermeta ) ) {
				foreach ( $usermeta as $k => $v ) {
					if ( ( $v['quiz'] == $post->ID ) ) {
						$attempts_count ++;
					}
				}
			}
		}

		return ( $repeats == '' || $repeats >= $attempts_count ) && $this->check_quiz_lock( $post, 'lock' );
	}

	/**
	 * @param        $post
	 * @param string $type
	 *
	 * @return bool
	 */
	public function check_quiz_lock( $post, $type = 'lock' ) {
		$quiz_settings = learndash_get_setting( $post->ID );
		$quiz_pro      = $quiz_settings['quiz_pro'];
		$quizMapper    = new WpProQuiz_Model_QuizMapper();
		$quiz          = $quizMapper->fetch( $quiz_pro );

		if ( $quiz->isQuizRunOnce() || $quiz->isPrerequisite() || $quiz->isStartOnlyRegisteredUser() ) {
			if ( ! isset( $_POST['quizId'] ) ) {
				$_POST['quizId'] = $quiz_pro;
			}
			$quizController = new WpProQuiz_Controller_Quiz();
			$lock           = $quizController->isLockQuiz( $_POST['quizId'] );

			if ( isset( $lock[ $type ] ) ) {
				if ( 'lock' == $type ) {
					return ! ( $lock['lock']['is'] && $lock['lock']['pre'] );
				} else {
					return ! ( $lock[ $type ] );
				}
			}
		}

		return true;
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function get_quiz_accessable( $post ) {
		$course_id                  = bbapp_learndash_get_course_id( $post->ID );
		$lesson_progression_enabled = bbapp_learndash_get_course_meta_setting( $course_id, 'course_disable_lesson_progression' );

		return 'on' == $lesson_progression_enabled || is_quiz_accessable( null, $post );
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function is_completed( $post ) {

		$user_id = get_current_user_id();

		return learndash_is_quiz_complete( $user_id, $post->ID );

	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_from( $post ) {
		$quiz_settings = learndash_get_setting( $post->ID );
		$quiz_pro      = $quiz_settings['quiz_pro'];
		$formMapper    = new WpProQuiz_Model_FormMapper();
		$forms         = $formMapper->fetch( $quiz_pro );
		$arr_form      = array();
		foreach ( $forms as $key => $form_field ) {
			$arr_form[ $key ]['id']        = $form_field->getFormId();
			$arr_form[ $key ]['fieldname'] = $form_field->getFieldname();
			$arr_form[ $key ]['type']      = $this->get_form_type( $form_field->getType() );
			$arr_form[ $key ]['required']  = $form_field->isRequired();
			$arr_form[ $key ]['data']      = $form_field->getData();
		}

		return $arr_form;
	}

	/**
	 * @param $type
	 *
	 * @return string
	 */
	private function get_form_type( $type ) {
		switch ( $type ) {
			case WpProQuiz_Model_Form::FORM_TYPE_TEXT:
				$type = 'text';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_EMAIL:
				$type = 'email';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_NUMBER:
				$type = 'number';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_TEXTAREA:
				$type = 'textarea';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_CHECKBOX:
				$type = 'checkbox';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_DATE:
				$type = 'date';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_RADIO:
				$type = 'radio';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_SELECT:
				$type = 'select';
				break;
			case WpProQuiz_Model_Form::FORM_TYPE_YES_NO:
				$type = 'yesno';
				break;
		}

		return $type;
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function get_settings( $post ) {
		$quiz_settings = learndash_get_setting( $post->ID );
		$quiz_pro      = $quiz_settings['quiz_pro'];
		$quizMapper    = new WpProQuiz_Model_QuizMapper();
		$quiz          = $quizMapper->fetch( $quiz_pro );

		$setting                 = $quiz->get_object_as_array();
		$setting['_toplistData'] = unserialize( $setting['_toplistData'] );

		return $setting;
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_error_message( $post ) {
		$course_id = bbapp_learndash_get_course_id( $post->ID );
		$user_id   = get_current_user_id();

		if ( ! bbapp_lms_is_content_access( $post, 'prerequities_completed' ) ) {
			$prerequisite_posts_all = learndash_get_course_prerequisites( $course_id );
			$course_str             = '';
			if ( ! empty( $prerequisite_posts_all ) ) {
				foreach ( $prerequisite_posts_all as $pre_post_id => $pre_status ) {
					if ( $pre_status === false ) {
						if ( ! empty( $course_str ) ) {
							$course_str .= ', ';
						}

						$course_str .= get_the_title( $pre_post_id );
					}
				}
			}

			return array(
				'code'    => 'learndash_json_course_prerequities',
				'message' => sprintf(
					__( 'To take this %1$s, you need to complete the following %2$s first: %3$s', 'bbapps' ),
					LearnDash_Custom_Label::label_to_lower( 'lesson' ),
					LearnDash_Custom_Label::label_to_lower( 'course' ),
					$course_str
				),
				'data'    => array( 'status' => 400 ),

			);
		}

		if ( ! bbapp_lms_is_content_access( $post, 'points_access' ) ) {
			$course_access_points = learndash_get_course_points_access( $course_id );
			$user_course_points   = learndash_get_user_course_points( $user_id );

			return array(
				'code'    => 'learndash_json_course_points_access',
				'message' => sprintf(
					__( 'To take this %s you need at least %.01f total points. You currently have %.01f points.', 'buddyboss-app' ),
					LearnDash_Custom_Label::label_to_lower( 'course' ),
					$course_access_points,
					$user_course_points
				),
				'data'    => array( 'status' => 400 ),

			);
		}

		if ( ! sfwd_lms_has_access( $post->ID, $user_id ) ) {
			return array(
				'code'    => 'learndash_json_quiz_invaild',
				'message' => __( "You do not have access to this quiz.", 'buddyboss-app' ),
				'data'    => array( 'status' => 400 ),

			);
		}

		if ( ! $this->check_quiz_lock( $post, 'prerequisite' ) ) {
			return array(
				'code'    => 'learndash_json_quiz_prerequities',
				'message' => __( "You have to pass the previous module's quiz in order to start this.", 'buddyboss-app' ),
				'data'    => array( 'status' => 400 ),

			);
		}
	}

	/**
	 * @param $answer
	 *
	 * @return string
	 */
	public function cleanupCurlyQuotes( $answer ) {
		if ( version_compare( LEARNDASH_VERSION, '2.5', '<' ) ) {
			$answer = strtolower( $answer );
		}
		$answer = trim( $answer );

		return $answer;
	}

}