<?php
namespace BuddyBossApp\Api\Learner\LearnDash;
use BuddyBossApp\Helpers\BBAPP_File;
use LDLMS_Factory_Post;
use LearnDash_Custom_Label;
use LearnDash_Settings_Section;
use WP_Error;
use WP_Query;

// NOTE : Old classname was class.bbapp_learner_learndash_lessons_rest. By Ketan, Oct-2019
// Contain functionality for required additional rest api endpoints for learndash - v1 Standard
class LessonsRest {

	public $post_type = 'sfwd-lessons';

	/**
	 * LessonsRest constructor.
	 */
	public function __construct() {
		$this->hooks();
	}

	/**
	 *
	 */
	public function hooks() {
		add_filter('bbapp_learner_get_lessons_items_permissions', array($this, 'get_items_permissions'), 10, 2);
		add_filter('bbapp_learner_get_lessons_item_permissions', array($this, 'get_item_permissions'), 10, 2);
		add_filter('bbapp_learner_get_lessons_read_permissions', array($this, 'get_read_permissions'), 10, 2);

		//add_filter('bbapp_learner_get_lessons', array($this, "get_lessons"), 10, 3);
		//add_filter('bbapp_learner_get_lesson', array($this, "get_lesson"), 10, 2);

		//add_filter('bbapp_learner_lesson_action', array($this, "lesson_action"), 10, 3);
		//add_filter('bbapp_learner_complete_lesson', array($this, "complete_lesson"), 10, 2);
	}

	/**
	 * @param $flag
	 * @param $request
	 *
	 * @return bool|WP_Error
	 */
	public function get_items_permissions($flag, $request) {

		$post_type = get_post_type_object($this->post_type);

		if ('edit' === $request['context'] && !current_user_can($post_type->cap->edit_posts)) {
			return new WP_Error('rest_forbidden_context', __('Sorry, you are not allowed to edit these posts in this post type', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		return true;
	}

	/**
	 * @param $flag
	 * @param $request
	 *
	 * @return bool|WP_Error
	 */
	public function get_item_permissions($flag, $request) {

		$post = get_post((int) $request['id']);

		if ('edit' === $request['context']) {
			return new WP_Error('rest_forbidden_context', __('Sorry, you are not allowed to edit this post', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if ($post) {
			return $this->get_read_permissions(false, $post);
		}

		return true;
	}

	/**
	 * @param $flag
	 * @param $post
	 *
	 * @return bool
	 */
	public function get_read_permissions($flag, $post) {

		$post_type = get_post_type_object($post->post_type);
		if (empty($post_type)) {
			return false;
		}

		// Can we read the post?
		if ('publish' === $post->post_status || current_user_can($post_type->cap->read_post, $post->ID)) {
			return true;
		}

		$post_status_obj = get_post_status_object($post->post_status);
		if ($post_status_obj && $post_status_obj->public) {
			return true;
		}

		// Can we read the parent if we're inheriting?
		if ('inherit' === $post->post_status && $post->post_parent > 0) {
			$parent = get_post($post->post_parent);
			return $this->get_read_permissions(false, $parent);
		}

		// If we don't have a parent, but the status is set to inherit, assume
		// it's published (as per get_post_status()).
		if ('inherit' === $post->post_status) {
			return true;
		}

		return false;
	}

	/**
	 * @param $result
	 * @param $query_args
	 * @param $request
	 *
	 * @return array
	 */
	public function get_lessons($result, $query_args, $request) {

		global $wpdb;
		$user_id = get_current_user_id();

		$query_args['post_type'] = $this->post_type;

		$parameter_mappings = array(
			'category' => 'ld_lesson_category',
			'tag' => 'ld_lesson_tag',
		);

		/*
			 * For each known parameter which is both registered and present in the request,
			 * set the parameter's value on the query $args.
		*/
		foreach ($parameter_mappings as $api_param => $wp_param) {
			if (isset($request[$api_param])) {
				$request[$wp_param] = $request[$api_param];
			}
		}

		/**
		 * if User pass course id then return Lesson list
		 *  - We want list to list out lesson on course page
		 * else retun lesson if user has access of the lesson course
		 * */
		if (isset($request['course_id'])) {
			$course_id = (int) (!empty($request['course_id']) ? $request['course_id'] : 0);

			//set order accoding to course setting
			$default_args = sfwd_lms_get_post_options($this->post_type);
			$order = bbapp_learndash_get_course_meta_setting($course_id, 'course_lesson_order');
			$orderby = bbapp_learndash_get_course_meta_setting($course_id, 'course_lesson_orderby');
			$query_args['order'] = !empty($order) ? $order : $default_args['order'];
			$query_args['orderby'] = !empty($orderby) ? $orderby : $default_args['orderby'];

			$_course_ids = array($course_id);
		} else {
			$_course_ids = ld_get_mycourses($user_id, array());
		}

		if (!empty($_course_ids)) {
			if (LearnDash_Settings_Section::get_section_setting('LearnDash_Settings_Courses_Builder', 'enabled') == 'yes') {

				//Code for Learndash Course builder
				if (isset($request['course_id'])) {
					//Filter topic if course id pass
					//Different logic to get lesson in order specified in backend
					$lesson_ids = array();
					foreach ($_course_ids as $course_id) {
						$ld_course_steps_object = LDLMS_Factory_Post::course_steps($course_id);
						if (isset($ld_course_steps_object)) {
							$_lesson_ids = $ld_course_steps_object->get_children_steps($course_id, $this->post_type);
							$lesson_ids = array_merge($lesson_ids, $_lesson_ids);
						}
					}
				} else {
					//Filter lesson by course which users has access
					//Insted of loop for all course use custom query to get all id which conntected with course.
					$_course_str = implode(',', $_course_ids);
					$lesson_ids = $wpdb->get_col("SELECT `post_id` FROM $wpdb->postmeta WHERE meta_value IN ($_course_str) AND (`meta_key` LIKE '%ld_course_%' OR  `meta_key`LIKE 'course_id' ) ORDER BY `meta_id` ASC");
				}

				if (!empty($lesson_ids)) {

					if (!empty($query_args['post__in'])) {
						if (!is_array($query_args['post__in'])) {
							$query_args['post__in'] = array($query_args['post__in']);
						}
						$lesson_ids = array_intersect($query_args['post__in'], $lesson_ids);
						if (empty($lesson_ids)) {
							$lesson_ids = array(-1);
						}
					}

					$query_args['post__in'] = $lesson_ids;
					$query_args['orderby'] = (isset($query_args['s'])) ? 'relevance' : 'post__in';
					unset($query_args['order']);
				} else {
					return array('posts' => array(), 'total_posts' => 0);
				}
			} else {
				//Code for Learndash old default structure
				//Filter topic by course which users has access
				$query_args['meta_query'][] = array(
					'key' => 'course_id',
					'value' => $_course_ids,
					'compare' => 'IN',
				);
			}
		} else {
			return array('posts' => array(), 'total_posts' => 0);
		}

		$taxonomies = wp_list_filter(get_object_taxonomies($this->post_type, 'objects'), array('show_in_rest' => true));
		foreach ($taxonomies as $taxonomy) {
			$base = !empty($taxonomy->rest_base) ? $taxonomy->rest_base : $taxonomy->name;

			if (!empty($request[$base])) {
				$query_args['tax_query'][] = array(
					'taxonomy' => $taxonomy->name,
					'field' => 'term_id',
					'terms' => $request[$base],
					'include_children' => false,
				);
			}
		}

		$posts_query = new WP_Query();
		$posts = $posts_query->query($query_args);

		/**
		 * Allow Full object when request for all lesson for single course
		 */
		//$full_obj = isset($request['course_id']) ? true : false;
		foreach ($posts as $post) {
			$post = $this->get_additional_data($post, true);
		}

		$total_posts = $posts_query->found_posts;

		if ($total_posts < 1) {
			// Out-of-bounds, run the query again without LIMIT for total count
			unset($query_args['paged']);
			$count_query = new WP_Query();
			$count_query->query($query_args);
			$total_posts = $count_query->found_posts;
		}

		return array('posts' => $posts, 'total_posts' => $total_posts);
	}

	/**
	 * @param $result
	 * @param $id
	 *
	 * @return mixed|void
	 */
	public function get_lesson($result, $id) {
		$post = get_post($id);

		if ($this->post_type !== $post->post_type) {
			return;
		}

		$post = $this->get_additional_data($post, true);
		return $post;
	}

	/**
	 * @param $result
	 * @param $action
	 * @param $lesson_id
	 *
	 * @return mixed
	 */
	public function lesson_action($result, $action, $lesson_id) {
		return $result;
	}

	/**
	 * @param $result
	 * @param $lesson_id
	 *
	 * @return bool|void|WP_Error
	 */
	public function complete_lesson($result, $lesson_id) {

		$user_id = get_current_user_id();

		if (!sfwd_lms_has_access($lesson_id, $user_id)) {
			return new WP_Error('learndash_json_lesson_invaild', __("You do not have access of this topic.", 'buddyboss-app'), array('status' => 400));
		}

		$lesson = get_post($lesson_id);

		if (!learndash_is_lesson_complete($user_id, $lesson_id)) {

			if (!bbapp_lms_is_content_access($lesson, 'points_access')) {
				return new WP_Error('learndash_json_course_points_access', __('Course required points does not match', 'buddyboss-app'), array('status' => 400));
			}

			if (!bbapp_lms_is_content_access($lesson, 'prerequities_completed')) {
				return new WP_Error('learndash_json_course_prerequities', __('Course prerequisite is not completed.', 'buddyboss-app'), array('status' => 400));
			}

			if (!bbapp_lms_is_content_access($lesson, 'previous_lesson_completed')) {
				return new WP_Error('learndash_json_previous_not_completed', __('You must complete each lesson in sequence', 'buddyboss-app'), array('status' => 400));
			}

			if (!learndash_lesson_topics_completed($lesson_id)) {
				if (!apply_filters('learndash_previous_step_completed', false, $lesson_id, $user_id)) {
					return new WP_Error('learndash_json_not_completed', __('Lesson\'s topic is not completed', 'buddyboss-app'), array('status' => 400));
				}
			}

			if (!bbapp_lms_quiz_completed($lesson)) {
				return new WP_Error('learndash_json_not_completed', __('Lesson\'s quiz not completed', 'buddyboss-app'), array('status' => 400));
			}

			if ($this->get_lesson_assignment_upload($lesson)) {

				if (!lesson_hasassignments($lesson_id) && !empty($_FILES['attachment'])) {

					if (!function_exists('wp_generate_attachment_metadata')) {
						require_once ABSPATH . "wp-admin" . '/includes/image.php';
						require_once ABSPATH . "wp-admin" . '/includes/file.php';
						require_once ABSPATH . "wp-admin" . '/includes/media.php';
					}

					if (!function_exists('media_handle_upload')) {
						require_once ABSPATH . 'wp-admin/includes/admin.php';
					}

					$_FILES['attachment']['name'] = learndash_clean_filename($_FILES['attachment']['name']);

					if (!function_exists('wp_get_current_user')) {
						include ABSPATH . 'wp-includes/pluggable.php';
					}

					$filetype = wp_check_filetype(basename($_FILES['attachment']['name']), null);
					$filetitle = preg_replace('/\.[^.]+$/', '', basename($_FILES['attachment']['name']));
					$filename = $filetitle . '.' . $filetype['ext'];

					$upload_dir = wp_upload_dir();
					$upload_dir_base = $upload_dir['basedir'];
					$upload_dir_path = $upload_dir_base . '/assignments';

					if (!file_exists($upload_dir_path)) {
						BBAPP_File::create_dir($upload_dir_path);
					}

					/**
					 * Check if the filename already exist in the directory and rename the
					 * file if necessary
					 */
					$i = 0;

					while (file_exists($upload_dir_path . '/' . $filename)) {
						$i++;
						$filename = $filetitle . '_' . $i . '.' . $filetype['ext'];
					}

					$_FILES['attachment']['name'] = $filename;
					$filedest = $upload_dir_path . '/' . $filename;

					/**
					 * Check write permissions
					 */
					if (!is_writeable($upload_dir_path)) {
						die(__('Unable to write to directory. Is this directory writable by the server?', 'buddyboss-app'));
						return;
					}

					add_filter('upload_dir', 'bbapp_lms_media_upload_dir');
					$aid = media_handle_upload('attachment', $lesson_id);
					remove_filter('upload_dir', 'bbapp_lms_media_upload_dir');

					// if has wp error then throw it.
					if (is_wp_error($aid)) {
						return $aid;
					}

					$attachment = get_post($aid);

					if (!empty($attachment)) {
						return bbapp_lms_upload_assignment_init($lesson_id, $filename, $filedest);
					}
				} else if (!lesson_hasassignments($lesson)) {
					return new WP_Error('learndash_json_not_completed', __('Please upload assignment for this Lesson', 'buddyboss-app'), array('status' => 400));
				}
			}

			$course_id = bbapp_learndash_get_course_id($lesson_id);
			if (!empty($course_id)) {
				ld_update_course_access($user_id, $course_id);
			}

            //To stop "Uncanny LearnDash Toolkit" Redirection after complete event for rest endpoint
            $_REQUEST['doing_rest'] = 1;

			return learndash_process_mark_complete($user_id, $lesson_id);
		} else {
			return true;
		}
	}

	/**
	 * @param $post
	 * @param $full_obj
	 *
	 * @return mixed
	 */
	public function get_additional_data($post, $full_obj) {

        $post->materials = $this->get_materials($post);
		$post->course = $this->get_course_id($post);
		$post->next_lesson = $this->get_next_lesson_id($post);
		$post->duration = $this->get_duration($post);
		$post->assignment_upload = $this->get_lesson_assignment_upload($post);
		$post->video = $this->get_video($post);
		$post->has_course_access = $this->get_has_course_access($post);
		$post->has_content_access = $this->get_has_content_access($post);
		$post->is_sample = $this->get_is_sample($post);
//		$post->categories = $this->get_category( $post );
		//		$post->tags = $this->get_tag( $post );
		$post->completed = $this->is_completed($post);
		$post->link = get_permalink($post->ID);
		$post->featured_media = $this->get_feature_media($post);
		$post->error_message = $this->get_error_message($post);
		$post->quizzes = array();
		$post->topics = array();
		$post->quiz_completed = false;

		/**
		 * Below data will only added if single lesson request as it's taking too much time for load.
		 */
		if ($full_obj) {
			$post->quizzes = $this->get_quizzes($post);
			$post->topics = $this->get_topics($post);
			$post->quiz_completed = $this->is_quiz_completed($post);
		}
		return $post;
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_feature_media($post) {
		$return = array(
			"large" => null,
			"small" => null,
		);
		$large = wp_get_attachment_image_src(get_post_thumbnail_id($post->ID), 'large');
		$small = wp_get_attachment_image_src(get_post_thumbnail_id($post->ID), 'medium');

		if (isset($large[0])) {
			$return["large"] = $large[0];
		}
		if (isset($small[0])) {
			$return["small"] = $small[0];
		}

		return $return;

	}

	/**
	 * @param $post
	 *
	 * @return int
	 */
	public function get_course_id($post) {
		$course_id = bbapp_learndash_get_course_id($post->ID);
		if (!isset($course_id)) {
			$course_id = 0;
		}
		return $course_id;
	}

	/**
	 * @param $post
	 *
	 * @return int|string
	 */
	public function get_next_lesson_id($post) {
		$next = bbapp_lms_next_post_id($post);
		if (!empty($next)) {
			return $next;
		} else {
			return 0;
		}
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_duration($post) {

		$timeval = learndash_get_setting($post->ID, 'forced_lesson_time');
		if (!empty($timeval)) {
			$time_sections = explode(' ', $timeval);
			$h = $m = $s = 0;

			foreach ($time_sections as $k => $v) {
				$value = trim($v);

				if (strpos($value, 'h')) {
					$h = intVal($value);
				} else if (strpos($value, 'm')) {
					$m = intVal($value);
				} else if (strpos($value, 's')) {
					$s = intVal($value);
				}
			}

			$time = $h * 60 * 60 + $m * 60 + $s;

			if ($time == 0) {
				$time = (int) $timeval;
			}
		} else {
			$time = 0;
		}

		return array(
			"min" => $time,
		);
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function get_lesson_assignment_upload($post) {
		return 'on' == learndash_get_setting($post->ID, 'lesson_assignment_upload');
	}

	/**
	 * @param $post
	 *
	 * @return string
	 */
    public function get_materials($post) {
        $lesson_settings = learndash_get_setting($post->ID);
        if ((isset($lesson_settings['lesson_materials_enabled'])) && ($lesson_settings['lesson_materials_enabled'] == 'on')) {

            $lesson_materials = learndash_get_setting($post->ID, 'lesson_materials');
            $lesson_materials = wp_kses_post(wp_specialchars_decode($lesson_materials, ENT_QUOTES));
            if (!empty($course_materials)) {
                $lesson_materials = do_shortcode($course_materials);
            };
            return $lesson_materials;
        }
        return '';
    }

	/**
	 * @param $post
	 *
	 * @return string
	 */
	public function get_video($post) {
		$lesson_settings = learndash_get_setting($post->ID);
		if ((isset($lesson_settings['lesson_video_enabled'])) && ($lesson_settings['lesson_video_enabled'] == 'on')) {
			return learndash_get_setting($post->ID, 'lesson_video_url');
		}
		return '';
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function get_has_course_access($post) {
		$course_id = bbapp_learndash_get_course_id($post->ID);
		return sfwd_lms_has_access($course_id);
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function get_has_content_access($post) {
		return bbapp_lms_lesson_access_from($post)
		&& bbapp_lms_is_content_access($post, 'prerequities_completed')
		&& bbapp_lms_is_content_access($post, 'points_access')
		&& bbapp_lms_is_content_access($post, 'previous_lesson_completed')
		&& sfwd_lms_has_access($post->ID);
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function get_is_sample($post) {
		return learndash_is_sample($post);
	}
//
	//	public function get_category( $post ) {
	//		$terms = get_the_terms( $post, 'ld_lesson_category' );
	//		return $terms ? wp_list_pluck( $terms, 'term_id' ) : array();
	//	}
	//
	//	public function get_tag( $post ) {
	//		$terms = get_the_terms( $post, 'ld_lesson_tag' );
	//		return $terms ? wp_list_pluck( $terms, 'term_id' ) : array();
	//	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_quizzes($post) {
		$quizes = learndash_get_lesson_quiz_list($post);
		$posts = array();
		foreach ($quizes as $quiz) {
			$posts[] = array(
				'id' => $quiz['post']->ID,
				'completed' => bbapp_learner_learndash()->c->bbapp_learner_learndash_quiz_rest->is_completed($quiz['post']),
			);
		}

		return $posts;
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_topics($post) {
		$course_id = bbapp_learndash_get_course_id($post);
		$topics = learndash_get_topic_list($post->ID, $course_id);
		$posts = array();
		if (!empty($topics)) {
			foreach ($topics as $topic) {
				$posts[] = array(
					'id' => $topic->ID,
					'completed' => bbapp_learner_learndash()->c->bbapp_learner_learndash_topics_rest->is_completed($topic),
				);
			}
		}
		return $posts;
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function is_completed($post) {
		$course_id = bbapp_learndash_get_course_id($post);
		return learndash_is_lesson_complete(null, $post->ID, $course_id);
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function is_quiz_completed($post) {
		return bbapp_lms_quiz_completed($post);
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_error_message($post) {
		$course_id = bbapp_learndash_get_course_id($post->ID);
		$user_id = get_current_user_id();

		if (!bbapp_lms_is_content_access($post, 'prerequities_completed')) {
			$prerequisite_posts_all = learndash_get_course_prerequisites($course_id);
			$course_str = '';
			if (!empty($prerequisite_posts_all)) {
				foreach ($prerequisite_posts_all as $pre_post_id => $pre_status) {
					if ($pre_status === false) {
						if (!empty($course_str)) {
							$course_str .= ', ';
						}

						$course_str .= get_the_title($pre_post_id);
					}
				}
			}
			return array(
				'code' => 'learndash_json_course_prerequities',
				'message' => sprintf(
					__('To take this %1$s, you need to complete the following %2$s first: %3$s', 'bbapps'),
					LearnDash_Custom_Label::label_to_lower('lesson'),
					LearnDash_Custom_Label::label_to_lower('course'),
					$course_str
				),
				'data' => array('status' => 400),

			);
		}

		if (!bbapp_lms_is_content_access($post, 'points_access')) {
			$course_access_points = learndash_get_course_points_access($course_id);
			$user_course_points = learndash_get_user_course_points($user_id);
			return array(
				'code' => 'learndash_json_course_points_access',
				'message' => sprintf(
					__('To take this %s you need at least %.01f total points. You currently have %.01f points.', 'buddyboss-app'),
					LearnDash_Custom_Label::label_to_lower('course'),
					$course_access_points,
					$user_course_points
				),
				'data' => array('status' => 400),

			);
		}

		$lesson_access_from = bbapp_lms_lesson_access_from($post, false);
		if (!empty($lesson_access_from) && true !== $lesson_access_from) {
			return array(
				'code' => 'learndash_json_lesson_not_available',
				'message' => __(sprintf("Available on: %s", learndash_adjust_date_time_display($lesson_access_from)), 'bbapp_api'),
				'data' => array('status' => 400),

			);
		}

		if (!sfwd_lms_has_access($post->ID, $user_id)) {
			return array(
				'code' => 'learndash_json_lesson_invaild',
				'message' => __("You do not have access to this lesson.", 'buddyboss-app'),
				'data' => array('status' => 400),

			);
		}

		if (!bbapp_lms_is_content_access($post, 'previous_lesson_completed')) {
			return array(
				'code' => 'learndash_json_previous_not_completed',
				'message' => __("You must complete each lessons in sequence.", 'buddyboss-app'),
				'data' => array('status' => 400),

			);
		}
	}
}