<?php
namespace BuddyBossApp\Api\Learner\LearnDash;
use LearnDash_Custom_Label;
use LearnDash_Settings_Section;
use WP_Error;
use WP_Query;
use WP_REST_Users_Controller;

// NOTE : Old classname was class.bbapp_learner_learndash_courses_rest. By Ketan, Oct-2019
// Contain functionality for required additional rest api endpoints for learndash - v1 Standard
class CoursesRest {

	public $post_type = 'sfwd-courses';

	/**
	 * CoursesRest constructor.
	 */
	public function __construct() {
		$this->hooks();
	}

	/**
	 *
	 */
	public function hooks() {
		add_filter('bbapp_learner_get_courses_items_permissions', array($this, 'get_items_permissions'), 10, 2);
		add_filter('bbapp_learner_get_courses_item_permissions', array($this, 'get_item_permissions'), 10, 2);
		add_filter('bbapp_learner_get_courses_read_permissions', array($this, 'get_read_permissions'), 10, 2);

		//add_filter('bbapp_learner_get_courses', array($this, 'get_courses'), 10, 3);

		//add_filter('bbapp_learner_get_courses_by_category', array($this, 'get_courses_by_category'), 10, 3);
		//add_filter('bbapp_learner_get_course_teachers', array($this, 'get_course_teachers'), 10, 3);

		//add_filter('bbapp_learner_get_course', array($this, 'get_course'), 10, 2);

		//add_filter('bbapp_learner_course_action', array($this, 'course_action'), 10, 3);
		//add_filter('bbapp_learner_enroll_course', array($this, 'enroll_course'), 10, 2);

		//add_filter('bbapp_learner_get_course_members', array($this, 'get_course_members'), 10, 4);

	}

	/**
	 * @param $flag
	 * @param $request
	 *
	 * @return bool|WP_Error
	 */
	public function get_items_permissions($flag, $request) {

		$post_type = get_post_type_object($this->post_type);

		if ('edit' === $request['context'] && !current_user_can($post_type->cap->edit_posts)) {
			return new WP_Error('rest_forbidden_context', __('Sorry, you are not allowed to edit these posts in this post type'), array('status' => rest_authorization_required_code()));
		}

		return true;
	}

	/**
	 * @param $flag
	 * @param $request
	 *
	 * @return bool|WP_Error
	 */
	public function get_item_permissions($flag, $request) {

		$post = get_post((int) $request['id']);

		if ('edit' === $request['context']) {
			return new WP_Error('rest_forbidden_context', __('Sorry, you are not allowed to edit this post'), array('status' => rest_authorization_required_code()));
		}

		if ($post) {
			return $this->get_read_permissions(false, $post);
		}

		return true;
	}

	/**
	 * @param $flag
	 * @param $post
	 *
	 * @return bool
	 */
	public function get_read_permissions($flag, $post) {

		$post_type = get_post_type_object($post->post_type);
		if (empty($post_type)) {
			return false;
		}

		// Can we read the post?
		if ('publish' === $post->post_status || current_user_can($post_type->cap->read_post, $post->ID)) {
			return true;
		}

		$post_status_obj = get_post_status_object($post->post_status);
		if ($post_status_obj && $post_status_obj->public) {
			return true;
		}

		// Can we read the parent if we're inheriting?
		if ('inherit' === $post->post_status && $post->post_parent > 0) {
			$parent = get_post($post->post_parent);
			return $this->get_read_permissions(false, $parent);
		}

		// If we don't have a parent, but the status is set to inherit, assume
		// it's published (as per get_post_status()).
		if ('inherit' === $post->post_status) {
			return true;
		}

		return false;
	}

	/**
	 * @param $result
	 * @param $query_args
	 * @param $request
	 *
	 * @return array
	 */
	public function get_courses($result, $query_args, $request) {

		$user_id = get_current_user_id();

		$query_args['post_type'] = $this->post_type;

		/**
		 * Tags & Category
		 */

		if (!empty($request["categories"])) {

			if (in_array(0, $request["categories"])) {

				$query_args['tax_query'][] = array(
					'taxonomy' => 'ld_course_category',
					'operator' => 'NOT EXISTS',
				);
				$categories = array();
				foreach ($request["categories"] as $k => $v) {
					if (!empty($v)) {
						$categories[] = $v;
					}

				}
				$request["categories"] = $categories;
				unset($categories);
			}

			if (!empty($request["categories"])) {
				$query_args['tax_query'][] = array(
					'taxonomy' => 'ld_course_category',
					'field' => 'term_id',
					'terms' => $request["categories"],
					'include_children' => false,
				);
			}

		}

		if (!empty($request["categories_exclude"])) {

			$query_args['tax_query'][] = array(
				'taxonomy' => 'ld_course_category',
				'field' => 'term_id',
				'terms' => $request["categories_exclude"],
				'include_children' => false,
				'operator' => 'NOT IN',
			);

		}

		if (!empty($request["tags"])) {
			$query_args['tax_query'][] = array(
				'taxonomy' => 'ld_course_tag',
				'field' => 'term_id',
				'terms' => $request["tags"],
				'include_children' => false,
			);
		}

		if (!empty($request["tags_exclude"])) {

			$query_args['tax_query'][] = array(
				'taxonomy' => 'ld_course_tag',
				'field' => 'term_id',
				'terms' => $request["tags_exclude"],
				'include_children' => false,
				'operator' => 'NOT IN',
			);

		}

		if (isset($query_args['recommended']) && !empty($query_args['recommended'])) {
			$query_args['meta_query'][] = array(
				'key' => '_is_recommended',
				'value' => 'yes',
				'compare' => '=',
			);
			unset($query_args['recommended']);
		}

		if (isset($query_args['mycourses']) && $query_args['mycourses']) {

			$mycourse_ids = ld_get_mycourses($user_id, array());

			if (!empty($mycourse_ids) && !is_wp_error($mycourse_ids)) {
				$query_args['post__in'] = !empty($query_args['post__in']) ? array_intersect($mycourse_ids, $query_args['post__in']) : $mycourse_ids;
				/*
					 * If we intersected, but there are no post ids in common,
					 * WP_Query won't return "no posts" for post__in = array()
					 * so we have to fake it a bit.
				*/
				if (!$query_args['post__in']) {
					$query_args['post__in'] = array(0);
				}
			} else {
				$query_args['post__in'] = array(0);
			}
		}

		$posts_query = new WP_Query();
		$posts = $posts_query->query($query_args);

		foreach ($posts as $post) {
			$post = $this->get_additional_data($post, true);
		}

		$total_posts = $posts_query->found_posts;

		if ($total_posts < 1) {
			// Out-of-bounds, run the query again without LIMIT for total count
			unset($query_args['paged']);
			$count_query = new WP_Query();
			$count_query->query($query_args);
			$total_posts = $count_query->found_posts;
		}

		return array('posts' => $posts, 'total_posts' => $total_posts);
	}

	/**
	 * @param $result
	 * @param $query_args
	 * @param $request
	 *
	 * @return array
	 */
	public function get_course_teachers($result, $query_args, $request) {
		global $wpdb;

		$limit = isset($query_args['posts_per_page']) ? (int) $query_args['posts_per_page'] : 5;

		$query = "SELECT u.post_author as post_author, COUNT(*) as count
							  FROM " . $wpdb->prefix . "posts u
							  WHERE (post_type = '{$this->post_type}' and post_status = 'publish' )
							  AND u.post_author IN (SELECT ID FROM  {$wpdb->prefix}users WHERE ID=u.post_author)
							  GROUP BY post_author
							  ORDER BY count DESC
							  LIMIT $limit";
		$result = $wpdb->get_results($query);

		$user_ids = array();
		if (!empty($result)) {
			foreach ($result as $r) {
				$user_ids[] = $r->post_author;
			}
		}

		return $user_ids;
	}

	/**
	 * @param $result
	 * @param $term
	 * @param $prepared_args
	 *
	 * @return mixed
	 */
	public function get_courses_by_category($result, $term, $prepared_args) {

		$user_id = get_current_user_id();

		$query_args = array(
			'post_type' => $this->post_type,
			'fields' => 'ids',
			'post__not_in' => $prepared_args["courses_exclude"],
			'order' => $prepared_args["course_order"],
			'orderby' => $prepared_args["course_orderby"],
		);

		if (isset($prepared_args['mycourses']) && $prepared_args['mycourses']) {
			$mycourse_ids = ld_get_mycourses($user_id, array());

			if (!empty($mycourse_ids) && !is_wp_error($mycourse_ids)) {
				$query_args['post__in'] = !empty($query_args['post__in']) ? array_intersect($mycourse_ids, $query_args['post__in']) : $mycourse_ids;
				/*
					 * If we intersected, but there are no post ids in common,
					 * WP_Query won't return "no posts" for post__in = array()
					 * so we have to fake it a bit.
				*/
				if (!$query_args['post__in']) {
					$query_args['post__in'] = array(0);
				}
			} else {
				$query_args['post__in'] = array(0);
			}
		}

		if (isset($prepared_args['courses_limit'])) {
			$query_args['posts_per_page'] = $prepared_args['courses_limit'];
		} else {
			$query_args['posts_per_page'] = 5;
		}

		if (isset($term->term_id) && empty($term->term_id)) {
			$query_args['tax_query'][] = array(
				'taxonomy' => 'ld_course_category',
				'operator' => 'NOT EXISTS',
			);
		} else {
			$query_args['tax_query'][] = array(
				'taxonomy' => 'ld_course_category',
				'field' => 'term_id',
				'terms' => $term->term_id,
				'include_children' => false,
			);
		}

		$posts_query = new WP_Query();
		$posts = $posts_query->query($query_args);

		return $posts;
	}

	/**
	 * @param $result
	 * @param $id
	 *
	 * @return mixed|void
	 */
	public function get_course($result, $id) {

		$post = get_post($id);

		if ($this->post_type !== $post->post_type) {
			return;
		}

		$post = $this->get_additional_data($post, true);

		return $post;

	}

	/**
	 * @param $result
	 * @param $action
	 * @param $course_id
	 *
	 * @return mixed
	 */
	public function course_action($result, $action, $course_id) {
		return $result;
	}

	/**
	 * @param $result
	 * @param $course_id
	 *
	 * @return bool|WP_Error
	 */
	public function enroll_course($result, $course_id) {

		$user_id = get_current_user_id();

		$course = get_post($course_id);
		if (!bbapp_lms_is_content_access($course, 'points_access')) {
			return new WP_Error('learndash_json_course_points_access',
				__('Course required points does not match', 'buddyboss-app'), array('status' => 400));
		}

		if (!bbapp_lms_is_content_access($course, 'prerequities_completed')) {
			return new WP_Error('learndash_json_course_prerequities',
				__('Course prerequisite is not completed.', 'buddyboss-app'), array('status' => 400));
		}

		$meta = get_post_meta($course_id, '_sfwd-courses', true);
		$price = (float) $meta['sfwd-courses_course_price'];
		if (
			(@$meta['sfwd-courses_course_price_type'] == 'free' || @$meta['sfwd-courses_course_price_type'] == 'open' // if price type is free or open
				 || (@$meta['sfwd-courses_course_price_type'] == 'paynow' && empty($price))) // or if price type paynow but price is 0
		) {

			if (!sfwd_lms_has_access($course_id, $user_id)) {
				ld_update_course_access($user_id, $course_id);
			}

			return true;
		} elseif (sfwd_lms_has_access($course_id, $user_id)) {
			return true; // if course already purchase
		}
	}

	/**
	 * @param $result
	 * @param $query_args
	 * @param $course_id
	 * @param $request
	 *
	 * @return array
	 */
	public function get_course_members($result, $query_args, $course_id, $request) {
		$post = get_post($course_id);

		if ($this->post_type !== $post->post_type) {
			return array();
		}

		$access_list = $this->get_access_list($post);

		$result = array();
		if (!empty($access_list)) {
			$result = array();
			$user_api = new WP_REST_Users_Controller();
			foreach ($access_list as $user_id) {
				$user = get_userdata((int) $user_id);
				if (empty($user) || !$user->exists()) {
					continue;
				}
				if (is_multisite() && !is_user_member_of_blog($user->ID)) {
					continue;
				}
				$result[] = $user_api->prepare_response_for_collection(
					$user_api->prepare_item_for_response($user, $request, false)
				);
				if (count($result) >= $query_args['number']) {
					break;
				}
			}
		}

		return $result;
	}

	/**
	 * @param $post
	 * @param $full_obj
	 *
	 * @return mixed
	 */
	public function get_additional_data($post, $full_obj) {

		$_GET['course_id'] = $post->ID;

		$post->materials = $this->get_materials($post);
		$post->purchasable = $this->is_purchasable($post);
		$post->is_closed = $this->is_closed($post);
		$post->price = $this->get_price($post);
		$post->has_course_access = $this->get_has_course_access($post);
		$post->has_content_access = $this->get_has_content_access($post);
		$post->hide_content_table = $this->get_hide_content_table($post);
		$post->progression = $this->get_progression($post);
		$post->points = $this->get_points($post);
		$post->access_list = $this->get_access_list($post);
		$post->categories = $this->get_categories($post);
		$post->tags = $this->get_tags($post);
		$post->completed = $this->is_completed($post);
		$post->featured_media = $this->get_feature_media($post);
		$post->link = get_permalink($post->ID);
		$post->error_message = $this->get_error_message($post);
		$post->quizzes = array();
		$post->lessons = array();
		$post->quiz_completed = false;

		/**
		 * Below data will only added if single course request as it's taking too much time for load.
		 */
		if ($full_obj) {
			$post->quizzes = $this->get_quizzes($post);
			$post->lessons = $this->get_lessons($post);
			$post->quiz_completed = $this->is_quiz_completed($post);
		}

		return $post;
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_feature_media($post) {
		$return = array(
			"large" => '',
			"small" => '',
		);
		$large = wp_get_attachment_image_src(get_post_thumbnail_id($post->ID), 'large');
		$small = wp_get_attachment_image_src(get_post_thumbnail_id($post->ID), 'medium');

		if (isset($large[0])) {
			$return["large"] = $large[0];
		}
		if (isset($small[0])) {
			$return["small"] = $small[0];
		}

		return $return;

	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function get_materials($post) {
		$course_materials = bbapp_learndash_get_course_meta_setting($post->ID, 'course_materials');
		$course_materials = wp_kses_post(wp_specialchars_decode($course_materials, ENT_QUOTES));
		if (!empty($course_materials)) {
			$course_materials = do_shortcode($course_materials);
		};
		return $course_materials;
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function is_purchasable($post) {
		$type = bbapp_learndash_get_course_meta_setting($post->ID, 'course_price_type');
		$price = (float) bbapp_learndash_get_course_meta_setting($post->ID, 'course_price');
		switch ($type) {
		case 'subscribe':
		case 'paynow':
		case 'closed':
			$is_paid = empty($price) ? false : true;
			break;
		default:
			$is_paid = false;
		}
		return $is_paid;
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function is_closed($post) {
		$type = bbapp_learndash_get_course_meta_setting($post->ID, 'course_price_type');
		$is_closed = false;
		if ('closed' == $type) {
			$is_closed = true;
		}
		return $is_closed;
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_price($post) {
		$paypal_settings = LearnDash_Settings_Section::get_section_settings_all('LearnDash_Settings_Section_PayPal');
		$price = (float) bbapp_learndash_get_course_meta_setting($post->ID, 'course_price');
		return array('value' => $price, 'code' => $paypal_settings['paypal_currency']);
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function get_has_course_access($post) {
		return sfwd_lms_has_access($post->ID);
	}

	/**
	 * @param      $post
	 * @param bool $message
	 *
	 * @return bool
	 */
	public function get_has_content_access($post, $message = false) {
		return bbapp_lms_is_content_access($post, 'prerequities_completed')
		&& bbapp_lms_is_content_access($post, 'points_access');
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function get_hide_content_table($post) {
	    if ( sfwd_lms_has_access($post->ID) ){
            return false;
        }
		return 'on' === bbapp_learndash_get_course_meta_setting($post->ID, 'course_disable_content_table');
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function get_progression($post) {
		$user_id = get_current_user_id();
		$progress = learndash_course_progress(array("user_id" => $user_id, "course_id" => $post->ID, "array" => true));
		return $progress['percentage'];
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function get_points($post) {
		return bbapp_learndash_get_course_meta_setting($post->ID, 'course_points');
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_access_list($post) {
		$access_list = bbapp_learndash_get_course_meta_setting($post->ID, 'course_access_list');

		if (!is_array($access_list)) {
			$access_list = array();
		}

		return $access_list;
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_categories($post) {
		if (LearnDash_Settings_Section::get_section_setting('LearnDash_Settings_Courses_Taxonomies', 'ld_course_category') == 'yes') {
			$terms = get_the_terms($post, 'ld_course_category');
			return $terms && !is_wp_error($terms) ? wp_list_pluck($terms, 'term_id') : array();
		}
		return array();
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_tags($post) {
		if (LearnDash_Settings_Section::get_section_setting('LearnDash_Settings_Courses_Taxonomies', 'ld_course_tag') == 'yes') {
			$terms = get_the_terms($post, 'ld_course_tag');
			return $terms && !is_wp_error($terms) ? wp_list_pluck($terms, 'term_id') : array();
		}
		return array();
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_quizzes($post) {
		$quizes = learndash_get_course_quiz_list($post);
		$posts = array();
		foreach ($quizes as $quiz) {
			$posts[] = array(
				'id' => $quiz['post']->ID,
				'completed' => bbapp_learner_learndash()->c->bbapp_learner_learndash_quiz_rest->is_completed($quiz['post']),
			);
		}

		return $posts;
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_lessons($post) {
		$posts = array();
		$lessons = learndash_get_lesson_list($post->ID, array('num' => -1));
		foreach ($lessons as $lesson) {
			$posts[] = array(
				'id' => $lesson->ID,
				'completed' => bbapp_learner_learndash()->c->bbapp_learner_learndash_lessons_rest->is_completed($lesson),
			);
		}
		return $posts;
	}

	/**
	 * @param $post
	 *
	 * @return mixed
	 */
	public function is_completed($post) {
		return learndash_course_completed(null, $post->ID);
	}

	/**
	 * @param $post
	 *
	 * @return bool
	 */
	public function is_quiz_completed($post) {
		return bbapp_lms_quiz_completed($post);
	}

	/**
	 * @param $post
	 *
	 * @return array
	 */
	public function get_error_message($post) {
		$user_id = get_current_user_id();

		if (!bbapp_lms_is_content_access($post, 'prerequities_completed')) {
			$prerequisite_posts_all = learndash_get_course_prerequisites($post->ID);
			$course_str = '';
			if (!empty($prerequisite_posts_all)) {
				foreach ($prerequisite_posts_all as $pre_post_id => $pre_status) {
					if ($pre_status === false) {
						if (!empty($course_str)) {
							$course_str .= ', ';
						}

						$course_str .= get_the_title($pre_post_id);
					}
				}
			}
			return array(
				'code' => 'learndash_json_course_prerequities',
				'message' => sprintf(
					__('To take this %1$s, you need to complete the following %2$s first: %3$s', 'buddyboss-app'),
					LearnDash_Custom_Label::label_to_lower('course'),
					LearnDash_Custom_Label::label_to_lower('course'),
					$course_str
				),
				'data' => array('status' => 400),

			);
		}

		if (!bbapp_lms_is_content_access($post, 'points_access')) {
			$course_access_points = learndash_get_course_points_access($post->ID);
			$user_course_points = learndash_get_user_course_points($user_id);
			return array(
				'code' => 'learndash_json_course_points_access',
				'message' => sprintf(
					__('To take this %s you need at least %.01f total points. You currently have %.01f points.', 'buddyboss-app'),
					LearnDash_Custom_Label::label_to_lower('course'),
					$course_access_points,
					$user_course_points
				),
				'data' => array('status' => 400),

			);
		}

		if (!sfwd_lms_has_access($post->ID, $user_id)) {
			return array(
				'code' => 'learndash_json_course_invaild',
				'message' => __("You do not have access to this course.", 'buddyboss-app'),
				'data' => array('status' => 400),

			);
		}
	}
}