<?php

namespace BuddyBossApp\Api\Learner;

use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

// NOTE : Old classname was class.bbapp_learner_courses_rest. By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for learndash.
class CoursesRest extends Rest {

	protected static $instance;

	private $post_type = 'sfwd-courses';
	protected $course_helper;

	/**
	 * CoursesRest constructor.
	 */
	public function __construct() {
		$this->rest_base     = 'courses';
		parent::__construct();
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * @return void|WP_Error
	 */
	public function register_routes() {

		$this->course_helper = bbapp_learner_learndash()->c->bbapp_learner_learndash_courses_rest;

		register_rest_route( $this->namespace, '/' . $this->rest_base, array(
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_items' ),
				'permission_callback' => array( $this, 'get_items_permissions_check' ),
				'args'                => $this->get_collection_params(),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)', array(
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_item' ),
				'permission_callback' => array( $this, 'get_item_permissions_check' ),
				'args'                => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/action/(?P<id>\d+)', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'callback' => array( $this, 'do_action' ),
				'permission_callback' => '__return_true',
				'args'     => array(
					'action' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
				),
			),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/enroll/(?P<id>\d+)', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'permission_callback' => '__return_true',
				'callback' => array( $this, 'enroll' ),
			),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/members/(?P<id>\d+)', array(
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_course_members' ),
				'permission_callback' => array( $this, 'get_items_permissions_check' ),
				'args'                => $this->get_collection_params(),
			),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/download', array(
			array(
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => array( $this, 'download_item' ),
				'permission_callback' => array( $this, 'get_item_permissions_check' ),
				'args'                => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
		) );

		register_rest_route( $this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/upload', array(
			array(
				'methods'  => WP_REST_Server::EDITABLE,
				'callback' => array( $this, 'upload_progress' ),
				'permission_callback' => '__return_true',
				'args'     => array(),
			),
		) );
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {

		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'course',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( "The date the object was published, in the site's timezone." ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit' ),
				),
				'modified'       => array(
					'description' => __( "The date the object was last modified, in the site's timezone." ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'           => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The id for the author of the object.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full URL of image.' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		$schema['properties']['categories'] = array(
			'description' => sprintf( __( 'The terms assigned to the object in the %s taxonomy.' ), 'categories' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['tags'] = array(
			'description' => sprintf( __( 'The terms assigned to the object in the %s taxonomy.' ), 'tags' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['module'] = array(
			'description' => sprintf( __( 'The terms assigned to the object in the %s taxonomy.' ), 'module' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['featured'] = array(
			'description' => __( 'Whether or not the object should be treated as featured.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['materials'] = array(
			'description' => __( 'Materials for the object.' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['purchasable'] = array(
			'description' => __( 'Whether or not object is purchasable' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['price'] = array(
			'description' => __( 'The price of the object' ),
			'type'        => 'object',
			'context'     => array( 'view', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'value'    => array(
						'description' => __( 'value for the object, as it exists in the database.' ),
						'type'        => 'integer',
						'context'     => array( 'view', 'edit' ),
					),
					'rendered' => array(
						'description' => __( 'Rendered value for object.' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'code'     => array(
						'description' => __( 'currency code for object.' ),
						'type'        => 'string',
						'context'     => array( 'view' ),
					),
				),
			),
		);

		$schema['properties']['video'] = array(
			'description' => __( 'video for the object.' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['group'] = array(
			'description' => __( 'BuddyPress group for the object.' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['forum'] = array(
			'description' => __( 'bbpress forum for the object.' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['is_closed'] = array(
			'description' => __( 'Whether object is closed or not.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['has_course_access'] = array(
			'description' => __( 'Whether or not user have the object access.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['has_content_access'] = array(
			'description' => __( 'Whether or not user have the object content access.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['hide_content_table'] = array(
			'description' => __( 'Whether or not user have the access of object content access.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['progression'] = array(
			'description' => __( 'The progression % of user for the object' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['points'] = array(
			'description' => __( 'Point for the object' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['duration'] = array(
			'description' => __( 'The time duration for the object' ),
			'type'        => 'object',
			'context'     => array( 'view', 'edit' ),
			'properties'  => array(
				'properties' => array(
					'min' => array(
						'description' => __( 'Minimum required time for this object.' ),
						'type'        => 'integer',
						'context'     => array( 'view', 'edit' ),
					),
				),
			),
		);

		$schema['properties']['access_list'] = array(
			'description' => __( 'The user list which has access of the object' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['quizzes'] = array(
			'description' => __( 'The quizzes list which has access of the object' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['lessons'] = array(
			'description' => __( 'The lessons list which has access of the object' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['completed'] = array(
			'description' => __( 'The object is completed by current user or not.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['quiz_completed'] = array(
			'description' => __( 'The object Quizzes is completed by current user or not.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.' ),
			'type'              => 'array',
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.' ),
			'type'              => 'array',
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.' ),
			'type'              => 'array',
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.' ),
			'type'              => 'array',
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent ids.' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default'           => array(),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent id.' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default'           => array(),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.' ),
		);

		$params['categories'] = array(
			'description' => sprintf( __( 'Limit result set to all items that have the specified term assigned in the %s.' ), 'category' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
			'default'     => array(),
		);

		$params['categories_exclude'] = array(
			'description' => sprintf( __( 'Limit result set to all items that have the specified term assigned in the %s.' ), 'category' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
			'default'     => array(),
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.' ),
			'type'              => 'array',
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['tags'] = array(
			'description' => sprintf( __( 'Limit result set to all items that have the specified term assigned in the %s.' ), 'tag' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
			'default'     => array(),
		);

		$params['tags_exclude'] = array(
			'description' => sprintf( __( 'Limit result set to all items except those that have the specified term assigned in the %s.' ), 'tag' ),
			'type'        => 'array',
			'items'       => array(
				'type' => 'integer',
			),
			'default'     => array(),
		);

		$params['recommended'] = array(
			'description'       => __( 'Limit result set to posts which is recommended' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['mycourses'] = array(
			'description'       => __( 'Limit response to resources which is taken by current user.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * @param array $prepared_args
	 * @param null  $request
	 *
	 * @return array
	 */
	protected function prepare_items_query( $prepared_args = array(), $request = null ) {
		$query_args = array();

		foreach ( $prepared_args as $key => $value ) {
			/**
			 * Filters the query_vars used in get_items() for the constructed query.
			 *
			 * The dynamic portion of the hook name, `$key`, refers to the query_var key.
			 *
			 * @param string $value The query_var value.
			 *
			 * @since 4.7.0
			 *
			 */
			$query_args[ $key ] = apply_filters( "rest_query_var-{$key}", $value ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		if ( 'post' !== $this->post_type || ! isset( $query_args['ignore_sticky_posts'] ) ) {
			$query_args['ignore_sticky_posts'] = true;
		}

		// Map to proper WP_Query orderby param.
		if ( isset( $query_args['orderby'] ) && isset( $request['orderby'] ) ) {
			$orderby_mappings = array(
				'id'            => 'ID',
				'include'       => 'post__in',
				'slug'          => 'post_name',
				'include_slugs' => 'post_name__in',
			);

			if ( isset( $orderby_mappings[ $request['orderby'] ] ) ) {
				$query_args['orderby'] = $orderby_mappings[ $request['orderby'] ];
			}
		}

		return $query_args;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {

		$links = parent::prepare_links( $data );

		if ( isset( $data["quizzes"] ) ) {
			$links['quizzes'] = array();
			foreach ( $data["quizzes"] as $quiz ) {
				$links['quizzes'][] = array(
					'href'       => rest_url( $this->namespace . '/quiz/' . $quiz["id"] ),
					'embeddable' => true
				);
			}
		}

		/**
		 * Categories Link
		 */
		if ( ! empty( $data["categories"] ) ) {
			$params = array();
			foreach ( $data["categories"] as $catid ) {
				$params[] = "include[]=" . $catid;
			}
			$links['categories'] = array(
				'href'       => rest_url( $this->namespace . '/course-categories?' . implode( "&", $params ) ),
				'embeddable' => true
			);
		}

		return $links;
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learner/v1/courses Courses
	 * @apiName        GetLearnerCourses
	 * @apiGroup       LearnerCourses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get All courses of/for learner component
	 * @apiUse         apidocForLearnerCoursesV1
	 * @apiParam {Number} recommended Limit result set to posts which is recommended
	 * @apiParam {Number{0-1}} mycourses Limit response to resources which is taken by current user.
	 * @apiDeprecated  Retrieve Courses. Check (#Courses:GetLDCourses)
	 */
	public function get_items( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		// Retrieve the list of registered collection query parameters.
		$registered = $this->get_collection_params();

		$request = apply_filters_deprecated( 'bbapp_learner_get_courses_request', array( $request ), '1.0.0', 'bbapp_ld_get_courses_request' );
		$request = apply_filters( 'bbapp_ld_get_courses_request', $request );

		$args = array();

		/*
			 * This array defines mappings between public API query parameters whose
			 * values are accepted as-passed, and their internal WP_Query parameter
			 * name equivalents (some are the same). Only values which are also
			 * present in $registered will be set.
		*/
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
		);

		// fallback
		if ( isset( $request["category"] ) && ! isset( $request["categories"] ) ) {
			$request["categories"] = (array) $request["category"];
		}
		if ( isset( $request["tag"] ) && ! isset( $request["tags"] ) ) {
			$request["tags"] = (array) $request["tag"];
		}

		/*
			 * For each known parameter which is both registered and present in the request,
			 * set the parameter's value on the query $args.
		*/
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			}
		}

		// Check for & assign any parameters which require special handling or setting.
		$args['date_query'] = array();

		// Set before into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['before'], $request['before'] ) ) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into date query. Date query must be specified as an array of an array.
		if ( isset( $registered['after'], $request['after'] ) ) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		// Ensure our per_page parameter overrides any provided posts_per_page filter.
		if ( isset( $registered['per_page'] ) ) {
			$args['posts_per_page'] = $request['per_page'];
		}

		if ( isset( $registered['sticky'], $request['sticky'] ) ) {
			$sticky_posts = get_option( 'sticky_posts', array() );
			if ( ! is_array( $sticky_posts ) ) {
				$sticky_posts = array();
			}
			if ( $request['sticky'] ) {
				/*
					 * As post__in will be used to only get sticky posts,
					 * we have to support the case where post__in was already
					 * specified.
				*/
				$args['post__in'] = $args['post__in'] ? array_intersect( $sticky_posts, $args['post__in'] ) : $sticky_posts;

				/*
					 * If we intersected, but there are no post ids in common,
					 * WP_Query won't return "no posts" for post__in = array()
					 * so we have to fake it a bit.
				*/
				if ( ! $args['post__in'] ) {
					$args['post__in'] = array( 0 );
				}
			} elseif ( $sticky_posts ) {
				/*
					 * As post___not_in will be used to only get posts that
					 * are not sticky, we have to support the case where post__not_in
					 * was already specified.
				*/
				$args['post__not_in'] = array_merge( $args['post__not_in'], $sticky_posts );
			}
		}

		$query_args = $this->prepare_items_query( $args, $request );

		if ( isset( $request['recommended'] ) ) {
			$query_args['recommended'] = $request['recommended'];
		}

		if ( isset( $request['mycourses'] ) ) {
			$query_args['mycourses'] = $request['mycourses'];
		}

		$query_args = apply_filters_deprecated( 'bbapp_learner_get_courses_args', array(
			$query_args,
			$request
		), '1.0.0', 'bbapp_ld_get_courses_args' );
		$query_args = apply_filters( 'bbapp_ld_get_courses_args', $query_args, $request );

		$query_result = $this->course_helper->get_courses( array(), $query_args, $request );
		$query_result = apply_filters_deprecated( 'bbapp_learner_get_courses', array( $query_result, $query_args, $request ), '1.0.0', 'bbapp_ld_get_courses' );
		$query_result = apply_filters( 'bbapp_ld_get_courses', $query_result, $query_args, $request );

		$posts = array();
		foreach ( $query_result['posts'] as $post ) {
			if ( ! $this->check_read_permission( $post ) ) {
				continue;
			}

			$data    = $this->prepare_item_for_response( $post, $request );
			$posts[] = $this->prepare_response_for_collection( $data );
		}
		$total_posts = $query_result['total_posts'];

		$max_pages = ceil( $total_posts / (int) $query_args['posts_per_page'] );

		$page = ( ! empty( $query_args['paged'] ) ) ? (int) $query_args['paged'] : 1;

		if ( $page > $max_pages && $total_posts > 0 ) {
			return new WP_Error( 'rest_post_invalid_page_number', __( 'The page number requested is larger than the number of pages available.' ), array( 'status' => 400 ) );
		}

		$response = rest_ensure_response( $posts );
		$response->header( 'X-WP-Total', (int) $total_posts );
		$response->header( 'X-WP-TotalPages', (int) $max_pages );

		$request_params = $request->get_query_params();
		$base           = add_query_arg( $request_params, rest_url( sprintf( '/%s/%s', $this->namespace, $this->rest_base ) ) );

		if ( $page > 1 ) {
			$prev_page = $page - 1;
			if ( $prev_page > $max_pages ) {
				$prev_page = $max_pages;
			}
			$prev_link = add_query_arg( 'page', $prev_page, $base );
			$response->link_header( 'prev', $prev_link );
		}
		if ( $max_pages > $page ) {
			$next_page = $page + 1;
			$next_link = add_query_arg( 'page', $next_page, $base );
			$response->link_header( 'next', $next_link );
		}

		return $response;
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learner/v1/courses/:id Course
	 * @apiName        GetLearnerCourse
	 * @apiGroup       LearnerCourses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get single course of/from learner component
	 * @apiUse         apidocGetLearnerCourseV1
	 * @apiDeprecated  Retrieve single Course. Check (#Courses:GetLDCourse)
	 */
	public function get_item( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$id = (int) $request['id'];

		$post = $this->course_helper->get_course( array(), $id );
		$post = apply_filters_deprecated( 'bbapp_learner_get_course', array(
			$post,
			$id
		), '1.0.0', 'bbapp_ld_get_course' );
		$post = apply_filters( 'bbapp_ld_get_course', $post, $id );

		if ( empty( $id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}

		$data     = $this->prepare_item_for_response( $post, $request );
		$response = rest_ensure_response( $data );

		return $response;
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/courses/action/:id Update course action
	 * @apiName        UpdateLearnerCourseAction
	 * @apiGroup       LearnerCourses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Update single course of/from learner component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Unique identifier
	 * @apiParam {String=enroll,mark_complete,access_expired} [action=''] Action to be performed
	 * @apiDeprecated  Course actions.
	 */
	public function do_action( $request ) {
		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$course_id = (int) $request['id'];
		$action    = $request['action'];

		// @todo : @ayush or @dipesh, why is possible_actions empty? I have added some values based on my awareness. By Ketan, May, 2019
		// Setup possible get actions
		$possible_actions = array( 'access_expired', 'enroll', 'mark_complete' );

		// Bail if actions aren't meant for this function
		if ( ! in_array( $action, $possible_actions ) ) {
			return new WP_Error( 'learndash_json_course_invalid_action', __( 'Action is invalid.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$user_id = get_current_user_id();

		if ( empty( $course_id ) ) {
			return new WP_Error( 'learndash_json_course_id', __( 'No course found! On which course are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$result = $this->course_helper->course_action( false, $action, $course_id );
		$result = apply_filters( 'bbapp_learner_course_action', $result, $action, $course_id );

		return rest_ensure_response( $result );

	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/courses/enroll/:id Update course enrollment
	 * @apiName        UpdateLearnerCourseEnrollment
	 * @apiGroup       LearnerCourses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Enroll course
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Course Id
	 * @apiDeprecated  Process Course enrollment. Check (#Courses:PostLDCourseEnroll)
	 */
	public function enroll( $request ) {

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}
		$course_id = (int) $request['id'];
		$user_id   = get_current_user_id();

		if ( empty( $course_id ) ) {
			return new WP_Error( 'learndash_json_course_id', __( 'No course was found! On which course are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$result = $this->course_helper->enroll_course( false, $course_id );
		$result = apply_filters( 'bbapp_learner_enroll_course', $result, $course_id );

		if ( is_wp_error( $result ) ) {
			return $result;
		} elseif ( true === $result ) {
			return $this->get_item( $request );
		}

		return new WP_Error( 'learndash_json_internal_error', __( 'You need to purchase this course before you can start.', 'buddyboss-app' ), array( 'status' => 400 ) );
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post;
		setup_postdata( $post );

		// Base fields for every post.
		$data = array();

		if ( ! $post->has_content_access ) {
			$post->materials = '';
			$post->video     = '';
		}

		if ( isset( $post->ID ) ) {
			$data['id'] = $post->ID;
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "ID" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( isset( $post->post_date_gmt ) && isset( $post->post_date ) ) {
			$data['date']     = $this->prepare_date_response( $post->post_date_gmt, $post->post_date );
			$data['date_gmt'] = $this->prepare_date_response( $post->post_date_gmt );
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Date or GMT Date" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( isset( $post->post_modified_gmt ) && isset( $post->post_modified ) ) {
			$data['modified']     = $this->prepare_date_response( $post->post_modified_gmt, $post->post_modified );
			$data['modified_gmt'] = $this->prepare_date_response( $post->post_modified_gmt );
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Modified Date or GMT Date" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( isset( $post->post_name ) ) {
			$data['slug'] = $post->post_name;
		} else {
			return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Slug" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$data['link'] = $post->link;

		$schema = $this->get_item_schema();

		if ( ! empty( $schema['properties']['title'] ) ) {
			if ( isset( $post->post_title ) ) {
				$data['title'] = array(
					'raw'      => $post->post_title,
					'rendered' => get_the_title( $post->ID ),
				);
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Title" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['content'] ) ) {

			if ( isset( $post->post_content ) ) {
				$data['content'] = array(
					'raw'      => bbapp_learners_fix_relative_urls_protocol( $post->post_content ),
					/** This filter is documented in wp-includes/post-template.php */
					'rendered' => bbapp_learners_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ),
				);
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Content" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}

		}

		if ( ! empty( $schema['properties']['excerpt'] ) ) {
			$data['excerpt'] = array(
				'raw'      => $post->post_excerpt,
				'rendered' => $this->prepare_excerpt_response( $post->post_excerpt ),
			);
		}

		if ( ! empty( $schema['properties']['author'] ) ) {
			if ( isset( $post->post_author ) ) {
				$data['author'] = (int) $post->post_author;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Author" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		/**
		 * Feature Media
		 */
		$data['featured_media']          = array();
		$data['featured_media']["small"] = ( is_array( $post->featured_media ) && isset( $post->featured_media["small"] ) ) ? $post->featured_media["small"] : null;
		$data['featured_media']["large"] = ( is_array( $post->featured_media ) && isset( $post->featured_media["large"] ) ) ? $post->featured_media["large"] : null;

		$data['parent']     = (int) $post->post_parent;
		$data['menu_order'] = (int) $post->menu_order;

		if ( ! empty( $schema['properties']['featured'] ) ) {
			$data['featured'] = (boolean) $post->featured;
		}

		if ( ! empty( $schema['properties']['materials'] ) ) {
			$data['materials'] = $post->materials;
		}

		if ( ! empty( $schema['properties']['purchasable'] ) ) {
			$data['purchasable'] = (boolean) $post->purchasable;
		}

		if ( ! empty( $schema['properties']['price'] ) ) {
			if ( is_array( $post->price ) ) {
				$data['price'] = array(
					'value'    => $post->price['value'],
					'rendered' => $this->prepare_price_response( $post->price ),
					'code'     => $post->price['code'],
				);
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "price" missing or incorrectly by addon plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['has_course_access'] ) ) {
			if ( isset( $post->has_course_access ) ) {
				$data['has_course_access'] = (boolean) $post->has_course_access;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Has course access" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['has_content_access'] ) ) {
			if ( isset( $post->has_content_access ) ) {
				$data['has_content_access'] = (boolean) $post->has_content_access;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Has content access" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['hide_content_table'] ) ) {
			if ( isset( $post->hide_content_table ) ) {
				$data['hide_content_table'] = (boolean) $post->hide_content_table;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Has content table" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['progression'] ) ) {
			if ( isset( $post->progression ) ) {
				$data['progression'] = (int) $post->progression;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Progression" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['is_closed'] ) ) {
			if ( isset( $post->is_closed ) ) {
				$data['is_closed'] = (boolean) $post->is_closed;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "is_closed" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['points'] ) ) {
			$data['points'] = (int) $post->points;
		}

		if ( ! empty( $schema['properties']['duration'] ) ) {
			$data['duration']        = array();
			$data['duration']["min"] = (int) ( is_array( $post->duration ) && isset( $post->duration["min"] ) ) ? $post->duration["min"] : 0;
		}

		if ( ! empty( $schema['properties']['access_list'] ) ) {
			if ( isset( $post->access_list ) ) {
				$data['access_list'] = $post->access_list;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Access List" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['categories'] ) ) {
			if ( isset( $post->categories ) ) {
				$data['categories'] = $post->categories;
			} else {
				return new WP_Error( 'learndash_json_internal_error', __( 'Required field "Categories" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['tags'] ) ) {
			if ( isset( $post->tags ) ) {
				$data['tags'] = $post->tags;
			} else {
				$data['tags'] = array();
			}
		}

		if ( ! empty( $schema['properties']['module'] ) ) {
			if ( isset( $post->module ) ) {
				$data['module'] = $post->module;
			} else {
				$data['module'] = array();
			}
		}

		if ( ! empty( $schema['properties']['video'] ) ) {
			$data['video'] = $post->video;
		}

		if ( ! empty( $schema['properties']['group'] ) ) {
			$data['group'] = (int) $post->group;
		}

		if ( ! empty( $schema['properties']['forum'] ) ) {
			$data['forum'] = (int) $post->forum;
		}

		if ( ! empty( $schema['properties']['lessons'] ) ) {
			if ( isset( $post->lessons ) ) {
				$data['lessons'] = $post->lessons;
			} else {
				$data['lessons'] = array();
			}
		}

		if ( ! empty( $schema['properties']['quizzes'] ) ) {
			if ( isset( $post->quizzes ) ) {
				$data['quizzes'] = $post->quizzes;
			} else {
				$data['quizzes'] = array();
			}
		}

		if ( ! empty( $schema['properties']['completed'] ) ) {
			$data['completed'] = (boolean) $post->completed;
		}

		if ( ! empty( $schema['properties']['quiz_completed'] ) ) {
			$data['quiz_completed'] = (boolean) $post->quiz_completed;
		}

		if ( ! empty( $schema['properties']['error_message'] ) ) {
			if ( ! empty( $post->error_message ) ) {
				$data['error_message'] = $post->error_message;
			} else {
				$data['error_message'] = array();
			}
		}

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		$response = apply_filters_deprecated( 'rest_prepare_bbapp_learner_course', array(
			$response,
			$post,
			$request
		), '1.0.0', 'bbapp_ld_rest_prepare_course' );

		return apply_filters( 'bbapp_ld_rest_prepare_course', $response, $post, $request );
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learner/v1/courses/members/:id Course members
	 * @apiName        GetLearnerCourseMembers
	 * @apiGroup       LearnerCourses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get members in particular course of/for learner component
	 * @apiDeprecated  Retrieve single Courses Members. Check (#Courses:GetLDCourseMembers)
	 * @apiUse         apidocForLearnerCourseMembersV1
	 */
	public function get_course_members( $request ) {
		$course_id = (int) $request['id'];

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		if ( empty( $course_id ) ) {
			return new WP_Error( 'learndash_json_course_id', __( 'No course found! On which course are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$args = array(
			'number' => 3,
		);
		if ( isset( $request['number'] ) ) {
			$args['number'] = $request['number'];
		}

		$result = $this->course_helper->get_course_members( false, $args, $course_id, $request );
		$result = apply_filters( 'bbapp_learner_get_course_members', $result, $args, $course_id, $request );

		return rest_ensure_response( $result );
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/courses/:id/download Download course content
	 * @apiName        DownloadLearnerCourseContent
	 * @apiGroup       LearnerCourses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Download course content(item) of/for learner component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Course ID
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiDeprecated  Process Course Download. Check (#Courses:PostLDCourseDownload)
	 */
	public function download_item( $request ) {

		$course_id = (int) $request['id'];

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		if ( empty( $course_id ) ) {
			return new WP_Error( 'learndash_json_course_id', __( 'No course was found! On which course are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		//Set course folder path
		$folder_name = 'course_' . $course_id . '/';

		//Delete course directory if exist
		$folder_path = bbapp_learners_getpath( $folder_name );
		bbapp_learners_delete_directory( $folder_path );

		//Create course directory
		$folder_path = bbapp_learners_setpath( $folder_name );

		//Create assests directory for course
		$assests_path = bbapp_learners_setpath( $folder_name ) . 'courses/';
		if ( ! file_exists( $assests_path ) ) {
			wp_mkdir_p( $assests_path );
		}

		//Create course data
		$post = $this->course_helper->get_course( array(), $course_id );
		$post = apply_filters( 'bbapp_learner_get_course', $post, $course_id );
		if ( empty( $course_id ) || empty( $post->ID ) ) {
			return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
		}
		$data = $this->prepare_item_for_response( $post, $request );

		//Download course html and assests

		$html_attrs = apply_filters_deprecated( 'bbapp_learner_course_html_attrs', array(
			array( 'content', 'materials' ),
		), '1.0.0', 'bbapp_ld_course_html_attrs' );
		$html_attrs = apply_filters( 'bbapp_ld_course_html_attrs', $html_attrs );

		foreach ( $html_attrs as $attr ) {
			if ( isset( $data->data[ $attr ]['rendered'] ) ) {
				$value = $data->data[ $attr ]['rendered'] = bbapp_learners_download_html( $assests_path, $data->data[ $attr ]['rendered'] );
			} else {
				$value = $data->data[ $attr ] = bbapp_learners_download_html( $assests_path, $data->data[ $attr ] );
			}
			$file_name = 'course_' . $attr . '.html';
			bbapp_learners_write_file( $assests_path . $file_name, $value );
		}

		//Create course.json
		$file_data = json_encode( $data );
		$file_name = 'course_' . $course_id . '.json';
		bbapp_learners_write_file( $folder_path . $file_name, $file_data );

		//Create info.json
		$info_data = json_encode( array(
			'id'       => $course_id,
			'hash_key' => sha1( $post->post_modified ),
			'modified' => $this->prepare_date_response( $post->post_modified_gmt, $post->post_modified ),
		) );
		$file_name = 'info.json';
		bbapp_learners_write_file( $folder_path . $file_name, $info_data );

		//Create zip file
		$zip_url = bbapp_learners_zip_create( $folder_path, 'course_' . $course_id );
		if ( empty( $zip_url ) ) {
			return new WP_Error( 'rest_post_invalid_zip', __( 'Zip file not created.' ), array( 'status' => 500 ) );
		}

		return rest_ensure_response( array( 'zip_url' => $zip_url ) );
	}

	/**
	 * @param $request
	 *
	 * @return array|WP_Error
     * @apiPrivate
	 * @api            {PATCH} /wp-json/appboss/learner/v1/courses/:id/upload Upload course progress
	 * @apiName        UploadLearnerCourseProgress
	 * @apiGroup       LearnerCourses
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Update course progress of/for learner component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Course ID
	 * @apiParam {Array} statements Statements consists of action, object and data. actions : 'completed' or
	 *           'something?'; object : 'lesson' or 'topics'; primary_item_id; secondary_item_id; item_id
	 * @apiDeprecated  Process Course Statements. Check (#Courses:PostLDCourseStatements)
	 */
	public function upload_progress( $request ) {
		$responses = array();
		$course_id = (int) $request['id'];

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_not_logged_in', __( 'You are not currently logged in.' ), array( 'status' => rest_authorization_required_code() ) );
		}

		if ( empty( $course_id ) ) {
			return new WP_Error( 'learndash_json_course_id', __( 'No course was found! On which course are you performing action?', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$statements = (array) $request['statements'];
		$_temp      = $_REQUEST;

		foreach ( $statements as $statement ) {
			$action      = $statement['action'];
			$object      = $statement['object'];
			$object_data = $statement['data'];

			if ( 'completed' == $action ) {
				switch ( $object ) {
					case 'lesson':
						$lesson_id = (int) $statement['item_id'];

						$lesson_request = new WP_REST_Request( 'POST', '/appboss/learner/v1/lessons/complete/' . $lesson_id );

						$lesson_request->set_url_params( array( 'id' => $lesson_id ) );
						$lesson_request->set_param( 'course_id', $course_id );
						$lesson_request->set_body( $object_data );
						if ( ! empty( $object_data ) ) {
							foreach ( $object_data as $key => $data ) {
								$lesson_request->set_param( $key, $data );
							}
						}

						$_GET       = $_POST = $_REQUEST = $lesson_request->get_params();
						$_temp_file = $_FILES;
						if ( isset( $_FILES[ $lesson_id ] ) ) {
							$_FILES = $_FILES[ $lesson_id ];
						} else {
							unset( $_FILES );
						}
						$_FILES = $_temp_file;

						$server      = rest_get_server();
						$responses[] = $server->dispatch( $lesson_request );

						break;
					case 'topic':

						$lesson_id = (int) $statement['secondary_item_id'];
						$topic_id  = (int) $statement['item_id'];

						$topic_request = new WP_REST_Request( 'POST', '/appboss/learner/v1/topics/complete/' . $topic_id );

						$topic_request->set_url_params( array( 'id' => $topic_id ) );
						$topic_request->set_param( 'course_id', $course_id );
						$topic_request->set_param( 'lesson_id', $lesson_id );
						$topic_request->set_body( $object_data );
						if ( ! empty( $object_data ) ) {
							foreach ( $object_data as $key => $data ) {
								$topic_request->set_param( $key, $data );
							}
						}

						$_GET       = $_POST = $_REQUEST = $topic_request->get_params();
						$_temp_file = $_FILES;
						if ( isset( $_FILES[ $topic_id ] ) ) {
							$_FILES = $_FILES[ $topic_id ];
						} else {
							unset( $_FILES );
						}
						$_FILES = $_temp_file;

						$server      = rest_get_server();
						$responses[] = $server->dispatch( $topic_request );

						break;
				}
			}
		}

		$_GET = $_POST = $_REQUEST = $_temp;

		return $responses;
	}
}