<?php
namespace BuddyBossApp\Api\Learner;
use BuddyBossApp\Api\Learner\Rest;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

// NOTE : Old classname was class.bbapp_learner_course_categories_rest. By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for learndash.
class CourseCategoriesRest extends Rest {

	protected static $instance;
	protected $course_helper;
	protected $courseCat_helper;

	/**
	 * CourseCategoriesRest constructor.
	 */
	public function __construct() {
		$this->rest_base = 'course-categories';
		parent::__construct();
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if (!isset(self::$instance)) {
			$class = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * @return void|WP_Error
	 */
	public function register_routes() {

		$this->course_helper = bbapp_learner_learndash()->c->bbapp_learner_learndash_courses_rest;
		$this->courseCat_helper = bbapp_learner_learndash()->c->bbapp_learner_learndash_courses_categories_rest;

		register_rest_route($this->namespace, '/' . $this->rest_base, array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_items'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
				'args' => $this->get_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)', array(
			'args' => array(
				'id' => array(
					'description' => __('Unique identifier for the term.'),
					'type' => 'integer',
				),
			),
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_item'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => array(
					'context' => $this->get_context_param(array('default' => 'view')),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

	}

	/**
	 * Retrieves the term's schema, conforming to JSON Schema.
	 *
	 * @since 4.7.0
	 *
	 * @return array Item schema data.
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema' => 'http://json-schema.org/schema#',
			'title' => 'lms-categories',
			'type' => 'object',
			'properties' => array(
				'id' => array(
					'description' => __('Unique identifier for the term.'),
					'type' => 'integer',
					'context' => array('view', 'embed', 'edit'),
					'readonly' => true,
				),
				'count' => array(
					'description' => __('Number of published posts for the term.'),
					'type' => 'integer',
					'context' => array('view', 'edit'),
					'readonly' => true,
				),
				'description' => array(
					'description' => __('HTML description of the term.'),
					'type' => 'string',
					'context' => array('view', 'edit'),
				),
				'link' => array(
					'description' => __('URL of the term.'),
					'type' => 'string',
					'format' => 'uri',
					'context' => array('view', 'embed', 'edit'),
					'readonly' => true,
				),
				'name' => array(
					'description' => __('HTML title for the term.'),
					'type' => 'string',
					'context' => array('view', 'embed', 'edit'),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_text_field',
					),
					'required' => true,
				),
				'slug' => array(
					'description' => __('An alphanumeric identifier for the term unique to its type.'),
					'type' => 'string',
					'context' => array('view', 'embed', 'edit'),
					'arg_options' => array(
						'sanitize_callback' => array($this, 'sanitize_slug'),
					),
				),
				'taxonomy' => array(
					'description' => __('Type attribution for the term.'),
					'type' => 'string',
					'enum' => array_keys(get_taxonomies()),
					'context' => array('view', 'embed', 'edit'),
					'readonly' => true,
				),
				'icon' => array(
					'description' => __('icon object containing thumb and full URL of image.'),
					'type' => 'array',
					'context' => array('view', 'edit', 'embed'),
				),
			),
		);

		$schema['properties']['parent'] = array(
			'description' => __('The parent term ID.'),
			'type' => 'integer',
			'context' => array('view', 'edit'),
		);

		$schema['properties']['courses'] = array(
			'description' => __('Courses for the term.'),
			'type' => 'array',
			'context' => array('view', 'edit'),
		);

		$schema['properties']['meta'] = array(
			'description' => __('Meta for the term.'),
			'type' => 'array',
			'context' => array('view', 'edit'),
		);

		$params['mycourses'] = array(
			'description' => __('Limit response to resources which is taked by current user.'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $this->add_additional_fields_schema($schema);
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		$params['exclude'] = array(
			'description' => __('Ensure result set excludes specific ids.'),
			'type' => 'array',
			'default' => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['exclude'] = array(
			'description' => __('Ensure result set excludes specific ids.'),
			'type' => 'array',
			'default' => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['courses_exclude'] = array(
			'description' => __('Ensure result set excludes specific courses ids.'),
			'type' => 'array',
			'default' => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['include'] = array(
			'description' => __('Limit result set to specific ids.'),
			'type' => 'array',
			'default' => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['order'] = array(
			'description' => __('Order sort attribute ascending or descending.'),
			'type' => 'string',
			'default' => 'asc',
			'enum' => array('asc', 'desc'),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['hide_empty'] = array(
			'description' => __('Whether to hide terms not assigned to any items.'),
			'type' => 'boolean',
			'default' => false,
		);

		$params['orderby'] = array(
			'description' => __('Sort collection by object attribute.'),
			'type' => 'string',
			'default' => 'name',
			'enum' => array(
				'date',
				'name',
				'id',
				'slug',
				'term_group',
				'term_id',
				'description',
				'parent',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['course_order'] = array(
			'description' => __('Order sort attribute ascending or descending of courses.'),
			'type' => 'string',
			'default' => 'desc',
			'enum' => array('asc', 'desc'),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['course_orderby'] = array(
			'description' => __('Sort courses collection by object attribute.'),
			'type' => 'string',
			'default' => 'date',
			'enum' => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
				'menu_order',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['slug'] = array(
			'description' => __('Limit result set to terms with one or more specific slugs.'),
			'type' => 'array',
			'items' => array(
				'type' => 'string',
			),
		);

		$query_params['courses_limit'] = array(
			'description' => __('Limit result set to courses populated on each category.'),
			'type' => 'integer',
		);

		return $params;

	}

	/**
	 * @param $request
	 * @return array|WP_Error
     * @apiPrivate
	 * @api {GET} /wp-json/appboss/learner/v1/course-categories Course categories
	 * @apiName GetLearnerCourseCategories
	 * @apiGroup LearnerCourseCategories
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription  Get all collection of course categories for learner component
	 * @apiDeprecated  Retrieve Course Categories. Check (#Courses:GetLDCourseCategories)
	 * @apiUse apidocForLearnerCourseCategoriesV1
	 */
	public function get_items($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		// Retrieve the list of registered collection query parameters.
		$registered = $this->get_collection_params();

		/*
			 * This array defines mappings between public API query parameters whose
			 * values are accepted as-passed, and their internal WP_Query parameter
			 * name equivalents (some are the same). Only values which are also
			 * present in $registered will be set.
		*/
		$parameter_mappings = array(
			'exclude' => 'exclude',
			'include' => 'include',
			'order' => 'order',
			'orderby' => 'orderby',
			'course_order' => 'course_order',
			'course_orderby' => 'course_orderby',
			'hide_empty' => 'hide_empty',
			'per_page' => 'number',
			'search' => 'search',
			'slug' => 'slug',
			'courses_limit' => 'courses_limit',
			'courses_exclude' => 'courses_exclude',
		);

		$prepared_args = array();

		/*
			 * For each known parameter which is both registered and present in the request,
			 * set the parameter's value on the query $prepared_args.
		*/
		foreach ($parameter_mappings as $api_param => $wp_param) {
			if (isset($registered[$api_param], $request[$api_param])) {
				$prepared_args[$wp_param] = $request[$api_param];
			}
		}

		if (isset($registered['offset']) && !empty($request['offset'])) {
			$prepared_args['offset'] = $request['offset'];
		} else {
			$prepared_args['offset'] = ($request['page'] - 1) * $prepared_args['number'];
		}

		if (isset($request['mycourses'])) {
			$prepared_args['mycourses'] = $request['mycourses'];
		}

		$query_result = $this->courseCat_helper->get_course_categories( array(), $prepared_args, $request );
		$query_result = apply_filters("bbapp_learner_get_course_categories", $query_result, $prepared_args, $request);

		$total_terms = $query_result['total_terms'];

		// wp_count_terms can return a falsy value when the term has no children.
		if (!$total_terms) {
			$total_terms = 0;
		}

		$response = array();

		foreach ($query_result['terms'] as $term) {
			$term->courses = $this->course_helper->get_courses_by_category( array(), $term, $prepared_args );
			$term->courses = apply_filters('bbapp_learner_get_courses_by_category', $term->courses, $term, $prepared_args);
			$data = $this->prepare_item_for_response($term, $request);
			$response[] = $this->prepare_response_for_collection($data);
		}

		$response = rest_ensure_response($response);

		// Store pagination values for headers.
		$per_page = (int) $prepared_args['number'];
		$page = ceil((((int) $prepared_args['offset']) / $per_page) + 1);

		$response->header('X-WP-Total', (int) $total_terms);

		$max_pages = ceil($total_terms / $per_page);

		$response->header('X-WP-TotalPages', (int) $max_pages);

		$base = add_query_arg($request->get_query_params(), rest_url($this->namespace . '/' . $this->rest_base));
		if ($page > 1) {
			$prev_page = $page - 1;

			if ($prev_page > $max_pages) {
				$prev_page = $max_pages;
			}

			$prev_link = add_query_arg('page', $prev_page, $base);
			$response->link_header('prev', $prev_link);
		}
		if ($max_pages > $page) {
			$next_page = $page + 1;
			$next_link = add_query_arg('page', $next_page, $base);

			$response->link_header('next', $next_link);
		}

		return $response;
	}

	/**
	 * @param $request
	 * @return WP_Error
     * @apiPrivate
	 * @api {GET} /wp-json/appboss/learner/v1/course-categories/:id Course category
	 * @apiName GetLearnerCourseCategoryItem
	 * @apiGroup LearnerCourseCategories
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription course category of/for learner component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Term id
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiDeprecated  Retrieve single Course category. Check (#Courses:GetLDCourseCategory)
	 */
	public function get_item($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		if ((int) $request['id'] < 0) {
			return new WP_Error('rest_term_invalid', __('Term does not exist.'), array('status' => 404));
		}

		$term = $this->courseCat_helper->get_course_category( array(), $request['id'], $request );
		$term = apply_filters("bbapp_learner_get_course_category", $term, $request['id'], $request);

		$response = $this->prepare_item_for_response($term, $request);

		return rest_ensure_response($response);
	}

	/**
	 * Prepares a single term output for response.
	 *
	 * @since 4.7.0
	 *
	 * @param obj             $item    Term object.
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response $response Response object.
	 */
	public function prepare_item_for_response($item, $request) {

		$schema = $this->get_item_schema();
		$data = array();

		if (!empty($schema['properties']['id'])) {
			$data['id'] = (int) $item->id;
		}

		if (!empty($schema['properties']['count'])) {
			$data['count'] = (int) $item->count;
		}

		if (!empty($schema['properties']['description'])) {
			$data['description'] = $item->description;
		}

		if (!empty($schema['properties']['link'])) {
			$data['link'] = $item->link;
		}

		if (!empty($schema['properties']['name'])) {
			$data['name'] = $item->name;
		}

		if (!empty($schema['properties']['slug'])) {
			$data['slug'] = $item->slug;
		}

		if (!empty($schema['properties']['parent'])) {
			$data['parent'] = (int) $item->parent;
		}

		if (!empty($schema['properties']['meta'])) {
			$data['meta'] = $item->meta;
		}

		if (!empty($schema['properties']['courses']) && isset($item->courses)) {
			$data['courses'] = $item->courses;
		}

		/**
		 * icon
		 */
		$data['icon'] = array();
		$data['icon']["small"] = (is_array($item->icon) && isset($item->icon["small"])) ? $item->icon["small"] : null;
		$data['icon']["large"] = (is_array($item->icon) && isset($item->icon["large"])) ? $item->icon["large"] : null;

		$context = !empty($request['context']) ? $request['context'] : 'view';
		$data = $this->add_additional_fields_to_object($data, $request);
		$data = $this->filter_response_by_context($data, $context);

		$response = rest_ensure_response($data);

		$response->add_links($this->prepare_links($data));

		/**
		 * Filters a term item returned from the API.
		 *
		 * The dynamic portion of the hook name, `$this->taxonomy`, refers to the taxonomy slug.
		 *
		 * Allows modification of the term data right before it is returned.
		 *
		 * @since 4.7.0
		 *
		 * @param WP_REST_Response  $response  The response object.
		 * @param object            $item      The original term object.
		 * @param WP_REST_Request   $request   Request used to generate the response.
		 */
		return apply_filters("rest_prepare_bbapp_learner_course-categories", $response, $item, $request);
	}
}