<?php
namespace BuddyBossApp\Api\Learner;
use BuddyBossApp\Api\Learner\Rest;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

// NOTE : Old classname was class.bbapp_learner_certificates_rest. By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for learndash.
class CertificatesRest extends Rest {

	protected static $instance;
	protected $certificate_helper;

	/**
	 * CertificatesRest constructor.
	 */
	public function __construct() {
		$this->rest_base = 'certificates';
		parent::__construct();
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if (!isset(self::$instance)) {
			$class = __CLASS__;
			self::$instance = new $class;
		}
		return self::$instance;
	}

	/**
	 * @return void|WP_Error
	 */
	public function register_routes() {

		$this->certificate_helper = bbapp_learner_learndash()->c->bbapp_learner_learndash_certificates_rest;

		register_rest_route($this->namespace, '/' . $this->rest_base, array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_items'),
				'permission_callback' => '__return_true',
				// 'permission_callback' => array($this, 'get_items_permissions_check'),
				'args' => $this->get_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {

		$schema = array(
			'$schema' => 'http://json-schema.org/draft-04/schema#',
			'title' => 'lesson',
			'type' => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id' => array(
					'description' => __('Unique identifier for the object.', 'buddyboss-app'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'title' => array(
					'description' => __('The title for the object.'),
					'type' => 'object',
					'context' => array('view', 'edit', 'embed'),
					'properties' => array(
						'raw' => array(
							'description' => __('Title for the object, as it exists in the database.', 'buddyboss-app'),
							'type' => 'string',
							'context' => array('edit'),
						),
						'rendered' => array(
							'description' => __('HTML title for the object, transformed for display.', 'buddyboss-app'),
							'type' => 'string',
							'context' => array('view', 'edit', 'embed'),
						),
					),
				),
				'date' => array(
					'description' => __("The date the object was published, in the site's timezone.", 'buddyboss-app'),
					'type' => 'string',
					'format' => 'date-time',
					'context' => array('view', 'edit', 'embed'),
				),
				'link' => array(
					'description' => __('URL to the object.', 'buddyboss-app'),
					'type' => 'string',
					'format' => 'uri',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'type' => array(
					'description' => __('type to the object.', 'buddyboss-app'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'filename' => array(
					'description' => __('Filename to the object.', 'buddyboss-app'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),

			),
		);
		return $this->add_additional_fields_schema($schema);
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		return $params;
	}

    /**
     * @param $request
     * @return WP_REST_Response
     * @apiPrivate
     * @api {GET} /wp-json/appboss/learner/v1/certificates Learner certificates
     * @apiName GetLearnerCertificates
     * @apiGroup LearnerCertificates
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get All certificates of/for learner component
     * @apiDeprecated  Retrieve certificates. Check (#Certificate:GetLDCertificates)
     * @apiUse apidocForLearnerCertificatesV1
     */
	public function get_items($request) {

		$query_result = $this->certificate_helper->getCertificates( array(), $request);
		$query_result = apply_filters("bbapp_learner_get_certificates", $query_result, $request);

		$posts = array();
		foreach ($query_result['posts'] as $post) {
			$data = $this->prepare_item_for_response($post, $request);
			$posts[] = $this->prepare_response_for_collection($data);
		}

		$response = rest_ensure_response($posts);

		return $response;
	}

	/**
	 * Get a single post.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_item($request) {
		return new WP_Error('rest_no_route', __('No route was found matching the URL and request method', 'buddyboss-app'), array('status' => 404));
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post $post Post object.
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response($post, $request) {
		$data = array();

		if (isset($post->ID)) {
			$data['id'] = $post->ID;
		} else {
			return new WP_Error('learndash_json_internal_error', __('Required field "ID" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
		}

		if (isset($post->date)) {
			$data['date'] = $this->prepare_date_response($post->date);
		} else {
			return new WP_Error('learndash_json_internal_error', __('Required field "Date or GMT Date" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
		}

		$data['link'] = $post->link;
		$data['filename'] = $post->filename;
		$data['type'] = $post->type;

		$schema = $this->get_item_schema();

		if (!empty($schema['properties']['title'])) {
			if (isset($post->title)) {
				$data['title'] = array(
					'raw' => $post->title,
					'rendered' => $post->title,
				);
			} else {
				return new WP_Error('learndash_json_internal_error', __('Required field "Title" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
			}
		}

		$context = !empty($request['context']) ? $request['context'] : 'view';
		$data = $this->add_additional_fields_to_object($data, $request);
		$data = $this->filter_response_by_context($data, $context);

		// Wrap the data in a response object.
		$response = rest_ensure_response($data);

		/**
		 * Filter the post data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
		 * prepared for the response.
		 *
		 * @param WP_REST_Response   $response   The response object.
		 * @param WP_Post            $post       Post object.
		 * @param WP_REST_Request    $request    Request object.
		 */
		return apply_filters("rest_prepare_learner_certificate", $response, $post, $request);
	}
}