<?php
namespace BuddyBossApp\Api\Learner;
use BuddyBossApp\Api\Learner\Rest;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

// NOTE : Old classname was class.bbapp_learner_badges_rest. By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for learndash.
class BadgesRest extends Rest {

	protected static $instance;

	/**
	 * BadgesRest constructor.
	 */
	public function __construct() {
		$this->rest_base = 'badges';
		parent::__construct();
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if (!isset(self::$instance)) {
			$class = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * @return void|WP_Error
	 */
	public function register_routes() {
		register_rest_route($this->namespace, '/' . $this->rest_base, array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_items'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
				'args' => $this->get_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/ranks', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_ranks'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
				'args' => $this->get_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {

		$schema = array(
			'$schema' => 'http://json-schema.org/draft-04/schema#',
			'title' => 'lesson',
			'type' => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id' => array(
					'description' => __('Unique identifier for the object.', 'buddyboss-app'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'title' => array(
					'description' => __('The title for the object.', 'buddyboss-app'),
					'type' => 'object',
					'context' => array('view', 'edit', 'embed'),
					'properties' => array(
						'raw' => array(
							'description' => __('Title for the object, as it exists in the database.', 'buddyboss-app'),
							'type' => 'string',
							'context' => array('edit'),
						),
						'rendered' => array(
							'description' => __('HTML title for the object, transformed for display.', 'buddyboss-app'),
							'type' => 'string',
							'context' => array('view', 'edit', 'embed'),
						),
					),
				),
				'content' => array(
					'description' => __('The content for the object.', 'buddyboss-app'),
					'type' => 'object',
					'context' => array('view', 'edit'),
					'properties' => array(
						'raw' => array(
							'description' => __('Content for the object, as it exists in the database.', 'buddyboss-app'),
							'type' => 'string',
							'context' => array('edit'),
						),
						'rendered' => array(
							'description' => __('HTML content for the object, transformed for display.', 'buddyboss-app'),
							'type' => 'string',
							'context' => array('view', 'edit'),
						),
					),
				),
				'link' => array(
					'description' => __('URL to the object.', 'buddyboss-app'),
					'type' => 'string',
					'format' => 'uri',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'image' => array(
					'description' => __('URL to the image object.', 'buddyboss-app'),
					'type' => 'image',
					'format' => 'uri',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),

			),
		);

		$schema['properties']['points'] = array(
			'description' => __('Point for the object', 'buddyboss-app'),
			'type' => 'integer',
			'context' => array('view', 'edit'),
		);

		return $this->add_additional_fields_schema($schema);
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		$params['user_id'] = array(
			'description' => __('Limit result set to specific user.', 'buddyboss-app'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['order'] = array(
			'description' => __('Order sort attribute ascending or descending.'),
			'type' => 'string',
			'default' => 'desc',
			'enum' => array('asc', 'desc'),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby'] = array(
			'description' => __('Sort collection by object attribute.'),
			'type' => 'string',
			'default' => 'date',
			'enum' => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

    /**
     * @param $request
     * @return array
     * @apiPrivate
     * @api {GET} /wp-json/appboss/learner/v1/badges Learner badges
     * @apiName GetLearnerBadges
     * @apiGroup LearnerBadges
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get learner badges
     * @apiHeader {String} accessToken Auth token
     * @apiParam {String=view,edit,embed} [context=view]
     * @apiParam {Number} user_id Limit result set to specific user.
     * @apiDeprecated  Get badges.
     */
	public function get_items($request) {

		$query_result = apply_filters("bbapp_learner_get_badges", array(), $request);

		$posts = array();
		foreach ($query_result['posts'] as $post) {
			$data = $this->prepare_item_for_response($post, $request);
			$posts[] = $this->prepare_response_for_collection($data);
		}

		$response = rest_ensure_response($posts);

		return $response;
	}

    /**
     * @param $request
     * @return array
     * @apiPrivate
     * @api {GET} /wp-json/appboss/learner/v1/badges/ranks Learner ranks
     * @apiName GetLearnerBadgesRanks
     * @apiGroup LearnerBadges
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get badges ranks for learner component
     * @apiHeader {String} accessToken Auth token
     * @apiParam {String=view,edit,embed} [context=view]
     * @apiParam {Number} user_id Limit result set to specific user.
     * @apiDeprecated  Get ranks.
     */
	public function get_ranks($request) {

		$query_result = apply_filters("bbapp_learner_get_ranks", array(), $request);

		$posts = array();
		foreach ($query_result['posts'] as $post) {
			$data = $this->prepare_item_for_response($post, $request);
			$posts[] = $this->prepare_response_for_collection($data);
		}

		$response = rest_ensure_response($posts);

		return $response;
	}

	/**
	 * Get a single post.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_item($request) {
		return new WP_Error('rest_no_route', __('No route was found matching the URL and request method', 'buddyboss-app'), array('status' => 404));
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post $post Post object.
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response($post, $request) {
		$data = array();

		if (isset($post->ID)) {
			$data['id'] = $post->ID;
		} else {
			return new WP_Error('learndash_json_internal_error', __('Required field "ID" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
		}

		$data['link'] = $post->link;

		$schema = $this->get_item_schema();

		if (!empty($schema['properties']['title'])) {
			if (isset($post->title)) {
				$data['title'] = array(
					'raw' => $post->title,
					'rendered' => $post->title,
				);
			} else {
				return new WP_Error('learndash_json_internal_error', __('Required field "Title" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
			}
		}

		if (!empty($schema['properties']['content'])) {
			if (isset($post->content)) {
				$data['content'] = array(
					'raw' => $post->content,
					/** This filter is documented in wp-includes/post-template.php */
					'rendered' => apply_filters('the_content', $post->content),
				);
			} else {
				return new WP_Error('learndash_json_internal_error', __('Required field "Content" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
			}
		}

		if (!empty($schema['properties']['points'])) {
			$data['points'] = (int) $post->points;
		}

		$data['image'] = $post->image;

		$context = !empty($request['context']) ? $request['context'] : 'view';
		$data = $this->add_additional_fields_to_object($data, $request);
		$data = $this->filter_response_by_context($data, $context);

		// Wrap the data in a response object.
		$response = rest_ensure_response($data);

		/**
		 * Filter the post data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
		 * prepared for the response.
		 *
		 * @param WP_REST_Response   $response   The response object.
		 * @param WP_Post            $post       Post object.
		 * @param WP_REST_Request    $request    Request object.
		 */
		return apply_filters("rest_prepare_learner_certificate", $response, $post, $request);
	}
}