<?php
namespace BuddyBossApp\Api\BuddyPress;
use BP_XProfile_Field_Type_Datebox;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Server;

// NOTE : Old classname was class.boss_buddypress_api_xprofile_rest_api. By Ketan, Oct-2019
class XprofileRestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/buddypress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
	}

	public function hooks() {

		add_action('rest_api_init', array($this, "register_routes"), 99);

	}

	/**
	 * Register the routes for the objects of the controller.
	 */
	public function register_routes() {

		register_rest_route($this->namespace, '/' . 'xprofile', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_fields'),
				'permission_callback' => array($this, 'bp_xprofile_permission'),
				// 'args'                => $this->get_activity_args(),
			),
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'update_fields'),
				'permission_callback' => array($this, 'bp_xprofile_permission'),
			),
			'schema' => array($this, 'get_public_item_schema'),
		)
		);

		register_rest_route($this->namespace, '/' . 'xprofile' . '/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_field'),
				'permission_callback' => array($this, 'bp_xprofile_permission'),
				// 'args'                => $this->get_activity_args(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		)
		);

	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema' => 'http://json-schema.org/draft-04/schema#',
			'title' => 'bp_xprofile',
			'type' => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id' => array(
					'description' => __('Unique identifier for the object.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'name' => array(
					'description' => __('xProfile field name.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'edit_value' => array(
					'description' => __('xProfile field edit value.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'value' => array(
					'description' => __('xProfile field value.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'current_user_can' => array(
					'description' => __('xProfile field permission.'),
					'type' => 'boolean',
					'context' => array('view', 'edit', 'embed'),
				),
				'description' => array(
					'description' => __('xProfile field description.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'type' => array(
					'description' => __('xProfile field type.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'required' => array(
					'description' => __('xProfile field is required or not.'),
					'type' => 'boolean',
					'context' => array('view', 'edit', 'embed'),
				),
				'visibility' => array(
					'description' => __('xProfile field visibility.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'options' => array(
					'description' => __('xProfile field option.'),
					'type' => 'array',
					'context' => array('view', 'edit', 'embed'),
				),
			),
		);

		return $this->add_additional_fields_schema($schema);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/xprofile/:id xProfile
	 * @apiName GetBuddyPressXprofile
	 * @apiGroup BuddyPressXprofile
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get xProfile of/for buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiDeprecated  Retrieve xProfile single Field. Check (#Profile Fields:GetBBxProfileField)
	 * @apiUse apidocForGetBuddyPressXprofileV1
	 * @apiPrivate
	 */
	public function get_field($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to get values.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$id = (int) $request['id'];

		$value = xprofile_get_field($id)->data;

		if ($value == null) {
			return new WP_Error('id_not_valid', __('Field id is not valid.', 'buddyboss-app'), array('status' => 404));
		}

		return xprofile_get_field($id)->data;

	}

	/**
	 * @param $request
	 * @return array|WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/xprofile/:id Update xprofile
	 * @apiName UpdateBuddyPressXprofile
	 * @apiGroup BuddyPressXprofile
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update xProfile of/for buddypress component
	 * @apiDeprecated  Update xProfile for user. Check (#Profile Fields:UpdateBBxProfile)
	 * @apiUse apidocForUpdateBuddyPressXprofileV1
	 * @apiPrivate
	 */
	public function update_fields($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to set values.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$displayed_user_id = $request->get_param('displayed_user_id');
		$user_id = $request->get_param('user_id');

		if (empty($user_id) && !empty($displayed_user_id)) {
			$user_id = $displayed_user_id;
		}

		if (empty($user_id)) {
			return new WP_Error('no_user_given', __('please mention user to update fields.', 'buddyboss-app'), array('status' => 404));
		}

		do_action('boss_rest_buddypress_xprofile_before_update_fields', $request);

		$fields = $request->get_param('fields');

		foreach ($fields as $field) {
			xprofile_set_field_data($field['id'], $user_id, $field['value']);
			xprofile_set_field_visibility_level($field['id'], $user_id, $field['visibility']);
		}

		do_action('boss_rest_buddypress_xprofile_after_update_fields', $request);

		$response_fields = $this->get_fields($request);

		return $response_fields;
	}

	/**
	 * bp_messages_permission function.
	 *
	 * allow permission to access data
	 *
	 * @access public
	 * @return void
	 */
	public function bp_xprofile_permission() {

		return apply_filters('boss_rest_buddypress_xprofile_permission', true);
	}

	/**
	 * @param $request
	 * @return array|WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/xprofile xProfiles
	 * @apiName GetAllBuddyPressXprofiles
	 * @apiGroup BuddyPressXprofile
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get all xProfile of/for buddypress's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} user_id
	 * @apiParam {Number} displayed_user_id
	 * @apiDeprecated  Retrieve Multiple xProfile Fields. Check (#Profile Fields:GetBBxProfileFields)
	 * @apiPrivate
	 */
	public function get_fields($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$displayed_user_id = $request->get_param('displayed_user_id');
		$user_id = $request->get_param('user_id');

		if (empty($user_id) && !empty($displayed_user_id)) {
			$user_id = $displayed_user_id;
		}

		if (empty($user_id)) {
			return new WP_Error('no_user_given', __('No displayed user.', 'buddyboss-app'), array('status' => 404));
		}

		do_action('boss_rest_buddypress_xprofile_before_get_fields', $request);

		$args = array(
			'user_id' => $user_id,
			'profile_group_id' => !empty($request['profile_group_id']) ? $request['profile_group_id'] : false,
			'hide_empty_groups' => !empty($request['hide_empty_groups']) ? $request['hide_empty_groups'] : false,
			'hide_empty_fields' => !empty($request['hide_empty_fields']) ? $request['hide_empty_fields'] : false,
			'fetch_fields' => !empty($request['fetch_fields']) ? $request['fetch_fields'] : true,
			'fetch_field_data' => !empty($request['fetch_field_data']) ? $request['fetch_field_data'] : true,
			'fetch_visibility_level' => !empty($request['fetch_visibility_level']) ? $request['fetch_visibility_level'] : true,
			'exclude_groups' => !empty($request['exclude_groups']) ? $request['exclude_groups'] : false,
			'exclude_fields' => !empty($request['exclude_fields']) ? $request['exclude_fields'] : false,
		);

		$response = array();
		if (bp_xprofile_get_settings_fields($args)) {

			while (bp_profile_groups()) {

				bp_the_profile_group();
				$response['field_groups'][bp_get_the_profile_group_id()]['name'] = bp_get_the_profile_group_name();

				while (bp_profile_fields()) {
					bp_the_profile_field();
					global $field;

					$current_field = array();
					//  xprofile_get_field( bp_get_the_profile_field_id())
					$current_field['id'] = bp_get_the_profile_field_id();
					$current_field['name'] = wp_specialchars_decode( bp_get_the_profile_field_name() );
					$current_field['edit_value'] = wp_specialchars_decode( bp_get_the_profile_field_edit_value() );
					if (bp_get_the_profile_field_type() == 'datebox') {
						$datebox_field = new BP_XProfile_Field_Type_Datebox();
						$current_field['settings']
						= $datebox_field::get_field_settings(bp_get_the_profile_field_id());
					}

					$type = bp_get_the_profile_field_type();

					$current_field['value'] = xprofile_get_field_data(bp_get_the_profile_field_id(), $user_id);

					if ('telephone' === $type || 'url' === $type) {
						$current_field['value'] = bp_get_the_profile_field_edit_value();
					}

					$current_field['current_user_can'] = bp_current_user_can('bp_xprofile_change_field_visibility');
					$current_field['description'] = bp_get_the_profile_field_description();

					$current_field['type'] = $type;

					if ('selectbox' == $type || 'multiselectbox' == $type || 'radio' == $type
						|| 'checkbox' == $type) {
						$current_field['options'] = $field->get_children();
					}

					if('membertypes' == $type && function_exists( 'bp_get_member_type_post_type' )) {
						$current_field['options'] = $this->get_member_type_options( $field, $request );
						// Get the profile type singular name same as platform.
						$current_field['value'] = get_post_meta( $current_field['edit_value'], '_bp_member_type_label_singular_name', true );
					}

					if ('datebox' == $type) {
						$current_field['options'][] = $this->get_date_field_options_array($field, array(
							'type' => 'day',
							'user_id' => $user_id,
						));
						$current_field['options'][] = $this->get_date_field_options_array($field, array(
							'type' => 'month',
							'user_id' => $user_id,
						));
						$current_field['options'][] = $this->get_date_field_options_array($field, array(
							'type' => 'year',
							'user_id' => $user_id,
						));
					}

					if ( isset( $current_field['value'] ) && ! empty( $current_field['value'] ) ) {
						if ( is_array( $current_field['value'] ) ) {
							$current_field['value'] = array_map( 'wp_specialchars_decode', $current_field['value'] );
						} else {
							$current_field['value'] = wp_specialchars_decode( $current_field['value'] );
						}
					}

					$current_field['required'] = bp_get_the_profile_field_is_required();
					$current_field['visibility'] = bp_get_the_profile_field_visibility_level();
					$response['field_groups'][bp_get_the_profile_group_id()]['fields'][] = $current_field;
				}
			}

			$visibility_levels = bp_xprofile_get_visibility_levels();
			$visibility_levels_arr = array();
			foreach ($visibility_levels as $level) {
				$visibility_levels_arr[$level['id']] = array('label' => $level['label'], 'value' => $level['id']);
			}
			$response['visibility_levels'] = $visibility_levels_arr;

		}

		do_action('boss_rest_buddypress_xprofile_after_get_fields', $request);

		return $response;
	}

	private function get_date_field_options_array($field, $args = array()) {
		$eng_months = array(
			'January',
			'February',
			'March',
			'April',
			'May',
			'June',
			'July',
			'August',
			'September',
			'October',
			'November',
			'December',
		);
		$options = array();

		// $type will be passed by calling function when needed.
		switch ($args['type']) {
		case 'day':
			for ($i = 1; $i < 32; ++$i) {
				$options[] = array("type" => "option", "name" => $i);
			}
			break;

		case 'month':
			for ($i = 0; $i < 12; ++$i) {
				$options[] = array("type" => "option", "name" => $eng_months[$i]);
			}
			break;

		case 'year':
			$settings = BP_XProfile_Field_Type_Datebox::get_field_settings($field->id);

			if ('relative' === $settings['range_type']) {
				$start = date('Y') + $settings['range_relative_start'];
				$end = date('Y') + $settings['range_relative_end'];
			} else {
				$start = $settings['range_absolute_start'];
				$end = $settings['range_absolute_end'];
			}

			for ($i = $end; $i >= $start; $i--) {
				$options[] = array("type" => "option", "name" => $i);
			}
			break;
		}

		return $options;
	}

    /**
     * Get Profile field options
     *
     * @param BP_XProfile_Field $field XProfile field object.
     * @param array             $request request argument.
     *
     * @return array
     */
    private function get_member_type_options( $field, $request ) {
        $posts = new \WP_Query(
            array(
                'posts_per_page' => - 1,
                'post_type'      => bp_get_member_type_post_type(),
                'orderby'        => 'title',
                'order'          => 'ASC',
            )
        );

        $member_type = bp_get_member_type( ! empty( $request['user_id'] ) ? (int) $request['user_id'] : get_current_user_id() );
        if ( '' !== $member_type ) {
            $post_selected = bp_member_type_post_by_type( $member_type );
        }

        $options = array();
        if ( $posts ) {
            foreach ( $posts->posts as $post ) {
                $enabled = get_post_meta( $post->ID, '_bp_member_type_enable_profile_field', true );
                $name    = get_post_meta( $post->ID, '_bp_member_type_label_singular_name', true );
                if ( '' === $enabled || '1' === $enabled ) {
                    $options[] = array(
                        'id'                => $post->ID,
                        'group_id'          => $field->group_id,
                        'parent_id'         => $field->id,
                        'type'              => 'option',
                        'name'              => $name,
                        'description'       => '',
                        'is_required'       => '0',
                        'is_default_option' => ( $post_selected === $post->ID ),
                    );
                }
            }
        }

        return $options;
    }

}
