<?php
namespace BuddyBossApp\Api\BuddyPress;
use BP_Messages_Box_Template;
use BP_Messages_Notice;
use BP_Messages_thread;
use stdClass;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Server;

defined('ABSPATH') || exit;
// NOTE : Old classname was class.boss_buddypress_api_messages_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Members Component
class MessagesRestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/buddypress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->rest_base = 'threads';
	}

	public function hooks() {

		add_action('rest_api_init', array($this, "register_routes"), 99);
	}

	/**
	 * Register the routes for the objects of the controller.
	 */
	public function register_routes() {

		register_rest_route($this->namespace, '/' . $this->rest_base, array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_items'),
				'permission_callback' => array($this, 'bp_messages_permission'),
				'args' => $this->get_items_args(),
			),
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'create_item'),
				'permission_callback' => array($this, 'bp_messages_permission'),
				'args' => $this->create_item_args(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));
		register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_item'),
				'permission_callback' => array($this, 'bp_messages_permission'),
				'args' => $this->get_item_args(),
			),
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'update_item'),
				'permission_callback' => array($this, 'bp_messages_permission'),
				'args' => array_merge($this->get_endpoint_args_for_item_schema(false), array(
					"message_id" => array(
						'description' => __('Id of message'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
					"action" => array(
						'type' => 'string',
						'default' => 'read',
						'enum' => array('read', 'unread', 'star', 'unstar'),
						'validate_callback' => 'rest_validate_request_arg',
					),
					"box" => array(
						'description' => __('box of message'),
						'default' => 'inbox',
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
				)),
			),
			array(
				'methods' => WP_REST_Server::DELETABLE,
				'callback' => array($this, 'delete_item'),
				'permission_callback' => array($this, 'bp_messages_permission'),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/notice/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'notice_action'),
				'permission_callback' => array($this, 'bp_messages_permission'),
				'args' => array_merge($this->get_endpoint_args_for_item_schema(false), array(
					"action" => array(
						'type' => 'string',
						'default' => 'activate',
						'enum' => array('activate', 'deactivate', 'delete'),
						'validate_callback' => 'rest_validate_request_arg',
					),
				)),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		//update a single MESSAGE
		register_rest_route($this->namespace, '/messages/action/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'do_action'),
				'permission_callback' => '__return_true',
				//'permission_callback' => array ( $this, 'bp_messages_permission' ),
				'args' => array_merge($this->get_endpoint_args_for_item_schema(false), array(
					"action" => array(
						'type' => 'string',
						'default' => 'star',
						'enum' => array('star', 'unstar'),
						'validate_callback' => 'rest_validate_request_arg',
					),
				)),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema' => 'http://json-schema.org/draft-04/schema#',
			'title' => 'bp_messages',
			'type' => 'object',
			/*
				             * Base properties for every Post.
			*/
			'properties' => array(
				'thread_id' => array(
					'description' => __('Unique identifier for the message.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'thread_url' => array(
					'description' => __('Link for the message.'),
					'type' => 'string',
					'format' => 'uri',
					'context' => array('view', 'edit', 'embed'),
				),
				'read' => array(
					'description' => __('Message read status.'),
					'type' => 'boolean',
					'context' => array('view', 'edit', 'embed'),
				),
				'total_messages' => array(
					'description' => __('Total message count.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'unread_messages' => array(
					'description' => __('Message unread status.'),
					'type' => 'boolean',
					'context' => array('view', 'edit', 'embed'),
				),
				'from' => array(
					'description' => __('Message sender data.'),
					'type' => 'array',
					'context' => array('view', 'edit', 'embed'),
				),
				'last_message_date' => array(
					'description' => __('Last message date.'),
					'type' => 'string',
					'format' => 'date-time',
					'context' => array('view', 'edit', 'embed'),
				),
				'subject' => array(
					'description' => __('Message subject.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'excerpt' => array(
					'description' => __('Message content excerpt.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'content' => array(
					'description' => __('Message content'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'avatar' => array(
					'description' => __('Avatar data for sender'),
					'type' => 'array',
					'context' => array('view', 'edit', 'embed'),
				),
				'to' => array(
					'description' => __('Recipients details.'),
					'type' => 'array',
					'context' => array('view', 'edit', 'embed'),
				),
				'star' => array(
					'description' => __('Message Stared or Un-stared.'),
					'type' => 'boolean',
					'context' => array('view', 'edit', 'embed'),
				),
			),
		);

		return $this->add_additional_fields_schema($schema);
	}

	/**
	 * bp_messages_permission function.
	 *
	 * allow permission to access data
	 *
	 * @access public
	 * @return void
	 */
	public function bp_messages_permission() {

		return apply_filters('boss_rest_buddypress_messages_permission', true);
	}

	/**
	 * Return args data for message endpoint
	 * helps for easily filter request data and validation
	 * @return array
	 */
	public function get_item_args() {

		$params = array(
			'context' => $this->get_context_param(array('default' => 'view')),
		);

		$params['box'] = array(
			'description' => __('Message folder.'),
			'type' => 'string',
			'default' => 'inbox',
			'enum' => array('inbox', 'sentbox', 'starred', 'notices'),
			'validate_callback' => 'rest_validate_request_arg',
		);

		return apply_filters('boss_rest_buddypress_message_args', $params);
	}

	/**
	 * Return args data for messages list endpoint
	 * helps for easily filter request data and validation
	 * @return array
	 */
	public function get_items_args() {

		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		unset($params['search']);

		$params['user_id'] = array(
			'description' => __('ID of the user you would like to message'),
			'type' => 'integer',
			'default' => 0,
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['box'] = array(
			'description' => __('Message folder.'),
			'type' => 'string',
			'default' => 'inbox',
			'enum' => array('inbox', 'sentbox', 'starred', 'notices'),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["search_terms"] = array(
			'description' => __('Limit results to those matching a string.'),
			'type' => 'string',
			'default' => "",
			'validate_callback' => 'rest_validate_request_arg',
		);

		// not included in api
		// @type string $page_arg     URL argument used for the pagination param.
		//                                 Default: 'mpage'.
		// @type array  $meta_query   Meta query arguments. Only applicable if $box is
		//                                 not 'notices'. See WP_Meta_Query more details.

		return apply_filters('boss_rest_buddypress_messages_args', $params);
	}

	public function create_item_args() {
		$args = array(
			"recipients" => array(
				'description' => __('Message recipients user ids'),
				'type' => 'array',
				'default' => array(),
				'items' => array(
					'type' => 'integer',
				),
				'sanitize_callback' => 'wp_parse_id_list',
				'validate_callback' => 'rest_validate_request_arg',
			),
			"thread_id" => array(
				'description' => __('Message thread id for reply'),
				'type' => 'integer',
				'default' => 0,
				'validate_callback' => 'rest_validate_request_arg',
			),
			"sender_id" => array(
				'description' => __('Message sender user id'),
				'type' => 'integer',
				'default' => 0,
				'validate_callback' => 'rest_validate_request_arg',
			),
			"subject" => array(
				'description' => __('Message Subject text for new message'),
				'type' => 'string',
				'default' => "",
				'validate_callback' => 'rest_validate_request_arg',
			),
			"content" => array(
				'description' => __('Message Content'),
				'type' => 'string',
				'default' => "",
				'validate_callback' => 'rest_validate_request_arg',
			),
			"date_sent" => array(
				'description' => __('Message Send date'),
				'type' => 'string',
				'default' => "",
				'validate_callback' => 'rest_validate_request_arg',
			),
		);

		return apply_filters('boss_rest_buddypress_create_message_args', $args);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {POST} /wp-json/appboss/buddypress/v1/threads Create thread
	 * @apiName CreateBuddypressMessage
	 * @apiGroup BuddyPressThreads
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Create thread for/of buddypress component
	 * @apiDeprecated  Create thread. Check (#Messages:CreateBBThread)
	 * @apiUse apidocForCreateBuddypressMessageV1
	 * @apiPrivate
	 */
	public function create_item($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$message = $this->prepare_item_for_database($request);

		$args = array(
			// 'sender_id'  => ($message->sender_id)?$message->sender_id:bp_loggedin_user_id(),
			'sender_id' => get_current_user_id(),
			'thread_id' => $message->thread_id, // false for a new message, thread id for a reply to a thread.
			'recipients' => $message->recipients, // Can be an array of usernames, user_ids or mixed.
			'subject' => $message->subject, // empty for a reply to a thread
			'content' => $message->content,
			'date_sent' => ($message->date_sent) ? $message->date_sent : bp_core_current_time(),
		);

		// return $args;

		$message_id = messages_new_message($args);

		if (!$message_id) {
			return new WP_Error('bp_json_message_create', __('Error creating new message.', BP_API_PLUGIN_SLUG), array('status' => 500));
		}

		//$this->update_additional_fields_for_object( $message, $request );

		/**
		 * Fires after a message is created via the REST API
		 *
		 * @param object $message Data used to create message
		 * @param WP_REST_Request $request Request object.
		 * @param bool $bool A boolean that is false.
		 */
		do_action('bp_json_insert_message', $message, $request, false);

		if ($message->thread_id) {
			$message_id = $message->thread_id;
		}

		$response = $this->get_item(array(
			'id' => $message_id,
			'context' => 'view',
			'box' => 'inbox',
		));

		$response = rest_ensure_response($response);
		$response->set_status(201);
		$response->header('Location', rest_url('/threads/' . $message_id));

		return $response;
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/threads/:id Update thread
	 * @apiName UpdateMessage
	 * @apiGroup BuddyPressThreads
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update thread for/of buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Thread(Message) Id
	 * @apiParam {Number} message_id Thread(Message) Id
	 * @apiParam {String=read,unread,star,unstar} [action=read] To be performed
	 * @apiParam {String} [box=inbox] box of message
	 * @apiDeprecated  Update thread. Check (#Messages:UpdateBBThread)
	 * @apiPrivate
	 */
	public function update_item($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$id = (int) $request['id'];

		if (!messages_check_thread_access($id)) {
			return new WP_Error('bp_json_message_invalid_id', __('Message ID is invalid.'), array('status' => 404));
		}

		switch ($request['action']) {
		case 'read':
			messages_mark_thread_read($id);
			break;
		case 'unread':
			messages_mark_thread_unread($id);
			break;
		case 'star':
			if (!empty($request['message_id'])) {
				$message_id = $request['message_id'];
			} else {
				$thread = new BP_Messages_thread($id);
				$mids = wp_list_pluck($thread->messages, 'id');
				$message_id = $mids[0];
			}
			bp_messages_star_set_action(array(
				'action' => 'star',
				'thread_id' => $id,
				'message_id' => $message_id,
				'user_id' => bp_loggedin_user_id(),
			));
			break;
		case 'unstar':
			$args = array(
				'action' => 'unstar',
				'thread_id' => $id,
				'user_id' => bp_loggedin_user_id(),
				'bulk' => true,
			);
			if (!empty($request['message_id'])) {
				$args['message_id'] = $request['message_id'];
			}
			bp_messages_star_set_action($args);
			break;
		}

		$response = $this->get_item(array(
			'id' => $id,
			'context' => 'view',
			'box' => $request['box'],
		));

		return rest_ensure_response($response);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {DELETE} /wp-json/appboss/buddypress/v1/threads/:id Delete thread
	 * @apiName DeleteMessage
	 * @apiGroup BuddyPressThreads
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Delete particular thread of/for buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Thread(Message) Id
	 * @apiDeprecated  Delete thread. Check (#Messages:DeleteBBThread)
	 * @apiPrivate
	 */
	public function delete_item($request) {
		$id = (int) $request['id'];

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (!messages_check_thread_access($id)) {
			return new WP_Error('messages_no_access', __('You can\'t access this thread.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (!bp_thread_has_messages(array('thread_id' => $id))) {
			return new WP_Error('bp_json_message', __('Message Not Found.', 'buddyboss-app'), array('status' => 404));
		}

		$deleted = messages_delete_thread($id);

		if (!$deleted) {
			return new WP_Error('bp_json_message_cannot_be_deleted', __('The message cannot be deleted.', 'buddyboss-app'), array('status' => 500));
		}

		return new WP_Error('bp_json_message_deleted', __('Message deleted successfully.', 'buddyboss-app'), array('status' => 200));
	}

	/**
	 * Prepare a single thread for create or update
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return object $message User object.
	 */
	protected function prepare_item_for_database($request) {
		$message = new stdClass;

		// required arguments.
		if (isset($request['recipients'])) {
			$message->recipients = $request['recipients'];
		}

		// optional arguments.
		if (isset($request['sender_id'])) {
			$message->sender_id = absint($request['sender_id']);
		}
		if (isset($request['thread_id'])) {
			$message->thread_id = absint($request['thread_id']);
		}
		if (isset($request['subject'])) {
			$message->subject = $request['subject'];
		}
		if (isset($request['content'])) {
			$message->content = $request['content'];
		}
		if (isset($request['date_sent'])) {
			$message->date_sent = $request['date_sent'];
		}

		/**
		 * Filter user data before inserting user via REST API
		 *
		 * @param object $message Message object.
		 * @param WP_REST_Request $request Request object.
		 */
		return apply_filters('bp_json_pre_insert_message', $message, $request);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/threads Threads
	 * @apiName GetAllMessages
	 * @apiGroup BuddyPressThreads
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get all threads of/for buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiParam {Number} [user_id=0] ID of the user you would like to message
	 * @apiParam {String=inbox,sentbox,starred,notices} [box=inbox] Message folder.
	 * @apiParam {String} [search_terms=""] Limit results to those matching a string.
	 * @apiDeprecated  Retrieve threads. Check (#Messages:GetBBThreads)
	 * @apiPrivate
	 */
	public function get_items($request) {

		global $messages_template;

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to get messages.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$args = array(
			'user_id' => !empty($request['user_id']) ? $request['user_id'] : bp_loggedin_user_id(),
			'box' => $request['box'],
			'per_page' => $request['per_page'],
			'page' => $request['page'],
			'search_terms' => $request['search_terms'],
		);

		if ($request['box'] == 'starred') {
			$args['box'] = 'inbox';
			$args['meta_query'] = array(array(
				'key' => 'starred_by_user',
				'value' => !empty($request['user_id']) ? $request['user_id'] : bp_loggedin_user_id(),
			));
		}

		$retval = array();
		if (bp_has_message_threads($args)) {
			if ('notices' == $args['box']) {
				add_filter('bp_current_action', array($this, 'temp_set_notices_as_bpaction'), 11);
			}

			while (bp_message_threads()): bp_message_thread();
				$retval[] = $this->prepare_response_for_collection(
					$this->prepare_thread_for_response($messages_template->thread, $request)
				);
			endwhile;

			if ('notices' == $args['box']) {
				remove_filter('bp_current_action', array($this, 'temp_set_notices_as_bpaction'), 11);
			}
		}

		$response = rest_ensure_response($retval);
		$response->header('X-WP-Total', (int) $messages_template->total_thread_count);

		return $response;
	}

	public function temp_set_notices_as_bpaction($action = '') {
		return 'notices';
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/threads/:id Thread
	 * @apiName GetMessage
	 * @apiGroup BuddyPressThreads
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get single thread of/for buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiHeader {Number} id Thread(Message) Id
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiParam {String=inbox,sentbox,starred,notices} [box=inbox] Message folder.
	 * @apiDeprecated  Retrieve single thread. Check (#Messages:GetBBThread)
	 * @apiPrivate
	 */
	public function get_item($request) {

		global $messages_template;

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to get messages.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (!messages_check_thread_access($request['id'])) {
			return new WP_Error('messages_no_access', __('You can\'t access this thread.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$box = !empty($request['box']) ? $request['box'] : 'inbox';

		$thread = new BP_Messages_Thread($request['id'], 'ASC', array(
			'update_meta_cache' => false,
		));

		if (!empty($thread)) {
			$messages_template = new BP_Messages_Box_Template(array('box' => $box));
			$messages_template->threads = array($thread);
			$messages_template->total_thread_count = 1;

			while (bp_message_threads()): bp_message_thread();
				$retval = $this->prepare_response_for_collection(
					$this->prepare_thread_for_response($thread, $request, true)
				);

				return rest_ensure_response($retval);
			endwhile;
		} else {
			return new WP_Error('invalid_thread', __('This thread no longer exists.', 'buddyboss-app'), array('status' => 404));
		}
	}

	public function prepare_thread_for_response($message_thread, $request) {

		if ('notices' == $request['box']) {
			//notices are different than others
			$data = array(
				'thread_id' => $message_thread->id,
				'read' => !empty($message_thread->is_active) ? true : false,
				'last_message_date' => boss_buddypress_rest_api_convert_date($message_thread->date_sent),
				'subject' => bp_get_message_notice_subject(),
				'content' => bp_get_message_notice_text(),
			);
		} else {

			$first_message = $message_thread->messages[0];

			$from = array(
				'id' => $first_message->sender_id,
				'name' => bp_core_get_user_displayname($first_message->sender_id),
				'url' => bp_core_get_user_domain($first_message->sender_id),
				'avatar' => html_entity_decode(bp_core_fetch_avatar(array(
					'item_id' => $first_message->sender_id,
					'avatar_dir' => 'avatars',
					'object' => 'user',
					'type' => 'thumb',
					'html' => false,
				))),
			);

			$data = array(
				'thread_id' => $message_thread->thread_id,
				'thread_url' => bp_get_message_thread_view_link($message_thread->thread_id, !empty($request['user_id']) ? $request['user_id'] : null),
				'read' => (bp_message_thread_has_unread()) ? false : true,
				'total_messages' => bp_get_message_thread_total_count($message_thread->thread_id),
				'unread_messages' => bp_get_message_thread_unread_count($message_thread->thread_id),
				'from' => $from,
				'last_message_date' => boss_buddypress_rest_api_convert_date($message_thread->last_message_date),
				'subject' => $message_thread->last_message_subject,
				'excerpt' => strip_tags(bp_create_excerpt($message_thread->last_message_content, 75, array('ending' => __('&hellip;', 'buddypress')))),
				'content' => $message_thread->last_message_content,
			);

//			$avatarFull = bp_core_fetch_avatar( array(
			//				'item_id'	 => $first_message->sender_id,
			//				'avatar_dir' => 'avatars',
			//				'object'	 => 'user',
			//				'type'		 => 'full',
			//				'html'		 => false
			//			) );
			//
			//			$avatarThumb = bp_core_fetch_avatar( array(
			//				'item_id'	 => $first_message->sender_id,
			//				'avatar_dir' => 'avatars',
			//				'object'	 => 'user',
			//				'type'		 => 'thumb',
			//				'html'		 => false
			//			) );
		}

		//$data[ "avatar" ] = null;

//		if ( !empty( $avatarFull ) ) {
		//
		//			$data[ "avatar" ] = array(
		//				"full"	 => html_entity_decode( $avatarFull ),
		//				"thumb"	 => html_entity_decode( $avatarThumb )
		//			);
		//		}

		// Recipients
		if (!empty($message_thread->recipients)) {
			foreach ($message_thread->recipients as $user => $userdata) {
				if ((int) $user !== bp_loggedin_user_id()) {
					$user_data = array(
						'id' => $user,
						'name' => bp_core_get_user_displayname($user),
						'url' => bp_core_get_user_domain($user),
						'avatar' => html_entity_decode(bp_core_fetch_avatar(array(
							'item_id' => $user,
							'avatar_dir' => 'avatars',
							'object' => 'user',
							'type' => 'thumb',
							'html' => false,
						))),
					);
					$data['to'][] = $user_data;
				}
			}
		}

		$messages = array();
		$is_starred = false;
		if (!empty($message_thread->messages)) {
			foreach ($message_thread->messages as $single_message) {
				$message = array(
					'id' => $single_message->id,
					'date_sent' => $single_message->date_sent,
					'message' => $single_message->message,
					'subject' => $single_message->subject,
					'sender' => array(
						'id' => $single_message->sender_id,
						'avatar' => html_entity_decode(bp_core_fetch_avatar(array(
							'item_id' => $single_message->sender_id,
							'avatar_dir' => 'avatars',
							'object' => 'user',
							'type' => 'thumb',
							'html' => false,
						))),
						'display_name' => bp_core_get_user_displayname($single_message->sender_id),
						'username' => bp_core_get_username($single_message->sender_id),
						'url' => bp_core_get_user_domain($single_message->sender_id),
					),
				);
				if (bp_is_active('messages', 'star')) {
					if (bp_messages_is_message_starred($single_message->id, bp_loggedin_user_id())) {
						$message['star'] = true;
						$is_starred = true;
					} else {
						$message['star'] = false;
					}
				}
				$messages[] = $message;
			}
		}

		if (isset($message_thread->messages_order) && 'ASC' == $message_thread->messages_order) {
			$messages = array_reverse($messages);
		}

		$data['messages'] = $messages;
		$data['star'] = $is_starred;

		$data = apply_filters('boss_rest_buddypress_message', $data, $request);

		$response = rest_ensure_response($data);
		$this->add_links($message_thread, $request, $response);

		return apply_filters('boss_rest_prepare_buddypress_message_response', $response, $request);
	}

	protected function _prepare_messages_for_response($thread) {
		$messages = $thread->messages;
		if (empty($messages)) {
			return false;
		}

		//we need to populate additional data about each message, like 'star' status
		//pull out individual message ids
		$message_ids = array();
		foreach ($messages as $key => $data) {
			$message_ids[] = (int) $messages[$key]->id;

			//set default value, to avoid another loop
			$messages[$key]->starred = false;
		}
		$message_ids_csv = implode(',', $message_ids);

		//single query to get star status of each message
		global $wpdb;
		$star_statuses = $wpdb->get_results(
			"SELECT message_id, meta_value FROM " . buddypress()->messages->table_name_meta . " WHERE message_id IN ( $message_ids_csv )"
		);

		foreach ($messages as $key => $data) {

			if (!empty($star_statuses && !is_wp_error($star_statuses))) {
				$is_starred = false;
				foreach ($star_statuses as $db_row) {
					if ($db_row->message_id == $messages[$key]->id && $db_row->meta_value == bp_loggedin_user_id()) {
						$is_starred = true;
						break;
					}
				}

				$messages[$key]->starred = $is_starred;
			}

			//get user information
			$messages[$key]->from = array(
				'id' => $data->sender_id,
				'name' => bp_core_get_user_displayname($data->sender_id),
				'email' => bp_core_get_user_email($data->sender_id),
				'url' => bp_core_get_user_domain($data->sender_id),
				'avatar' => bp_core_fetch_avatar(array(
					'item_id' => $data->sender_id,
					'avatar_dir' => 'avatars',
					'width' => 100,
					'height' => 100,
					'object' => 'user',
					'type' => 'thumb',
					'html' => false,
				)),
			);
		}

		return $messages;
	}

	/**
	 * Get a single message
	 *
	 * @param WP_REST_Request $request
	 * @return array|WP_Error
	 */
	public function get_message($request) {

		if (!bp_loggedin_user_id()) {
			return newWP_Error('not_logged_in', __('Please login to get messages.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		global $wpdb;
		$bp = buddypress();

		$message_id = $request['id'];
		$message = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$bp->messages->table_name_messages} WHERE id = %d ", $message_id));

		if (empty($message) || is_wp_error($message)) {
			return newWP_Error('invalid_message', __('This message no longer exists.', 'buddyboss-app'), array('status' => 404));
		}

		$thread_id = $message->thread_id;

		if (!messages_check_thread_access($thread_id)) {
			return newWP_Error('messages_no_access', __('You can\'t access this thread.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$message->starred = bp_messages_is_message_starred($message_id, bp_loggedin_user_id());
		return rest_ensure_response($message);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/messages/action/:id Update thread action
	 * @apiName UpdateMessageNotice
	 * @apiGroup BuddyPressThreads
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update thread-action of/for buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Thread(Message) Id
	 * @apiParam {String=star,unstar} [action=star] Action to be performed
	 * @apiDeprecated  Update starred thread. Check (#Messages:UpdateBBThreadStarred)
	 * @apiPrivate
	 */
	public function do_action($request) {
		$message_id = (int) $request['id'];

		global $wpdb;

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$thread_id = $wpdb->get_var($wpdb->prepare("SELECT thread_id FROM " . buddypress()->messages->table_name_messages . " WHERE id = %d", $message_id));

		if (!messages_check_thread_access($thread_id)) {
			return newWP_Error('messages_no_access', __('You can\'t access this thread.', 'buddyboss-app'), array('status' => 403));
		}

		switch ($request['action']) {
		case 'star':
		case 'unstar':
			bp_messages_star_set_action(array(
				'action' => $request['action'],
				'message_id' => $message_id,
				'user_id' => bp_loggedin_user_id(),
				'bulk' => false,
			));
			break;
		}

		$response = $this->get_message(array(
			'id' => $message_id,
		));

		return rest_ensure_response($response);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/threads/notice/:id Update thread notice
	 * @apiName UpdateMessageNotice
	 * @apiGroup BuddyPressThreads
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update message notice
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Thread(Message) Id
	 * @apiParam {String=activate,deactivate,delete} [action=activate] Action to be performed
	 * @apiDeprecated  Message Notice action.
	 * @apiPrivate
	 */
	public function notice_action($request) {

		global $messages_template;

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$notice_id = (int) $request['id'];
		$action = $request['action'];

		// Get the notice from database.
		$notice = new BP_Messages_Notice($notice_id);

		// Take action.
		switch ($action) {

		// Deactivate.
		case 'deactivate':
			$success = $notice->deactivate();
			$feedback = false === $success
			? __('There was a problem deactivating that notice.', 'buddyboss-app')
			: '';
			break;

		// Activate.
		case 'activate':
			$success = $notice->activate();
			$feedback = false === $success
			? __('There was a problem activating that notice.', 'buddyboss-app')
			: '';
			break;

		// Delete.
		case 'delete':
			$success = $notice->delete();
			$feedback = false === $success
			? __('There was a problem deleting that notice.', 'buddyboss-app')
			: '';
			break;
		}

		// Feedback.
		if (!empty($feedback)) {
			// Add feedback message.
			return new WP_Error('messages_error', $feedback, array('status' => 500));
		}

		$thread = $notice;

		if (!empty($thread)) {
			$messages_template = new BP_Messages_Box_Template(array('box' => 'notices'));
			$messages_template->threads = array($thread);
			$messages_template->total_thread_count = 1;

			add_filter('bp_current_action', array($this, 'temp_set_notices_as_bpaction'), 11);

			while (bp_message_threads()): bp_message_thread();
				$retval = $this->prepare_response_for_collection(
					$this->prepare_thread_for_response($thread, array('box' => 'notices'))
				);
				return rest_ensure_response($retval);
			endwhile;

			remove_filter('bp_current_action', array($this, 'temp_set_notices_as_bpaction'), 11);
		}

		return new WP_Error('messages_error', $feedback, array('status' => 500));

		//return $this->prepare_thread_for_response( $notice, array( 'box' => 'notices' ) );

	}

	/**
	 * Add links to request.
	 * @param $message_thread
	 * @param WP_REST_Request $request .
	 * @param $response
	 * @return void
	 */
	protected function add_links($message_thread, $request, $response) {

		$base = sprintf('/%s/%s', $this->namespace, $this->rest_base);

		$thread_id = isset($message_thread->thread_id) ? $message_thread->thread_id : $message_thread->id;

		// Entity meta.
		$links = array(
			'self' => array(
				'href' => rest_url($base . '/' . $thread_id),
			),
			'collection' => array(
				'href' => rest_url($base . '?user_id=' . bp_loggedin_user_id() . '&box=' . $request['box']),
			),
		);

		$user_domain = bp_loggedin_user_domain();
		if ($user_domain) {
			$slug = bp_get_messages_slug();
			$messages_link = trailingslashit($user_domain . $slug . '/compose');
			$response->add_link('compose_url', $messages_link);
		}

		// Add basic links
		$response->add_links($links);
	}

}
