<?php
namespace BuddyBossApp\Api\BuddyPress;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

defined('ABSPATH') || exit;

// NOTE : Old classname was class.boss_buddypress_api_members_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Members Component
class MembersRestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/buddypress/v1';
    protected $member_id = '';

    public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->rest_base = buddypress()->members->id;
		// @todo : Verify below line of code
		$this->rest_base = 'members';
	}

	public function hooks() {

		add_action("rest_api_init", array($this, "register_routes"), 99);

		add_filter("rest_prepare_user", array($this, "rest_prepare_user"), 10, 3);

	}

	/**
	 * Register the plugin routes.
	 */
	public function register_routes() {
		register_rest_route($this->namespace, '/' . $this->rest_base . '/details', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_page_details'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => array(),
			),
		)
		);

		register_rest_route($this->namespace, '/' . $this->rest_base, array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_items'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
				'args' => $this->get_members_args(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_item'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => array(
					'context' => $this->get_context_param(array('default' => 'view')),
				),
			),
			array(
				'methods' => WP_REST_Server::DELETABLE,
				'callback' => array($this, 'delete_item'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/slug/(?P<slug>\S+)', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_item_by_slug'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => array(
					'context' => $this->get_context_param(array('default' => 'view')),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/action/(?P<id>[\d]+)', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'do_action'),
				'permission_callback' => '__return_true',
				'args' => $this->get_members_action_args(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));
	}

	/**
	 *
	 * @param WP_REST_Response $response
	 *
	 * @return WP_REST_Response
	 */
	public function rest_prepare_user($response, $user, $request) {

		if (isset($response->data["id"])) {

			$user_id = (int) $response->data["id"];

			/**
			 * We add member single request on users API on member_data object.
			 */

			$request['id'] = $user_id;
			$member_response = $this->get_item( $request );

			if(isset($member_response->data)) {
                $response->data["member_rest"] = $member_response->data;
            }

		}

		return $response;

	}

	/**
	 * @param $url
	 *
	 * @return mixed|string
	 */
	public function fix_url_scheme($url) {

		if (substr($url, 0, 2) == "//") {
			$url = str_replace("//", "", $url);
		}

		$url = str_replace(array("http://", "https://"), "", $url);

		if (is_ssl()) {
			$url = "https://{$url}";
		} else {
			$url = "https://{$url}";
		}

		return $url;
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema' => 'http://json-schema.org/draft-04/schema#',
			'title' => 'bp_members',
			'type' => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id' => array(
					'description' => __('Unique identifier for the object.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'user_nicename' => array(
					'description' => __('Member Nicename.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'mention_name' => array(
					'description' => __('Member Mention name.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'user_email' => array(
					'description' => __('Email of Member.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'display_name' => array(
					'description' => __('Display of Member.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'fullname' => array(
					'description' => __('Full name of Member.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'friendship_status' => array(
					'description' => __('Friendship status of Member with current user.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'last_activity' => array(
					'description' => __('Last activity time.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'total_friend_count' => array(
					'description' => __('Total friend count of user.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'domain' => array(
					'description' => __('Permalink of member.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'private_msg_url' => array(
					'description' => __('Private message URL of member.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'action_links' => array(
					'description' => __('action links of member.'),
					'type' => 'array',
					'context' => array('view', 'edit', 'embed'),
				),
				'avatar_urls' => array(
					'description' => __('Avatar URL of member.'),
					'type' => 'array',
					'context' => array('view', 'edit', 'embed'),
				),
				'avatar' => array(
					'description' => __('Avatar of member.'),
					'type' => 'array',
					'context' => array('view', 'edit', 'embed'),
				),
				'cover_image' => array(
					'description' => __('Cover image of member.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'unread_messages_count' => array(
					'description' => __('Unread message count of member.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'unread_notifications_count' => array(
					'description' => __('Unread notification count of member.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'created_groups_count' => array(
					'description' => __('Group created count of member.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'latest_cover_activity' => array(
					'description' => __('latest cover activity id of member.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'latest_avatar_activity' => array(
					'description' => __('latest avatar activity id of member.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'followers' => array(
					'description' => __('followers count of member.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
			),
		);

		return $this->add_additional_fields_schema($schema);
	}

	/**
	 * Return args data for Members list endpoint
	 * helps for easily fiilter request data and validation
	 * @return array
	 */
	public function get_members_args() {

		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';
		$params['per_page']['default'] = 20;

		unset($params['search']);

		$params["scope"] = array(
			"description" => "Scope of member like all, friends.",
			'default' => 'all',
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["type"] = array(
			"description" => "Sort order. Active, newest, alphabetical, random or popular.",
			'type' => 'string',
			'default' => 'active',
			'enum' => array('active', 'newest', 'alphabetical', 'random', 'popular'),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["user_id"] = array(
			"description" => "Limit results to friends of a user.",
			'type' => 'integer',
			'default' => 0,
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["exclude"] = array(
			"description" => "IDs to exclude from results.",
			'type' => 'array',
			'default' => array(),
			'items' => array(
				'type' => 'integer',
			),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["include"] = array(
			"description" => "Limit results by user IDs.",
			'type' => 'array',
			'default' => array(),
			'items' => array(
				'type' => 'integer',
			),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["search_terms"] = array(
			"description" => "Limit to users matching search terms.",
			'type' => 'string',
			'default' => "",
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["meta_key"] = array(
			"description" => "Limit to users with a meta_key.",
			'type' => 'string',
			'default' => "",
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["meta_value"] = array(
			"description" => "Limit to users with a meta_value.",
			'type' => 'string',
			'default' => "",
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["member_type"] = array(
			"description" => "Limit results by member types.",
			'type' => 'array',
			'default' => array(),
			'items' => array(
				'type' => 'string',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["member_type__in"] = array(
			"description" => "Limit results by member types.",
			'type' => 'array',
			'default' => array(),
			'items' => array(
				'type' => 'string',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["member_type__not_in"] = array(
			"description" => "member types to be excluded from result.",
			'type' => 'array',
			'default' => array(),
			'items' => array(
				'type' => 'string',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["populate_extras"] = array(
			"description" => "Fetch optional extras.",
			'type' => 'boolean',
			'default' => true,
		);

		$params["count_total"] = array(
			"description" => "How to do total user count.",
			'type' => 'string',
			'default' => "count_query",
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	public function get_members_action_args() {
		$args = array(
			"action" => array(
				'type' => 'string',
				'enum' => apply_filters('boss_rest_buddypress_members_action_enum', array()),
				'validate_callback' => 'rest_validate_request_arg',
			),
		);

		return apply_filters('boss_rest_buddypress_members_action_args', $args);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/members Buddpress members
	 * @apiName GetBuddpressMembersItems
	 * @apiGroup BuddyPressMembers
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get Buddpress members items
	 * @apiDeprecated  Retrieve Members. Check (#Members:GetBBMembers)
	 * @apiUse apidocForGetBuddpressMembersItemsV1
	 * @apiPrivate
	 */
	public function get_items($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		// Parse the user query arguments.
		$args = array(
			'type' => isset($request['type']) ? $request['type'] : '',
			'user_id' => ($request['user_id']) ? $request['user_id'] : false,
			'exclude' => ($request['exclude']) ? $request['exclude'] : false,
			'search_terms' => ($request['search_terms']) ? $request['search_terms'] : false,
			'meta_key' => ($request['meta_key']) ? $request['meta_key'] : false,
			'meta_value' => ($request['meta_value']) ? $request['meta_value'] : false,
			'member_type' => $request['member_type'],
			'member_type__in' => $request['member_type__in'],
			'member_type__not_in' => $request['member_type__not_in'],
			'include' => ($request['include']) ? $request['include'] : array(),
			'per_page' => $request['per_page'],
			'page' => $request['page'],
			'populate_extras' => $request['populate_extras'],
			'count_total' => $request['count_total'],
		);

		if ($request->get_param('scope') == 'requests') {
			$args['user_id'] = !empty($args['user_id']) ? $args['user_id'] : bp_loggedin_user_id();
			if (empty($args['include'])) {
				$request_ids = bp_get_friendship_requests($args['user_id']);
				$args['include'] = $request_ids;
				unset($args['user_id']);
			}
		} elseif ($request->get_param('scope') == 'friends') {
			$args['user_id'] = !empty($args['user_id']) ? $args['user_id'] : bp_loggedin_user_id();
		}

		//BuddyBoss Platform exclude member type user support
		if ( in_array( $request->get_param('scope'), array( 'all' ) ) && function_exists( 'bp_get_removed_member_types' )  ){
            $bp_member_type_ids = bp_get_removed_member_types();
            // get removed profile type names/slugs
            $bp_member_type_names = array();
            if ( isset( $bp_member_type_ids ) && ! empty( $bp_member_type_ids ) ) {
                foreach ( $bp_member_type_ids as $single ) {
                    $bp_member_type_names[] = $single['name'];
                }
            }
            $args['member_type__not_in'] = $bp_member_type_names;
        }

		$args = apply_filters('boss_rest_buddypress_members_args', $args, $request);

		$members = bp_core_get_users($args);

		$retval = array();
		$members_array = array();
		foreach ($members['users'] as $member) {
			$members_array[] = $this->prepare_response_for_collection(
				$this->prepare_member_item_for_response($member, $request)
			);
		}

		$response = rest_ensure_response($members_array);
		$response->header('X-WP-Total', (int) $members['total']);

		return $response;
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/members/:id Member activity
	 * @apiName GetBuddypressMembersItem
	 * @apiGroup BuddyPressMembers
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get Buddypress members item
	 * @apiDeprecated  Retrieve single Member. Check (#Members:GetBBMember)
	 * @apiUse apidocForGetBuddypressMembersItemV1
	 * @apiPrivate
	 */
	public function get_item($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$member = bp_core_get_users(array(
			'include' => (int) $request['id'],
			'type' => '',
		));

		// bp_core_get_core_userdata((int) $request['id']);

		$retval = $this->prepare_member_item_for_response($member['users'][0], $request, true);

		return rest_ensure_response($retval);
	}

	/**
	 * @param $request
	 * @return bool|WP_Error
	 * @api {DELETE} /wp-json/appboss/buddypress/v1/members/:id Delete member activity
	 * @apiName DeleteMembersActivity
	 * @apiGroup BuddyPressMembers
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Delete members activity
	 * @apiHeader {String} accessToken Auth token
	 * @apiDeprecated  Delete a member. Check (#Members:DeleteBBMembers)
	 * @apiPrivate
	 */
	public function delete_item($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		// Bail if account deletion is disabled.
		if (bp_disable_account_deletion() && !bp_current_user_can('delete_users')) {
			return new WP_Error('account_deletion_disabled', __('Acccount deletion is disabled on the site.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if ($request['id'] != get_current_user_id() && !bp_current_user_can('delete_users')) {
			return new WP_Error('permission_error', __('You cannot delete other user\'s account.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		// Delete the users account.
		if (!bp_core_delete_account($request['id'])) {
			return new WP_Error('delete_error', __('There was an error deleting this account.', 'buddyboss-app'), array('status' => 500));
		}

		return true;

	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/members/slug/:slug Member activity by slug
	 * @apiName RetrieveMembersItemBySlug
	 * @apiGroup BuddyPressMembers
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Retrieve members item by slug
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {String} slug Member Slug
	 * @apiDeprecated  Retrieve single Member. Check (#Members:GetBBMember)
	 * @apiPrivate
	 */
	public function get_item_by_slug($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$user = get_user_by('slug', $request->get_param('slug'));
		if (!$user) {
			return new WP_Error('wrong_member_slug', __('Wrong member slug.', 'TEXTDOMAIN'), array('status' => 404));
		}

		$member = bp_core_get_users(array(
			'include' => (int) $user->ID,
			'type' => '',
		));

		// bp_core_get_core_userdata((int) $request['id']);

		$retval = $this->prepare_member_item_for_response($member['users'][0], $request, true);

		return rest_ensure_response($retval);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/members/details Member page details
	 * @apiName GetMembersPageDetails
	 * @apiGroup BuddyPressMembers
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get member page details, such as tabs and further
	 * @apiDeprecated  Retrieve Members details(includes tabs and order_options). Check (#Members:GetBBMembersDetails)
	 * @apiUse apidocForGetMembersPageDetailsV1
	 * @apiPrivate
	 */
	public function get_page_details($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$type = 'active';

		if (isset($request['type'])) {
			$type = $request['type'];
		}

		// NOTE : old way was => bp_get_total_member_count()
		$count = get_total_members($type);

		/**
		 * Get Navigation item for directory page
		 **/
		$tabs = array(
			'all' => array(
				'title' => __('All Members ', 'buddypress'),
				'count' => $count,
			),
		);

		if (is_user_logged_in()) {

			if (bp_is_active('friends')) {
				//NOTE : old way was => bp_get_total_friend_count(bp_loggedin_user_id()).
				$count = get_total_friends( bp_loggedin_user_id(), $type );

				/**
				 * Note :- Buddypress uses personal but friends look more logical.
				 */
				$tabs['friends'] = array(
					'title' => __('My Friends', 'buddypress'),
					'count' => $count,
				);
			}
		}

		$tabs = apply_filters('boss_rest_buddypress_members_directory_types', $tabs);

		// Order options
		$order_options['active'] = __('Last Active', 'buddypress');
		$order_options['newest'] = __('Newest Registered', 'buddypress');

		if (bp_is_active('xprofile')) {
			$order_options['alphabetical'] = __('Alphabetical', 'buddypress');
		}
		$order_options = apply_filters('boss_rest_buddypress_members_directory_order_options', $order_options);

		return rest_ensure_response(array(
			'tabs' => $tabs,
			'order_options' => $order_options,
		));
	}

	/**
	 * Check if a given request has access to get information about a specific activity.
	 * @param WP_REST_Request $request Full data about the request.
	 * @return bool
	 */
	public function get_item_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_members_item_permission', true);
	}

	/**
	 * Check if a given request has access to activity items.
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function get_items_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_members_items_permission', true);
	}

	/**
	 * Prepares member data for return as an object.
	 * @param WP_User $member Activity data.
	 * @param WP_REST_Request $request
	 * @param bool $show_details
	 * @param boolean $is_raw Optional, not used. Defaults to false.
	 * @return WP_REST_Response
	 */
	public function prepare_member_item_for_response($member, $request, $show_details = false, $is_raw = false) {

		/**
		 * Categories common objects values into array with desired name for API
		 */
		$data = array(
			'user_nicename' => $member->user_nicename,
			'user_email' => $member->user_email,
			'display_name' => $member->display_name,
			'mention_name' => function_exists( 'bp_activity_get_user_mentionname' ) ? bp_activity_get_user_mentionname( $member->ID ) : $member->user_nicename,
			'id' => $member->ID,
			'fullname' => isset($member->fullname) ? $member->fullname : '',
			'friendship_status' => isset($member->friendship_status) ? $member->friendship_status : '',
			'last_activity' => boss_buddypress_rest_api_convert_date($member->last_activity),
			'total_friend_count' => $member->total_friend_count,
			'domain' => bp_core_get_user_domain($member->ID),
		);

		if (bp_is_active('messages')) {
		    $this->member_id = $member->ID;
            add_filter('bp_displayed_user_id', array( $this, 'filter__bp_displayed_user_id' ) );
			$data['private_msg_url'] = bp_get_send_private_message_link();
            remove_filter('bp_displayed_user_id', array( $this, 'filter__bp_displayed_user_id' ) );
		}

		$action_links = array();
		if (bp_loggedin_user_id() && $member->id != bp_loggedin_user_id()) {
			if (bp_is_active('activity')) {
				$args = array('r' => bp_activity_get_user_mentionname($member->id));
				$url = add_query_arg($args, bp_get_activity_directory_permalink());
				$action_links['public_msg_url'] = array(
					'link_text' => __('Public Message', 'buddyboss-app'),
					'link_url' => wp_nonce_url($url),
				);
			}
		}
		$data['action_links'] = $action_links;

		/**
		 * online and offline check
		 */
		$current_time = current_time('mysql', 1);
		$diff = strtotime($current_time) - strtotime($member->last_activity);
		if ($diff < 300) {
			// 5 minutes  =  5 * 60
			$data['online'] = true;
		}

		// /**
		//  * ========== Inject Member Avatars ============
		//  */

		$data['avatar_urls'] = rest_get_avatar_urls($member->user_email);

		$avatarFull = bp_core_fetch_avatar(array(
			'item_id' => $member->id,
			'avatar_dir' => 'avatars',
			'object' => 'user',
			'type' => 'full',
			'html' => false,
		));

		$avatarThumb = bp_core_fetch_avatar(array(
			'item_id' => $member->id,
			'avatar_dir' => 'avatars',
			'object' => 'user',
			'type' => 'thumb',
			'html' => false,
		));

		$data["avatar"] = null;

		if (!empty($avatarFull)) {

			$data["avatar"] = array(
				"full" => html_entity_decode($avatarFull),
				"thumb" => html_entity_decode($avatarThumb),
			);
		}

		if(function_exists('bp_attachments_is_wp_version_supported')) {
			if (bp_is_active('xprofile', 'cover_image') && !bp_disable_cover_image_uploads() && bp_attachments_is_wp_version_supported()) {
				$data['cover_image'] = bp_attachments_get_attachment('url', array(
					'object_dir' => 'members',
					'item_id' => $member->id,
				));
			} else {
				$data['cover_image'] = '';
			}
		}

		if ($show_details) {
			/**
			 * ========= Inject Messages Information =========
			 */
			if (function_exists('messages_get_unread_count')) {
				$data["unread_messages_count"] = messages_get_unread_count($member->id);
			}

			/**
			 * ========= Inject Notifications Information =========
			 */
			if (function_exists('bp_notifications_get_unread_notification_count')) {
				$data["unread_notifications_count"] = bp_notifications_get_unread_notification_count($member->id);
			}

			/**
			 * ========= Inject Member Information =========
			 */
			if (bp_is_active("groups")) {
				global $bp, $wpdb;
				$data["created_groups_count"] = (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$bp->groups->table_name} WHERE creator_id = %d", $member->id));
			}
		}

		$context = !empty($request['context']) ? $request['context'] : 'view';
		$data = $this->add_additional_fields_to_object($data, $request);
		$data = apply_filters('boss_rest_buddypress_member_item', $data, $request, $member);
		//$data    = $this->filter_response_by_context( $data, $context ); // No need as we don't have context of each param
		$response = rest_ensure_response($data);
		$this->add_links($member, $response);

		/**
		 * Filter an activity value returned from the API.
		 * * */
		return apply_filters('boss_rest_prepare_buddypress_member_response', $response, $request);
	}

	/**
	 * Add links to request.
	 * @param $member
	 * @param $response
	 * @return void
	 */
	protected function add_links($member, $response) {

		$base = sprintf('/%s/%s/', $this->namespace, $this->rest_base);
		// Entity meta.
		$links = array(
			'self' => array(
				'href' => rest_url($base . $member->id),
			),
			'collection' => array(
				'href' => rest_url($base),
			),
			'details_url' => array(
				'href' => bp_core_get_user_domain($member->id),
			),
		);

		// Add basic links
		$response->add_links($links);

		// Groups
		$response->add_link('groups', rest_url('/' . $this->namespace_slug . '/buddypress/v1/groups/?order=desc&user_id=' . $member->id), array('embeddable' => true));
	}

	/**
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/members/action/:id Update members action
	 * @apiName UpdateMembersAction
	 * @apiGroup BuddyPressGroups
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update members action
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Unique identifier for member
	 * @apiParam {String} action Action to perform
	 * @apiDeprecated  Member action. Check (#Members:GetBBMembers-UpdateMembersAction)
	 * @apiPrivate
	 */
	public function do_action($request) {
		$member_id = (int) $request['id'];
		$action = $request['action'];

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$member = bp_core_get_core_userdata($member_id);

		// throw error when no member found.
		if (empty($member->ID)) {
			return new WP_Error('member_not_found', __('Requested member not found.', 'buddyboss-app'), array('status' => 404));
		}

		switch ($action) {
		default:
			do_action('boss_rest_buddypress_members_action', $request);
		}

		return $this->get_item($request);
	}

    /**
     * Fixed Private message url.
     * @param $user_id
     * @return int
     */
	public function filter__bp_displayed_user_id( $user_id ) {
        if( ! empty( $this->member_id ) ) {
            return $this->member_id;
        }
        return $user_id;
    }
}
