<?php

namespace BuddyBossApp\Api\BuddyPress;

use BuddyBossApp\Api\BuddyPress\MembersRestApi;
use BP_Groups_Group;
use BP_Group_Member_Query;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

defined('ABSPATH') || exit;

// NOTE : Old classname was class.boss_buddypress_api_groups_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Group Component
class GroupsRestApi extends WP_REST_Controller
{

    protected $namespace_slug = '';
    protected $namespace = '/buddypress/v1';

    public function __construct($slug)
    {
        /** Nothing here */
        $this->namespace_slug = $slug;
        $this->namespace = $this->namespace_slug . $this->namespace;
        $this->rest_base = buddypress()->groups->id;
        $this->rest_base = 'groups';
    }

    public function hooks()
    {

        add_action('rest_api_init', array($this, "register_routes"), 99);

        add_filter('boss_rest_buddypress_member_item', array($this, 'add_group_member_data'), 10, 3);

    }

    /**
     * Register the plugin routes.
     */
    public function register_routes()
    {
        register_rest_route($this->namespace, '/' . $this->rest_base . '/details', array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_page_details'),
                    'permission_callback' => array($this, 'get_item_permissions_check'),
                    'args' => array(),
                ),
            )
        );
        register_rest_route($this->namespace, '/' . $this->rest_base, array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_items'),
                'permission_callback' => array($this, 'get_items_permissions_check'),
                'args' => $this->get_groups_args(),
            ),
            'schema' => array($this, 'get_public_item_schema'),
        ));

        register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_item'),
                'permission_callback' => array($this, 'get_item_permissions_check'),
                'args' => array(
                    'context' => $this->get_context_param(array('default' => 'view')),
                ),
            ),
            array(
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => array($this, 'delete_item'),
                'permission_callback' => array($this, 'get_item_permissions_check'),
            ),
            'schema' => array($this, 'get_public_item_schema'),
        ));

        register_rest_route($this->namespace, '/' . $this->rest_base . '/slug/(?P<slug>\S+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_item_by_slug'),
                'permission_callback' => array($this, 'get_item_permissions_check'),
                'args' => array(
                    'context' => $this->get_context_param(array('default' => 'view')),
                ),
            ),
            'schema' => array($this, 'get_public_item_schema'),
        ));

        register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/members', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_members'),
                'permission_callback' => array($this, 'get_item_permissions_check'),
                'args' => array(
                    'context' => $this->get_context_param(array('default' => 'view')),
                    'exclude_admins' => array(
                        'description' => 'Exclude admin user from results.',
                        'type' => 'boolean',
                        'default' => false,
                        'validate_callback' => 'rest_validate_request_arg',
                    ),
                    'exclude_banned' => array(
                        'description' => 'Exclude banned user from results.',
                        'type' => 'boolean',
                        'default' => false,
                        'validate_callback' => 'rest_validate_request_arg',
                    ),
                    'per_page' => array(
                        'description' => __('Maximum number of items to be returned in result set.'),
                        'type' => 'integer',
                        'default' => 10,
                        'minimum' => 1,
                        'maximum' => 100,
                        'sanitize_callback' => 'absint',
                        'validate_callback' => 'rest_validate_request_arg',
                    ),
                    'page' => array(
                        'description' => __('Current page of the collection.'),
                        'type' => 'integer',
                        'default' => 1,
                        'sanitize_callback' => 'absint',
                        'validate_callback' => 'rest_validate_request_arg',
                        'minimum' => 1,
                    ),
                    'search_terms' => array(
                        'description' => 'Limit results to those matching a string.',
                        'type' => 'string',
                        'validate_callback' => 'rest_validate_request_arg',
                    ),
                    'group_role' => array(
                        'description' => 'Limit results to specific group role.',
                        'type' => 'string',
                        'validate_callback' => 'rest_validate_request_arg',
                    ),
                    'type' => array(
                        'description' => 'Sort the order of results.',
                        'type' => 'string',
                        'validate_callback' => 'rest_validate_request_arg',
                    ),
                ),
            ),
        ));

        register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/requests', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_member_requests'),
                'permission_callback' => array($this, 'get_item_permissions_check'),
                'args' => array(
                    'context' => $this->get_context_param(array('default' => 'view')),
                    'page' => array(
                        'description' => __('Current page of the collection.'),
                        'type' => 'integer',
                        'default' => 1,
                        'sanitize_callback' => 'absint',
                        'validate_callback' => 'rest_validate_request_arg',
                        'minimum' => 1,
                    ),
                    'per_page' => array(
                        'description' => __('Maximum number of items to be returned in result set.'),
                        'type' => 'integer',
                        'default' => 10,
                        'minimum' => 1,
                        'maximum' => 100,
                        'sanitize_callback' => 'absint',
                        'validate_callback' => 'rest_validate_request_arg',
                    ),
                ),
            ),
        ));

        register_rest_route($this->namespace, '/' . $this->rest_base . '/action/(?P<id>[\d]+)', array(
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'item_actions'),
                'permission_callback' => '__return_true',
                'args' => $this->get_group_action_args(),
            ),
            'schema' => array($this, 'get_public_item_schema'),
        ));

    }

    /**
     * Get the Post's schema, conforming to JSON Schema.
     *
     * @return array
     */
    public function get_item_schema()
    {
        $schema = array(
            '$schema' => 'http://json-schema.org/draft-04/schema#',
            'title' => 'bp_groups',
            'type' => 'object',
            /*
                 * Base properties for every Post.
            */
            'properties' => array(
                'id' => array(
                    'description' => __('Unique identifier for the Group.'),
                    'type' => 'integer',
                    'context' => array('view', 'edit', 'embed'),
                    'readonly' => true,
                ),
                'creator' => array(
                    'description' => __('Creator id of the group.'),
                    'type' => 'integer',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'name' => array(
                    'description' => __('Name for the group.'),
                    'type' => 'string',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'slug' => array(
                    'description' => __('Slug for the group.'),
                    'type' => 'string',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'description' => array(
                    'description' => __('Description of the group.'),
                    'type' => 'string',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'link' => array(
                    'description' => __('Link for the group.'),
                    'type' => 'string',
                    'format' => 'uri',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'status' => array(
                    'description' => __('Status for the group.'),
                    'type' => 'string',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'enable_forum' => array(
                    'description' => __('Whether forums enable for the group.'),
                    'type' => 'integer',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'forum' => array(
                    'description' => __('Forum id of the group.'),
                    'type' => 'integer',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'featured_media' => array(
                    'description' => __('Media id of the group forum'),
                    'type' => 'string',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'date_created' => array(
                    'description' => __('Date of group created.'),
                    'type' => 'string',
                    'format' => 'date-time',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'last_activity' => array(
                    'description' => __('Last activity date of group.'),
                    'type' => 'string',
                    'format' => 'date-time',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'can_invite' => array(
                    'description' => __('Current user can invite user for group.'),
                    'type' => 'boolean',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'is_member' => array(
                    'description' => __('Current user is member of group.'),
                    'type' => 'boolean',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'is_invited' => array(
                    'description' => __('Current user invited for group.'),
                    'type' => 'boolean',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'is_pending' => array(
                    'description' => __('Current user invitation pending of group.'),
                    'type' => 'boolean',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'is_admin' => array(
                    'description' => __('Current user is admin of group.'),
                    'type' => 'boolean',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'is_mod' => array(
                    'description' => __('Current user is moderator of group.'),
                    'type' => 'boolean',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'avatar' => array(
                    'description' => __('avatar of the group.'),
                    'type' => 'array',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'cover_image' => array(
                    'description' => __('Cover image of the group.'),
                    'type' => 'string',
                    'context' => array('view', 'edit', 'embed'),
                ),
                'members_count' => array(
                    'description' => __('Member count of the group.'),
                    'type' => 'integer',
                    'context' => array('view', 'edit', 'embed'),
                ),
            ),
        );

        return $this->add_additional_fields_schema($schema);
    }

    /**
     * Return args data for Group action endpoint
     * helps for easily fiilter request data and validation
     * @return array
     */
    public function get_group_action_args()
    {

        $args = array(
            "action" => array(
                'type' => 'string',
                'enum' => array('leave_group', 'join_group', 'accept_invite', 'reject_invite', 'delete_pending_request', 'request_membership', 'accept_membership', 'reject_membership'),
                'validate_callback' => 'rest_validate_request_arg',
            ),
        );

        return $args;
    }

    /**
     * Return args data for Group lists endpoint
     * helps for easily fiilter request data and validation
     * @return array
     */
    public function get_groups_args()
    {

        $params = parent::get_collection_params();

        $params['context']['default'] = 'view';

        $params["scope"] = array(
            "description" => "Scope of group like all, personal.",
            'default' => 'all',
            'type' => 'string',
            'validate_callback' => 'rest_validate_request_arg',
        );

        $params['type'] = array(
            'description' => __('Object sorting type.'),
            'type' => 'string',
            'default' => 'active',
            'enum' => array('active', 'newest', 'alphabetical', 'random', 'popular', 'most-forum-topics', 'most-forum-posts'),
            'validate_callback' => 'rest_validate_request_arg',
        );
        $params["order"] = array(
            'description' => __('Order sort attribute ascending or descending.'),
            'type' => 'string',
            'default' => 'desc',
            'enum' => array('asc', 'desc'),
            'validate_callback' => 'rest_validate_request_arg',
        );
        $params["orderby"] = array(
            'description' => __('Sort collection by object attribute.'),
            'type' => 'string',
            'default' => 'date_created',
            'enum' => array('date_created', 'last_activity', 'total_member_count', 'name', 'random'),
            'validate_callback' => 'rest_validate_request_arg',
        );
        $params["user_id"] = array(
            'description' => __('Limit results to specefic user.'),
            'type' => 'integer',
            'default' => 0,
            'validate_callback' => 'rest_validate_request_arg',
        );
        $params["include"] = array(
            'description' => __('Limit result set to specific ids.'),
            'type' => 'array',
            'default' => array(),
            'items' => array(
                'type' => 'integer',
            ),
            'sanitize_callback' => 'wp_parse_id_list',
            'validate_callback' => 'rest_validate_request_arg',
        );
        $params["exclude"] = array(
            'description' => __('Ensure result set excludes specific ids.'),
            'type' => 'array',
            'default' => array(),
            'items' => array(
                'type' => 'integer',
            ),
            'sanitize_callback' => 'wp_parse_id_list',
            'validate_callback' => 'rest_validate_request_arg',
        );
        $params["only_forum"] = array(
            'description' => __('Limit results of object which has forum'),
            'type' => 'integer',
            'default' => 0,
            'sanitize_callback' => 'absint',
            'validate_callback' => 'rest_validate_request_arg',
        );
        $params["extras"] = array(
            'type' => 'array',
            'description' => __('Shows Extras Values Depending Available (admins,mods) format array'),
            'default' => array(),
            'items' => array(
                'type' => 'string',
            ),
        );

        return $params;
    }

    /**
     * @param $request
     * @return array|WP_Error
     * @api {DELETE} /wp-json/appboss/buddypress/v1/groups/:id Delete group
     * @apiName DeleteGroup
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Delete buddypress group
     * @apiHeader {String} accessToken Auth token
     * @apiParam {String} id Group id
     * @apiDeprecated  Delete a group. Check (#Groups:DeleteBBGroup)
     * @apiPrivate
     */
    public function delete_item($request)
    {
        $group_id = $request['id'];

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        /**
         * Fires before the deletion of a group from the Delete Group page.
         *
         * @param int $id ID of the group being deleted.
         * @since 1.5.0
         *
         */
        do_action('groups_before_group_deleted', $group_id);

        // Group admin has deleted the group, now do it.
        if (!groups_delete_group($group_id)) {
            return new WP_Error('error', __('There was an error deleting the group. Please try again.', 'buddyboss-app'), array('status' => 500));
        }

        /**
         * Fires after the deletion of a group from the Delete Group page.
         *
         * @param int $id ID of the group being deleted.
         * @since 1.0.0
         *
         */
        do_action('groups_group_deleted', $group_id);

        return array('id' => $group_id, 'success' => true);
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {PATCH} /wp-json/appboss/buddypress/v1/groups/action/:id Update group actions
     * @apiName UpdateGroupsAction
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Update buddypress's group action
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id
     * @apiParam {String=leave_group,join_group,accept_invite,reject_invite,delete_pending_request,request_membership,accept_membership,reject_membership} action Action to perform
     * @apiDeprecated  Group action. Check (#Groups:AddBBGroupsMembers)
     * @apiPrivate
     */
    public function item_actions($request)
    {

        $group_id = (int)$request['id'];
        $action = $request['action'];

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group(array(
            'group_id' => (int)$group_id,
        ));

        // throw error when no group found.
        if (empty($group->id)) {
            return new WP_Error('group_not_found', __('Requested group not found.', 'buddyboss-app'), array('status' => 404));
        }

        switch ($action) {

            case "join_group":

                $join = groups_join_group($group_id);

                if (!$join) {
                    return new WP_Error('join_group_error', __('Error while joining group.', 'buddyboss-app'), array('status' => 500));
                }

                break;

            case "leave_group":

                $leave = groups_leave_group($group_id);

                if (!$leave) {
                    return new WP_Error('leave_group_error', __('Error while leaving group.', 'buddyboss-app'), array('status' => 500));
                }

                break;
            case "accept_invite":

                $accept = groups_accept_invite(get_current_user_id(), $group_id);

                if (!$accept) {
                    return new WP_Error('accept_invite_group_error', __('Error while accepting the invite for the group.', 'buddyboss-app'), array('status' => 500));
                }

                break;
            case "reject_invite":

                $reject = groups_reject_invite(get_current_user_id(), $group_id);

                if (!$reject) {
                    return new WP_Error('reject_invite_group_error', __('Error while leaving group.', 'buddyboss-app'), array('status' => 500));
                }

                break;
            case "accept_membership":

                $accept = groups_accept_membership_request(false, get_current_user_id(), $group_id);

                if (!$accept) {
                    return new WP_Error('accept_group_membership_error', __('Error while accepting the group request.', 'buddyboss-app'), array('status' => 500));
                }

                break;
            case "reject_membership":

                $reject = groups_reject_membership_request(false, get_current_user_id(), $group_id);

                if (!$reject) {
                    return new WP_Error('reject_group_membership_error', __('Error while rejecting the group request.', 'buddyboss-app'), array('status' => 500));
                }

                break;

            case "request_membership":

                $request_membership = groups_send_membership_request(get_current_user_id(), $group_id);

                if (!$request_membership) {
                    return new WP_Error('request_group_membership_error', __('Error while requesting group.', 'buddyboss-app'), array('status' => 500));
                }

                break;
            case "delete_pending_request":

                $pending_request = groups_delete_membership_request(false, get_current_user_id(), $group_id);

                if (!$pending_request) {
                    return new WP_Error('pending_request_group_error', __('Error while deleting pending request.', 'buddyboss-app'), array('status' => 500));
                }

                break;
        }

        /**
         * Return Response on Array as from starting app's were using it on array.
         */
        return $this->get_item($request, true);
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups Groups
     * @apiName GetBuddypressGroups
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get all groups (of buddypress component)
     * @apiDeprecated  Retrieve groups. Check (#Groups:GetBBGroups)
     * @apiUse apidocForGetBuddypressGroupsV1
     * @apiPrivate
     */
    public function get_items($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
        }

        $args = array();
        $args['type'] = $request['type'];
        $args['user_id'] = $request['user_id'];
        $args['order'] = strtoupper($request['order']);
        $args['orderby'] = $request['orderby'];
        $args['page'] = $request['page'];
        $args['per_page'] = $request['per_page'];
        $args['include'] = $request['include'];
        $args['exclude'] = $request['exclude'];
        $args['search_terms'] = $request['search'];

        if ($request->get_param('scope') == 'personal') {
            $args['user_id'] = bp_loggedin_user_id();
        }

        if ((!empty($args['user_id']) && $args['user_id'] == bp_loggedin_user_id()) || bp_current_user_can('bp_moderate')) {
            $args['show_hidden'] = true;
        }

        // Query only group has forums
        if ($request['only_forum'] == '1') {
            $args["meta_query"][] = array(
                'key' => 'forum_id',
                'value' => '',
                'compare' => '!=',
            );
        }

        $args = apply_filters('boss_rest_buddypress_group_args', $args, $request);

        if ('invites' == $request->get_param('scope')) {
            $groups = groups_get_invites_for_user($args['user_id'], $args['per_page'], $args['page']);
        } else {
            $groups = groups_get_groups($args);
        }

        $ret_groups = array();
        foreach ($groups['groups'] as $group) {

            if ('invites' == $request->get_param('scope')) {
                $group = new BP_Groups_Group($group->id);
            }

            $ret_groups[] = $this->prepare_response_for_collection(
                $this->prepare_group_item_for_response($group, $request)
            );
        }

        $response = rest_ensure_response($ret_groups);
        $response->header('X-WP-Total', (int)$groups['total']);

        return $response;
    }

    /**
     * @param $request
     * @param bool $return_array
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/:id Retrieve group
     * @apiName GetGroupsManageSettings
     * @apiGroup BuddyPressGroups
     * @apiDescription Group settings for buddypress component
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiHeader {String} accessToken Auth token
     * @apiParam {String=view,edit,embed} [context=view]
     * @apiParam {Number} id Group Id
     * @apiDeprecated  Retrieve single group. Check (#Groups:GetBBGroup)
     * @apiPrivate
     */
    public function get_item($request, $return_array = false)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
        }

        $group = groups_get_group(array(
            'group_id' => (int)$request['id'],
        ));

        if (empty($group->id)) {
            return new WP_Error('no_group', __('No group found.', 'buddyboss-app'), array('status' => 404));
        }

        $retval = $this->prepare_response_for_collection(
            $this->prepare_group_item_for_response($group, $request)
        );

        /**
         * Convert Response to Array.
         */
        if ($return_array) {
            $retval = array($retval);
        }

        return rest_ensure_response($retval);
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/slug/:slug Group item by slug
     * @apiName Groups
     * @apiGroup BuddyPressGroups
     * @apiDescription Get group-item by slug of/for buddypress component
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiHeader {String} accessToken Auth token
     * @apiParam {String} slug Slug of item to retrieve
     * @apiParam {String=view,edit,embed} [context=view]
     * @apiDeprecated  Retrieve single group. Check (#Groups:GetBBGroup)
     * @apiPrivate
     */
    public function get_item_by_slug($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
        }

        $group_id = groups_get_id($request->get_param('slug'));

        if (empty($group_id)) {
            return new WP_Error('no_group', __('No group found.', 'buddyboss-app'), array('status' => 404));
        }

        $group = groups_get_group(array(
            'group_id' => (int)$group_id,
        ));

        if (empty($group->id)) {
            return new WP_Error('no_group', __('No group found.', 'buddyboss-app'), array('status' => 404));
        }

        $retval = array($this->prepare_response_for_collection(
            $this->prepare_group_item_for_response($group, $request)
        ));

        return rest_ensure_response($retval);
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/details Groups page details
     * @apiName GetBpGroupsPageDetails
     * @apiGroup BuddyPressGroups
     * @apiDescription Get group's page details such as tabs and further
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDeprecated  Retrieve groups details(includes tabs and order_options). Check (#Groups:GetBBGroupsDetails)
     * @apiUse apidocForGetBpGroupsPageDetailsV1
     * @apiPrivate
     */
    public function get_page_details($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
        }

        $type = 'active';

        if (isset($request['type'])) {
            $type = $request['type'];
        }

        // NOTE : old way was => bp_get_total_group_count()
        $count = get_bp_total_groups($type);

        // Get Navigation item for directory page
        $tabs = array(
            'all' => array(
                'title' => __('All Groups', 'buddypress'),
                'count' => $count,
            ),
        );

        if (is_user_logged_in()) {
            if (bp_is_active('groups')) {
                // NOTE : old way was => bp_get_total_group_count_for_user()
                $count = get_bp_total_groups_for_user($type, bp_loggedin_user_id());

                $tabs['personal'] = array(
                    'title' => __('My Groups', 'buddypress'),
                    'count' => $count,
                );

            }
        }

        $tabs = apply_filters('boss_rest_buddypress_groups_directory_filter', $tabs);

        // Get order options
        $order_options = array(
            'active' => __('Last Active', 'buddypress'),
            'popular' => __('Most Members', 'buddypress'),
            'newest' => __('Newly Created', 'buddypress'),
            'alphabetical' => __('Alphabetical', 'buddypress'),
        );
        $order_options = apply_filters('boss_rest_buddypress_group_directory_order_options', $order_options);

        return rest_ensure_response(array(
            'tabs' => $tabs,
            'order_options' => $order_options,
        ));
    }

    /**
     * Check if a given request has access to get information about a specific activity.
     * @param WP_REST_Request $request Full data about the request.
     * @return bool
     */
    public function get_item_permissions_check($request)
    {
        return apply_filters('boss_rest_buddypress_groups_item_permission', true);
    }

    /**
     * Check if a given request has access to activity items.
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|bool
     */
    public function get_items_permissions_check($request)
    {
        return apply_filters('boss_rest_buddypress_groups_items_permission', true);
    }

    /**
     * Prepares group data for return as an object.
     * @param $group
     * @param WP_REST_Request $request
     * @param boolean $is_raw Optional, not used. Defaults to false.
     * @return WP_REST_Response
     */
    public function prepare_group_item_for_response($group, $request, $is_raw = false)
    {

        /**
         * Categories common objects values into array with desired name for API
         */
        $data = array(
            'id' => $group->id,
            'creator' => $group->creator_id,
            'name' => $group->name,
            'slug' => $group->slug,
            'description' => $group->description,
            'link' => bp_get_group_permalink($group),
            'status' => $group->status,
            'enable_forum' => (isset($group->enable_forum) && function_exists('bbpress')) ? $group->enable_forum : 0,
            'date_created' => boss_buddypress_rest_api_convert_date($group->date_created),
            'last_activity' => boss_buddypress_rest_api_convert_date($group->last_activity),
            'is_member' => ($group->__get("is_member")) ? true : false,
            'is_invited' => $group->__get("is_invited"),
            'is_pending' => $group->__get("is_pending"),
            'is_admin' => false,
            'is_mod' => false,
        );

        /**
         * ========== Invite Information  ============
         */
        $data["can_invite"] = false;
        if (is_user_logged_in()) {
            $invite_type = bp_group_get_invite_status($group->id);
            if ('admins' === $invite_type) {
                $data["can_invite"] = groups_is_user_admin(get_current_user_id(), $group->id);
            } elseif ('mods' === $invite_type) {
                $data["can_invite"] = groups_is_user_mod(get_current_user_id(), $group->id);
            } elseif ('members' === $invite_type) {
                $data["can_invite"] = groups_is_user_member(get_current_user_id(), $group->id);
            }
        }

        /**
         * ========== Inject Current User Role Information  ============
         */
        if (is_user_logged_in()) {
            $data["is_admin"] = groups_is_user_admin(get_current_user_id(), $group->id);
            $data["is_mod"] = groups_is_user_mod(get_current_user_id(), $group->id);
        }

        /**
         * ========== Inject Forum Information  ============
         */
        if ($group->enable_forum && function_exists('bbpress')) {
            $data["forum"] = groups_get_groupmeta($group->id, 'forum_id', true);
            if (is_array($data["forum"])) {
                if (!empty($data["forum"][0])) {
                    $data["forum"] = $data["forum"][0];
                } else {
                    $data["forum"] = 0;
                }
                $data['featured_media'] = (int)get_post_thumbnail_id($data["forum"]);
            }
        }

        /**
         * ========== Inject Group Avatars ============
         */
        $avatarFull = bp_core_fetch_avatar(array(
            'item_id' => $group->id,
            'avatar_dir' => 'group-avatars',
            'object' => 'group',
            'width' => 300,
            'height' => 300,
            'type' => 'full',
            'html' => false,
        ));

        $avatarThumb = bp_core_fetch_avatar(array(
            'item_id' => $group->id,
            'avatar_dir' => 'group-avatars',
            'object' => 'group',
            'width' => 300,
            'height' => 300,
            'type' => 'thumb',
            'html' => false,
        ));

        $data["avatar"] = null;

        if (!empty($avatarFull)) {

            $data["avatar"] = array(
                "full" => html_entity_decode($avatarFull),
                "thumb" => html_entity_decode($avatarThumb),
            );

        }

        //cover image of group
        $group_cover_image_url = bp_attachments_get_attachment('url', array(
            'object_dir' => 'groups',
            'item_id' => $group->id,
        ));

        $data['cover_image'] = $group_cover_image_url;

        /**
         * ========= Inject Member Information =========
         */
        $data["members_count"] = groups_get_total_member_count($group->id);

        /**
         * ========= Extras Information =========
         */
        // Group admins
        if ($request["extras"] && in_array("admins", $request["extras"])) {

            $vals = groups_get_group_admins($group->id);
            $data["admins"] = array();
            foreach ($vals as $val) {
                $data["admins"][] = $val->user_id;
            }
        }

        // Group Mods
        if ($request["extras"] && in_array("mods", $request["extras"])) {

            $vals = groups_get_group_mods($group->id);
            $data["mods"] = array();
            foreach ($vals as $val) {
                $data["mods"][] = $val->user_id;
            }
        }

        $context = !empty($request['context']) ? $request['context'] : 'view';
        $data = $this->add_additional_fields_to_object($data, $request);

        $data = apply_filters('boss_rest_buddypress_group_item', $data, $request);

        $response = rest_ensure_response($data);
        $this->add_links($group, $response);

        /**
         * Filter an activity value returned from the API.
         * * */
        return apply_filters('boss_rest_prepare_buddypress_group_response', $response, $request);
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/:id/members Groups members
     * @apiName GetGroupsMembers
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription All members of groups of/for buddypress component
     * @apiDeprecated  Retrieve group Members. Check (#Groups:GetBBGroupsMembers)
     * @apiUse apidocForAllFriendsV1
     * @apiPrivate
     */
    public function get_members($request)
    {
        if (!is_user_logged_in()) {
            return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
        }

        $member_args = array(
            'group_id' => $request['id'],
            'exclude_admins_mods' => !empty($request['exclude_admins']) ? $request['exclude_admins'] : false,
            'exclude_banned' => !empty($request['exclude_banned']) ? $request['exclude_banned'] : false,
            'per_page' => !empty($request['per_page']) ? $request['per_page'] : 20,
            'page' => !empty($request['page']) ? $request['page'] : 1,
            'search_terms' => !empty($request['search_terms']) ? $request['search_terms'] : false,
            'group_role' => !empty($request['group_role']) ? $request['group_role'] : array(),
            'type' => !empty($request['type']) ? $request['type'] : 'last_joined',
        );

        $members = groups_get_group_members($member_args);

        $members_array = array();
        $request['group_members'] = true;
        foreach ($members['members'] as $member) {
            $boss_rest_bp_members = new MembersRestApi($this->namespace_slug);
            $members_array[] = $this->prepare_response_for_collection(
                $boss_rest_bp_members->prepare_member_item_for_response($member, $request)
            );
        }

        $response = rest_ensure_response($members_array);
        $response->header('X-WP-Total', (int)$members['count']);

        return $response;
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/:id/requests Groups member requests
     * @apiName GetGroupsMemberRequests
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get Member Requests
     * @apiDeprecated  Retrieve group membership requests. Check (#Groups:GetBBGroupsMembershipsRequest)
     * @apiUse apidocForGetGroupsMemberRequestsV1
     * @apiPrivate
     */
    public function get_member_requests($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $is_admin = groups_is_user_admin(get_current_user_id(), $request['id']);
        $is_mod = groups_is_user_mod(get_current_user_id(), $request['id']);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => 500));
        }

        $mquery = new BP_Group_Member_Query(array(
            'group_id' => $request['id'],
            'type' => 'first_joined',
            'per_page' => !empty($request['per_page']) ? $request['per_page'] : 20,
            'page' => !empty($request['page']) ? $request['page'] : 1,
            'max' => false,

            // These filters ensure we only get pending requests.
            'is_confirmed' => false,
            'inviter_id' => 0,
        ));

        $members = array_values($mquery->results);
        $total_request_count = $mquery->total_users;

        $members_array = array();
        $request['group_members'] = true;
        $boss_rest_bp_members = new MembersRestApi($this->namespace_slug);
        if (!empty($members)) {
            foreach ($members as $member) {
                $members_array[] = $this->prepare_response_for_collection(
                    $boss_rest_bp_members->prepare_member_item_for_response($member, $request)
                );
            }
        }

        $response = rest_ensure_response($members_array);
        $response->header('X-WP-Total', (int)$total_request_count);

        return $response;
    }

    public function add_group_member_data($data, $request, $member)
    {

        if (isset($request['group_members'])) {
            $data['is_admin'] = !empty($member->is_admin) ? $member->is_admin : 0;
            $data['is_mod'] = !empty($member->is_mod) ? $member->is_mod : 0;
            $data['is_banned'] = !empty($member->is_banned) ? $member->is_banned : 0;
            $data['invite_sent'] = !empty($member->invite_sent) ? $member->invite_sent : 0;
            $data['inviter_id'] = !empty($member->inviter_id) ? $member->inviter_id : 0;
            $data['is_confirmed'] = !empty($member->is_confirmed) ? $member->is_confirmed : 0;
            $data['membership_id'] = !empty($member->membership_id) ? $member->membership_id : 0;
            $data['joined_date'] = !empty($member->date_modified) ? boss_buddypress_rest_api_convert_date($member->date_modified) : 0;
            $data['comments'] = !empty($member->comments) ? $member->comments : 0;
        }

        return $data;
    }

    /**
     * Add links to request.
     * @param $group
     * @param $response
     * @return void
     */
    protected function add_links($group, $response)
    {

        $base = sprintf('/%s/%s/', $this->namespace, $this->rest_base);
        // Entity meta.
        $links = array(
            'self' => array(
                'href' => rest_url($base . $group->id),
            ),
            'collection' => array(
                'href' => rest_url($base),
            ),
            'details_url' => array(
                'href' => bp_get_group_permalink($group),
            ),
        );

        // If we have a featured media, add that.
        if (isset($response->data["featured_media"])) {
            $image_url = rest_url('wp/v2/media/' . $response->data["featured_media"]);
            $links['https://api.w.org/featuredmedia'] = array(
                'href' => $image_url,
                'embeddable' => true,
            );
        }

        // Add basic links
        $response->add_links($links);

        $users = array($group->creator_id);

        if (isset($response->data["admins"])) {
            $users = array_merge($users, $response->data["admins"]);
        }
        if (isset($response->data["mods"])) {
            $users = array_merge($users, $response->data["admins"]);
        }

        $users = array_unique($users);

        // Users
        foreach ($users as $user_id) {
            $response->add_link('author', rest_url('/wp/v2/users/' . $user_id), array('embeddable' => true));
        }

        // Link Forum
        if (isset($response->data["forum"])) {
            $response->add_link('forum', rest_url('/' . $this->namespace_slug . '/bbpress/v1/forum/' . $response->data["forum"]), array('embeddable' => true));
        }

        $response->add_link('members', rest_url('/' . $this->namespace_slug . '/buddypress/v1/groups/' . $group->id . '/members'), array('embeddable' => true));
    }
}
