<?php

namespace BuddyBossApp\Api\BuddyPress;

use BuddyBossApp\Api\BuddyPress\MembersRestApi;
use BP_Attachment_Cover_Image;
use BP_Groups_Member;
use stdClass;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Server;

defined('ABSPATH') || exit;

// NOTE : Old classname was class.boss_buddypress_api_groups_manage_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Groups Manage Component
class GroupsManageRestApi extends WP_REST_Controller
{

    protected $namespace_slug = '';
    protected $namespace = '/buddypress/v1';

    public function __construct($slug)
    {
        /** Nothing here */
        $this->namespace_slug = $slug;
        $this->namespace = $this->namespace_slug . $this->namespace;
        $this->rest_base = buddypress()->groups->id;
        // @todo : Verify below line of code
        $this->rest_base = 'groups';
    }

    public function hooks()
    {
        add_action('rest_api_init', array($this, "register_routes"), 99);
    }

    /**
     * Register the routes for the objects of the controller.
     */
    public function register_routes()
    {
        register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>\d+)/manage/general', array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_general_details'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                ),
                array(
                    'methods' => WP_REST_Server::EDITABLE,
                    'callback' => array($this, 'update_general_details'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                    'args' => array(
                        'name' => array(
                            'description' => __('Name of group.'),
                            'type' => 'string',
                            'validate_callback' => 'rest_validate_request_arg',
                        ),
                        'description' => array(
                            'description' => __('Description of group.'),
                            'type' => 'string',
                            'validate_callback' => 'rest_validate_request_arg',
                        ),
                        'notify' => array(
                            'description' => __('Whether members notify for changes or not.'),
                            'type' => 'integer',
                            'validate_callback' => 'rest_validate_request_arg',
                        ),
                    ),
                ),
            )
        );

        register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>\d+)/manage/settings', array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_settings'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                ),
                array(
                    'methods' => WP_REST_Server::EDITABLE,
                    'callback' => array($this, 'update_settings'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                    'args' => array(
                        'group-show-forum' => array(
                            'description' => __('Whether the Group has a forum or not.'),
                            'type' => 'integer',
                            'validate_callback' => 'rest_validate_request_arg',
                        ),
                        'group-status' => array(
                            'description' => __('Status of group.'),
                            'type' => 'string',
                            'validate_callback' => 'rest_validate_request_arg',
                        ),
                        'group-invite-status' => array(
                            'description' => __('Invite status of group.'),
                            'type' => 'string',
                            'validate_callback' => 'rest_validate_request_arg',
                        ),
                        'group-types' => array(
                            'description' => __('Types of group.'),
                            'type' => 'string',
                            'validate_callback' => 'rest_validate_request_arg',
                        ),
                    ),
                ),
            )
        );

        if (function_exists('is_bbpress')) {

            register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>\d+)/manage/forum', array(
                    array(
                        'methods' => WP_REST_Server::READABLE,
                        'callback' => array($this, 'get_forum_settings'),
                        'permission_callback' => array($this, 'bp_groups_manage_permission'),
                    ),
                    array(
                        'methods' => WP_REST_Server::EDITABLE,
                        'callback' => array($this, 'update_forum_settings'),
                        'permission_callback' => array($this, 'bp_groups_manage_permission'),
                        'args' => array(
                            'bbp_edit_group_forum' => array(
                                'description' => __('Whether the Group has a forum or not.'),
                                'type' => 'boolean',
                                'validate_callback' => 'rest_validate_request_arg',
                            ),
                            'bbp_group_forum_id' => array(
                                'description' => __('Id of group forum.'),
                                'type' => 'integer',
                                'validate_callback' => 'rest_validate_request_arg',
                            ),
                        ),
                    ),
                )
            );
        }

        register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/manage/members/action', array(
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'item_actions'),
                'permission_callback' => '__return_true',
                'args' => $this->get_group_manage_action_args(),
            ),
        ));

        register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>\d+)/cover', array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_cover'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                ),
                array(
                    'methods' => WP_REST_Server::EDITABLE,
                    'callback' => array($this, 'update_cover'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                ),
                array(
                    'methods' => WP_REST_Server::DELETABLE,
                    'callback' => array($this, 'delete_cover'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                ),
            )
        );

        register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>\d+)/avatar', array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_avatar'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                ),
                array(
                    'methods' => WP_REST_Server::EDITABLE,
                    'callback' => array($this, 'update_avatar'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                ),
                array(
                    'methods' => WP_REST_Server::DELETABLE,
                    'callback' => array($this, 'delete_avatar'),
                    'permission_callback' => array($this, 'bp_groups_manage_permission'),
                ),
            )
        );

    }

    /**
     * Check if a given request has access to settings items.
     *
     * @return WP_Error|bool
     */
    public function bp_groups_manage_permission()
    {
        return apply_filters('boss_rest_buddypress_groups_manage_permission', true);
    }

    /**
     * Return args data for Group action endpoint
     * helps for easily fiilter request data and validation
     * @return array
     */
    public function get_group_manage_action_args()
    {

        $args = array(
            "user_id" => array(
                'description' => __('User id.'),
                'type' => 'integer',
                'validate_callback' => 'rest_validate_request_arg',
            ),
            "action" => array(
                'description' => __('Name of member action.'),
                'type' => 'string',
                'enum' => array('ban', 'unban', 'promote_to_mod', 'promote_to_admin', 'demote_to_member', 'remove', 'accept_membership', 'reject_membership'),
                'validate_callback' => 'rest_validate_request_arg',
            ),
            "membership_id" => array(
	            'description' => __('Group membership id.'),
	            'type' => 'integer',
	            'validate_callback' => 'rest_validate_request_arg',
            ),
        );

        return $args;
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/:id/manage/general Group general details
     * @apiName GetBpGroupsManageDetails
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get BP groups manage details
     * @apiDeprecated  Retrieve groups settings. Check (#Groups:GetBBGroupsSettings)
     * @apiUse apidocForGetBpGroupsManageDetailsV1
     * @apiPrivate
     */
    public function get_general_details($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group_id = $request['id'];

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        $response = array(
            'name' => $group->name,
            'description' => $group->description,
        );

        return rest_ensure_response($response);

    }

    /**
     * @param $request
     * @return WP_Error
     * @api {PATCH} /wp-json/appboss/buddypress/v1/groups/:id/manage/general Update group general details
     * @apiName UpdateGroupsManageDetails
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Update groups manage details
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id
     * @apiParam {String} name Name of group.
     * @apiParam {String} description Description of group.
     * @apiParam {Number} notify Whether members notify for changes or not.
     * @apiDeprecated  Update Group settings. Check (#Groups:UpdateBBGroupsSettings)
     * @apiPrivate
     */
    public function update_general_details($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group_id = $request['id'];

        // you can change if you are admin or mod
        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        $group_notify_members = isset($request['notify']) ? (int)$request['notify'] : 0;

        // Name and description are required and may not be empty.
        if (empty($request['name']) || empty($request['description'])) {
            return new WP_Error('no_data', __('Groups must have a name and a description. Please try again.', 'buddyboss-app'), array('status' => 500));
        } elseif (!groups_edit_base_group_details(array(
            'group_id' => $group_id,
            'name' => $request['name'],
            'slug' => null,
            'description' => $request['description'],
            'notify_members' => $group_notify_members,
        ))) {
            return new WP_Error('error', __('There was an error updating group details. Please try again.', 'buddyboss-app'), array('status' => 500));
        }

        /**
         * Fires before the redirect if a group details has been edited and saved.
         *
         * @param int $id ID of the group that was edited.
         * @since 1.0.0
         *
         */
        do_action('groups_group_details_edited', $group_id);

        $response = array(
            'name' => $request['name'],
            'description' => $request['description'],
        );

        return rest_ensure_response($response);
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/:id/manage/settings Group settings
     * @apiName GetBpGroupsSettings
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get settings for particular group of/for buddypress component
     * @apiDeprecated  Retrieve groups settings. Check (#Groups:GetBBGroupsSettings)
     * @apiUse apidocForGetBpGroupsSettingsV1
     * @apiPrivate
     */
    public function get_settings($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group_id = $request['id'];

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        $response = array();

        if (bp_is_active('forums') && bp_forums_is_installed_correctly()) {
            $response['enable_forum'] = $group->enable_forum;
        }

        $response['status'] = $group->status;

        $response['types'] = bp_groups_get_group_types(array('show_in_create_screen' => true), 'objects');
        $response['current_types'] = bp_groups_get_group_type($group_id, false);

        $response['invite_status'] = bp_group_get_invite_status($group_id);

        return rest_ensure_response($response);

    }

    /**
     * @param $request
     * @return WP_Error
     * @api {PATCH} /wp-json/appboss/buddypress/v1/groups/:id/manage/settings Update group settings
     * @apiName UpdateGroupsManageSettings
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Update group settings for buddypress component
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id
     * @apiParam {Number=1,0} [group-show-forum=0] Whether the Group has a forum or not.
     * @apiParam {String=public,private,hidden} group-status Status of group.
     * @apiParam {String=members,mods,admins} group-invite-status Invite status of group.
     * @apiParam {String} group-types Types of group.
     * @apiDeprecated  Update Group settings. Check (#Groups:UpdateBBGroupsSettings)
     * @apiPrivate
     */
    public function update_settings($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group_id = $request['id'];

        // you can change if you are admin or mod
        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        $enable_forum = (isset($request['group-show-forum'])) ? 1 : 0;

        // Checked against a whitelist for security.
        /** This filter is documented in bp-groups/bp-groups-admin.php */
        $allowed_status = apply_filters('groups_allowed_status', array('public', 'private', 'hidden'));
        $status = (in_array($request['group-status'], (array)$allowed_status)) ? $request['group-status'] : 'public';

        // Checked against a whitelist for security.
        /** This filter is documented in bp-groups/bp-groups-admin.php */
        $allowed_invite_status = apply_filters('groups_allowed_invite_status', array('members', 'mods', 'admins'));
        $invite_status = isset($request['group-invite-status']) && in_array($request['group-invite-status'], (array)$allowed_invite_status) ? $request['group-invite-status'] : 'members';

        /*
             * Save group types.
             *
             * Ensure we keep types that have 'show_in_create_screen' set to false.
        */
        $current_types = bp_groups_get_group_type($group_id, false);
        $current_types = array_intersect(bp_groups_get_group_types(array('show_in_create_screen' => false)), (array)$current_types);
        if (isset($request['group-types'])) {
            $current_types = array_merge($current_types, $request['group-types']);

            // Set group types.
            bp_groups_set_group_type($group_id, $current_types);

            // No group types checked, so this means we want to wipe out all group types.
        } else {
            /*
                 * Passing a blank string will wipe out all types for the group.
                 *
                 * Ensure we keep types that have 'show_in_create_screen' set to false.
            */
            $current_types = empty($current_types) ? '' : $current_types;

            // Set group types.
            bp_groups_set_group_type($group_id, $current_types);
        }

        if (!groups_edit_group_settings($group_id, $enable_forum, $status, $invite_status)) {
            return new WP_Error('error', __('There was an error updating group settings. Please try again.', 'buddyboss-app'), array('status' => 500));
        }

        /**
         * Fires before the redirect if a group settings has been edited and saved.
         *
         * @param int $id ID of the group that was edited.
         * @since 1.0.0
         *
         */
        do_action('groups_group_settings_edited', $group_id);

        $response = array();

        if (bp_is_active('forums') && bp_forums_is_installed_correctly()) {
            $response['enable_forum'] = $group->enable_forum;
        }

        $response['status'] = $group->status;

        $response['types'] = bp_groups_get_group_types(array('show_in_create_screen' => true), 'objects');
        $response['current_types'] = bp_groups_get_group_type($group_id, false);

        $response['invite_status'] = bp_group_get_invite_status($group_id);

        return rest_ensure_response($response);

    }

    /**
     * @param $request
     * @return array|WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/:id/manage/forum Group forum settings
     * @apiName GetBpGroupsForumSettings
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get Forum settings (from groups of buddypress component)
     * @apiDeprecated  Retrieve groups settings. Check (#Groups:GetBBGroupsSettings)
     * @apiUse apidocForGetBpGroupsForumSettingsV1
     * @apiPrivate
     */
    public function get_forum_settings($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        if (!function_exists('is_bbpress')) {
            return new WP_Error('error', __('bbPress is not installed.', 'buddyboss-app'), array('status' => 404));
        }

        $group_id = $request['id'];

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 400));
        }

        $response = array();

        $response['bbp_edit_group_forum'] = false;
        if (!empty($group->enable_forum)) {
            $response['bbp_edit_group_forum'] = $group->enable_forum;
        }

        $forum_id = 0;
        $forum_ids = array_values(bbp_get_group_forum_ids($group_id));

        // Normalize group forum relationships now
        if (!empty($forum_ids)) {

            // Loop through forums, and make sure they exist
            foreach ($forum_ids as $forum_id) {

                // Look for forum
                $forum = bbp_get_forum($forum_id);

                // No forum exists, so break the relationship
                if (empty($forum)) {
                    bbp_remove_forum_id_from_group($group_id, $forum_id);
                    bbp_remove_group_id_from_forum($forum_id, $group_id);
                    unset($forum_ids[$forum_id]);
                }
            }

            // No support for multiple forums yet
            $forum_id = (int)(is_array($forum_ids) ? $forum_ids[0] : $forum_ids);
        }

        $response['bbp_group_forum_id'] = $forum_id;

        return $response;

    }

    /**
     * @param $request
     * @return array|WP_Error
     * @api {PATCH} /wp-json/appboss/buddypress/v1/groups/:id/manage/forum Update group forum settings
     * @apiName UpdateGroupForumSettings
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Update group forum settings
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id(bbp_group_forum_id) of group forum.
     * @apiParam {Boolean=true,false} [bbp_edit_group_forum=false] Whether the Group has a forum or not.
     * @apiParam {Array} [bbp_group_forum_id=array()] Forum id(s)
     * @apiDeprecated  Update Group settings. Check (#Groups:UpdateBBGroupsSettings)
     * @apiPrivate
     */
    public function update_forum_settings($request)
    {

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        if (!function_exists('is_bbpress')) {
            return new WP_Error('error', __('bbPress is not installed.', 'buddyboss-app'), array('status' => 404));
        }

        $group_id = $request['id'];

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $edit_forum = !empty($request['bbp_edit_group_forum']) ? true : false;

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        // Keymasters have the ability to reconfigure forums
        if (bbp_is_user_keymaster()) {
            $forum_ids = !empty($request['bbp_group_forum_id']) ? (array)(int)$request['bbp_group_forum_id'] : array();

            // Use the existing forum IDs
        } else {
            $forum_ids = array_values(bbp_get_group_forum_ids($group_id));
        }

        $forum_id = 0;
        // Normalize group forum relationships now
        if (!empty($forum_ids)) {

            // Loop through forums, and make sure they exist
            foreach ($forum_ids as $forum_id) {

                // Look for forum
                $forum = bbp_get_forum($forum_id);

                // No forum exists, so break the relationship
                if (empty($forum)) {
                    bbp_remove_forum_id_from_group($group_id, $forum_id);
                    bbp_remove_group_id_from_forum($forum_id, $group_id);
                    unset($forum_ids[$forum_id]);
                }
            }

            // No support for multiple forums yet
            $forum_id = (int)(is_array($forum_ids) ? $forum_ids[0] : $forum_ids);
        }

        // Update the group ID and forum ID relationships
        bbp_update_group_forum_ids($group_id, (array)$forum_ids);
        bbp_update_forum_group_ids($forum_id, (array)$group_id);

        // Update the group forum setting
        // Set forum enabled status
        $group->enable_forum = (int)$edit_forum;

        // Save the group
        $group->save();

        // Maybe disconnect forum from group
        if (empty($edit_forum)) {
            bbp_remove_forum_id_from_group($group_id, $forum_id);
            bbp_remove_group_id_from_forum($forum_id, $group_id);
        }

        // Update bbPress' internal private and forum ID variables
        bbp_repair_forum_visibility();

        // Create a new forum
        if (empty($forum_id) && (true === $edit_forum)) {

            // Set the default forum status
            switch ($group->status) {
                case 'hidden':
                    $status = bbp_get_hidden_status_id();
                    break;
                case 'private':
                    $status = bbp_get_private_status_id();
                    break;
                case 'public':
                default:
                    $status = bbp_get_public_status_id();
                    break;
            }

            // Create the initial forum
            $forum_id = bbp_insert_forum(array(
                'post_parent' => bbp_get_group_forums_root_id(),
                'post_title' => $group->name,
                'post_content' => $group->description,
                'post_status' => $status,
            ));

            // Run the BP-specific functions for new groups
            bbp_add_forum_id_to_group($group_id, $forum_id);
            bbp_add_group_id_to_forum($forum_id, $group_id);
        }

        $response = array('bbp_edit_group_forum' => $edit_forum ? 1 : 0, 'bbp_group_forum_id' => $edit_forum ? $forum_id : 0);

        return $response;

    }

    /**
     * @param $request
     * @return array|WP_Error
     * @api {DELETE} /wp-json/appboss/buddypress/v1/groups/:id/avatar Delete an avatar.
     * @apiName DeleteGroupsManageAvatar
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Delete groups manage avatar
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id
     * @apiDeprecated  Delete group avatar. Check (#Groups:DeleteBBGroupAvatar)
     * @apiPrivate
     */
    public function delete_avatar($request)
    {

        $args = $request->get_params();

        $group_id = $request['id'];

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        if (bp_core_delete_existing_avatar($args)) {
            return array(
                'avatar' => '',
                'feedback_code' => 3,
                'item_id' => $request['id'],
            );
        } else {
            return new WP_Error('error', __('There was a problem deleting that avatar. Please try again.', 'buddyboss-app'), array('status' => 200));
        }
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/:id/avatar Group avatar
     * @apiName GetGroupsAvatar
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get Avatar (from groups of buddypress component)
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id
     * @apiDeprecated  Retrieve group avatar. Check (#Groups:GetBBGroupAvatar)
     * @apiPrivate
     */
    public function get_avatar($request)
    {

        $group_id = $request['id'];

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || is_super_admin())) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        return bp_core_fetch_avatar(array(
            'object' => 'group',
            'item_id' => $group_id,
            'html' => false,
        ));
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {PATCH} /wp-json/appboss/buddypress/v1/groups/:id/avatar Update group avatar
     * @apiName UpdateGroupsAvatar
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription @since 2.3.0. Update(upload) avatar for particular group of/for buddypress component
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id
     * @apiDeprecated  Create group avatar. Check (#Groups:CreateBBGroupAvatar)
     * @apiPrivate
     */
    public function update_avatar($request)
    {

        $group_id = $request['id'];

        // you can change if you are admin or mod
        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        // Init the BuddyPress parameters.
        $bp_params = array(
            'object' => 'group',
            'item_id' => $group_id,
        );

        // We need the object to set the uploads dir filter.
        if (empty($bp_params['object'])) {
            return new WP_Error('no_args', __('No arguments', 'buddyboss-app'), array('status' => 500));
        }

        // Capability check.
        if (!bp_attachments_current_user_can('edit_avatar', $bp_params)) {
            return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => 500));
        }

        $bp = buddypress();
        $bp_params['upload_dir_filter'] = 'groups_avatar_upload_dir';
        $bp->groups->current_group = $group;
        $needs_reset = array(
            'component' => 'groups',
            'key' => 'current_group',
            'value' => $bp->groups->current_group,
        );

        if (!isset($bp->avatar_admin)) {
            $bp->avatar_admin = new stdClass();
        }

        /**
         * The BuddyPress upload parameters is including the Avatar UI Available width,
         * add it to the avatar_admin global for a later use.
         */
        if (isset($bp_params['ui_available_width'])) {
            $bp->avatar_admin->ui_available_width = (int)$bp_params['ui_available_width'];
        }

        // upload will be rejected without this
        add_filter('bp_attachment_upload_overrides', array($this, 'change_overrides'));

        // Upload the avatar.
        $avatar = bp_core_avatar_handle_upload($_FILES, $bp_params['upload_dir_filter']);

        // Reset objects.
        if (!empty($needs_reset)) {
            if (!empty($needs_reset['component'])) {
                $bp->{$needs_reset['component']}->{$needs_reset['key']} = $needs_reset['value'];
            } else {
                $bp->{$needs_reset['key']} = $needs_reset['value'];
            }
        }

        // Init the feedback message.
        $feedback_message = false;

        if (!empty($bp->template_message)) {
            $feedback_message = $bp->template_message;

            // Remove template message.
            $bp->template_message = false;
            $bp->template_message_type = false;

            @setcookie('bp-message', false, time() - 1000, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
            @setcookie('bp-message-type', false, time() - 1000, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
        }

        if (empty($avatar)) {
            // Default upload error.
            $message = __('Upload failed.', 'buddyboss-app');

            // Use the template message if set.
            if (!empty($feedback_message)) {
                $message = $feedback_message;
            }

            // Upload error reply.
            return new WP_Error('upload_error', $message, array('status' => 500));
        }

        if (empty($bp->avatar_admin->image->file)) {
            return new WP_Error('upload_error', __('Upload failed.', 'buddyboss-app'), array('status' => 500));
        }

        $uploaded_image = @getimagesize($bp->avatar_admin->image->file);

        // Set the name of the file.
        $name = $_FILES['file']['name'];
        $name_parts = pathinfo($name);
        $name = trim(substr($name, 0, -(1 + strlen($name_parts['extension']))));

        // Further code is copied from bp_avatar_ajax_set, reall croping will be done in client app

        $avatar_data = array(
            'original_file' => $bp->avatar_admin->image->url,
            'crop_w' => $uploaded_image[0],
            'crop_h' => $uploaded_image[1],
            'crop_x' => 0,
            'crop_y' => 0,
            'item_id' => $bp_params['item_id'],
            'object' => $bp_params['object'],
            'type' => 'crop',
        );

        if (empty($avatar_data['object']) || empty($avatar_data['item_id']) || empty($avatar_data['original_file'])) {
            return new WP_Error('no_args', __('No arguments.', 'buddyboss-app'), array('status' => 500));
        }

        $original_file = str_replace(bp_core_avatar_url(), '', $avatar_data['original_file']);

        // Set avatars dir & feedback part.
        $avatar_dir = sanitize_key($avatar_data['object']) . '-avatars';

        // Crop args.
        $r = array(
            'item_id' => $avatar_data['item_id'],
            'object' => $avatar_data['object'],
            'avatar_dir' => $avatar_dir,
            'original_file' => $original_file,
            'crop_w' => $avatar_data['crop_w'],
            'crop_h' => $avatar_data['crop_h'],
            'crop_x' => $avatar_data['crop_x'],
            'crop_y' => $avatar_data['crop_y'],
        );

        // Handle crop.
        if (bp_core_avatar_handle_crop($r)) {
            $return = array(
                'avatar' => html_entity_decode(bp_core_fetch_avatar(array(
                    'object' => $avatar_data['object'],
                    'item_id' => $avatar_data['item_id'],
                    'html' => false,
                    'type' => 'full',
                ))),
                'feedback_code' => 1,
                'item_id' => $avatar_data['item_id'],
            );

            do_action($avatar_data['object'] . '_avatar_image_uploaded', (int)$bp_params['item_id']);

            return rest_ensure_response($return);
        } else {
            return new WP_Error('upload_failed', __('Upload failed.', 'buddyboss-app'), array('status' => 500));
        }
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {DELETE} /wp-json/appboss/buddypress/v1/groups/:id/cover Delete group cover
     * @apiName DeleteGroupsManageCoverImage
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription @since 2.4.0. Delete cover image of particular group of/for buddypress component
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id
     * @apiDeprecated  Delete group cover. Check (#Groups:DeleteBBGroupCover)
     * @apiPrivate
     */
    public function delete_cover($request)
    {

        $group_id = $request['id'];

        // you can change if you are admin or mod
        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        $cover_image_data = array(
            'object' => 'groups',
            'item_id' => $group_id,
        );

        if (empty($cover_image_data['object']) || empty($cover_image_data['item_id'])) {
            return new WP_Error('no_args', __('Provide all arguments', 'buddyboss-app'), array('status' => 500));
        }

        // Capability check.
        if (!bp_attachments_current_user_can('edit_cover_image', $cover_image_data)) {
            return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => 500));
        }

        $component = $cover_image_data['object'] . 's';
        $dir = $component;
        // you can change if you are admin or mod
        $is_admin = groups_is_user_admin(get_current_user_id(), $cover_image_data['item_id']);
        $is_mod = groups_is_user_mod(get_current_user_id(), $cover_image_data['item_id']);
        if (!($is_admin || $is_mod)) {
            return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => 500));
        }

        // Handle delete.
        if (bp_attachments_delete_file(array('item_id' => $cover_image_data['item_id'],
            'object_dir' => $dir,
            'type' => 'cover-image',
        ))) {

            // Defaults no cover image.
            $response = array(
                'reset_url' => '',
                'feedback_code' => 3,
            );

            // Get cover image settings in case there's a default header.
            $cover_params = bp_attachments_get_cover_image_settings($component);

            // Check if there's a default cover.
            if (!empty($cover_params['default_cover'])) {
                $response['reset_url'] = $cover_params['default_cover'];
            }

            // Finally send the reset url.
            return rest_ensure_response($response);
        } else {
            return new WP_Error('couldn\'t_be_deleted', __('Couldn\'t be deleted', 'buddyboss-app'), array('status' => 500));
        }
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {PATCH} /wp-json/appboss/buddypress/v1/groups/:id/cover Update cover image
     * @apiName UpdateGroupsManageCoverImage
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription @since 2.4.0 Update(upload) cover image for particular group of/for buddypress component
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id
     * @apiDeprecated  Create group cover. Check (#Groups:CreateBBGroupCover)
     * @apiPrivate
     */
    public function update_cover($request)
    {

        $group_id = $request['id'];

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        // Init the BuddyPress parameters.
        $bp_params = bp_parse_args(array(), array(
            'object' => 'group',
            'item_id' => $group_id,
        ), 'attachments_cover_image_ajax_upload');

        // We need the object to set the uploads dir filter.
        if (empty($bp_params['object'])) {
            return new WP_Error('no_args', __('Provide all arguments', 'buddyboss-app'), array('status' => 500));
        }

        // Capability check.
        if (!bp_attachments_current_user_can('edit_cover_image', $bp_params)) {
            return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => 500));
        }

        $bp = buddypress();
        $object_data = array('dir' => 'groups', 'component' => 'groups');
        $bp->groups->current_group = groups_get_group($bp_params['item_id']);
        $needs_reset = array(
            'component' => 'groups',
            'key' => 'current_group',
            'value' => $bp->groups->current_group,
        );

        // Stop here in case of a missing parameter for the object.
        if (empty($object_data['dir']) || empty($object_data['component'])) {
            return new WP_Error('error', __('Error', 'buddyboss-app'), array('status' => 500));
        }

        /**
         * Filters whether or not to handle cover image uploading.
         *
         * If you want to override this function, make sure you return an array with the 'result' key set.
         *
         * @param array $value
         * @param array $bp_params
         * @param array $needs_reset Stores original value of certain globals we need to revert to later.
         * @param array $object_data
         * @since 2.5.1
         *
         */
        $pre_filter = apply_filters('bp_attachments_pre_cover_image_ajax_upload', array(), $bp_params, $needs_reset, $object_data);
        if (isset($pre_filter['result'])) {
            // bp_attachments_json_response( $pre_filter['result'], $is_html4, $pre_filter );
            return rest_ensure_response($pre_filter);
        }

        add_filter('bp_attachment_upload_overrides', array($this, 'change_overrides'));
        add_filter('bp_attachments_cover_image_upload_dir', array($this, 'bp_attachments_cover_image_upload_dir'), 99);

        // Need for `bp_attachments_cover_image_upload_dir`
        $_REQUEST['item_id'] = $group_id;
        $cover_image_attachment = new BP_Attachment_Cover_Image();
        $uploaded = $cover_image_attachment->upload($_FILES);

        // Reset objects.
        if (!empty($needs_reset)) {
            if (!empty($needs_reset['component'])) {
                $bp->{$needs_reset['component']}->{$needs_reset['key']} = $needs_reset['value'];
            } else {
                $bp->{$needs_reset['key']} = $needs_reset['value'];
            }
        }

        if (!empty($uploaded['error'])) {
            // Upload error response.
            return new WP_Error('upload_error', sprintf(__('Upload Failed! Error was: %s', 'buddypress'), $uploaded['error']), array('status' => 500));
        }

        // Default error message.
        $error_message = __('There was a problem uploading the cover image.', 'buddypress');

        // Get BuddyPress Attachments Uploads Dir datas.
        $bp_attachments_uploads_dir = bp_attachments_uploads_dir_get();

        // The BP Attachments Uploads Dir is not set, stop.
        if (!$bp_attachments_uploads_dir) {
            return new WP_Error('upload_error', $error_message, array('status' => 500));
        }

        $cover_subdir = $object_data['dir'] . '/' . $bp_params['item_id'] . '/cover-image';
        $cover_dir = trailingslashit($bp_attachments_uploads_dir['basedir']) . $cover_subdir;

        if (!is_dir($cover_dir)) {
            // Upload error response.
            return new WP_Error('upload_error', $error_message, array('status' => 500));
        }

        /**
         * Generate the cover image so that it fit to feature's dimensions
         *
         * Unlike the Avatar, Uploading and generating the cover image is happening during
         * the same Ajax request, as we already instantiated the BP_Attachment_Cover_Image
         * class, let's use it.
         */
        $cover = bp_attachments_cover_image_generate_file(array(
            'file' => $uploaded['file'],
            'component' => $object_data['component'],
            'cover_image_dir' => $cover_dir,
        ), $cover_image_attachment);

        if (!$cover) {
            // Upload error response.
            return new WP_Error('upload_error', $error_message, array('status' => 500));
        }

        // Build the url to the file.
        $cover_url = trailingslashit($bp_attachments_uploads_dir['baseurl']) . $cover_subdir . '/' . $cover['cover_basename'];

        // Init Feedback code, 1 is success.
        $feedback_code = 1;

        // 0 is the size warning.
        if ($cover['is_too_small']) {
            $feedback_code = 0;
        }

        // Set the name of the file.
        $name = $_FILES['file']['name'];
        $name_parts = pathinfo($name);
        $name = trim(substr($name, 0, -(1 + strlen($name_parts['extension']))));

        /**
         * Fires if the new cover image was successfully uploaded.
         *
         * The dynamic portion of the hook will be xprofile in case of a user's
         * cover image, groups in case of a group's cover image. For instance:
         * Use add_action( 'xprofile_cover_image_uploaded' ) to run your specific
         * code once the user has set his cover image.
         *
         * @param int $item_id Inform about the item id the cover image was set for.
         * @since 2.4.0
         *
         */
        do_action($object_data['component'] . '_cover_image_uploaded', (int)$bp_params['item_id']);

        // Finally return the cover image url to the UI.
        return rest_ensure_response(array(
            'name' => $name,
            'url' => $cover_url,
            'feedback_code' => $feedback_code,
        ));
    }

    public function change_overrides($overrides)
    {
        $overrides['test_form'] = false;
        return $overrides;
    }

    public function bp_attachments_cover_image_upload_dir($args)
    {

        if (!empty($_REQUEST['item_id'])) {
            $upload_dir = bp_upload_dir();
            $base_dir = bp_attachments_uploads_dir_get('dir');
            $upload_path = trailingslashit($upload_dir['basedir']) . $base_dir;
            $url = trailingslashit($upload_dir['baseurl']) . $base_dir;
            // We're in a group, edit default values.
            $object_id = $_REQUEST['item_id'];
            $object_directory = 'groups';
            // Set the subdir.
            $subdir = '/' . $object_directory . '/' . $object_id . '/cover-image';
            $args['subdir'] = $subdir;
            $args['path'] = $upload_path . $subdir;
            $args['url'] = $url . $subdir;
        }

        return $args;
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/buddypress/v1/groups/:id/cover Group cover
     * @apiName GetGroupsCover
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get cover image of particular group of/for buddypress component
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id
     * @apiDeprecated  Retrieve group cover. Check (#Groups:GetBBGroupCover)
     * @apiPrivate
     */
    public function get_cover($request)
    {

        $group_id = $request['id'];

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        if (empty($group->id)) {
            return new WP_Error('no_group', __('Group does not exist.', 'buddyboss-app'), array('status' => 404));
        }

        return bp_attachments_get_attachment('url', array(
            'object_dir' => 'groups',
            'item_id' => $group_id,
            'type' => 'cover-image',
            'file' => '',
        ));
    }

    /**
     * @param $request
     * @return WP_Error
     * @api {PATCH} /wp-json/appboss/buddypress/v1/groups/:id/manage/members/action Update group manage actions
     * @apiName UpdateGroupsManageActions
     * @apiGroup BuddyPressGroups
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Admin Moderation level of action triggers.
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} id Group Id
     * @apiParam {String} user_id User id.
     * @apiParam {String=ban,unban,promote_to_mod,promote_to_admin,demote_to_member,remove,accept_membership,reject_membership} action Name of member action.
     * @apiParam {Number} membership_id Group membership id.
     * @apiDeprecated  Update user status on a group (add, remove, promote, demote or ban). Check (#Groups:UpdateBBGroupsMembers)
     * @apiPrivate
     */
    public function item_actions($request)
    {

        $group_id = (int)$request['id'];
        $action = $request['action'];

        if (!is_user_logged_in()) {
            return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $is_admin = groups_is_user_admin(get_current_user_id(), $group_id);
        $is_mod = groups_is_user_mod(get_current_user_id(), $group_id);
        if (!($is_admin || $is_mod || bp_current_user_can('bp_moderate'))) {
            return new WP_Error('permission_error', __('You need to be an admin to change details.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
        }

        $group = groups_get_group($group_id);

        // throw error when no group found.
        if (empty($group->id)) {
            return new WP_Error('group_not_found', __('Requested group not found.', 'buddyboss-app'), array('status' => 404));
        }

        if (empty($request['user_id'])) {
            return new WP_Error('error', __('Please provide user.', 'buddyboss-app'), array('status' => 500));
        }

        $user_id = $request['user_id'];

	    $pre_filter = apply_filters( 'bbapp_before_groups_member_manage_action', true, $group_id, $user_id, $action );
	    if ( is_wp_error( $pre_filter ) ) {
		    return $pre_filter;
	    }

        $member = new BP_Groups_Member($user_id, $group_id);

        switch ($action) {

            case "ban":

                do_action('groups_ban_member', $group_id, $user_id);

                $ban = $member->ban();

                if (!$ban) {
                    return new WP_Error('ban_user_error', __('Error while banning user.', 'buddyboss-app'), array('status' => 500));
                }

                /**
                 * Fires before the redirect after a group member has been banned.
                 *
                 * @param int $user_id ID of the user being banned.
                 * @param int $id ID of the group user is banned from.
                 * @since 1.0.0
                 *
                 */
                do_action('groups_banned_member', $user_id, $group_id);

                break;

            case "unban":

                do_action('groups_unban_member', $group_id, $user_id);

                $unban = $member->unban();

                if (!$unban) {
                    return new WP_Error('unban_user_error', __('Error while unbanning user.', 'buddyboss-app'), array('status' => 500));
                }

                /**
                 * Fires before the redirect after a group member has been unbanned.
                 *
                 * @param int $user_id ID of the user being unbanned.
                 * @param int $id ID of the group user is unbanned from.
                 * @since 1.0.0
                 *
                 */
                do_action('groups_unbanned_member', $user_id, $group_id);

                break;
            case "promote_to_mod":

                do_action('groups_promote_member', $group_id, $user_id, 'mod');

                $mod = $member->promote();

                if (!$mod) {
                    return new WP_Error('promote_to_mod_error', __('Error while promoting to mod.', 'buddyboss-app'), array('status' => 500));
                }

                /**
                 * Fires before the redirect after a group member has been promoted.
                 *
                 * @param int $user_id ID of the user being promoted.
                 * @param int $id ID of the group user is promoted within.
                 * @since 1.0.0
                 *
                 */
                do_action('groups_promoted_member', $user_id, $group_id);

                break;
            case "promote_to_admin":

                do_action('groups_promote_member', $group_id, $user_id, 'admin');

                $admin = $member->promote('admin');

                if (!$admin) {
                    return new WP_Error('promote_to_admin_error', __('Error while promoting to admin.', 'buddyboss-app'), array('status' => 500));
                }

                /**
                 * Fires before the redirect after a group member has been promoted.
                 *
                 * @param int $user_id ID of the user being promoted.
                 * @param int $id ID of the group user is promoted within.
                 * @since 1.0.0
                 *
                 */
                do_action('groups_promoted_member', $user_id, $group_id);

                break;
            case "demote_to_member":

                do_action('groups_demote_member', $group_id, $user_id);

                $demote = $member->demote();

                if (!$demote) {
                    return new WP_Error('demote_to_member_error', __('Error while promoting to admin.', 'buddyboss-app'), array('status' => 500));
                }

                /**
                 * Fires before the redirect after a group member has been demoted.
                 *
                 * @param int $user_id ID of the user being demoted.
                 * @param int $id ID of the group user is demoted within.
                 * @since 1.0.0
                 *
                 */
                do_action('groups_demoted_member', $user_id, $group_id);

                break;
            case "remove":

                do_action('groups_remove_member', $group_id, $user_id);

                $remove = $member->remove();

                if (!$remove) {
                    return new WP_Error('remove_user_error', __('Error while removing user from group.', 'buddyboss-app'), array('status' => 500));
                }

                /**
                 * Fires before the redirect after a group member has been removed.
                 *
                 * @param int $user_id ID of the user being removed.
                 * @param int $id ID of the group the user is removed from.
                 * @since 1.2.6
                 *
                 */
                do_action('groups_removed_member', $user_id, $group_id);

                break;
            case "accept_membership":

	            $accept = groups_accept_membership_request( $request['membership_id'], $user_id, $group_id );

                if (!$accept) {
                    return new WP_Error('accept_membership_group_error', __('There was an error accepting the membership request. Please try again.', 'buddyboss-app'), array('status' => 500));
                }

                /**
                 * Fires after a group membership request has been accepted.
                 *
                 * @param int $user_id ID of the user who accepted membership.
                 * @param int $group_id ID of the group that was accepted membership to.
                 * @param bool $value If membership was accepted.
                 * @since 1.0.0
                 *
                 */
                do_action('groups_membership_accepted', $member->user_id, $member->group_id, true);

                break;
            case "reject_membership":

	            $reject = groups_reject_membership_request( $request['membership_id'], $user_id, $group_id );

                if (!$reject) {
                    return new WP_Error('reject_membership_group_error', __('There was an error rejecting the membership request. Please try again.', 'buddyboss-app'), array('status' => 500));
                }

                /**
                 * Fires after a group membership request has been rejected.
                 *
                 * @param int $user_id ID of the user who rejected membership.
                 * @param int $group_id ID of the group that was rejected membership to.
                 * @param bool $value If membership was accepted.
                 * @since 1.0.0
                 *
                 */
                do_action('groups_membership_rejected', $member->user_id, $member->group_id, false);

                break;
        }

        $member_args = array(
            'group_id' => $group_id,
            'exclude_admins_mods' => false,
            'exclude_banned' => false,
        );
        $members = groups_get_group_members($member_args);
        $members = $members['members'];

        $members_array = array();
        $request['group_members'] = true;
        foreach ($members as $member) {
            if ($member->ID == $user_id || in_array($action, array('reject_membership'))) {
                $boss_rest_bp_members = new MembersRestApi($this->namespace_slug);
                $members_array[] = $this->prepare_response_for_collection(
                    $boss_rest_bp_members->prepare_member_item_for_response($member, $request)
                );
            }
        }

        return rest_ensure_response($members_array);
    }
}
