<?php
namespace BuddyBossApp\Api\BuddyPress;
use BuddyBossApp\Api\BuddyPress\MembersRestApi;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Server;

defined('ABSPATH') || exit;

// NOTE : Old classname was class.boss_buddypress_api_groups_invite_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Group Invite Component
class GroupsInviteRestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/buddypress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->rest_base = buddypress()->groups->id;
		// @todo : Verify below line of code
		$this->rest_base = 'groups';
	}

	public function hooks() {
		add_action('rest_api_init', array($this, "register_routes"), 99);
	}

	/**
	 * Register the plugin routes.
	 */
	public function register_routes() {

		register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/invite/', array(
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'create_invite'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => array(
					'users' => array(
						'description' => __('User ids for invite.'),
						'type' => 'array',
						'items' => array(
							'type' => 'integer',
						),
						'sanitize_callback' => 'wp_parse_id_list',
						'default' => array(),
					),
					'message' => array(
						'description' => __('add message.'),
						'type' => 'string',
						'default' => '',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::DELETABLE,
				'callback' => array($this, 'delete_invite'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => array(
					'user' => array(
						'description' => __('User id for delete invite.'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
						'default' => '',
					),
				),
			),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)/invite/members', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_invite_list'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
				'args' => array(
					'context' => $this->get_context_param(array('default' => 'view')),
					"scope" => array(
						"description" => "Scope of member like members, friends.",
						'default' => 'all',
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'per_page' => array(
						'description' => __('Maximum number of items to be returned in result set.'),
						'type' => 'integer',
						'default' => 10,
						'minimum' => 1,
						'maximum' => 100,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'page' => array(
						'description' => __('Current page of the collection.'),
						'type' => 'integer',
						'default' => 1,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
						'minimum' => 1,
					),
					'search_terms' => array(
						'description' => 'Limit results to those matching a string.',
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		));
	}

	/**
	 * Check if a given request has access to get information about a specific activity.
	 * @param WP_REST_Request $request Full data about the request.
	 * @return bool
	 */
	public function get_item_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_groups_invite_item_permission', true);
	}

	/**
	 * Check if a given request has access to activity items.
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function get_items_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_groups_invite_items_permission', true);
	}

	/**
	 * @param $request
	 * @return array|WP_Error
	 * @api {POST} /wp-json/appboss/buddypress/v1/groups/:id/invite Invite for groups
	 * @apiName CreateGroupsInviteMembers
	 * @apiGroup BuddyPressGroups
	 * @apiVersion 1.0.0
	 * @apiPermission Public
	 * @apiDescription Create member invite for groups of/for buddypress component
	 * User ids for invite
	 * @apiDeprecated  Create group invitation. Check (#Groups:CreateBBGroupsInvites)
	 * @apiUse apidocForCreateGroupInvitesV1
	 * @apiPrivate
	 */
	public function create_invite($request) {
		$bp = buddypress();

		$group_id = (int) $request['id'];

		$response = array(
			'feedback' => __('Invites could not be sent. Please try again.', 'buddypress'),
			'type' => 'error',
		);

		if (!bp_groups_user_can_send_invites($group_id)) {
			return new WP_Error('group_invite_not_allow', __('You are not allowed to send invitations for this group.', 'buddypress'), array('status' => 401));
		}

		if (empty($request['users'])) {
			return new WP_Error('group_invite_fail', __('Invites could not be sent. Please try again.', 'buddypress'), array('status' => 500));
		}

		$invited = array();

		foreach ((array) $request['users'] as $user_id) {
			$invited[(int) $user_id] = groups_invite_user(
				array(
					'user_id' => $user_id,
					'group_id' => $group_id,
				)
			);
		}

		if (!empty($_POST['message'])) {
			$bp->groups->invites_message = wp_kses(wp_unslash($request['message']), array());

			add_filter('groups_notification_group_invites_message', 'bp_nouveau_groups_invites_custom_message', 10, 1);
		}

		// Send the invites.
		groups_send_invites(bp_loggedin_user_id(), $group_id);

		if (!empty($_POST['message'])) {
			unset($bp->groups->invites_message);

			remove_filter('groups_notification_group_invites_message', 'bp_nouveau_groups_invites_custom_message', 10, 1);
		}

		if (array_search(false, $invited)) {
			$errors = array_keys($invited, false);

			$error_count = count($errors);
			$error_message = sprintf(
				/* translators: count of users affected */
				_n(
					'Invitation failed for %s user.',
					'Invitation failed for %s users.',
					$error_count, 'buddypress'
				),
				number_format_i18n($error_count)
			);

			$error_message .= sprintf(__('User: %s', 'buddyboss-app'), implode($errors, ','));

			return new WP_Error('group_invite_fail', $error_message, array('status' => 500));
		}

		return array(
			'success' => 1,
		);
	}

	/**
	 * @param $request
	 * @return array|WP_Error
	 * @api {DELETE} /wp-json/appboss/buddypress/v1/groups/:id/invite Delete invite for groups
	 * @apiName DeleteGroupsInviteMembers
	 * @apiGroup BuddyPressGroups
	 * @apiVersion 1.0.0
	 * @apiPermission Public
	 * @apiDescription Delete member invite for groups of/for buddypress component
	 * @apiParam {Number} id Group Id
	 * @apiParam {Number} user User id for delete invite
	 * @apiDeprecated  Delete group invitation. Check (#Groups:DeleteBBGroupsInvite)
	 * @apiPrivate
	 */
	public function delete_invite($request) {
		$group_id = (int) $request['id'];
		$user_id = (int) $request['user'];

		// Verify pending invite.
		$invites_args = array(
			'is_confirmed' => false,
			'is_banned' => null,
			'is_admin' => null,
			'is_mod' => null,
		);
		$invites = bp_get_user_groups($user_id, $invites_args);
		if (empty($invites)) {
			return new WP_Error('group_invite_remove_failed', __('Group invitation could not be removed.', 'buddypress'), array('status' => 500));
		}

		if (!groups_is_user_admin(bp_loggedin_user_id(), $group_id)) {
			return new WP_Error('group_invite_remove_failed', __('Group invitation could not be removed.', 'buddypress'), array('status' => 500));
		}

		if (BP_Groups_Member::check_for_membership_request($user_id, $group_id)) {
			return array(
				'message' => __('The member is already a member of the group.', 'buddypress'),
				'success' => 1,
			);
		}

		// Remove the unsent invitation.
		if (!groups_uninvite_user($user_id, $group_id)) {
			return new WP_Error('group_invite_remove_failed', __('Group invitation could not be removed.', 'buddypress'), array('status' => 500));
		}

		return array(
			'success' => 1,
		);
	}

	/**
	 * @param $request
	 *
	 * @return          array
	 * @api            {GET} /wp-json/appboss/buddypress/v1/groups/:id/invite/members Groups invite members
	 * @apiName        GetGroupsInviteMembers
	 * @apiGroup       BuddyPressGroups
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription All members for groups invitation of/for buddypress component
	 * @apiDeprecated  Retrieve invites for group. Check (#Groups:GetBBGroupsInvites)
	 * @apiUse         apidocForAllFriendsV1
	 * @apiPrivate
	 */
	public function get_invite_list($request) {
		$member_args = array(
			'type' => 'alphabetical',
			'scope' => !empty($request['scope']) ? $request['scope'] : 'members',
			'group_id' => $request['id'],
			'per_page' => !empty($request['per_page']) ? $request['per_page'] : 20,
			'page' => !empty($request['page']) ? $request['page'] : 1,
			'search_terms' => !empty($request['search_terms']) ? $request['search_terms'] : false,
		);

        $group_members = groups_get_group_members( [
            'group_id'            => $member_args['group_id'],
            'exclude_admins_mods' => false,
        ] );

        $group_members_ids = wp_list_pluck( $group_members['members'], 'ID' );

        $type = 'exclude';
		if ('friends' === $request['scope']) {
			$member_args['user_id'] = $request['user_id'];
        }

		if ('invited' === $request['scope']) {
			$member_args['is_confirmed'] = false;
			$type = 'include';
            $group_members_ids = groups_get_invites_for_group( $member_args['user_id'], $member_args['group_id'] );
		}

		if (!empty($group_members_ids)) {
			$member_args[$type] = $group_members_ids;
		} else {
			$member_args[$type] = array(0);
		}

		$members = bp_core_get_users($member_args);

		$members_array = array();
		foreach ($members['users'] as $member) {
			$boss_rest_bp_members = new MembersRestApi($this->namespace_slug);
			$members_array[] = $this->prepare_response_for_collection(
				$boss_rest_bp_members->prepare_member_item_for_response($member, $request)
			);
		}

		$response = rest_ensure_response($members_array);
		$response->header('X-WP-Total', (int) $members['count']);

		return $response;
	}
}
