<?php
namespace BuddyBossApp\Api\BuddyPress;
use BP_Attachment_Cover_Image;
use stdClass;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Server;

defined('ABSPATH') || exit;

// NOTE : Old classname was class.boss_buddypress_api_groups_create_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Groups Manage Component
class GroupsCreateRestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/buddypress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->rest_base = buddypress()->groups->id;
		// @todo : Verify below line of code
		$this->rest_base = 'groups';
	}

	public function hooks() {
		add_action('rest_api_init', array($this, "register_routes"), 99);
	}

	/**
	 * Register the routes for the objects of the controller.
	 */
	public function register_routes() {
		register_rest_route($this->namespace, '/' . $this->rest_base . '/create', array(
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'create'),
				'permission_callback' => array($this, 'bp_groups_create_permission'),
				'args' => array(
					'group_id' => array(
						'description' => __('Group id'),
						'type' => 'integer',
						'default' => 0,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'step' => array(
						'description' => __('Name of group step.'),
						'type' => 'string',
						'default' => 'group-details',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'group-name' => array(
						'description' => __('Name of group.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'group-desc' => array(
						'description' => __('Description of group.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'group-status' => array(
						'description' => __('Status of group.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'group-types' => array(
						'description' => __('Types of group.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'hasforum' => array(
						'description' => __('Whether the Group has a forum or not.'),
						'type' => 'string',
						'enum' => array('true', 'false', ''),
						'validate_callback' => 'rest_validate_request_arg',
					),
					'group-invite-status' => array(
						'description' => __('Invite status of group.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'friends' => array(
						'description' => __('User ids of group invitees.'),
						'type' => 'array',
						'items' => array(
							'type' => 'integer',
						),
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		));
	}

	/**
	 * Check if a given request has access to settings items.
	 *
	 * @return WP_Error|bool
	 */
	public function bp_groups_create_permission() {
		return apply_filters('boss_rest_buddypress_groups_create_permission', true);
	}

	/**
	 * @param $request
	 * @return array|WP_Error
	 * @api {POST} /wp-json/appboss/buddypress/v1/groups/create Create group
	 * @apiName CreateGroup
	 * @apiGroup BuddyPressGroups
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Create group of/for buddypress
	 * @apiHeader {String} accessToken Auth token
	 * @apiDeprecated  Create groups. Check (#Groups:CreateBBGroups)
	 * @apiUse apidocForCreateGroupV1
	 * @apiPrivate
	 */
	public function create($request) {

		$response = array();

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', 'Please login to perform any action.', array('status' => rest_authorization_required_code()));
		}

		if (!bp_user_can_create_groups()) {
			return new WP_Error('group_create_no_access', __('Sorry, you are not allowed to create groups.'), array('status' => rest_authorization_required_code()));
		}

		do_action('boss_bp_api_before_group_create');

		$bp = buddypress();

		// If no current step is set, reset everything so we can start a fresh group creation.
		$bp->groups->current_create_step = $request['step'];

		if ('group-details' == $request['step']) {
			//reset cookies before create group
			setcookie('bp_new_group_id', false, time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
			setcookie('bp_completed_create_steps', false, time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
		}

		if (!empty($request['group_id'])) {
			$bp->groups->new_group_id = (int) $request['group_id'];
		}

		if (bp_get_groups_current_create_step() && empty($bp->groups->group_creation_steps[bp_get_groups_current_create_step()])) {
			return new WP_Error('error', __('There was an error saving group details. Please try again.'), array('status' => 500));
		}

		// Fetch the currently completed steps variable.
		if (isset($_COOKIE['bp_completed_create_steps']) && !isset($reset_steps)) {
			$bp->groups->completed_create_steps = json_decode(base64_decode(stripslashes($_COOKIE['bp_completed_create_steps'])));
		}

		if (bp_get_new_group_id()) {
			$bp->groups->current_group = groups_get_group($bp->groups->new_group_id);

			// Only allow the group creator to continue to edit the new group.
			if (!bp_is_group_creator($bp->groups->current_group, bp_loggedin_user_id())) {
				return new WP_Error('error', __('Only the group creator may continue editing this group.'), array('status' => 500));
			}
		}

		$completed_create_steps = isset($bp->groups->completed_create_steps) ? $bp->groups->completed_create_steps : array();

		//manipulate Multiple group step into one step
		do_action('boss_before_group_details_creation_step');

		/***************
			 * Save Group Details page
		*/
		if ('group-details' == $request['step']) {
			if (empty($request['group-name']) || empty($request['group-desc']) || !strlen(trim($request['group-name'])) || !strlen(trim($request['group-desc']))) {
				return new WP_Error('error', __('Please fill in all of the required fields'), array('status' => 500));
			}

			$new_group_id = isset($bp->groups->new_group_id) ? $bp->groups->new_group_id : 0;

			/***************
				 * Create Group
			*/
			if (!$bp->groups->new_group_id = groups_create_group(
				array(
					'group_id' => $new_group_id,
					'name' => $request['group-name'],
					'description' => $request['group-desc'],
					'slug' => groups_check_slug(sanitize_title(esc_attr($request['group-name']))),
					'date_created' => bp_core_current_time(),
					'status' => 'public',
				))) {
				return new WP_Error('error', __('There was an error saving group details. Please try again.'), array('status' => 500));
			}

			$bp->groups->current_group = groups_get_group($bp->groups->new_group_id);

			/***************
				 * Create forums for groups
			*/
			$group_enable_forum = false;
			if (function_exists('bbp_is_group_forums_active') && bbp_is_group_forums_active()) {
				$group_enable_forum = (!empty($request['hasforum']) && "true" === $request['hasforum']) ? true : false;

				$this->create_group_forums($request);

				if (!in_array('forum', $completed_create_steps)) {
					$bp->groups->completed_create_steps[] = 'forum';
				}

			}

			/***************
				 * Save group status
			*/
			$group_status = 'public';

			if ('private' == $request['group-status']) {
				$group_status = 'private';
			} elseif ('hidden' == $request['group-status']) {
				$group_status = 'hidden';
			}

			if (!$bp->groups->new_group_id = groups_create_group(array('group_id' => $bp->groups->new_group_id, 'status' => $group_status, 'enable_forum' => $group_enable_forum))) {
				return new WP_Error('error', __('There was an error saving group details. Please try again.'), array('status' => 500));
			}

			/***************
				 * Save group types.
			*/
			if (!empty($request['group-types'])) {
				bp_groups_set_group_type($bp->groups->new_group_id, $_POST['group-types']);
			}

			/***************
				 * Save invite statuses.
			*/
			$allowed_invite_status = apply_filters('groups_allowed_invite_status', array('members', 'mods', 'admins'));
			$invite_status = !empty($request['group-invite-status']) && in_array($request['group-invite-status'], (array) $allowed_invite_status) ? $request['group-invite-status'] : 'members';

			groups_update_groupmeta($bp->groups->new_group_id, 'invite_status', $invite_status);

			if (!in_array('group-settings', $completed_create_steps)) {
				$bp->groups->completed_create_steps[] = 'group-settings';
			}

			/***************
				 * Save group Cover - group-cover-image
			*/
			if (!bp_disable_group_cover_image_uploads()) {
				if (!empty($_FILES['cover'])) {
					$this->create_group_cover($request);
				}
				if (!in_array('group-cover-image', $completed_create_steps)) {
					$bp->groups->completed_create_steps[] = 'group-cover-image';
				}
			}

			/***************
				 * Save group Avtar - group-avatar
			*/
			if (!bp_disable_group_avatar_uploads()) {
				if (!empty($_FILES['avatar'])) {
					$this->create_group_avatar($request);
				}
				if (!in_array('group-avatar', $completed_create_steps)) {
					$bp->groups->completed_create_steps[] = 'group-avatar';
				}
			}

			$response['group_id'] = $bp->groups->new_group_id;
		}

		//manipulate Multiple group step into one step
		do_action('boss_after_group_details_creation_step');

		if ('group-invites' === $request['step']) {
			if (!empty($request['friends'])) {
				foreach ((array) $request['friends'] as $friend) {
					$temp = groups_invite_user(array(
						'user_id' => (int) $friend,
						'group_id' => $bp->groups->new_group_id,
					));
				}
			}

			groups_send_invites(bp_loggedin_user_id(), $bp->groups->new_group_id);
		}

		/**
		 * Fires before finalization of group creation and cookies are set.
		 *
		 * This hook is a variable hook dependent on the current step
		 * in the creation process.
		 *
		 * @since 1.1.0
		 */
		do_action('groups_create_group_step_save_' . bp_get_groups_current_create_step());

		/**
		 * Fires after the group creation step is completed.
		 *
		 * Mostly for clearing cache on a generic action name.
		 *
		 * @since 1.1.0
		 */
		do_action('groups_create_group_step_complete');

		/**
		 * Once we have successfully saved the details for this step of the creation process
		 * we need to add the current step to the array of completed steps, then update the cookies
		 * holding the information
		 */
		if (!in_array(bp_get_groups_current_create_step(), $completed_create_steps)) {
			$bp->groups->completed_create_steps[] = bp_get_groups_current_create_step();
		}

		/***************
			 * Store cookie for groups
		*/
		setcookie('bp_new_group_id', $bp->groups->new_group_id, time() + 60 * 60 * 24, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
		setcookie('bp_completed_create_steps', base64_encode(json_encode($bp->groups->completed_create_steps)), time() + 60 * 60 * 24, COOKIEPATH, COOKIE_DOMAIN, is_ssl());

		/***************
			 * groups create completed or not
		*/
		$keys = array_keys($bp->groups->group_creation_steps);
		$filter_key = array_diff($keys, array('forum', 'group-settings', 'group-cover-image', 'group-avatar'));
		if (count($bp->groups->completed_create_steps) == count($keys) && bp_get_groups_current_create_step() == array_pop($filter_key)) {
			unset($bp->groups->current_create_step);
			unset($bp->groups->completed_create_steps);

			setcookie('bp_new_group_id', false, time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
			setcookie('bp_completed_create_steps', false, time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl());

			// Once we completed all steps, record the group creation in the activity stream.
			groups_record_activity(array(
				'type' => 'created_group',
				'item_id' => $bp->groups->new_group_id,
			));

			/**
			 * Fires after the group has been successfully created.
			 *
			 * @since 1.1.0
			 *
			 * @param int $new_group_id ID of the newly created group.
			 */
			do_action('groups_group_create_complete', $bp->groups->new_group_id);

			$response['grouplink'] = bp_get_group_permalink($bp->groups->current_group);
		} else {
			//Next Step
			$keys = array_keys($bp->groups->group_creation_steps);
			foreach ($keys as $key) {
				if (in_array($key, $bp->groups->completed_create_steps)) {
					$next = 1;
					continue;
				}
				if (isset($next)) {
					$response['next'] = $key;
					break;
				}
			}
		}

		return $response;
	}

	private function create_group_forums($request) {
		$bp = buddypress();

		$group_enable_forum = (!empty($request['hasforum']) && "true" === $request['hasforum']) ? true : false;
		$forum_id = 0;
		$forum_ids = bbp_get_group_forum_ids($bp->groups->new_group_id);

		if (!empty($forum_ids)) {
			$forum_id = (int) is_array($forum_ids) ? $forum_ids[0] : $forum_ids;
		}

		// Create a forum, or not
		switch ($group_enable_forum) {
		case true:
			// Bail if initial content was already created
			if (empty($forum_id)) {
				// Set the default forum status
				switch (bp_get_new_group_status()) {
				case 'hidden':
					$status = bbp_get_hidden_status_id();
					break;
				case 'private':
					$status = bbp_get_private_status_id();
					break;
				case 'public':
				default:
					$status = bbp_get_public_status_id();
					break;
				}

				// Create the initial forum
				$forum_id = bbp_insert_forum(array(
					'post_parent' => bbp_get_group_forums_root_id(),
					'post_title' => bp_get_new_group_name(),
					'post_content' => bp_get_new_group_description(),
					'post_status' => $status,
				));

				// Run the BP-specific functions for new groups
				$this->new_forum(array('forum_id' => $forum_id));

				// Update forum active
				groups_update_groupmeta(bp_get_new_group_id(), '_bbp_forum_enabled_' . $forum_id, true);

				// Toggle forum on
				$this->toggle_group_forum(bp_get_new_group_id(), true);
			}

			break;
		case false:

			// Forum was created but is now being undone
			if (!empty($forum_id)) {

				// Delete the forum
				wp_delete_post($forum_id, true);

				// Delete meta values
				groups_delete_groupmeta(bp_get_new_group_id(), 'forum_id');
				groups_delete_groupmeta(bp_get_new_group_id(), '_bbp_forum_enabled_' . $forum_id);

				// Toggle forum off
				$this->toggle_group_forum(bp_get_new_group_id(), false);
			}

			break;
		}
	}

	private function create_group_cover($request) {
		$bp = buddypress();

		$bp_params = bp_parse_args(array(), array(
			'object' => 'group',
			'item_id' => bp_get_new_group_id(),
		), 'attachments_cover_image_ajax_upload');

		$object_data = array('dir' => 'groups', 'component' => 'groups');
		$needs_reset = array(
			'component' => 'groups',
			'key' => 'current_group',
			'value' => $bp->groups->current_group,
		);

		$is_admin = groups_is_user_admin(get_current_user_id(), bp_get_new_group_id());
		$is_mod = groups_is_user_mod(get_current_user_id(), bp_get_new_group_id());
		if (!($is_admin || $is_mod)) {
			return new WP_Error('error', __('No permission to upload cover.', 'buddyboss-app'), array('status' => 500));
		}

		if (empty($object_data['dir']) || empty($object_data['component'])) {
			return new WP_Error('error', __('Error:', 'buddyboss-app'), array('status' => 500));
		}

		$pre_filter = apply_filters('bp_attachments_pre_cover_image_ajax_upload', array(), $bp_params, $needs_reset, $object_data);
		if (isset($pre_filter['result'])) {
			return $pre_filter['result'];
		}

		$_REQUEST['group_cover_upload'] = true;
		add_filter('bp_attachment_upload_overrides', array($this, 'change_overrides'));
		add_filter('bp_attachments_cover_image_upload_dir', array($this, 'bp_attachments_cover_image_upload_dir'), 99);

		$cover_image_attachment = new BP_Attachment_Cover_Image();
		$temp = array('file' => $_FILES['cover']);
		$uploaded = $cover_image_attachment->upload($temp);

		// Reset objects.
		if (!empty($needs_reset)) {
			if (!empty($needs_reset['component'])) {
				$bp->{$needs_reset['component']}->{$needs_reset['key']} = $needs_reset['value'];
			} else {
				$bp->{$needs_reset['key']} = $needs_reset['value'];
			}
		}

		if (!empty($uploaded['error'])) {
			return new WP_Error('error', __('Upload Failed!', 'buddyboss-app'), array('status' => 500));
		}

		// Default error message.
		$error_message = __('There was a problem uploading the cover image.', 'buddypress');

		// Get BuddyPress Attachments Uploads Dir datas.
		$bp_attachments_uploads_dir = bp_attachments_uploads_dir_get();

		// The BP Attachments Uploads Dir is not set, stop.
		if (!$bp_attachments_uploads_dir) {
			return new WP_Error('error', $error_message, array('status' => 500));
		}

		$cover_subdir = $object_data['dir'] . '/' . $bp_params['item_id'] . '/cover-image';
		$cover_dir = trailingslashit($bp_attachments_uploads_dir['basedir']) . $cover_subdir;

		if (!is_dir($cover_dir)) {
			return new WP_Error('upload_error', $error_message, array('status' => 500));
		}

		/**
		 * Generate the cover image so that it fit to feature's dimensions
		 *
		 * Unlike the Avatar, Uploading and generating the cover image is happening during
		 * the same Ajax request, as we already instantiated the BP_Attachment_Cover_Image
		 * class, let's use it.
		 */
		$cover = bp_attachments_cover_image_generate_file(array(
			'file' => $uploaded['file'],
			'component' => $object_data['component'],
			'cover_image_dir' => $cover_dir,
		), $cover_image_attachment);

		if (!$cover) {
			return new WP_Error('upload_error', $error_message, array('status' => 500));
		}

		/**
		 * Fires if the new cover image was successfully uploaded.
		 *
		 * The dynamic portion of the hook will be xprofile in case of a user's
		 * cover image, groups in case of a group's cover image. For instance:
		 * Use add_action( 'xprofile_cover_image_uploaded' ) to run your specific
		 * code once the user has set his cover image.
		 *
		 * @since 2.4.0
		 *
		 * @param int $item_id Inform about the item id the cover image was set for.
		 */
		do_action($object_data['component'] . '_cover_image_uploaded', (int) bp_get_new_group_id());
	}

	private function create_group_avatar($request) {
		$bp = buddypress();

		$bp_params = array(
			'object' => 'group',
			'item_id' => bp_get_new_group_id(),
			'upload_dir_filter' => 'groups_avatar_upload_dir',
		);

		$needs_reset = array(
			'component' => 'groups',
			'key' => 'current_group',
			'value' => $bp->groups->current_group,
		);

		$is_admin = groups_is_user_admin(get_current_user_id(), bp_get_new_group_id());
		$is_mod = groups_is_user_mod(get_current_user_id(), bp_get_new_group_id());
		if (!($is_admin || $is_mod)) {
			return new WP_Error('error', __('No permission to upload cover.', 'buddyboss-app'), array('status' => 500));
		}

		if (!isset($bp->avatar_admin)) {
			$bp->avatar_admin = new stdClass();
		}

		/**
		 * The BuddyPress upload parameters is including the Avatar UI Available width,
		 * add it to the avatar_admin global for a later use.
		 */
		if (isset($bp_params['ui_available_width'])) {
			$bp->avatar_admin->ui_available_width = (int) $bp_params['ui_available_width'];
		}

		// upload will be rejected without this
		add_filter('bp_attachment_upload_overrides', array($this, 'change_overrides'));

		// Upload the avatar.
		$temp = array('file' => $_FILES['avatar']);
		$avatar = bp_core_avatar_handle_upload($temp, $bp_params['upload_dir_filter']);

		// Reset objects.
		if (!empty($needs_reset)) {
			if (!empty($needs_reset['component'])) {
				$bp->{$needs_reset['component']}->{$needs_reset['key']} = $needs_reset['value'];
			} else {
				$bp->{$needs_reset['key']} = $needs_reset['value'];
			}
		}

		// Init the feedback message.
		$feedback_message = false;

		if (!empty($bp->template_message)) {
			$feedback_message = $bp->template_message;

			// Remove template message.
			$bp->template_message = false;
			$bp->template_message_type = false;

			@setcookie('bp-message', false, time() - 1000, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
			@setcookie('bp-message-type', false, time() - 1000, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
		}

		if (empty($avatar)) {
			// Default upload error.
			$message = __('Upload failed.', 'buddyboss-app');

			// Use the template message if set.
			if (!empty($feedback_message)) {
				$message = $feedback_message;
			}

			// Upload error reply.
			return new WP_Error('upload_error', $message, array('status' => 500));
		}

		if (empty($bp->avatar_admin->image->file)) {
			return new WP_Error('upload_error', __('Upload failed.', 'buddyboss-app'), array('status' => 500));
		}

		$uploaded_image = @getimagesize($bp->avatar_admin->image->file);

		// Set the name of the file.
		$name = $_FILES['avatar']['name'];
		$name_parts = pathinfo($name);
		$name = trim(substr($name, 0, -(1 + strlen($name_parts['extension']))));

		// Further code is copied from bp_avatar_ajax_set, reall croping will be done in client app

		$size = $uploaded_image[0];

		// take the smallest part. so we don't see any black empty area.
		if ($uploaded_image[0] > $uploaded_image[1]) {
			$size = $uploaded_image[1];
		}

		$avatar_data = array(
			'original_file' => $bp->avatar_admin->image->url,
			'crop_w' => $size,
			'crop_h' => $size,
			'crop_x' => 0,
			'crop_y' => 0,
			'item_id' => $bp_params['item_id'],
			'object' => $bp_params['object'],
			'type' => 'crop',
		);

		if (empty($avatar_data['object']) || empty($avatar_data['item_id']) || empty($avatar_data['original_file'])) {
			return new WP_Error('no_args', __('No arguments.', 'buddyboss-app'), array('status' => 500));
		}

		$original_file = str_replace(bp_core_avatar_url(), '', $avatar_data['original_file']);

		// Set avatars dir & feedback part.
		$avatar_dir = sanitize_key($avatar_data['object']) . '-avatars';

		// Crop args.
		$r = array(
			'item_id' => $avatar_data['item_id'],
			'object' => $avatar_data['object'],
			'avatar_dir' => $avatar_dir,
			'original_file' => $original_file,
			'crop_w' => $avatar_data['crop_w'],
			'crop_h' => $avatar_data['crop_h'],
			'crop_x' => $avatar_data['crop_x'],
			'crop_y' => $avatar_data['crop_y'],
		);

		// Handle crop.
		if (bp_core_avatar_handle_crop($r)) {
			do_action($avatar_data['object'] . '_avatar_image_uploaded', (int) $bp_params['item_id']);
			return true;
		} else {
			return new WP_Error('upload_failed', __('Upload failed.', 'buddyboss-app'), array('status' => 500));
		}
	}

	public function change_overrides($overrides) {
		$overrides['test_form'] = false;
		return $overrides;
	}

	public function bp_attachments_cover_image_upload_dir($args) {
		if (!empty($_REQUEST['group_cover_upload'])) {
			$upload_dir = bp_upload_dir();
			$base_dir = bp_attachments_uploads_dir_get('dir');
			$upload_path = trailingslashit($upload_dir['basedir']) . $base_dir;
			$url = trailingslashit($upload_dir['baseurl']) . $base_dir;
			// We're in a group, edit default values.
			$object_id = bp_get_new_group_id();
			$object_directory = 'groups';
			// Set the subdir.
			$subdir = '/' . $object_directory . '/' . $object_id . '/cover-image';
			$args['subdir'] = $subdir;
			$args['path'] = $upload_path . $subdir;
			$args['url'] = $url . $subdir;
		}
		return $args;
	}

	/**
	 * bbpress/includes/extend/buddypress/groups.php
	 *
	 * Creating a group forum or category (including root for group)
	 *
	 * @param array $forum_args
	 * @return void no forum_id is available
	 * @uses bp_get_current_group_id()
	 * @uses bbp_add_forum_id_to_group()
	 * @uses bbp_add_group_id_to_forum()
	 * @since bbPress (r3653)
	 * @uses bbp_get_forum_id()
	 */
	private function new_forum($forum_args = array()) {

		// Bail if no forum_id was passed
		if (empty($forum_args['forum_id'])) {
			return;
		}

		// Validate forum_id
		$forum_id = bbp_get_forum_id($forum_args['forum_id']);
		$group_id = !empty($forum_args['group_id']) ? $forum_args['group_id'] : bp_get_current_group_id();

		bbp_add_forum_id_to_group($group_id, $forum_id);
		bbp_add_group_id_to_forum($forum_id, $group_id);
	}

	/**
	 * bbpress/includes/extend/buddypress/groups.php
	 *
	 * Removing a group forum or category (including root for group)
	 *
	 * @param array $forum_args
	 * @return void no forum_id is available
	 * @uses bp_get_current_group_id()
	 * @uses bbp_add_forum_id_to_group()
	 * @uses bbp_add_group_id_to_forum()
	 * @since bbPress (r3653)
	 * @uses bbp_get_forum_id()
	 */
	private function remove_forum($forum_args = array()) {

		// Bail if no forum_id was passed
		if (empty($forum_args['forum_id'])) {
			return;
		}

		// Validate forum_id
		$forum_id = bbp_get_forum_id($forum_args['forum_id']);
		$group_id = !empty($forum_args['group_id']) ? $forum_args['group_id'] : bp_get_current_group_id();

		bbp_remove_forum_id_from_group($group_id, $forum_id);
		bbp_remove_group_id_from_forum($forum_id, $group_id);
	}

	/**
	 * bbpress/includes/extend/buddypress/groups.php
	 *
	 * Toggle the enable_forum group setting on or off
	 *
	 * @since bbPress (r4612)
	 *
	 * @param int $group_id The group to toggle
	 * @param bool $enabled True for on, false for off
	 * @uses groups_get_group() To get the group to toggle
	 * @return False if group is not found, otherwise return the group
	 */
	private function toggle_group_forum($group_id = 0, $enabled = false) {

		// Get the group
		$group = groups_get_group(array('group_id' => $group_id));

		// Bail if group cannot be found
		if (empty($group)) {
			return false;
		}

		// Set forum enabled status
		$group->enable_forum = (int) $enabled;

		// Save the group
		$group->save();

		// Maybe disconnect forum from group
		if (empty($enabled)) {
			$this->disconnect_forum_from_group($group_id);
		}

		// Update bbPress' internal private and forum ID variables
		bbp_repair_forum_visibility();

		// Return the group
		return $group;
	}

	/**
	 * bbpress/includes/extend/buddypress/groups.php
	 *
	 * Listening to BuddyPress Group deletion to remove the forum
	 *
	 * @param int $group_id The group ID
	 * @uses bbp_get_group_forum_ids()
	 * @uses BBP_Forums_Group_Extension::remove_forum()
	 */
	private function disconnect_forum_from_group($group_id = 0) {

		// Bail if no group ID available
		if (empty($group_id)) {
			return;
		}

		// Get the forums for the current group
		$forum_ids = bbp_get_group_forum_ids($group_id);

		// Use the first forum ID
		if (empty($forum_ids)) {
			return;
		}

		// Get the first forum ID
		$forum_id = (int) is_array($forum_ids) ? $forum_ids[0] : $forum_ids;
		$this->remove_forum(array(
			'forum_id' => $forum_id,
			'group_id' => $group_id,
		));
	}

	/*public function get_step( $request ){
		$response = array();

		$bp = buddypress();

		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'error', 'Not login', array( 'status' => 500 ) );
		}

		if ( ! bp_user_can_create_groups() ) {
			//bp_core_add_message( __( 'Sorry, you are not allowed to create groups.', 'buddypress' ), 'error' );
			$response[] = 'Sorry, you are not allowed to create groups.';
		}

		$bp = buddypress();

		// If no current step is set, reset everything so we can start a fresh group creation.
		$bp->groups->current_create_step = $request['step'];
		$bp->groups->new_group_id = (int) $_COOKIE['bp_new_group_id'];
		$bp->current_component = 'groups';
		$bp->current_action = 'create';

		if ( bp_get_groups_current_create_step() && empty( $bp->groups->group_creation_steps[bp_get_groups_current_create_step()] ) ) {
			//bp_core_add_message( __('There was an error saving group details. Please try again.', 'buddypress'), 'error' );
		}

		// Fetch the currently completed steps variable.
		if ( isset( $_COOKIE['bp_completed_create_steps'] ) && !isset( $reset_steps ) )
			$bp->groups->completed_create_steps = json_decode( base64_decode( stripslashes( $_COOKIE['bp_completed_create_steps'] ) ) );

		if ( bp_get_new_group_id() ) {
			$bp->groups->current_group = groups_get_group( $bp->groups->new_group_id );

			// Only allow the group creator to continue to edit the new group.
			if ( ! bp_is_group_creator( $bp->groups->current_group, bp_loggedin_user_id() ) ) {
				//bp_core_add_message( __( 'Only the group creator may continue editing this group.', 'buddypress' ), 'error' );
				$response[] = 'Only the group creator may continue editing this group.';
			}
		}

		ob_start();

		bp_core_load_template( apply_filters( 'groups_template_create_group', 'groups/create' ) );
		do_action( 'template_include' );
		do_action( 'the_content' );
		$response['html'] = ob_get_clean();

		return $response;
	}*/

}
