<?php
namespace BuddyBossApp\Api\BuddyPress;
use BuddyBossApp\Api\BuddyPress\MembersRestApi;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Server;

defined('ABSPATH') || exit;
// NOTE : Old classname was class.boss_buddypress_api_friends_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Friends Component
class FriendsRestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/buddypress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->rest_base = buddypress()->friends->id;
		// @todo : Verify below line of code
		$this->rest_base = 'friends';
	}

	public function hooks() {

		add_action('rest_api_init', array($this, "register_routes"), 99);

	}

	/**
	 * Register the plugin routes.
	 */
	public function register_routes() {

		register_rest_route($this->namespace, '/' . $this->rest_base, array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_items'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
				'args' => $this->get_friends_args(),
			),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/action/(?P<id>[\d]+)', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'do_action'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => $this->get_friends_action_args(),
			),
		));
	}

	/**
	 * Return args data for Friend action endpoint
	 * helps for easily fiilter request data and validation
	 * @return array
	 */
	public function get_friends_action_args() {

		$args = array(
			"action" => array(
				'type' => 'string',
				'default' => 'add_friend',
				'enum' => array('add_friend', 'remove_friend', 'withdraw_friendship', 'accept_friend', 'reject_friend'),
				'validate_callback' => 'rest_validate_request_arg',
			),
		);

		return $args;
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/friends/action/:id Update friend action
	 * @apiName UpdateFriendAction
	 * @apiGroup BuddyPressFriends
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update BuddyPress's Friend action
	 * @apiDeprecated  friendship actions. Check (#Connections:CreateBBFriendship)
	 * @apiUse apidocForUpdateFriendActionV1
	 * @apiPrivate
	 */
	public function do_action($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$me_id = bp_loggedin_user_id();
		$member_id = (int) $request['id'];
		$action = $request['action'];

		// throw error when no user is found.
		if (!bp_is_user_active($member_id)) {
			return new WP_Error('member_not_found', __('Requested member not found.', 'buddyboss-app'), array('status' => 404));
		}

		switch ($action) {

		case "add_friend":
			$status = friends_check_friendship_status($me_id, $member_id);
			if ('not_friends' == $status) {
				friends_add_friend($me_id, $member_id);
			} else {
				if ('is_friends' == $status) {
					return new WP_Error('friend_added_already', __('You are already friends with this member.', 'buddyboss-app'), array('status' => 400));
				} else {
					return new WP_Error('friendship_requested_already', __('The already is a pending friendship request.', 'buddyboss-app'), array('status' => 400));
				}
			}
			break;

		case "remove_friend":
			$status = friends_check_friendship_status($me_id, $member_id);
			if ('is_friend' == $status) {
				friends_remove_friend($me_id, $member_id);
			} elseif ('pending' == $status) {
				friends_withdraw_friendship($me_id, $member_id);
			} else {
				if ('not_friends' == $status) {
					return new WP_Error('friend_removed_already', __('The member is not a friend.', 'buddyboss-app'), array('status' => 400));
				} else {
					return new WP_Error('friendship_requested_already', __('The already is a pending friendship request.', 'buddyboss-app'), array('status' => 400));
				}
			}
			break;

		case "withdraw_friendship":
			$status = friends_check_friendship_status($me_id, $member_id);
			if ('pending' == $status) {
				friends_withdraw_friendship($me_id, $member_id);
			} else {
				if ('not_friends' == $status) {
					return new WP_Error('friend_removed_already', __('The member is not a friend.', 'buddyboss-app'), array('status' => 400));
				} else {
					return new WP_Error('friendship_requested_already', __('The already is a pending friendship request.', 'buddyboss-app'), array('status' => 400));
				}
			}
			break;

		case "accept_friend":
			$status = friends_check_friendship_status($me_id, $member_id);
			if ('awaiting_response' == $status) {

				if (!$friendship_id = wp_cache_get('friendship_id_' . $member_id . '_' . bp_loggedin_user_id())) {
					$friendship_id = friends_get_friendship_id($member_id, bp_loggedin_user_id());
					wp_cache_set('friendship_id_' . $member_id . '_' . bp_loggedin_user_id(), $friendship_id, 'bp');
				}
				friends_accept_friendship($friendship_id);
			} else {
				if ('is_friends' == $status) {
					return new WP_Error('friend_added_already', __('You are already friends with this member.', 'buddyboss-app'), array('status' => 400));
				} else {
					return new WP_Error('friendship_requested_already', __('The already is a pending friendship request.', 'buddyboss-app'), array('status' => 400));
				}
			}
			break;

		case "reject_friend":
			$status = friends_check_friendship_status($me_id, $member_id);
			if ('awaiting_response' == $status) {

				if (!$friendship_id = wp_cache_get('friendship_id_' . $member_id . '_' . bp_loggedin_user_id())) {
					$friendship_id = friends_get_friendship_id($member_id, bp_loggedin_user_id());
					wp_cache_set('friendship_id_' . $member_id . '_' . bp_loggedin_user_id(), $friendship_id, 'bp');
				}
				friends_reject_friendship($friendship_id);
			} else {
				if ('is_friends' == $status) {
					return new WP_Error('friend_added_already', __('You are already friends with this member.', 'buddyboss-app'), array('status' => 400));
				} else {
					return new WP_Error('friendship_requested_already', __('The already is a pending friendship request.', 'buddyboss-app'), array('status' => 400));
				}
			}
			break;
		}

		$args = array('id' => $member_id);
		if (!empty($request['card_type'])) {
			$args['card_type'] = $request['card_type'];
		}

		$boss_rest_bp_members = new MembersRestApi($this->namespace_slug);
		return $boss_rest_bp_members->get_item($args);
	}

	/**
	 * Return args data for Friends lists endpoint
	 * helps for easily fiilter request data and validation
	 * @return array
	 */
	public function get_friends_args() {

		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';
		$params['per_page']['default'] = 0;
		$params['per_page']['minimum'] = 0;

		unset($params['search']);

		$params["sort"] = array(
			'description' => __('Sorting option.', 'buddyboss-app'),
			'type' => 'string',
			'default' => 'recently_active',
			'enum' => array('recently_active', 'newest', 'alphabetical'),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["user_id"] = array(
			'description' => __('ID of the user whose friends are being retrieved.', 'buddyboss-app'),
			'type' => 'integer',
			'default' => get_current_user_id(),
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["filter"] = array(
			'description' => __('Limit results to those matching a search string.', 'buddyboss-app'),
			'type' => 'string',
			'default' => "",
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/friends Friends
	 * @apiName AllFriends
	 * @apiGroup BuddyPressFriends
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get all friends
	 * @apiDeprecated  Retrieve Friendships. Check (#Connections:GetBBFriendships)
	 * @apiUse apidocForAllFriendsV1
	 * @apiPrivate
	 */
	public function get_items($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		/**
		 * @todo: we should have param validator of sort.
		 */
		$get_function = 'friends_get_' . $request['sort'];

		$friends = $get_function($request['user_id'], $request['per_page'], $request['page'], $request['filter']);

		$retval = array();

		if (!$friends) {
			return rest_ensure_response($retval);
		}

		$friends_array = array();
		foreach ($friends['users'] as $friend) {
			$boss_rest_bp_members = new MembersRestApi($this->namespace_slug);
			$friends_array[] = $this->prepare_response_for_collection(
				$boss_rest_bp_members->prepare_member_item_for_response($friend, $request)
			);
		}

		$retval['users'] = $friends_array;
		$retval['total'] = $friends['total'];

		return rest_ensure_response($retval);
	}

	/**
	 * Check if a given request has access to get information about a specific activity.
	 * @param WP_REST_Request $request Full data about the request.
	 * @return bool
	 */
	public function get_item_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_friends_item_permission', true);
	}

	/**
	 * Check if a given request has access to activity items.
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function get_items_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_friends_items_permission', true);
	}
}
