<?php
namespace BuddyBossApp\Api\BuddyPress;
use BP_Activity_Activity;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

defined('ABSPATH') || exit;

// NOTE : Old classname was class.boss_buddypress_api_activity_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Activity Component
class ActivityRestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/buddypress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->rest_base = buddypress()->activity->id;
		// @todo : Verify below line of code
		$this->rest_base = 'activity';
	}

	public function hooks() {

		add_action('rest_api_init', array($this, "register_routes"), 99);

	}

	/**
	 * Register the plugin routes.
	 */
	public function register_routes() {

		register_rest_route($this->namespace, '/' . $this->rest_base . '/details', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_page_details'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => array(),
			),
		)
		);

		register_rest_route($this->namespace, '/' . $this->rest_base, array(
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'create_item'),
				'permission_callback' => array($this, 'get_create_item_permissions_check'),
				'args' => array(
					'object' => array(
						'description' => __('Name of component.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'item_id' => array(
						'description' => __('The ID of the specific item being recorded.'),
						'type' => 'integer',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'content' => array(
						'description' => __('The content of the activity.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_items'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
				'args' => $this->get_activity_args(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		)
		);

		register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_item'),
				'permission_callback' => '__return_true',
				'args' => array(
					'context' => $this->get_context_param(array('default' => 'view')),
					'object' => array(
						'description' => __('Filters by the `component` column in the database.'),
						'type' => 'array',
						'items' => array(
							'type' => 'string',
						),
						'validate_callback' => 'rest_validate_request_arg',
					),

					'primary_id' => array(
						'description' => __('Filters by the `item_id` column in the database.'),
						'type' => 'integer',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
				'permission_callback' => array($this, 'get_item_permissions_check'),
			),
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'create_item'),
				'permission_callback' => array($this, 'get_manage_item_permissions_check'),
				'args' => array(
					'object' => array(
						'description' => __('Name of component.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'item_id' => array(
						'description' => __('The ID of the specific item being recorded.'),
						'type' => 'integer',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'content' => array(
						'description' => __('The content of the activity.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::DELETABLE,
				'callback' => array($this, 'delete_item'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
			),
			'schema' => array($this, 'get_public_item_schema'),
		)
		);

		register_rest_route($this->namespace, '/' . $this->rest_base . '/action/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'do_action'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => array(
					"action" => array(
						'type' => 'string',
						'default' => 'favorite',
						'enum' => array('favorite', 'unfavorite'),
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		)
		);
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema' => 'http://json-schema.org/draft-04/schema#',
			'title' => 'bp_activity',
			'type' => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id' => array(
					'description' => __('Unique identifier for the object.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'action' => array(
					'description' => __('HTML action for the object, transformed for display.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'content' => array(
					'description' => __('HTML content for the object, transformed for display.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'content_raw' => array(
					'description' => __('content for the object, as it exists in the database.'),
					'type' => 'string',
					'context' => array('view', 'edit'),
				),
				'type' => array(
					'description' => __('Type for the object unique to its type.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'avatar' => array(
					'description' => __('Avatar for the object unique to its type.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'date_recorded' => array(
					'description' => __("The date the object was published, in the site's timezone."),
					'type' => 'string',
					'format' => 'date-time',
					'context' => array('view', 'edit', 'embed'),
				),
				'comment_count' => array(
					'description' => __("Comment count for the object."),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'favorite_count' => array(
					'description' => __("Favorite count for the object."),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'can_comment' => array(
					'description' => __("Whether or not user have the comment access for this object."),
					'type' => 'boolean',
					'context' => array('view', 'edit', 'embed'),
				),
				'can_favorite' => array(
					'description' => __("Whether or not user have the favorite access for this object."),
					'type' => 'boolean',
					'context' => array('view', 'edit', 'embed'),
				),
				'is_favorite' => array(
					'description' => __("Whether or not user has favorite for this object."),
					'type' => 'boolean',
					'context' => array('view', 'edit', 'embed'),
				),
				'can_delete' => array(
					'description' => __("Whether or not user have the delete access for this object."),
					'type' => 'boolean',
					'context' => array('view', 'edit', 'embed'),
				),
				'children' => array(
					'description' => __("comments for this object."),
					'type' => 'array',
					'context' => array('view', 'edit', 'embed'),
				),
				'item_id' => array(
					'description' => __("The ID of the specific item being recorded."),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'secondary_item_id' => array(
					'description' => __("A second ID used to further filter."),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
				'url' => array(
					'description' => __("An URL to single activity."),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
				),
			),
		);

		return $this->add_additional_fields_schema($schema);
	}

	/**
	 * Return args data for Group lists endpoint
	 * helps for easily fiilter request data and validation
	 * @return array
	 */
	public function get_activity_args() {

		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';
		$params['per_page']['default'] = 20;

		unset($params['search']);

		$params["max"] = array(
			'description' => __('Maximum number of results to return.'),
			'type' => 'integer',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["fields"] = array(
			'description' => __('Activity fields to retrieve.'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["sort"] = array(
			'description' => __('Order sort attribute ascending or descending.'),
			'type' => 'string',
			'default' => 'desc',
			'enum' => array('asc', 'desc'),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["include"] = array(
			'description' => __('Array of exact activity IDs to query.'),
			'type' => 'array',
			'items' => array(
				'type' => 'integer',
			),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["exclude"] = array(
			'description' => __('Array of activity IDs to exclude.'),
			'type' => 'array',
			'items' => array(
				'type' => 'integer',
			),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["meta_query"] = array(
			'description' => __('Limit by activity meta by passing an array of meta_query.'),
			'type' => 'array',
			'items' => array(
				'type' => 'array',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["date_query"] = array(
			'description' => __('Limit by date by passing an array of date_query condition.'),
			'type' => 'array',
			'items' => array(
				'type' => 'array',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["search_terms"] = array(
			'description' => __('Limit results by a search term.'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["scope"] = array(
			'description' => __('Use a BuddyPress pre-built filter.'),
			'type' => 'string',
			'default' => 'all',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["user_id"] = array(
			'description' => __('The ID(s) of user(s) whose activity should be fetched.'),
			'type' => 'integer',
			'default' => 0,
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["object"] = array(
			'description' => __('Filters by the `component` column in the database.'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["action"] = array(
			'description' => __('Filters by the `type` column in the database.'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["primary_id"] = array(
			'description' => __('Filters by the `item_id` column in the database.'),
			'type' => 'integer',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["secondary_id"] = array(
			'description' => __('Filters by the `secondary_item_id` column in the database.'),
			'type' => 'integer',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["offset"] = array(
			'description' => __('Return only activity items with an ID greater than or equal to this one.'),
			'type' => 'integer',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["display_comments"] = array(
			'description' => __('How to handle activity comments.'),
			'type' => 'string',
			'enum' => array('threaded', 'stream', false),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params["spam"] = array(
			'description' => __('Spam status. \'ham_only\', \'spam_only\', or false to show all activity regardless of spam status.'),
			'type' => 'string',
			'enum' => array('ham_only', 'spam_only', false),
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {POST} /wp-json/appboss/buddypress/v1/activity Create activity update
	 * @apiName CreateActivityUpdates
	 * @apiGroup BuddyPressActvty
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Create activity update for buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} [id=0] Activity Id
	 * @apiParam {Number} [item_id=0] The ID of the specific item being recorded.
	 * @apiParam {Number} [user_id=bp_loggedin_user_id()] User Id
	 * @apiParam {String} [object=''] Name of component.
	 * @apiParam {String} content The content of the activity.
	 * @apiParam {Array} bbplatform_media_upload_ids (optional) IDS of the platform Media Uploads.
	 * @apiDeprecated  Create activity. Check (#Activity:CreateBBActivity)
	 * @apiPrivate
	 */
	public function create_item($request) {
		$bp = buddypress();

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$validate = apply_filters('boss_rest_buddypress_activity_validate', !empty($request['content']), $request);

		if (!$validate) {
			return new WP_Error('blank_content', __('Please enter some content to post.', 'buddyboss-app'), array('status' => 400));
		}

		$activity_id = !empty($request['id']) ? (int) $request['id'] : 0;
		$item_id = !empty($request['item_id']) ? (int) $request['item_id'] : 0;
		$object = !empty($request['object']) ? $request['object'] : '';
		$user_id = !empty($request['user_id']) ? $request['user_id'] : bp_loggedin_user_id();

		if (!$object && bp_is_active('activity')) {

			//Update user activity start
			$add_content = apply_filters('bp_activity_new_update_content', $request['content']);
			$primary_link = bp_core_get_userlink($user_id, false, true);
			$add_primary_link = apply_filters('bp_activity_new_update_primary_link', $primary_link);

			$args = array(
				'user_id' => $user_id,
				'content' => $add_content,
				'primary_link' => $add_primary_link,
				'component' => buddypress()->activity->id,
				'type' => 'activity_update',
				'error_type' => 'wp_error',
			);
			if (!empty($activity_id)) {
				$args['id'] = $activity_id;
			}

			$activity_id = bp_activity_add($args);

			if (false !== $activity_id && !is_wp_error($activity_id)) {
				$latest_content = (!empty($request['content'])) ? $request['content'] : __('Posted a update.');
				$activity_content = apply_filters('bp_activity_latest_update_content', $latest_content, $latest_content);

				// Add this update to the "latest update" usermeta so it can be fetched anywhere.
				bp_update_user_meta(bp_loggedin_user_id(), 'bp_latest_update', array(
					'id' => $activity_id,
					'content' => $activity_content,
				));

				do_action('bp_activity_posted_update', $request['content'], $user_id, $activity_id);
			}
			//Update activity end

		} elseif ('groups' === $object) {
			if ($item_id && bp_is_active('groups')) {
				$bp = buddypress();
				$bp->current_component = 'groups';
				$bp->groups->current_group = groups_get_group($item_id);

				if (!bp_current_user_can('bp_moderate') && !groups_is_user_member($user_id, $item_id)) {
					return new WP_Error('post_error', __('You are not allowed to post anything to this group until joined.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
				}

				$activity_action = sprintf(__('%1$s posted an update in the group %2$s', 'buddypress'), bp_core_get_userlink($user_id), '<a href="' . bp_get_group_permalink($bp->groups->current_group) . '">' . esc_attr($bp->groups->current_group->name) . '</a>');
				$activity_content = $request['content'];

				$action = apply_filters('groups_activity_new_update_action', $activity_action);
				$content_filtered = apply_filters('groups_activity_new_update_content', $activity_content);

				$args = array(
					'user_id' => $user_id,
					'action' => $action,
					'content' => $content_filtered,
					'type' => 'activity_update',
					'item_id' => $item_id,
					'error_type' => 'wp_error',
				);
				if (!empty($activity_id)) {
					$args['id'] = $activity_id;
				}

				//update group activity start
				$activity_id = groups_record_activity($args);

				if (false !== $activity_id && !is_wp_error($activity_id)) {
					groups_update_groupmeta($item_id, 'last_activity', bp_core_current_time());
					do_action('bp_groups_posted_update', $request['content'], $user_id, $item_id, $activity_id);
				}
			}
		} else {
			/** This filter is documented in bp-activity/bp-activity-actions.php */
			$activity_id = apply_filters('bp_activity_custom_update', false, $object, $item_id, $request['content']);
		}

		if (false === $activity_id) {
			return new WP_Error('post_error', __('There was a problem posting your update. Please try again.', 'buddyboss-app'), array('status' => 500));
		} elseif (is_wp_error($activity_id) && $activity_id->get_error_code()) {
			return new WP_Error('post_error', $activity_id->get_error_message(), array('status' => 500));
		}

		$show_hidden = false;
		if ('groups' == $object && !empty($item_id)) {
			$show_hidden = (bool) (groups_is_user_member(bp_loggedin_user_id(), $item_id) || bp_current_user_can('bp_moderate'));
		}

		$activity = bp_activity_get(array('in' => array($activity_id), 'show_hidden' => $show_hidden));
		if (!empty($activity['activities'])) {
			$fields_update = $this->update_additional_fields_for_object((array) $activity['activities'][0], $request);

			if (is_wp_error($fields_update)) {
				return $fields_update;
			}
		}

		$last_recorded = !empty($request['since']) ? date('Y-m-d H:i:s', intval($request['since'])) : 0;
		if ($last_recorded) {
			$bp->activity->last_recorded = $last_recorded;
			add_filter('bp_get_activity_css_class', 'bp_activity_newest_class', 10, 1);
		}

		if (!empty($last_recorded)) {
			remove_filter('bp_get_activity_css_class', 'bp_activity_newest_class', 10, 1);
		}

		return $this->get_item(array('id' => $activity_id, 'object' => $object, 'primary_id' => $item_id));
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {DELETE} /wp-json/appboss/buddypress/v1/activity/:id Delete activity update
	 * @apiName DeleteActivityUpdate
	 * @apiGroup BuddyPressActvty
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Delete activity update
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Activity ID
	 * @apiDeprecated  Delete single activity. Check (#Activity:DeleteBBActivity)
	 * @apiPrivate
	 */
	public function delete_item($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to delete activity.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (empty($request['id']) || !is_numeric($request['id'])) {
			return new WP_Error('wrong_activity_id', __('Wrong activity id.', 'buddyboss-app'), array('status' => 404));
		}

		$activity = new BP_Activity_Activity((int) $request['id']);

		// Check access.
		if (!bp_activity_user_can_delete($activity)) {
			return new WP_Error('no_activity_access', __('You dont have permission to delete this activity.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		/** This action is documented in bp-activity/bp-activity-actions.php */
		do_action('bp_activity_before_action_delete_activity', $activity->id, $activity->user_id);

		if (!bp_activity_delete(array('id' => $activity->id, 'user_id' => $activity->user_id))) {
			return new WP_Error('deleting_problem', __('There was a problem when deleting. Please try again.', 'buddyboss-app'), array('status' => 500));
		}

		/** This action is documented in bp-activity/bp-activity-actions.php */
		do_action('bp_activity_action_delete_activity', $activity->id, $activity->user_id);

		return rest_ensure_response(array("deleted" => true));
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/activity/details Activity page details
	 * @apiName GetBpPageDetails
	 * @apiGroup BuddyPressActvty
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get activity page details(includes nav, filters, post_in and further)
	 * @apiDeprecated  Retrieve activity details (includes nav, filters and post_in). Check (#Activity:GetBBActivitiesDetails)
	 * @apiUse apidocForGetBpPageDetailsV1
	 * @apiPrivate
	 */
	public function get_page_details($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		return rest_ensure_response(array(
			'nav' => $this->activities_tabs(),
			'filters' => $this->activities_filters(),
			'post_in' => $this->activities_post_in(),
		));

	}

	/**
	 * Return Activity Post in details.
	 * @return array
	 */
	public function activities_post_in() {

		$post_in = array();

		$post_in[0] = __('My Profile', 'buddypress');

		if(function_exists('bp_has_groups')) {

		if (bp_has_groups('user_id=' . bp_loggedin_user_id() . '&type=alphabetical&max=100&per_page=100&populate_extras=0')):
			while (bp_groups()): bp_the_group();
				$post_in[bp_get_group_id()] = bp_get_group_name();
			endwhile;
		endif;

        }

        return $post_in;

	}

	public function activities_filters() {

		/**
		 * 3rd party plugins support.
		 */

		// BuddyPress Docs @https://wordpress.org/plugins/buddypress-docs/
		if (function_exists("bp_docs_load_activity_filter_options")) {
			bp_docs_load_activity_filter_options();
		}

		/**
		 * Get filters options
		 **/
		$filters = array(
			'-1' => __('-- Everything --', 'buddypress'),
		);
		$actions = bp_activity_get_actions_for_context('activity');

		foreach ($actions as $action) {
			// Friends activity collapses two filters into one.
			if (in_array($action['key'], array('friendship_accepted', 'friendship_created'))) {
				$action['key'] = 'friendship_accepted,friendship_created';
			}

			$filters[$action['key']] = $action['label'];
		}

		/**
		 * Filters the options available in the activity filter dropdown.
		 *
		 * @see plugins/buddypress/bp-activity/bp-activity-template.php:3862
		 */
		$filters = apply_filters('bp_get_activity_show_filters_options', $filters, 'activity');

		/**
		 * Capture html version of 3rd party filters.
		 */
		ob_start();

		/**
		 * Fires inside the select input for group activity filter options.
		 *
		 * @since 1.2.0
		 */
		do_action('bp_group_activity_filter_options');

		$html_filters_options = ob_get_contents();

		ob_get_clean();

		preg_match_all('/<option.*?[value=][\'|"](.*?)[\'|"][(.*?>)|>](.*?)<\/option>/', $html_filters_options, $matches);

		if (isset($matches[1])) {
			foreach ($matches[1] as $k => $match) {
				$filters[$match] = $matches[2][$k];
			}
		}

		/**
		 * Filters the options available in the activity filter only for api.
		 */
		$filters = apply_filters('boss_rest_buddypress_activity_show_filters_options', $filters);

		return $filters;
	}

	/**
	 * Return Activities Tabs.
	 * @return array|mixed|void
	 */
	public function activities_tabs() {

		/**
		 * Get Navigation item for directory page
		 **/
		$tabs = array(
			'all' => array(
				'title' => __('All Members ', 'buddypress'),
				'count' => bp_get_total_member_count(),
			),
		);

		if (is_user_logged_in()) {

			/**
			 * Retrieves items belonging to the friends of a user.
			 */
			if (bp_is_active('friends')) {
				if (bp_get_total_friend_count(bp_loggedin_user_id())) {
					$tabs['friends'] = array(
						'title' => __('My Friends', 'buddypress'),
						'count' => bp_get_total_friend_count(bp_loggedin_user_id()),
					);
				}
			}

			/**
			 * Retrieves items belonging to groups to which a user belongs to.
			 */
			if (bp_is_active('groups')) {
				if (bp_get_total_group_count_for_user(bp_loggedin_user_id())) {
					$tabs['groups'] = array(
						'title' => __('My Groups', 'buddypress'),
						'count' => bp_get_total_group_count_for_user(bp_loggedin_user_id()),
					);
				}
			}

			/**
			 * Retrieves a user's favorited activity items.
			 */
			if (bp_get_total_favorite_count_for_user(bp_loggedin_user_id())) {
				$tabs['favorites'] = array(
					'title' => __('My Favorites', 'buddypress'),
					'count' => bp_get_total_favorite_count_for_user(bp_loggedin_user_id()),
				);
			}

			/**
			 * Retrieves items where a user has received an @-mention.
			 */
			if (bp_activity_do_mentions()) {
				$tabs['mentions'] = array(
					'title' => __('Mentions ', 'buddypress'),
					'count' => bp_get_total_mention_count_for_user(bp_loggedin_user_id()),
				);
			}

		}

		$tabs = apply_filters('boss_rest_buddypress_activity_type_tabs', $tabs);

		return $tabs;
	}

	/**
	 * Check if a given request has access to activity items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_items_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_activity_items_permission', true);
	}

	/**
	 * Check if a given request has access to activity item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_item_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_activity_item_permission', true);
	}

	/**
	 * Check if a given request has access to create activity.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_create_item_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_activity_create_permission', true);
	}

	/**
	 * Check if a given request has access to manage activity.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_manage_item_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_groups_manage_permission', true);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/activity Activities
	 * @apiName GetBpActivities
	 * @apiGroup BuddyPressActvty
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get BP activities
	 * @apiDeprecated  Retrieve activities. Check (#Activity:GetBBActivities)
	 * @apiUse apidocForGetBpActivitiesV1
	 * @apiPrivate
	 */
	public function get_items($request) {
		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		global $activities_template;

		$args = apply_filters('boss_rest_buddypress_activity_args', $request->get_params(), $request);
		$activities = $this->get_activities($args);

		$data = array();
		foreach ($activities as $activity) {
			$post = $this->prepare_item_for_response($activity, $request);
			$data[] = $this->prepare_response_for_collection($post);
		}

		$response = rest_ensure_response($data);
		$response->header('X-WP-Total', (int) $activities_template->total_activity_count);

		return $response;
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/activity/:id Activity
	 * @apiName GetBpActivity
	 * @apiGroup BuddyPressActvty
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get BP activity
	 * @apiDeprecated  Retrieve single activity. Check (#Activity:GetBBActivity)
	 * @apiUse apidocForGetBpActivityV1
	 * @apiPrivate
	 */
	public function get_item($request) {
		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		global $activities_template;

		$filter = array(
			'include' => $request['id'],
		);

		if (!empty($request['object'])) {
			$filter['object'] = $request['object'];
		}

		if (!empty($request['primary_id'])) {
			$filter['primary_id'] = $request['primary_id'];
		}

		// New platform required scope and old platfrom and buddypress not required scope.
		$filter['scope'] = isset( $request['scope'] ) ? $request['scope'] : 'all';

		// Ensure all activity come in get item. some can be missed in thread thats why we need to provide some extra param.
		$filter["display_comments"] = "stream"; # include thread comments.
		$filter["show_hidden"] = true; # include hidden activity from user.

		/**
		 * If we want to change the single activity argument so we can use this filter.
		 */
		$filter = apply_filters( 'boss_rest_buddypress_single_activity_args', $filter, $request );

		$activities = $this->get_activities($filter);

		$data = array();
		/**
		 * Catch the first item from loop.
		 */
		foreach ($activities as $activity) {
			$post = $this->prepare_item_for_response($activity, $request);
			$data = $this->prepare_response_for_collection($post);
			break;
		}

		$response = rest_ensure_response($data);
		$response->header('X-WP-Total', (int) $activities_template->total_activity_count);

		return $response;
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/activity/action/:id Update activity
	 * @apiName UpdateActivity
	 * @apiGroup BuddyPressActvty
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update action under activity for buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Activity ID
	 * @apiParam {String=favorite,unfavorite} [action=favorite] Action to perform
	 * @apiDeprecated  Make activity favorite/unfavorite. Check (#Activity:UpdateBBActivityFavorite)
	 * @apiPrivate
	 */
	public function do_action($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (empty($request['id']) || !is_numeric($request['id'])) {
			return new WP_Error('wrong_activity_id', __('Wrong activity id.', 'buddyboss-app'), array('status' => 404));
		}

		$id = (int) $request['id'];

		$activity = new BP_Activity_Activity($id);

		switch ($request['action']) {
		case 'favorite':
			if (!bp_activity_can_favorite($activity)) {
				return new WP_Error('no_activity_access', __('You don\'t have permission to favorite this activity.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
			} else {
				$success = bp_activity_add_user_favorite($activity->id);
			}
			break;
		case 'unfavorite':
			if (!bp_activity_can_favorite($activity)) {
				return new WP_Error('no_activity_access', __('You don\'t have permission to favorite this activity.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
			} else {
				$success = bp_activity_remove_user_favorite($activity->id);
			}
		}

		global $wp_rest_server;

		$primary_id = $activity->item_id;
		$object = $activity->component;

		$args = array('id' => $id, 'object' => $object, 'primary_id' => $primary_id);
		if (!empty($request['_embed'])) {
			$args['_embed'] = true;
		}

		$response = $this->get_item($args);

		return $wp_rest_server->response_to_data(rest_ensure_response($response), isset($request['_embed']) ? 1 : 0);
	}

	/**
	 * get_activity function.
	 *
	 * @access public
	 *
	 * @param $args
	 *
	 * @return void
	 */
	public function get_activities($args) {

		$data = array();

		if (!empty($args['object']) && 'groups' == $args['object'] && !empty($args['primary_id'])) {
			$args['show_hidden'] = (bool) (groups_is_user_member(bp_loggedin_user_id(), $args['primary_id']) || bp_current_user_can('bp_moderate'));
		}

		$args['count_total'] = true;
		if (bp_has_activities($args)) {
			$activities = array();

			while (bp_activities()) {
				bp_the_activity();
				$activity = $this->prepare_activity_for_response();
				$activity = apply_filters('boss_rest_buddypress_activity_item', $activity);
				$activities[] = $activity;
			}

			$data = apply_filters('boss_rest_buddypress_activities_items', $activities);
		}
//		else {
		//			return new WP_Error( 'bp_json_activity', __( 'No Activity Found.',  'buddyboss-app' ), array( 'status' => 200 ) );
		//		}

		return $data;
	}

	public function prepare_activity_for_response($comment = false) {
		global $activities_template;

		/**
		 * Remove read more link from activity.
		 */
		remove_filter('bp_get_activity_content_body', 'bp_activity_truncate_entry', 5);

		$avatar_thumb_rendered = bp_get_activity_avatar(array(
			'type' => 'thumb',
		));

		$avatar_thumb_url = boss_bp_get_activity_avatar_url(array(
			'type' => 'thumb',
		));

		if (!$comment) {

			$activity = array(
				'id' => bp_get_activity_id(),
				'avatar' => array(
					'thumb_raw' => $avatar_thumb_url,
					'thumb_rendered' => $avatar_thumb_rendered,
				),
				'type' => bp_get_activity_type(),
				'action' => bp_get_activity_action(),
				'content' => bp_get_activity_content_body(),
				'content_stripped' => html_entity_decode(strip_tags($activities_template->activity->content)),
				'date_recorded' => boss_buddypress_rest_api_convert_date(bp_get_activity_date_recorded()),
				'date_gmt' => mysql_to_rfc3339( $activities_template->activity->date_recorded ),
				'user' => array(
					'user_id' => bp_get_activity_user_id(),
					'display_name' => bp_core_get_user_displayname(bp_get_activity_user_id()),
					'user_domain' => bp_core_get_user_domain(bp_get_activity_user_id()),
				),
				'comment_count' => bp_activity_get_comment_count(),
				'favorite_count' => $this->get_activity_favorite_count(bp_get_activity_id()),
				'can_comment' => ('activity_comment' == bp_get_activity_type()) ? bp_activity_can_comment_reply() : bp_activity_can_comment(),
				'can_favorite' => bp_activity_can_favorite(),
				'is_favorite' => bp_get_activity_is_favorite(),
				'can_delete' => bp_activity_user_can_delete(),
				'item_id' => bp_get_activity_item_id(),
				'secondary_item_id' => bp_get_activity_secondary_item_id(),
				'url' => bp_activity_get_permalink(bp_get_activity_id()),
				'children' => array(),
			);

			$activity['children'] = $this->bp_api_activity_get_comments($activities_template->activity);

		} else {

			// BuddyPress don't have any action return function for comment but in api we have.
			$activity_comment_action = apply_filters("rest_boss_activity_comment_action", $activities_template->activity->current_comment->action);

			$activity = array(
				'id' => bp_get_activity_comment_id(),
				'avatar' => array(
					'thumb_raw' => $avatar_thumb_url,
					'thumb_rendered' => $avatar_thumb_rendered,
				),
				'type' => 'activity_comment',
				'action' => $activity_comment_action,
				'content' => bp_get_activity_comment_content(),
				'content_stripped' => html_entity_decode(strip_tags($activities_template->activity->current_comment->content)),
				'date_recorded' => boss_buddypress_rest_api_convert_date(bp_get_activity_comment_date_recorded_raw()),
				'date_gmt' => mysql_to_rfc3339($activities_template->activity->current_comment->date_recorded),
				'user' => array(
					'user_id' => bp_get_activity_comment_user_id(),
					'display_name' => bp_core_get_user_displayname(bp_get_activity_comment_user_id()),
					'user_domain' => bp_core_get_user_domain(bp_get_activity_comment_user_id()),
				),
				'comment_count' => bp_activity_get_comment_count(),
				'favorite_count' => $this->get_activity_favorite_count(bp_get_activity_comment_id()),
				'can_comment' => bp_activity_can_comment_reply(),
				'can_favorite' => bp_activity_can_favorite(),
				'is_favorite' => bp_get_activity_is_favorite(),
				'can_delete' => bp_activity_user_can_delete(),
				'item_id' => $activities_template->activity->current_comment->item_id,
				'secondary_item_id' => $activities_template->activity->current_comment->secondary_item_id,
				'url' => bp_activity_get_permalink(bp_get_activity_comment_id()),
				'children' => array(),
			);

			$activity['children'] = $this->bp_api_activity_get_comments($comment);
			$activity['depth'] = bp_activity_get_comment_depth($activities_template->activity->current_comment);
		}

		return $activity;
	}

	public function bp_api_activity_get_comments($activity) {

		global $activities_template;

		$children = array();

		if (empty($activity->children)) {
			return $children;
		}

		foreach ($activity->children as $child) {

			if (empty($child)) {
				continue;
			}

			// Put the comment into the global so it's available to filters.
			$activities_template->activity = $child;
			$activities_template->activity->current_comment = $child;

			$children[] = $this->prepare_activity_for_response($child);

			$activities_template->activity = $activity;

		}

		/**
		 * Filters Activity's comments object before rest prepared
		 *
		 * @param array $children Children Activity.
		 * @param object $activity Activity Object.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'boss_rest_buddypress_activity_children_items', $children, $activity );
	}

	/**
	 * get favorite count for activity
	 *
	 * @param $activity_id
	 *
	 * @return int|mixed
	 */
	public function get_activity_favorite_count($activity_id) {

		$fav_count = bp_activity_get_meta($activity_id, 'favorite_count', true);
		$count = !empty($fav_count) ? $fav_count : 0;

		return $count;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array item object.
	 *
	 * @return array Links for the given data.
	 */
	public function prepare_links($data) {

		$base = sprintf('/%s/%s', $this->namespace, $this->rest_base);

		// Entity meta
		$links = array(
			'self' => array(
				'href' => rest_url(trailingslashit($base) . $data["id"]),
			),
			'collection' => array(
				'href' => rest_url($base),
			),
		);

		if (bp_is_active('groups') && in_array($data['type'], array('joined_group', 'group_new_avatar', 'group_new_cover', 'created_group', 'group_details_updated'))) {
			$links['bp:group'] = array(
				'href' => rest_url(sprintf('/%s/%s', $this->namespace, buddypress()->groups->id) . '/' . $data['item_id']),
				'embeddable' => true,
			);
		}

		/**
		 * Users Embed
		 */
		$users = array();
		$users[] = $data["user"];
		foreach ($users as $user) {
			$links['users'] = array(
				'href' => rest_url('/wp/v2/users/' . $user['user_id']),
				'embeddable' => true,
			);
		}

		return $links;

	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param                 $data
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response($data, $request) {

		$context = !empty($request['context']) ? $request['context'] : 'view';
		$data = $this->add_additional_fields_to_object($data, $request);
		$data = $this->filter_response_by_context($data, $context);

		// Wrap the data in a response object.
		$response = rest_ensure_response($data);

		$response->add_links($this->prepare_links($data));

		return apply_filters('rest_prepare_boss_buddypress_activity', $response, $data, $request);

	}
}
