<?php
namespace BuddyBossApp\Api\BuddyPress;
use BuddyBossApp\Api\BuddyPress\ActivityRestApi;
use BP_Activity_Activity;
use stdClass;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Server;

defined('ABSPATH') || exit;

// NOTE : Old classname was class.boss_buddypress_api_activity_comment_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Activity Component
class ActivityCommentRestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/buddypress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->rest_base = buddypress()->activity->id;
		// @todo : Verify below line of code
		$this->rest_base = 'activity';
	}

	public function hooks() {
		add_action('rest_api_init', array($this, "register_routes"), 99);
	}

	/**
	 * Register the plugin routes.
	 */
	public function register_routes() {
		register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>\d+)/comment/', array(
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'create_item'),
				'permission_callback' => array($this, 'get_create_item_permissions_check'),
				'args' => array(
					'parent_id' => array(
						'description' => __('The ID of the parent activity item.'),
						'type' => 'integer',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'content' => array(
						'description' => __('The content of the comment.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_comments'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
			),
		)
		);
		register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<a_id>\d+)/comment/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::DELETABLE,
				'callback' => array($this, 'delete_comment'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
			),
		)
		);
	}

	/**
	 * Check if a given request has access to comment items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_items_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_comment_items_permission', true);
	}

	/**
	 * Check if a given request has access to comment item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_item_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_comment_item_permission', true);
	}

	/**
	 * Check if a given request has access to create comment.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_create_item_permissions_check($request) {
		return apply_filters('boss_rest_buddypress_comment_create_permission', true);
	}

	/**
	 * @api {POST} /wp-json/appboss/buddypress/v1/activity/:id/comment Create activity comment
	 * @apiName CreateActivityComment
	 * @apiGroup BuddyPressActvty.Comment
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Create comment under activity for buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} parent_id ID of the parent activity item.
	 * @apiParam {Number} id Root activity ID
	 * @apiParam {String} content The content of the comment.
	 * @apiDeprecated  Create comment under activity. Check (#Activity:CreateActivityComment)
	 * @apiPrivate
	 */
	/**
	 * @return WP_Error
	 * @global BP_Activity_Template $activities_template
	 * @since 1.2.0
	 */
	public function create_item($request) {
		global $activities_template;

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform this action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (empty($request['content'])) {
			return new WP_Error('blank_content', __('Please do not leave the comment content blank.', 'buddyboss-app'), array('status' => 400));
		}

		if (empty($request['id'])) {
			return new WP_Error('no_parent', __('Root activity not found.', 'buddyboss-app'), array('status' => 404));
		}

		$comment_id = bp_activity_new_comment(array(
			'activity_id' => $request['id'],
			'content' => $request['content'],
			'parent_id' => $request['parent_id'],
			'error_type' => 'wp_error',
		));

		if (is_wp_error($comment_id)) {
			return new WP_Error('comment_error', esc_html($comment_id->get_error_message()), array('status' => 500));
		}

		// Load the new activity item into the $activities_template global.
		bp_has_activities('display_comments=stream&hide_spam=false&show_hidden=true&include=' . $comment_id);

		// Swap the current comment with the activity item we just loaded.
		if (isset($activities_template->activities[0])) {
			$activities_template->activity = new stdClass();
			$activities_template->activity->id = $activities_template->activities[0]->item_id;
			$activities_template->activity->current_comment = $activities_template->activities[0];

			// Because the whole tree has not been loaded, we manually
			// determine depth.
			$depth = 1;
			$parent_id = (int) $activities_template->activities[0]->secondary_item_id;
			while ($parent_id !== (int) $activities_template->activities[0]->item_id) {
				$depth++;
				$p_obj = new BP_Activity_Activity($parent_id);
				$parent_id = (int) $p_obj->secondary_item_id;
			}
			$activities_template->activity->current_comment->depth = $depth;
		}

		unset($activities_template);

		$activity = bp_activity_get(array(
			'in' => $request['id'],
			'display_comments' => 'stream',
			'show_hidden' => true,
		));

		if (!empty($activity['activities'][0])) {
			$object = $activity['activities'][0]->component;
			$item_id = $activity['activities'][0]->item_id;

			return rest_ensure_response($this->get_single_comment(array('id' => $comment_id, 'object' => $object, 'primary_id' => $item_id)));
		}

		return new WP_Error('no_parent', __('Root activity not found.', 'buddyboss-app'), array('status' => 404));
	}

	/**
	 * @param $request
	 * @return array|bool|WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/activity/:id/comment Activity comments
	 * @apiName ListActivityComments
	 * @apiGroup BuddyPressActvty.Comment
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get all comments under activity for buddypress component
	 * @apiDeprecated  Get all comments for an activity. Check (#Activity:GetActivityComment)
	 * @apiUse apidocForListActivityCommentsV1
	 * @apiPrivate
	 */
	public function get_comments($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$data = array();

		if (empty($request['id'])) {
			return new WP_Error('no_parent', __('Activity not found.', 'buddyboss-app'), array('status' => 500));
		}

		if (bp_has_activities(array('include' => $request['id']))) {
			while (bp_activities()) {
				bp_the_activity();
				$data = $this->bp_api_activity_get_comments();
			}
		}

		return $data;
	}

	/**
	 * @param $request
	 * @return bool|WP_Error
	 * @api {DELETE} /wp-json/appboss/buddypress/v1/activity/:a_id/comment/:id  Delete activity comment
	 * @apiName DeleteActivityComment
	 * @apiGroup BuddyPressActvty.Comment
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Delete comment under activity for buddypress component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} a_id Activity ID
	 * @apiParam {Number} id Comment ID
	 * @apiDeprecated  Delete single activity. Check (#Activity:DeleteBBActivity)
	 * @apiPrivate
	 */
	public function delete_comment($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform this action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (empty($request['id']) || !is_numeric($request['id'])) {
			return new WP_Error('wrong_comment_id', __('Wrong comment id.', 'buddyboss-app'), array('status' => 404));
		}

		$id = (int) $request['id'];

		$comment = new BP_Activity_Activity($id);

		// Check access
		if (!bp_current_user_can('bp_moderate') && $comment->user_id != bp_loggedin_user_id()) {
			return new WP_Error('permission_error', __('You do not have permission to delete this item.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		// Call the action before the delete so plugins can still fetch information about it
		do_action('bp_activity_before_action_delete_activity', $id, $comment->user_id);

		if (!bp_activity_delete_comment($comment->item_id, $comment->id)) {
			return new WP_Error('error', __('There was a problem deleting this item.', 'buddyboss-app'), array('status' => 500));
		}

		do_action('bp_activity_action_delete_activity', $id, $comment->user_id);

		return true; //rest_ensure_response( $this->get_comments( $comment->item_id ) );

	}

	/**
	 * Get the comment markup for an activity item.
	 *
	 * @since 1.2.0
	 *
	 * @return bool
	 * @global object $activities_template {@link BP_Activity_Template}
	 *
	 */
	public function bp_api_activity_get_comments() {
		global $activities_template;

		if (empty($activities_template->activity->children)) {
			return false;
		}
		$boss_rest_bp_activity = new ActivityRestApi($this->namespace_slug);
		return $boss_rest_bp_activity->prepare_activity_for_response();
	}

	public function get_single_comment($request) {
		global $activities_template;

		$item_id = !empty($request['primary_id']) ? (int) $request['primary_id'] : 0;
		$object = !empty($request['object']) ? $request['object'] : '';

		if (empty($request['id'])) {
			return new WP_Error('no_parent', __('There was an error posting your reply. Please try again.', 'buddyboss-app'), array('status' => 500));
		}

		$show_hidden = false;
		if ('groups' == $object && !empty($item_id)) {
			$show_hidden = (bool) (groups_is_user_member(bp_loggedin_user_id(), $item_id) || bp_current_user_can('bp_moderate'));
		}

		$activity = bp_activity_get(array(
			'in' => $request['id'],
			'display_comments' => 'stream',
			'show_hidden' => $show_hidden,
		));

		if (!empty($activity['activities'][0])) {

			$activities_template->activity = $activity['activities'][0];
			$activities_template->activity->depth = bp_activity_get_comment_depth($activities_template->activity->id);

			$boss_rest_bp_activity = new ActivityRestApi($this->namespace_slug);

			return $boss_rest_bp_activity->prepare_activity_for_response();
		}

		return new WP_Error('no_comment', __('Comment activity not found.', 'buddyboss-app'), array('status' => 404));
	}
}
