<?php
namespace BuddyBossApp\Api\BuddyPress;
use BP_Attachment_Cover_Image;
use stdClass;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Server;
use WP_User;
use WP_Post;
use WP_Comment;

defined('ABSPATH') || exit;

// NOTE : Old classname was class.boss_buddypress_api_account_settings_rest_api. By Ketan, Oct-2019
// Endpoint for BuddyBoss Activity Component
class AccountSettingsRestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/buddypress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->rest_base = buddypress()->settings->id;
		// @todo : Verify below line of code
		$this->rest_base = 'settings';
	}

	public function hooks() {
		add_action('rest_api_init', array($this, "register_routes"), 99);
	}

	/**
	 * Register the routes for the objects of the controller.
	 */
	public function register_routes() {
		register_rest_route($this->namespace, '/' . 'settings/general', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_general_settings_fields'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					'displayed_user_id' => array(
						'description' => __('Displayed user id.'),
						'type' => 'integer',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'update_general_settings'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					'displayed_user_id' => array(
						'description' => __('Displayed user id.'),
						'type' => 'integer',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'email' => array(
						'description' => __('User email id.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'pwd' => array(
						'description' => __('User current password.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'pass1' => array(
						'description' => __('New password.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'pass2' => array(
						'description' => __('Confirm password.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		)
		);
		register_rest_route($this->namespace, '/' . 'settings/visibility', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_visibility_levels'),
				'permission_callback' => array($this, 'bp_settings_permission'),
			),
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'update_visibility_settings'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					'displayed_user_id' => array(
						'description' => __('Displayed user id.'),
						'type' => 'integer',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'field_ids' => array(
						'description' => __('User xprofile fields ids.'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		)
		);
		register_rest_route($this->namespace, '/' . 'settings/notifications', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_notification_settings'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					'displayed_user_id' => array(
						'description' => __('Displayed user id.'),
						'type' => 'integer',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'update_notification_settings'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					'displayed_user_id' => array(
						'description' => __('Displayed user id.'),
						'type' => 'integer',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'notifications' => array(
						'description' => __('Notification settings.'),
						'type' => 'object',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		)
		);
		register_rest_route($this->namespace, '/' . 'bp_attachment/cover', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_cover'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					"item_id" => array(
						'description' => __('User id.'),
						'type' => 'integer',
						'default' => 0,
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'update_cover'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					"item_id" => array(
						'description' => __('User id.'),
						'type' => 'integer',
						'default' => 0,
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::DELETABLE,
				'callback' => array($this, 'delete_cover'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					"item_id" => array(
						'description' => __('User id.'),
						'type' => 'integer',
						'default' => 0,
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		)
		);

		register_rest_route($this->namespace, '/' . 'bp_attachment/avatar', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_avatar'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					"item_id" => array(
						'description' => __('User id.'),
						'type' => 'integer',
						'default' => 0,
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'update_avatar'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					"item_id" => array(
						'description' => __('User id.'),
						'type' => 'integer',
						'default' => 0,
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			array(
				'methods' => WP_REST_Server::DELETABLE,
				'callback' => array($this, 'delete_avatar'),
				'permission_callback' => array($this, 'bp_settings_permission'),
				'args' => array(
					"item_id" => array(
						'description' => __('User id.'),
						'type' => 'integer',
						'default' => 0,
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		)
		);

		// Override gravatar to buddypress avatar if available
		add_filter("pre_get_avatar_data", array($this, "pre_get_avatar_data"), 9, 2);
	}

	/**
	 * @param $request
	 * @return array|WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/settings/general Account settings
	 * @apiName BuddyPressAccountSettings
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get Buddypress account settings
	 * @apiDeprecated  Retrieve account setting options based on navigation tab. Check (#Account Settings:GetBBAccountSettingsOptions)
	 * @apiUse apidocForGeneralSettingsFieldsV1
	 * @apiPrivate
	 */
	public function get_general_settings_fields($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		if (!empty($request->get_param('displayed_user_id')) && get_current_user_id() != $request->get_param('displayed_user_id')) {
			if ((is_super_admin() || current_user_can('administrator'))) {
				$displayed_user_id = $request->get_param('displayed_user_id');
			} else {
				return new WP_Error('member_permission_denied', __('You are not allowed to get member data.', 'buddyboss-app'), array('status' => 500));
			}
		} else {
			$displayed_user_id = get_current_user_id();
		}

		$displayed_user = bp_core_get_core_userdata($displayed_user_id);

		$response = array(
			'user_email' => $displayed_user->user_email,
			'username' => $displayed_user->user_login,
			'user_profile_url' => bp_core_get_user_domain($displayed_user->ID),
		);

		$pending_email = bp_get_user_meta($displayed_user_id, 'pending_email_change', true);

		if (!empty($pending_email['newemail'])) {
			$pending_email['dismiss_change_url'] = esc_url(wp_nonce_url(bp_core_get_user_domain(get_current_user_id()) . bp_get_settings_slug() . '/?dismiss_email_change=1', 'bp_dismiss_email_change'));
			$response['pending_email'] = $pending_email;
		}

		return $response;

	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/settings/general Update general settings
	 * @apiName UpdateBuddyPressAccount
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDeprecated  Update account setting options based on navigation tab. Check (#Account Settings:UpdateBBAccountSettingsOptions)
	 * @apiUse apidocForUpdateBuddyPressAccountV1
	 * @apiPrivate
	 */
	public function update_general_settings($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		if (!empty($request->get_param('displayed_user_id')) && get_current_user_id() != $request->get_param('displayed_user_id')) {
			if ((is_super_admin() || current_user_can('administrator'))) {
				$displayed_user_id = $request->get_param('displayed_user_id');
			} else {
				return new WP_Error('member_permission_denied', __('You are not allowed to update member data.', 'buddyboss-app'), array('status' => 500));
			}
		} else {
			$displayed_user_id = get_current_user_id();
		}
		$displayed_user = bp_core_get_core_userdata($displayed_user_id);

		$email_error = false; // invalid|blocked|taken|empty|nochange
		$pass_error = false; // invalid|mismatch|empty|nochange
		$pass_changed = false; // true if the user changes their password
		$email_changed = false; // true if the user changes their email
		$feedback_type = 'error'; // success|error
		$feedback = array(); // array of strings for feedback.
		$login = false;

		if (!is_super_admin() && (!empty($request['pwd']) && !wp_check_password($request['pwd'], $displayed_user->user_pass, get_current_user_id()))) {
			$pass_error = 'invalid';
		} else {
			$login = true;
		}

		// Validate the user again for the current password when making a big change.
		if ((is_super_admin()) || $login) {

			$update_user = get_userdata($displayed_user_id);

			/* Email Change Attempt ***************************************** */

			if (!empty($request['email'])) {

				// What is missing from the profile page vs signup -
				// let's double check the goodies.
				$user_email = sanitize_email(esc_html(trim($request['email'])));
				$old_user_email = $displayed_user->user_email;

				// User is changing email address.
				if ($old_user_email != $user_email) {

					// Run some tests on the email address.
					$email_checks = bp_core_validate_email_address($user_email);

					if (true !== $email_checks) {
						if (isset($email_checks['invalid'])) {
							$email_error = 'invalid';
						}

						if (isset($email_checks['domain_banned']) || isset($email_checks['domain_not_allowed'])) {
							$email_error = 'blocked';
						}

						if (isset($email_checks['in_use'])) {
							$email_error = 'taken';
						}
					}

					// Store a hash to enable email validation.
					if (false === $email_error) {
						$hash = bbapp_generate_password( 32, false );

						$pending_email = array(
							'hash' => $hash,
							'newemail' => $user_email,
						);

						bp_update_user_meta(get_current_user_id(), 'pending_email_change', $pending_email);
						$verify_link = bp_core_get_user_domain($displayed_user_id) . bp_get_settings_slug() . '/?verify_email_change=' . $hash;

						// Send the verification email.
						$args = array(
							'tokens' => array(
								'displayname' => bp_core_get_user_displayname(get_current_user_id()),
								'old-user.email' => $old_user_email,
								'user.email' => $user_email,
								'verify.url' => esc_url($verify_link),
							),
						);
						bp_send_email('settings-verify-email-change', get_current_user_id(), $args);

						// We mark that the change has taken place so as to ensure a
						// success message, even though verification is still required.
						$request['email'] = $update_user->user_email;
						$pending_email = bp_get_user_meta($displayed_user_id, 'pending_email_change', true);

						if (!empty($pending_email)) {
							$pending_email['dismiss_change_url'] = esc_url(wp_nonce_url(bp_core_get_user_domain($displayed_user_id) . bp_get_settings_slug() . '/?dismiss_email_change=1'), 'bp_dismiss_email_change');
							$displayed_user->pending_email = $pending_email;
						}

						$email_changed = true;
					}

					// No change.
				} else {
					$email_error = false;
				}

				// Email address cannot be empty.
			} else {
				$email_error = 'empty';
			}

			/* Password Change Attempt ************************************** */

			if (!empty($request['pass1']) && !empty($request['pass2'])) {

				if (($request['pass1'] == $request['pass2']) && !strpos(" " . $request['pass1'], "\\")) {

					// Password change attempt is successful.
					if ((!empty($request['pwd']) && $request['pwd'] != $request['pass1']) || is_super_admin()) {
						$update_user->user_pass = $request['pass1'];
						$pass_changed = true;

						// The new password is the same as the current password.
					} else {
						$pass_error = 'same';
					}

					// Password change attempt was unsuccessful.
				} else {
					$pass_error = 'mismatch';
				}

				// Both password fields were empty.
			} elseif (empty($request['pass1']) && empty($request['pass2'])) {
				$pass_error = false;

				// One of the password boxes was left empty.
			} elseif ((empty($request['pass1']) && !empty($request['pass2'])) || (!empty($request['pass1']) && empty($request['pass2']))) {
				$pass_error = 'empty';
			}

			// The structure of the $update_user object changed in WP 3.3, but
			// wp_update_user() still expects the old format.
			if (isset($update_user->data) && is_object($update_user->data)) {
				$update_user = $update_user->data;
				$update_user = get_object_vars($update_user);

				// Unset the password field to prevent it from emptying out the
				// user's user_pass field in the database.
				// @see wp_update_user().
				if (false === $pass_changed) {
					unset($update_user['user_pass']);
				}
			}

			// Clear cached data, so that the changed settings take effect
			// on the current page load.
			if ((false === $email_error) && (false === $pass_error) && (wp_update_user($update_user))) {
				wp_cache_delete('bp_core_userdata_' . get_current_user_id(), 'bp');
				// $displayed_user = bp_core_get_core_userdata( get_current_user_id() );
			}

			// Password Error.
		}

		// Email feedback.
		switch ($email_error) {
		case 'invalid':
			$feedback['email_invalid'] = __('That email address is invalid. Check the formatting and try again.', 'buddyboss-app');
			break;
		case 'blocked':
			$feedback['email_blocked'] = __('That email address is currently unavailable for use.', 'buddyboss-app');
			break;
		case 'taken':
			$feedback['email_taken'] = __('That email address is already taken.', 'buddyboss-app');
			break;
		case 'empty':
			$feedback['email_empty'] = __('Email address cannot be empty.', 'buddyboss-app');
			break;
		case false:
			// No change.
			break;
		}

		// Password feedback.
		switch ($pass_error) {
		case 'invalid':
			$feedback['pass_error'] = __('Your current password is invalid.', 'buddyboss-app');
			break;
		case 'mismatch':
			$feedback['pass_mismatch'] = __('The new password fields did not match.', 'buddyboss-app');
			break;
		case 'empty':
			$feedback['pass_empty'] = __('One of the password fields was empty.', 'buddyboss-app');
			break;
		case 'same':
			$feedback['pass_same'] = __('The new password must be different from the current password.', 'buddyboss-app');
			break;
		case false:
			// No change.
			break;
		}

		// No errors so show a simple success message.
		if (((false === $email_error) || (false == $pass_error)) && ((true === $pass_changed) || (true === $email_changed))) {
			$feedback[] = __('Your settings have been saved.', 'buddyboss-app');
			$feedback_type = 'success';

			// Some kind of errors occurred.
		} elseif (((false === $email_error) || (false === $pass_error)) && ((false === $pass_changed) || (false === $email_changed))) {
			if (bp_is_my_profile()) {
				$feedback['nochange'] = __('No changes were made to your account.', 'buddyboss-app');
			} else {
				$feedback['nochange'] = __('No changes were made to this account.', 'buddyboss-app');
			}
		}

		/**
		 * Fires after the general settings have been saved, and before redirect.
		 *
		 * @since 1.5.0
		 */
		do_action('bp_core_general_settings_after_save');

		if ($feedback_type == 'success') {

			$response = array(
				'user_email' => $displayed_user->user_email,
				'username' => $displayed_user->user_login,
				'user_profile_url' => bp_core_get_user_domain($displayed_user->ID),
			);

			$pending_email = bp_get_user_meta($displayed_user->ID, 'pending_email_change', true);

			if (!empty($pending_email['newemail'])) {
				$pending_email['dismiss_change_url'] = esc_url(wp_nonce_url(bp_core_get_user_domain(get_current_user_id()) . bp_get_settings_slug() . '/?dismiss_email_change=1', 'bp_dismiss_email_change'));
				$response['pending_email'] = $pending_email;
			}

			return rest_ensure_response($response);
		}

		return new WP_Error('error', implode("\n", $feedback), array('status' => 500));

	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/settings/visibility Visibility level
	 * @apiName GetVisibilityLevel
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get profile visibility levels from Buddypress
	 * @apiDeprecated  Retrieve account setting options based on navigation tab. Check (#Account Settings:GetBBAccountSettingsOptions)
	 * @apiUse apidocGetVisibilityLevelV1
	 * @apiPrivate
	 */
	public function get_visibility_levels($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		return bp_xprofile_get_visibility_levels();
	}

	/**
	 * @param $request
	 * @return bool|WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/settings/visibility Update visibility level
	 * @apiName UpdateVisibilityLevel
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription todo
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} displayed_user_id Displayed user id.
	 * @apiParam {String} field_ids User xprofile fields ids.
	 * @apiDeprecated  Update account setting options based on navigation tab. Check (#Account Settings:UpdateBBAccountSettingsOptions)
	 * @apiPrivate
	 */
	public function update_visibility_settings($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (!empty($request->get_param('displayed_user_id')) && get_current_user_id() != $request->get_param('displayed_user_id')) {
			if ((is_super_admin() || current_user_can('administrator'))) {
				$displayed_user_id = $request->get_param('displayed_user_id');
			} else {
				return new WP_Error('member_permission_denied', __('You are not allowed to update member data.', 'buddyboss-app'), array('status' => 500));
			}
		} else {
			$displayed_user_id = get_current_user_id();
		}

		/**
		 * Fires before saving xprofile field visibilities.
		 *
		 * @since 2.0.0
		 */
		do_action('bp_xprofile_settings_before_save');

		/* Save ***************************************************************** */

		// Only save if there are field ID's being posted.
		if (!empty($request['field_ids'])) {

			// Get the POST'ed field ID's.
			$posted_field_ids = explode(',', $request['field_ids']);

			// Backward compatibility: a bug in BP 2.0 caused only a single
			// group's field IDs to be submitted. Look for values submitted
			// in the POST request that may not appear in 'field_ids', and
			// add them to the list of IDs to save.
			foreach ($request as $posted_key => $posted_value) {
				preg_match('/^field_([0-9]+)_visibility$/', $posted_key, $matches);
				if (!empty($matches[1]) && !in_array($matches[1], $posted_field_ids)) {
					$posted_field_ids[] = $matches[1];
				}
			}

			// Save the visibility settings.
			foreach ($posted_field_ids as $field_id) {

				$visibility_level = 'public';

				if (!empty($request['field_' . $field_id . '_visibility'])) {
					$visibility_level = $request['field_' . $field_id . '_visibility'];
				}

				xprofile_set_field_visibility_level($field_id, $displayed_user_id, $visibility_level);
			}
		}

		/* Other **************************************************************** */

		/**
		 * Fires after saving xprofile field visibilities.
		 *
		 * @since 2.0.0
		 */
		do_action('bp_xprofile_settings_after_save');

		// Redirect to the root domain.
		return true;
	}

	/**
	 * Check if a given request has access to settings items.
	 *
	 * @return WP_Error|bool
	 */
	public function bp_settings_permission() {
		return apply_filters('boss_rest_buddypress_settings_permission', true);
	}

	/**
	 * @param $request
	 * @return array|WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/settings/notifications Notification settings
	 * @apiName GetNotificationSettings
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get notification settings
	 * @apiDeprecated  Retrieve account setting options based on navigation tab. Check (#Account Settings:GetBBAccountSettingsOptions)
	 * @apiUse apidocForGetNotificationSettingsV1
	 * @apiPrivate
	 */
	public function get_notification_settings($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		if (!empty($request->get_param('displayed_user_id')) && get_current_user_id() != $request->get_param('displayed_user_id')) {
			if ((is_super_admin() || current_user_can('administrator'))) {
				$displayed_user_id = $request->get_param('displayed_user_id');
			} else {
				return new WP_Error('member_permission_denied', __('You are not allowed to update member data.', 'buddyboss-app'), array('status' => 500));
			}
		} else {
			$displayed_user_id = get_current_user_id();
		}

		$settings = array();
		foreach (array_unique(bp_settings_get_registered_notification_keys()) as $key => $value) {
			$setting_value = bp_get_user_meta($displayed_user_id, $value, true);
			$setting_value = 'yes' == $setting_value ? true : false;
			$settings[$value] = $setting_value;
		}

		return $settings;
	}

	/**
	 * @param $request
	 * @return array|WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/settings/notifications Update notification settings
	 * @apiName UpdateNotificationSettings
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription @since 2.3.5
	 * @apiDeprecated  Update account setting options based on navigation tab. Check (#Account Settings:UpdateBBAccountSettingsOptions)
	 * @apiUse apidocForUpdateNotificationSettingsV1
	 * @apiPrivate
	 */
	public function update_notification_settings($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (!empty($request->get_param('displayed_user_id')) && get_current_user_id() != $request->get_param('displayed_user_id')) {
			if ((is_super_admin() || current_user_can('administrator'))) {
				$displayed_user_id = $request->get_param('displayed_user_id');
			} else {
				return new WP_Error('member_permission_denied', __('You are not allowed to get member data.', 'buddyboss-app'), array('status' => 500));
			}
		} else {
			$displayed_user_id = get_current_user_id();
		}
		$settings = $request->get_params()['notifications'];

		if (isset($settings['access_token'])) {
			unset($settings['access_token']);
		}

		foreach ((array) $settings as $key => $value) {
			// Force core keys to 'yes' or 'no' values.
			$value = $value ? 'yes' : 'no';
			$settings[$key] = $value;
		}

		$settings = bp_settings_sanitize_notification_settings($settings);
		foreach ($settings as $setting_key => $setting_value) {
			bp_update_user_meta($displayed_user_id, $setting_key, $setting_value);
		}

		$response_settings = $this->get_notification_settings($request);

		return $response_settings;
	}

	/**
	 * Changes the avatar_urls to buddypress urls
	 *
	 * @param $args
	 * @param $id_or_email
	 *
	 * @return mixed
	 */
	public function pre_get_avatar_data($args, $id_or_email) {

		$user = false;

		// Ugh, hate duplicating code; process the user identifier.
		if (is_numeric($id_or_email)) {
			$user = get_user_by('id', absint($id_or_email));
		} elseif ($id_or_email instanceof WP_User) {
			// User Object
			$user = $id_or_email;
		} elseif ($id_or_email instanceof WP_Post) {
			// Post Object
			$user = get_user_by('id', (int) $id_or_email->post_author);
		} elseif ($id_or_email instanceof WP_Comment) {
			if (!empty($id_or_email->user_id)) {
				$user = get_user_by('id', (int) $id_or_email->user_id);
			}
		} elseif (is_email($id_or_email)) {
			$user = get_user_by('email', $id_or_email);
		}

		// No user, so bail.
		if (false === $user instanceof WP_User) {
			return $args;
		}

		// Set BuddyPress-specific avatar args.
		$bpargs['item_id'] = $user->ID;
		$bpargs['html'] = false;

		// Use the 'full' type if size is larger than BP's thumb width.
		if ((int) $args['size'] > bp_core_avatar_thumb_width()) {
			$bpargs['type'] = 'full';
		}

		// Get the BuddyPress avatar URL.
		if ($args["url"] = bp_core_fetch_avatar($bpargs)) {
			return $args;
		}

		return $args;
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {DELETE} /wp-json/appboss/buddypress/v1/bp_attachment/avatar Delete attachment avatar
	 * @apiName DeleteAttachmentAvatar
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Delete attachment avatar
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} [item_id="logged-in-user-id"] User Id.
	 * @apiDeprecated  Delete member avatar. Check (#Members:DeleteBBMemberAvatar)
	 * @apiPrivate
	 */
	public function delete_avatar($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$args = array(
			'object' => 'user',
			'item_id' => !empty($request['item_id']) ? $request['item_id'] : get_current_user_id(),
		);

		if ('user' == $args['object']) {
			// you can change only your own cover
			if ($args['item_id'] != get_current_user_id()) {
				return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
			}
		}

		if ('group' == $args['object']) {
			// you can change if you are admin or mod
			$is_admin = groups_is_user_admin(get_current_user_id(), $args['item_id']);
			$is_mod = groups_is_user_mod(get_current_user_id(), $args['item_id']);
			if (!($is_admin || $is_mod)) {
				return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
			}
		}

		return bp_core_delete_existing_avatar($args);
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/bp_attachment/avatar Attachment avatar
	 * @apiName GetAttachmentAvatar
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get attachment avatar
	 * @apiDeprecated  Retrieve member avatar. Check (#Members:GetBBMemberAvatar)
	 * @apiUse apidocForGetAttachmentAvatarV1
	 * @apiPrivate
	 */
	public function get_avatar($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		return bp_core_fetch_avatar(array(
			'object' => 'user',
			'item_id' => !empty($request['item_id']) ? $request['item_id'] : get_current_user_id(),
			'html' => false,
		));
	}

	/**
	 * @param $request
	 * @return void|WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/bp_attachment/avatar Update attachment avatar
	 * @apiName UpdateAttachmentAvatar
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription @since buddypress 2.3.0
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} [item_id="current user id if logged-in"] User Id.
	 * @apiDeprecated  Create member avatar. Check (#Members:CreateBBMemberAvatar)
	 * @apiPrivate
	 */
	public function update_avatar($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		// Init the BuddyPress parameters.
		$bp_params = array();

		// We need it to carry on.
		$bp_params = array(
			'object' => 'user',
			'item_id' => !empty($request['item_id']) ? $request['item_id'] : get_current_user_id(),
		);

		// Capability check.
		if (!bp_attachments_current_user_can('edit_avatar', $bp_params)) {
			return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$bp = buddypress();
		$bp_params['upload_dir_filter'] = function_exists( 'bp_members_avatar_upload_dir' ) ? 'bp_members_avatar_upload_dir' : 'xprofile_avatar_upload_dir';
		$needs_reset = array('key' => 'displayed_user', 'value' => $bp->displayed_user);
		$bp->displayed_user->id = $bp_params['item_id'];

		if (!isset($bp->avatar_admin)) {
			$bp->avatar_admin = new stdClass();
		}

		/**
		 * The BuddyPress upload parameters is including the Avatar UI Available width,
		 * add it to the avatar_admin global for a later use.
		 */
		if (isset($bp_params['ui_available_width'])) {
			$bp->avatar_admin->ui_available_width = (int) $bp_params['ui_available_width'];
		}

		// upload will be rejected without this
		add_filter('bp_attachment_upload_overrides', array($this, 'change_overrides'));

		// Upload the avatar.
		$avatar = bp_core_avatar_handle_upload($_FILES, $bp_params['upload_dir_filter']);

		// Reset objects.
		if (!empty($needs_reset)) {
			if (!empty($needs_reset['component'])) {
				$bp->{$needs_reset['component']}->{$needs_reset['key']} = $needs_reset['value'];
			} else {
				$bp->{$needs_reset['key']} = $needs_reset['value'];
			}
		}

		// Init the feedback message.
		$feedback_message = false;

		if (!empty($bp->template_message)) {
			$feedback_message = $bp->template_message;

			// Remove template message.
			$bp->template_message = false;
			$bp->template_message_type = false;

			@setcookie('bp-message', false, time() - 1000, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
			@setcookie('bp-message-type', false, time() - 1000, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
		}

		if (empty($avatar)) {
			// Default upload error.
			$message = __('Upload failed.', 'buddyboss-app');

			// Use the template message if set.
			if (!empty($feedback_message)) {
				$message = $feedback_message;
			}

			// Upload error reply.
			return new WP_Error('upload_error', $message, array('status' => 500));
		}

		if (empty($bp->avatar_admin->image->file)) {
			return new WP_Error('upload_error', __('Upload failed.', 'buddyboss-app'), array('status' => 500));
		}

		$uploaded_image = @getimagesize($bp->avatar_admin->image->file);

		// Set the name of the file.
		$name = $_FILES['file']['name'];
		$name_parts = pathinfo($name);
		$name = trim(substr($name, 0, -(1 + strlen($name_parts['extension']))));

		// Further code is copied from bp_avatar_ajax_set, reall croping will be done in client app

		$size = $uploaded_image[0];

		// take the smallest part. so we don't see any black empty area.
		if ($uploaded_image[0] > $uploaded_image[1]) {
			$size = $uploaded_image[1];
		}

		$avatar_data = array(
			'original_file' => $bp->avatar_admin->image->url,
			'crop_w' => $size,
			'crop_h' => $size,
			'crop_x' => 0,
			'crop_y' => 0,
			'item_id' => $bp_params['item_id'],
			'object' => $bp_params['object'],
			'type' => 'crop',
		);

		if (empty($avatar_data['object']) || empty($avatar_data['item_id']) || empty($avatar_data['original_file'])) {
			return new WP_Error('no_args', __('No arguments.', 'buddyboss-app'), array('status' => 500));
		}

		if (!empty($avatar_data['type']) && 'camera' === $avatar_data['type'] && 'user' === $avatar_data['object']) {
			$webcam_avatar = false;

			if (!empty($avatar_data['original_file'])) {
				$webcam_avatar = str_replace(array('data:image/png;base64,', ' '), array('', '+'), $avatar_data['original_file']);
				$webcam_avatar = base64_decode($webcam_avatar);
			}

			if (!bp_avatar_handle_capture($webcam_avatar, $avatar_data['item_id'])) {
				return new WP_Error('upload_failed', __('Upload failed.', 'buddyboss-app'), array('status' => 500));

			} else {
				$return = array(
					'avatar' => html_entity_decode(bp_core_fetch_avatar(array(
						'object' => $avatar_data['object'],
						'item_id' => $avatar_data['item_id'],
						'html' => false,
						'type' => 'full',
					))),
					'feedback_code' => 2,
					'item_id' => $avatar_data['item_id'],
				);

				/**
				 * Fires if the new avatar was successfully captured.
				 *
				 * @since 1.1.0 Used to inform the avatar was successfully cropped
				 * @since 2.3.4 Add two new parameters to inform about the user id and
				 *              about the way the avatar was set (eg: 'crop' or 'camera')
				 *              Move the action at the right place, once the avatar is set
				 *
				 * @param string $item_id Inform about the user id the avatar was set for
				 * @param string $type    Inform about the way the avatar was set ('camera')
				 */
				do_action('xprofile_avatar_uploaded', (int) $avatar_data['item_id'], $avatar_data['type']);

				wp_send_json_success($return);
			}

			return;
		}

		$original_file = str_replace(bp_core_avatar_url(), '', $avatar_data['original_file']);

		// Set avatars dir & feedback part.
		if ('user' === $avatar_data['object']) {
			$avatar_dir = 'avatars';

			// Defaults to object-avatars dir.
		} else {
			$avatar_dir = sanitize_key($avatar_data['object']) . '-avatars';
		}

		// Crop args.
		$r = array(
			'item_id' => $avatar_data['item_id'],
			'object' => $avatar_data['object'],
			'avatar_dir' => $avatar_dir,
			'original_file' => $original_file,
			'crop_w' => $avatar_data['crop_w'],
			'crop_h' => $avatar_data['crop_h'],
			'crop_x' => $avatar_data['crop_x'],
			'crop_y' => $avatar_data['crop_y'],
		);

		// Handle crop.
		if (bp_core_avatar_handle_crop($r)) {
			$return = array(
				'avatar' => html_entity_decode(bp_core_fetch_avatar(array(
					'object' => $avatar_data['object'],
					'item_id' => $avatar_data['item_id'],
					'html' => false,
					'type' => 'full',
				))),
				'feedback_code' => 2,
				'item_id' => $avatar_data['item_id'],
			);

			if ('user' === $avatar_data['object']) {
				/**
				 * Fires if the new avatar was successfully cropped.
				 *
				 * @since 1.1.0 Used to inform the avatar was successfully cropped
				 * @since 2.3.4 Add two new parameters to inform about the user id and
				 *              about the way the avatar was set (eg: 'crop' or 'camera')
				 *              Move the action at the right place, once the avatar is set
				 *
				 * @param string $item_id Inform about the user id the avatar was set for
				 * @param string $type Inform about the way the avatar was set ('crop')
				 */
				do_action('xprofile_avatar_uploaded', (int) $avatar_data['item_id'], $avatar_data['type']);
			}

			return rest_ensure_response($return);
		} else {
			return new WP_Error('upload_failed', __('Upload failed.', 'buddyboss-app'), array('status' => 500));
		}
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {DELETE} /wp-json/appboss/buddypress/v1/bp_attachment/cover Delete attachment cover
	 * @apiName DeleteAttachmentCover
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription @since buddypress 2.4.0
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} [item_id="current user id if logged-in"] User Id.
	 * @apiDeprecated  Delete member cover. Check (#Members:DeleteBBMemberCover)
	 * @apiPrivate
	 */
	public function delete_cover($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$cover_image_data = array(
			'object' => 'user',
			'item_id' => !empty($request['item_id']) ? $request['item_id'] : get_current_user_id(),
		);

		// Capability check.
		if (!bp_attachments_current_user_can('edit_cover_image', $cover_image_data)) {
			return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		// Set object for the user's case.
		if ('user' === $cover_image_data['object']) {
			$component = 'xprofile';
			$dir = 'members';
			// you can change only your own cover
			if ($cover_image_data['item_id'] != get_current_user_id()) {
				return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
			}

			// Set it for any other cases.
		} else {
			$component = $cover_image_data['object'] . 's';
			$dir = $component;
			// you can change if you are admin or mod
			$is_admin = groups_is_user_admin(get_current_user_id(), $cover_image_data['item_id']);
			$is_mod = groups_is_user_mod(get_current_user_id(), $cover_image_data['item_id']);
			if (!($is_admin || $is_mod)) {
				return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
			}
		}

		// Handle delete.
		if (bp_attachments_delete_file(array('item_id' => $cover_image_data['item_id'], 'object_dir' => $dir, 'type' => 'cover-image'))) {

			// Defaults no cover image.
			$response = array(
				'reset_url' => '',
				'feedback_code' => 3,
			);

			// Get cover image settings in case there's a default header.
			$cover_params = bp_attachments_get_cover_image_settings($component);

			// Check if there's a default cover.
			if (!empty($cover_params['default_cover'])) {
				$response['reset_url'] = $cover_params['default_cover'];
			}

			// Finally send the reset url.
			return rest_ensure_response($response);
		} else {
			return new WP_Error('cover_not_deleted', __('Couldn\'t be deleted', 'buddyboss-app'), array('status' => 500));
		}
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {PATCH} /wp-json/appboss/buddypress/v1/bp_attachment/cover Update attachment cover
	 * @apiName UpdateAttachmentCover
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription @since buddypress 2.4.0
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} [item_id=0] User Id.
	 * @apiDeprecated  Create member cover. Check (#Members:CreateBBMemberCover)
	 * @apiPrivate
	 */
	public function update_cover($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('not_logged_in', __('Please login to perform any action.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		// Init the BuddyPress parameters.
		$bp_params = array();

		// We need it to carry on.
		$bp_params = array(
			'object' => 'user',
			'item_id' => !empty($request['item_id']) ? $request['item_id'] : get_current_user_id(),
		);

		// Capability check.
		if (!bp_attachments_current_user_can('edit_cover_image', $bp_params)) {
			return new WP_Error('no_permission', __('No permission', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$bp = buddypress();

		$object_data = array('dir' => 'members', 'component' => 'xprofile');

		// Stop here in case of a missing parameter for the object.
		if (empty($object_data['dir']) || empty($object_data['component'])) {
			return new WP_Error('error', __('Error', 'buddyboss-app'), array('status' => 500));
		}

		$needs_reset = array('key' => 'displayed_user', 'value' => $bp->displayed_user);
		$bp->displayed_user->id = $bp_params['item_id'];

		/**
		 * Filters whether or not to handle cover image uploading.
		 *
		 * If you want to override this function, make sure you return an array with the 'result' key set.
		 *
		 * @since 2.5.1
		 *
		 * @param array $value
		 * @param array $bp_params
		 * @param array $needs_reset Stores original value of certain globals we need to revert to later.
		 * @param array $object_data
		 */
		$pre_filter = apply_filters('bp_attachments_pre_cover_image_ajax_upload', array(), $bp_params, $needs_reset, $object_data);
		if (isset($pre_filter['result'])) {
			// bp_attachments_json_response( $pre_filter['result'], $is_html4, $pre_filter );
			return rest_ensure_response($pre_filter);
		}

		// upload will be rejected without this
		add_filter('bp_attachment_upload_overrides', array($this, 'change_overrides'));

		$cover_image_attachment = new BP_Attachment_Cover_Image();
		$uploaded = $cover_image_attachment->upload($_FILES);

		// Reset objects.
		if (!empty($needs_reset)) {
			if (!empty($needs_reset['component'])) {
				$bp->{$needs_reset['component']}->{$needs_reset['key']} = $needs_reset['value'];
			} else {
				$bp->{$needs_reset['key']} = $needs_reset['value'];
			}
		}

		if (!empty($uploaded['error'])) {
			// Upload error response.
			return new WP_Error('upload_error', sprintf(__('Upload Failed! Error was: %s', 'buddypress'), $uploaded['error']), array('status' => 500));
		}

		// Default error message.
		$error_message = __('There was a problem uploading the cover image.', 'buddypress');

		// Get BuddyPress Attachments Uploads Dir datas.
		$bp_attachments_uploads_dir = bp_attachments_uploads_dir_get();

		// The BP Attachments Uploads Dir is not set, stop.
		if (!$bp_attachments_uploads_dir) {
			return new WP_Error('upload_error', $error_message, array('status' => 500));
		}

		$cover_subdir = $object_data['dir'] . '/' . $bp_params['item_id'] . '/cover-image';
		$cover_dir = trailingslashit($bp_attachments_uploads_dir['basedir']) . $cover_subdir;

		if (!is_dir($cover_dir)) {
			// Upload error response.
			return new WP_Error('upload_error', $error_message, array('status' => 500));
		}

		/**
		 * Generate the cover image so that it fit to feature's dimensions
		 *
		 * Unlike the Avatar, Uploading and generating the cover image is happening during
		 * the same Ajax request, as we already instantiated the BP_Attachment_Cover_Image
		 * class, let's use it.
		 */
		$cover = bp_attachments_cover_image_generate_file(array(
			'file' => $uploaded['file'],
			'component' => $object_data['component'],
			'cover_image_dir' => $cover_dir,
		), $cover_image_attachment);

		if (!$cover) {
			// Upload error response.
			return new WP_Error('upload_error', $error_message, array('status' => 500));
		}

		// Build the url to the file.
		$cover_url = trailingslashit($bp_attachments_uploads_dir['baseurl']) . $cover_subdir . '/' . $cover['cover_basename'];

		// Init Feedback code, 1 is success.
		$feedback_code = 1;

		// 0 is the size warning.
		if ($cover['is_too_small']) {
			$feedback_code = 0;
		}

		// Set the name of the file.
		$name = $_FILES['file']['name'];
		$name_parts = pathinfo($name);
		$name = trim(substr($name, 0, -(1 + strlen($name_parts['extension']))));

		/**
		 * Fires if the new cover image was successfully uploaded.
		 *
		 * The dynamic portion of the hook will be xprofile in case of a user's
		 * cover image, groups in case of a group's cover image. For instance:
		 * Use add_action( 'xprofile_cover_image_uploaded' ) to run your specific
		 * code once the user has set his cover image.
		 *
		 * @since 2.4.0
		 *
		 * @param int $item_id Inform about the item id the cover image was set for.
		 */
		do_action($object_data['component'] . '_cover_image_uploaded', (int) $bp_params['item_id']);

		// Finally return the cover image url to the UI.
		return rest_ensure_response(array(
			'name' => $name,
			'url' => $cover_url,
			'feedback_code' => $feedback_code,
		));
	}

	public function change_overrides($overrides) {
		$overrides['test_form'] = false;
		return $overrides;
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @api {GET} /wp-json/appboss/buddypress/v1/bp_attachment/cover Attachment cover
	 * @apiName GetAttachmentCoverImage
	 * @apiGroup BuddyPressAct.Settings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get attachment cover image
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} [item_id=0] User Id.
	 * @apiDeprecated  Retrieve member cover. Check (#Members:GetBBMemberCover)
	 * @apiPrivate
	 */
	public function get_cover($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		return bp_attachments_get_attachment('url', array(
			'object' => 'user',
			'item_id' => !empty($request['item_id']) ? $request['item_id'] : get_current_user_id(),
			'html' => false,
		));
	}
}
