<?php
namespace BuddyBossApp\Api\BbPress;
use BuddyBossApp\Api\BbPress\PostController;
use stdClass;
use WP_Error;
use WP_REST_Request;
use WP_REST_Server;

// NOTE : Old classname was class.boss_bbpress_api_topic_rest By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for bbpress.
class TopicRest extends PostController {

	protected $namespace_slug = '';
	protected $namespace = '/bbpress/v1';

	public function __construct($slug) {

		parent::__construct($slug, bbp_get_topic_post_type());

		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->hooks();
	}

	public function hooks() {

		add_action('rest_api_init', array($this, "register_routes"), 99);
		add_filter('boss_bp_json_pre_insert_topic', [$this, 'set_group'], 10, 2);

	}

	public function register_routes() {

		parent::register_routes();

		register_rest_route($this->namespace, '/topic/action/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'do_action'),
				'permission_callback' => '__return_true',
				'args' => array(
					'favourite' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'subscribe' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'close' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'stick' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'is_super' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'spam' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'trash' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'delete' => array(
						'type' => 'integer',
						'enum' => array("1"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/topic', array(
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'create'),
				'permission_callback' => '__return_true',
				'args' => $this->get_create_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/topic/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::DELETABLE,
				'callback' => array($this, 'delete'),
				'permission_callback' => '__return_true',
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/topic/edit/', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'update'),
				'permission_callback' => '__return_true',
				'args' => $this->get_create_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/topic/merge/', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'merge'),
				'permission_callback' => '__return_true',
				'args' => array(
					'bbp_topic_id' => array(
						'description' => __('Id of topic'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
						'required' => true,
					),
					'bbp_destination_topic' => array(
						'description' => __('Id of destination topic'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
						'required' => true,
					),
					'bbp_topic_subscribers' => array(
						'description' => __('Whether topic is subscribed or not.'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
					'bbp_topic_favoriters' => array(
						'description' => __('Whether topic is favourite or not.'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
					'bbp_topic_tags' => array(
						'description' => __('Whether topic has tag or not.'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/topic/split/', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'split'),
				'permission_callback' => '__return_true',
				'args' => array(
					'bbp_reply_id' => array(
						'description' => __('Id of topic'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
						'required' => true,
					),
					'bbp_destination_topic' => array(
						'description' => __('Id of destination topic'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
					'bbp_topic_split_option' => array(
						'description' => __('option of split topic'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
						'required' => true,
					),
					'bbp_topic_split_destination_title' => array(
						'description' => __('Title of destination topic'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'bbp_topic_subscribers' => array(
						'description' => __('Whether topic is subscribed or not.'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
					'bbp_topic_favoriters' => array(
						'description' => __('Whether topic is favourite or not.'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
					'bbp_topic_tags' => array(
						'description' => __('Whether topic has tag or not.'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		/**
		 *  @since 1.0.1
		 */
		register_rest_route($this->namespace, '/topic/dropdown/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'dropdown'),
				'permission_callback' => '__return_true',
				'args' => array(
					'page' => array(
						'description' => __('Current page of the collection.'),
						'type' => 'integer',
						'default' => 1,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
						'minimum' => 1,
					),
					'per_page' => array(
						'description' => __('Maximum number of items to be returned in result set.'),
						'type' => 'integer',
						'default' => 10,
						'minimum' => 1,
						'maximum' => 100,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		));

		/**
		 * This is fallback old endpoint later 1.0.1
		 */
		register_rest_route($this->namespace, '/topic/dropdown/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'dropdown'),
				'permission_callback' => '__return_true',
				'args' => array(
					'page' => array(
						'description' => __('Current page of the collection.'),
						'type' => 'integer',
						'default' => 1,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
						'minimum' => 1,
					),
					'per_page' => array(
						'description' => __('Maximum number of items to be returned in result set.'),
						'type' => 'integer',
						'default' => 10,
						'minimum' => 1,
						'maximum' => 100,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		));

	}

	public function set_group($topic, $request) {
		global $bp;

		$forum_id = filter_var($topic->bbp_forum_id, FILTER_VALIDATE_INT);

		if ($forum_id && function_exists('bbp_get_forum_group_ids')) {
			$group_ids = bbp_get_forum_group_ids($forum_id);
			$group_id = reset($group_ids);
			if ($group_id) {
				$bp->groups->current_group = groups_get_group($group_id);
			}
		}

		return $topic;
	}

	/**
	 * @api {PATCH} /wp-json/appboss/bbpress/v1/topic/action/:id Update topic action
	 * @apiName UpdateBBpressTopicAction
	 * @apiGroup BBPressTopic
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update BBPress Topic's action
	 * @apiParam {Number} id Topic Id
	 * @apiParam {String} favorite set it as favorite
	 * @apiParam {String} subscribe set it as subscribe
	 * @apiUse apidocForUpdateTopicActionV1
	 * @apiParam {Number} id Unique identifier for topic
	 * @apiDeprecated  Actions on Topic. Check (#Forum Topics:ActionBBPTopic)
	 * @apiPrivate
	 */
	public function do_action($request) {

		$topic_id = (int) $request['id'];
		$success = true;

		if (!is_user_logged_in()) {
			return new WP_Error('rest_cannot_view', __("You don't have permission.", 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$is_valid = $this->check_topic($topic_id);

		if (is_wp_error($is_valid)) {
			return $is_valid;
		}

		$user_id = get_current_user_id();

		/**
		 * Functionality Presence Check
		 */

		if ((isset($request['favorite']) && !bbp_is_favorites_active()) || (isset($request['subscribe']) && !bbp_is_subscriptions_active())) {
			return new WP_Error('bp_json_topic_action_disabled', __('This action is disabled.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		/**
		 * Permissions Check
		 */

		# Common Permission Challenge
		if (!current_user_can('edit_user', $user_id)) {
			return new WP_Error('bp_json_topic_no_permission_to_edit', __('You don\'t have the permission to edit favorites of that user!', 'buddyboss-app'), array('status' => 403));
		}

		# Trash Permission
		if (isset($request['trash']) && !current_user_can('delete_topic', $topic_id)) {
			return new WP_Error('bp_json_topic_no_permission_to_trash', __('You don\'t have the permission to trash topics!', 'buddyboss-app'), array('status' => 403));
		}

		if (isset($request['delete']) && !current_user_can('delete_topic', $topic_id)) {
			return new WP_Error('bp_json_topic_no_permission_to_delete', __('You don\'t have the permission to delete topics!', 'buddyboss-app'), array('status' => 403));
		}

		/**
		 * Close Topic Action
		 */
		if (isset($request['close']) && $request['close'] == 1) {

			$is_opened = bbp_is_topic_open($topic_id);
			if ($is_opened) {
				$success = bbp_close_topic($topic_id);
			}

		} elseif (isset($request['close']) && $request['close'] == 0) {

			$is_opened = bbp_is_topic_open($topic_id);
			if (!$is_opened) {
				$success = bbp_open_topic($topic_id);
			}

		}

		/**
		 * Stick Topic Action
		 */
		if (isset($request['stick']) && $request['stick'] == 1) {
			$is_sticky = bbp_is_topic_sticky($topic_id);
			$is_super = isset($request['is_super']) ? true : false;
			if (!$is_sticky || ($is_sticky && $is_super)) {
				$success = bbp_stick_topic($topic_id, $is_super);
			}
		} elseif (isset($request['stick']) && $request['stick'] == 0) {
			$is_sticky = bbp_is_topic_sticky($topic_id);
			if ($is_sticky) {
				$success = bbp_unstick_topic($topic_id);
			}
		}

		/**
		 * Spam Topic Action
		 */
		if (isset($request['spam']) && $request['spam'] == 1) {
			$is_spam = bbp_is_topic_spam($topic_id);
			if (!$is_spam) {
				$success = bbp_spam_topic($topic_id);
			}
		} elseif (isset($request['spam']) && $request['spam'] == 0) {
			$is_spam = bbp_is_topic_spam($topic_id);
			if ($is_spam) {
				$success = bbp_unspam_topic($topic_id);
			}
		}

		/**
		 * Trash Topic Action
		 */
		if (isset($request['trash']) && $request['trash'] == 1) {
			$success = wp_trash_post($topic_id);
		} else if (isset($request['trash']) && $request['trash'] == 0) {
			$success = wp_untrash_post($topic_id);
		}

		/**
		 * Delete topic
		 */
		if (isset($request['delete'])) {
			$success = wp_delete_post($topic_id, true);
		}

		/**
		 * Subscribe Topic Action
		 */
		if (isset($request['subscribe']) && $request['subscribe'] == 1) {

			$user_id = bbp_get_user_id(0, true, true);
			$is_subscription = bbp_is_user_subscribed($user_id, $topic_id);
			if (!$is_subscription) {
				$success = bbp_add_user_subscription($user_id, $topic_id);
			}

		} elseif (isset($request['subscribe']) && $request['subscribe'] == 0) {

			$user_id = bbp_get_user_id(0, true, true);
			$is_subscription = bbp_is_user_subscribed($user_id, $topic_id);
			if ($is_subscription) {
				$success = bbp_remove_user_subscription($user_id, $topic_id);
			}

		}

		/**
		 * Fav Topic Action
		 */
		if (isset($request['favourite']) && $request['favourite'] == 1) {

			$user_id = bbp_get_user_id(0, true, true);
			$is_favorite = bbp_is_user_favorite($user_id, $topic_id);
			if (!$is_favorite) {
				$success = bbp_add_user_favorite($user_id, $topic_id);
			}

		} elseif (isset($request['favourite']) && $request['favourite'] == 0) {

			$user_id = bbp_get_user_id(0, true, true);
			$is_favorite = bbp_is_user_favorite($user_id, $topic_id);
			if ($is_favorite) {
				$success = bbp_remove_user_favorite($user_id, $topic_id);
			}

		}

		/**
		 * Error Handling.
		 */
		if (false == $success || is_wp_error($success)) {
			return new WP_Error('bp_json_topic_action_failure', __('Action failed.', 'buddyboss-app'), array('status' => 500));
		}

		return $this->_get_item($topic_id, $request);

	}

	/**
	 * @api {DELETE} /wp-json/appboss/bbpress/v1/topic/:id Delete topic
	 * @apiName DeleteTopic
	 * @apiGroup BBPressTopic
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Delete BBPress topic
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Unique Identifier
	 * @apiDeprecated  Trash OR Delete a topic. Check (#Forum Topics:DeleteBBPTopic)
	 * @apiPrivate
	 */
	public function delete($request) {

		$topic_id = (int) $request['id'];

		if (!is_user_logged_in()) {
			return new WP_Error('rest_cannot_view', __("You don't have permission.", 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		$is_valid = $this->check_topic($topic_id);

		if (is_wp_error($is_valid)) {
			return $is_valid;
		}

		/**
		 * Permissions Check
		 */
		if (!current_user_can('delete_topic', $topic_id)) {
			return new WP_Error('bp_json_topic_no_permission_to_delete', __('You don\'t have the permission to delete topics!', 'buddyboss-app'), array('status' => 403));
		}

		/**
		 * Delete topic
		 */
		$success = wp_delete_post($topic_id);

		/**
		 * Error Handling.
		 */
		if (false == $success || is_wp_error($success)) {
			return new WP_Error('bp_json_topic_action_failure', __('Action failed.', 'buddyboss-app'), array('status' => 500));
		}

		return true;

	}

	/**
	 * @api {POST} /wp-json/appboss/bbpress/v1/topic Create topic
	 * @apiName CreateBBPressTopic
	 * @apiGroup BBPressTopic
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Create BBPress Topic
	 * @apiDeprecated  Create a topic. Check (#Forum Topics:CreateBBPTopic)
	 * @apiUse apidocForBBPressTopicV1
	 * @apiPrivate
	 */
	public function create($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$topic = $this->prepare_topic_for_database($request);

		// Define local variable(s)
		$view_all = false;
		$forum_id = $topic_author = $anonymous_data = 0;
		$topic_title = $topic_content = '';
		$terms = array(bbp_get_topic_tag_tax_id() => array());

		/** Topic Author **********************************************************/

		// User is anonymous
		if (bbp_is_anonymous()) {

			$bbp_anonymous_name = !empty($request['bbp_anonymous_name']) ? $request['bbp_anonymous_name'] : '';
			$bbp_anonymous_email = !empty($request['bbp_anonymous_email']) ? $request['bbp_anonymous_email'] : '';
			$bbp_anonymous_website = !empty($request['bbp_anonymous_website']) ? $request['bbp_anonymous_website'] : '';

			$a_data = array(
				'bbp_anonymous_name' => $bbp_anonymous_name,
				'bbp_anonymous_email' => $bbp_anonymous_email,
				'bbp_anonymous_website' => $bbp_anonymous_website,
			);

			// Filter anonymous data
			$anonymous_data = bbp_filter_anonymous_post_data($a_data);

			// Anonymous data checks out, so set cookies, etc...
			if (!empty($anonymous_data) && is_array($anonymous_data)) {
				bbp_set_current_anonymous_user_data($anonymous_data);
			}

			// User is logged in
		} else {

			// User cannot create topics
			if (!current_user_can('publish_topics')) {
				return new WP_Error('bp_json_topic_permissions', __('You do not have permission to create new topics.', 'buddyboss-app'), array('status' => 403));
			}

			// Topic author is current user
			$topic_author = bbp_get_current_user_id();
		}

		// Remove kses filters from title and content for capable users and if the nonce is verified
		if (current_user_can('unfiltered_html')) {
			remove_filter('bbp_new_topic_pre_title', 'wp_filter_kses');
			remove_filter('bbp_new_topic_pre_content', 'bbp_encode_bad', 10);
			remove_filter('bbp_new_topic_pre_content', 'bbp_filter_kses', 30);
		}

		/** Topic Title ***********************************************************/

		if (!empty($topic->bbp_topic_title)) {
			$topic_title = esc_attr(strip_tags($topic->bbp_topic_title));
		}

		// Filter and sanitize
		$topic_title = apply_filters('bbp_new_topic_pre_title', $topic_title);

		// No topic title
		if (empty($topic_title)) {
			return new WP_Error('bp_json_topic_title', __('Your topic needs a title.', 'buddyboss-app'), array('status' => 400));
		}

		/** Topic Content *********************************************************/

		if (!empty($topic->bbp_topic_content)) {
			$topic_content = $topic->bbp_topic_content;
		}

		// Filter and sanitize
		$topic_content = apply_filters('bbp_new_topic_pre_content', $topic_content);

		// No topic content
		if (empty($topic_content)) {
			return new WP_Error('bp_json_topic_content', __('Your topic cannot be empty.', 'buddyboss-app'), array('status' => 400));
		}

		/** Topic Forum ***********************************************************/

		// Error check the POST'ed topic id
		if (isset($topic->bbp_forum_id)) {

			// Empty Forum id was passed
			if (empty($topic->bbp_forum_id)) {
				return new WP_Error('bp_json_topic_forum_id', __('Forum ID is missing.', 'buddyboss-app'), array('status' => 400));

				// Forum id is not a number
			} elseif (!is_numeric($topic->bbp_forum_id)) {
				return new WP_Error('bp_json_topic_forum_id_not_number', __('Forum ID must be a number.', 'buddyboss-app'), array('status' => 400));

				// Forum id might be valid
			} else {

				// Get the forum id
				$posted_forum_id = intval($topic->bbp_forum_id);

				// Forum id is empty
				if (0 === $posted_forum_id) {
					return new WP_Error('bp_json_topic_forum_id', __('Forum ID is missing.', 'buddyboss-app'), array('status' => 400));

					// Forum id is a negative number
				} elseif (0 > $posted_forum_id) {
					return new WP_Error('bp_json_topic_forum_id_negative', __('Forum ID cannot be a negative number.', 'buddyboss-app'), array('status' => 400));

					// Forum does not exist
				} elseif (!bbp_get_forum($posted_forum_id)) {
					return new WP_Error('bp_json_topic_forum_not_exist', __('Forum does not exist.', 'buddyboss-app'), array('status' => 400));

					// Use the POST'ed forum id
				} else {
					$forum_id = $posted_forum_id;
				}
			}
		}

		// Forum exists
		if (!empty($forum_id)) {

			// Forum is a category
			if (bbp_is_forum_category($forum_id)) {
				return new WP_Error('bp_json_topic_forum_category', __('This forum is a category. No topics can be created in this forum.', 'buddyboss-app'), array('status' => 403));

				// Forum is not a category
			} else {

				// Forum is closed and user cannot access
				if (bbp_is_forum_closed($forum_id) && !current_user_can('edit_forum', $forum_id)) {
					return new WP_Error('bp_json_topic_forum_closed', __('This forum has been closed to new topics.', 'buddyboss-app'), array('status' => 403));
				}

				/**
				 * Added logic for group forum
				 * Current user is part of that group or not.
				 * We need to check manually because bbpress updating that caps only on group forum page and
				 * in API those conditional tag will not work.
				 */
                if(function_exists('bbp_get_forum_group_ids')) {
                    $group_ids = bbp_get_forum_group_ids($forum_id);
                    if (!empty($group_ids)) {
                        $is_member = false;
                        foreach ($group_ids as $group_id) {
                            if (groups_is_user_member(get_current_user_id(), $group_id)) {
                                $is_member = true;
                                break;
                            }
                        }
                    }
                }


                // Forum is private and user cannot access
				if (bbp_is_forum_private($forum_id) && (!bbp_is_user_keymaster())) {
					if ((empty($group_ids) && !current_user_can('read_private_forums')) || (!empty($group_ids) && !$is_member)) {
						return new WP_Error('bp_json_topic_forum_private', __('This forum is private and you do not have the capability to read or create new topics in it.', 'buddyboss-app'), array('status' => 403));
					}

					// Forum is hidden and user cannot access
				} elseif (bbp_is_forum_hidden($forum_id) && (!bbp_is_user_keymaster())) {
					if ((empty($group_ids) && !current_user_can('read_hidden_forums')) || (!empty($group_ids) && !$is_member)) {
						return new WP_Error('bbp_new_topic_forum_hidden', __('This forum is hidden and you do not have the capability to read or create new topics in it.', 'buddyboss-app'), array('status' => 403));
					}
				}
			}
		}

		/** Topic Flooding ********************************************************/

		if (!bbp_check_for_flood($anonymous_data, $topic_author)) {
			return new WP_Error('bp_json_topic_flood', __('Slow down; you move too fast.', 'buddyboss-app'), array('status' => 400));
		}

		/** Topic Duplicate *******************************************************/

		if (!bbp_check_for_duplicate(array('post_type' => bbp_get_topic_post_type(), 'post_author' => $topic_author, 'post_content' => $topic_content, 'anonymous_data' => $anonymous_data))) {
			return new WP_Error('bp_json_duplicate', __('Duplicate topic detected; it looks as though you&#8217;ve already said that!', 'buddyboss-app'), array('status' => 400));
		}

		/** Topic Blacklist *******************************************************/

		if (!bbp_check_for_blacklist($anonymous_data, $topic_author, $topic_title, $topic_content)) {
			return new WP_Error('bp_json_blacklist', __('Your topic cannot be created at this time.', 'buddyboss-app'), array('status' => 400));
		}

		/** Topic Status **********************************************************/

		// Maybe put into moderation
		if (!bbp_check_for_moderation($anonymous_data, $topic_author, $topic_title, $topic_content)) {
			$topic_status = bbp_get_pending_status_id();

			// Check a whitelist of possible topic status ID's
		} elseif (!empty($topic->bbp_topic_status) && in_array($topic->bbp_topic_status, array_keys(bbp_get_topic_statuses()))) {
			$topic_status = $topic->bbp_topic_status;

			// Default to published if nothing else
		} else {
			$topic_status = bbp_get_public_status_id();
		}

		/** Topic Tags ************************************************************/

		if (bbp_allow_topic_tags() && !empty($topic->bbp_topic_tags)) {

			// Escape tag input
			$terms = esc_attr(strip_tags($topic->bbp_topic_tags));

			// Explode by comma
			if (strstr($terms, ',')) {
				$terms = explode(',', $terms);
			}

			// Add topic tag ID as main key
			$terms = array(bbp_get_topic_tag_tax_id() => $terms);
		}

		/** Additional Actions (Before Save) **************************************/

		do_action('bbp_new_topic_pre_extras', $forum_id);

		// Bail if errors
		if (bbp_has_errors()) {
			return new WP_Error('bp_json_unknown', __('Unknown error.', 'buddyboss-app'), array('status' => 400));
		}

		/** No Errors *************************************************************/

		// Add the content of the form to $topic_data as an array.
		// Just in time manipulation of topic data before being created
		$topic_data = apply_filters('bbp_new_topic_pre_insert', array(
			'post_author' => $topic_author,
			'post_title' => $topic_title,
			'post_content' => $topic_content,
			'post_status' => $topic_status,
			'post_parent' => $forum_id,
			'post_type' => bbp_get_topic_post_type(),
			'tax_input' => $terms,
			'comment_status' => 'closed',
		));

		// Insert topic
		$topic_id = wp_insert_post($topic_data);

		/** No Errors *************************************************************/

		if (!empty($topic_id) && !is_wp_error($topic_id)) {

			/** Trash Check *******************************************************/

			// If the forum is trash, or the topic_status is switched to
			// trash, trash it properly
			if ((get_post_field('post_status', $forum_id) === bbp_get_trash_status_id()) || ($topic_data['post_status'] === bbp_get_trash_status_id())) {

				// Trash the reply
				wp_trash_post($topic_id);

				// Force view=all
				$view_all = true;
			}

			/** Spam Check ********************************************************/

			// If reply or topic are spam, officially spam this reply
			if ($topic_data['post_status'] === bbp_get_spam_status_id()) {
				add_post_meta($topic_id, '_bbp_spam_meta_status', bbp_get_public_status_id());

				// Force view=all
				$view_all = true;
			}

			/** Update counts, etc... *********************************************/

			do_action('bbp_new_topic', $topic_id, $forum_id, $anonymous_data, $topic_author);

			/** Stickies **********************************************************/

			// Sticky check after 'bbp_new_topic' action so forum ID meta is set
			if (!empty($topic->bbp_stick_topic) && in_array($topic->bbp_stick_topic, array('stick', 'super', 'unstick'))) {

				// What's the caps?
				if (current_user_can('moderate')) {

					// What's the haps?
					switch ($topic->bbp_stick_topic) {

					// Sticky in this forum
					case 'stick':
						bbp_stick_topic($topic_id);
						break;

					// Super sticky in all forums
					case 'super':
						bbp_stick_topic($topic_id, true);
						break;

					// We can avoid this as it is a new topic
					case 'unstick':
					default:
						break;
					}
				}
			}

			/** Subscription **********************************************************/

			if (isset($topic->bbp_topic_subscription)) {

				// What's the caps?
				if (current_user_can('moderate')) {
					$user_id = bbp_get_user_id(0, true, true);
					if ($topic->bbp_topic_subscription) {
						$is_subscription = bbp_is_user_subscribed($user_id, $topic_id);
						if (!$is_subscription) {
							bbp_add_user_subscription($user_id, $topic_id);
						}
					} else {
						$is_subscription = bbp_is_user_subscribed($user_id, $topic_id);
						if ($is_subscription) {
							bbp_remove_user_subscription($user_id, $topic_id);
						}
					}
				}
			}

			/** Additional Actions (After Save) ***********************************/

			do_action('bbp_new_topic_post_extras', $topic_id);

			/**
			 * Rest Hook
			 */
			do_action('boss_rest_bbpress_create_topic', $topic_id, $request);

			$request["id"] = $topic_id;

			return $this->_get_item($topic_id, $request);

			// Errors
		} else {
			$append_error = (is_wp_error($topic_id) && $topic_id->get_error_message()) ? $topic_id->get_error_message() . ' ' : '';
			return new WP_Error('bp_json_unknown', __('The following problem(s) have been found with your topic:' . $append_error, 'buddyboss-app'), array('status' => 400));
		}
	}

	/**
	 * @api {PATCH} /wp-json/appboss/bbpress/v1/topic/edit Update topic
	 * @apiName UpdateTopic
	 * @apiGroup BBPressTopic
	 * @apiVersion 1.0.0
	 * @apiDescription Update BBpress Topic
	 * @apiUse apidocForBBPressReplyV1
	 * @apiParam {Number} id Id of topic
	 * @apiDeprecated  Update a topic. Check (#Forum Topics:UpdateBBPTopic)
	 * @apiPrivate
	 */
	public function update($request) {

		$topic_new = $this->prepare_topic_for_database($request);

		// Define local variable(s)
		$revisions_removed = false;
		$topic = $topic_id = $topic_author = $forum_id = $anonymous_data = 0;
		$topic_title = $topic_content = $topic_edit_reason = '';

		/** Topic *****************************************************************/

		// Topic id was not passed
		if (empty($topic_new->bbp_topic_id)) {
			return new WP_Error('bp_json_edit_topic_id', __('Topic ID not found.', 'buddyboss-app'), array('status' => 404));

			// Topic id was passed
		} elseif (is_numeric($topic_new->bbp_topic_id)) {
			$topic_id = (int) $topic_new->bbp_topic_id;
			$topic = bbp_get_topic($topic_id);
		}

		// Topic does not exist
		if (empty($topic)) {
			return new WP_Error('bp_json_edit_topic_not_found', __('The topic you want to edit was not found.', 'buddyboss-app'), array('status' => 404));

			// Topic exists
		} else {

			// Check users ability to create new topic
			if (!bbp_is_topic_anonymous($topic_id)) {

				if (!current_user_can('moderate', $topic_id)) {
					// User cannot edit this topic
					if (!current_user_can('edit_topic', $topic_id) || bbp_past_edit_lock($topic->post_date_gmt)) {
						return new WP_Error('bp_json_edit_topic_permissions', __('You do not have permission to edit that topic.', 'buddyboss-app'), array('status' => 403));
					}
				}

				// Set topic author
				$topic_author = bbp_get_topic_author_id($topic_id);

				// It is an anonymous post
			} else {

				$bbp_anonymous_name = !empty($request['bbp_anonymous_name']) ? $request['bbp_anonymous_name'] : '';
				$bbp_anonymous_email = !empty($request['bbp_anonymous_email']) ? $request['bbp_anonymous_email'] : '';
				$bbp_anonymous_website = !empty($request['bbp_anonymous_website']) ? $request['bbp_anonymous_website'] : '';

				$a_data = array(
					'bbp_anonymous_name' => $bbp_anonymous_name,
					'bbp_anonymous_email' => $bbp_anonymous_email,
					'bbp_anonymous_website' => $bbp_anonymous_website,
				);

				// Filter anonymous data
				$anonymous_data = bbp_filter_anonymous_post_data($a_data);
			}
		}

		// Remove kses filters from title and content for capable users and if the nonce is verified
		if (current_user_can('unfiltered_html')) {
			remove_filter('bbp_edit_topic_pre_title', 'wp_filter_kses');
			remove_filter('bbp_edit_topic_pre_content', 'bbp_encode_bad', 10);
			remove_filter('bbp_edit_topic_pre_content', 'bbp_filter_kses', 30);
		}

		/** Topic Forum ***********************************************************/

		// Forum id was not passed
		if (empty($topic_new->bbp_forum_id)) {
			return new WP_Error('bp_json_topic_forum_id', __('Forum ID is missing.', 'buddyboss-app'), array('status' => 400));
			// Forum id was passed
		} elseif (is_numeric($topic_new->bbp_forum_id)) {
			$forum_id = (int) $topic_new->bbp_forum_id;
		}

		// Current forum this topic is in
		$current_forum_id = bbp_get_topic_forum_id($topic_id);

		// Forum exists
		if (!empty($forum_id) && ($forum_id !== $current_forum_id)) {

			// Forum is a category
			if (bbp_is_forum_category($forum_id)) {
				return new WP_Error('bp_json_topic_forum_category', __('This forum is a category. No topics can be updated in it.', 'buddyboss-app'), array('status' => 400));

				// Forum is not a category
			} else {

				// Forum is closed and user cannot access
				if (bbp_is_forum_closed($forum_id) && !current_user_can('edit_forum', $forum_id)) {
					return new WP_Error('bp_json_edit_topic_forum_closed', __('This forum has been closed to update topics.', 'buddyboss-app'), array('status' => 400));
				}

				/**
				 * Added logic for group forum
				 * Current user is part of that group or not.
				 * We need to check manually because bbpress updating that caps only on group forum page and
				 * in API those conditional tag will not work.
				 */
				$group_ids = bbp_get_forum_group_ids($forum_id);
				if (!empty($group_ids)) {
					$is_member = false;
					foreach ($group_ids as $group_id) {
						if (groups_is_user_member(get_current_user_id(), $group_id)) {
							$is_member = true;
							break;
						}
					}
				}

				// Forum is private and user cannot access
				if (bbp_is_forum_private($forum_id) && (!bbp_is_user_keymaster())) {
					if ((empty($group_ids) && !current_user_can('read_private_forums')) || (!empty($group_ids) && !$is_member)) {
						return new WP_Error('bp_json_edit_topic_forum_private', __('This forum is private and you do not have the capability to read or update new topics in it.', 'buddyboss-app'), array('status' => 400));

					}

					// Forum is hidden and user cannot access
				} elseif (bbp_is_forum_hidden($forum_id) && (!bbp_is_user_keymaster())) {
					if ((empty($group_ids) && !current_user_can('read_hidden_forums')) || (!empty($group_ids) && !$is_member)) {
						return new WP_Error('bp_json_edit_topic_forum_hidden', __('This forum is hidden and you do not have the capability to read or update new topics in it.', 'buddyboss-app'), array('status' => 400));
					}
				}
			}
		}

		/** Topic Title ***********************************************************/

		if (!empty($topic_new->bbp_topic_title)) {
			$topic_title = esc_attr(strip_tags($topic_new->bbp_topic_title));
		}

		// Filter and sanitize
		$topic_title = apply_filters('bbp_edit_topic_pre_title', $topic_title, $topic_id);

		// No topic title
		if (empty($topic_title)) {
			return new WP_Error('bp_json_edit_topic_title', __('Your topic needs a title.', 'buddyboss-app'), array('status' => 400));
		}

		/** Topic Content *********************************************************/

		if (!empty($topic_new->bbp_topic_content)) {
			$topic_content = $topic_new->bbp_topic_content;
		}

		// Filter and sanitize
		$topic_content = apply_filters('bbp_edit_topic_pre_content', $topic_content, $topic_id);

		// No topic content
		if (empty($topic_content)) {
			return new WP_Error('bp_json_edit_topic_content', __('Your topic cannot be empty.', 'buddyboss-app'), array('status' => 400));
		}

		/** Topic Blacklist *******************************************************/

		if (!bbp_check_for_blacklist($anonymous_data, $topic_author, $topic_title, $topic_content)) {
			return new WP_Error('bp_json_topic_blacklist', __('Your topic cannot be edited at this time.', 'buddyboss-app'), array('status' => 400));
		}

		/** Topic Status **********************************************************/

		// Maybe put into moderation
		if (!bbp_check_for_moderation($anonymous_data, $topic_author, $topic_title, $topic_content)) {

			// Set post status to pending if public or closed
			if (in_array($topic->post_status, array(bbp_get_public_status_id(), bbp_get_closed_status_id()))) {
				$topic_status = bbp_get_pending_status_id();
			}

			// Check a whitelist of possible topic status ID's
		} elseif (!empty($topic_new->bbp_topic_status) && in_array($topic_new->bbp_topic_status, array_keys(bbp_get_topic_statuses()))) {
			$topic_status = $topic_new->bbp_topic_status;

			// Use existing post_status
		} else {
			$topic_status = $topic->post_status;
		}

		/** Topic Tags ************************************************************/

		// Either replace terms
		if (bbp_allow_topic_tags() && current_user_can('assign_topic_tags') && !empty($topic_new->bbp_topic_tags)) {

			// Escape tag input
			$terms = esc_attr(strip_tags($topic_new->bbp_topic_tags));

			// Explode by comma
			if (strstr($terms, ',')) {
				$terms = explode(',', $terms);
			}

			// Add topic tag ID as main key
			$terms = array(bbp_get_topic_tag_tax_id() => $terms);

			// ...or remove them.
		} elseif (isset($this->bbp_topic_tags)) {
			$terms = array(bbp_get_topic_tag_tax_id() => array());

			// Existing terms
		} else {
			$terms = array(bbp_get_topic_tag_tax_id() => explode(',', bbp_get_topic_tag_names($topic_id, ',')));
		}

		/** Additional Actions (Before Save) **************************************/

		do_action('bbp_edit_topic_pre_extras', $topic_id);

		/** No Errors *************************************************************/

		// Add the content of the form to $topic_data as an array
		// Just in time manipulation of topic data before being edited
		$topic_data = apply_filters('bbp_edit_topic_pre_insert', array(
			'ID' => $topic_id,
			'post_title' => $topic_title,
			'post_content' => $topic_content,
			'post_status' => $topic_status,
			'post_parent' => $forum_id,
			'post_author' => $topic_author,
			'post_type' => bbp_get_topic_post_type(),
			'tax_input' => $terms,
		));

		// Toggle revisions to avoid duplicates
		if (post_type_supports(bbp_get_topic_post_type(), 'revisions')) {
			$revisions_removed = true;
			remove_post_type_support(bbp_get_topic_post_type(), 'revisions');
		}

		// Insert topic
		$topic_id = wp_update_post($topic_data);

		// Toggle revisions back on
		if (true === $revisions_removed) {
			$revisions_removed = false;
			add_post_type_support(bbp_get_topic_post_type(), 'revisions');
		}

		/** No Errors *************************************************************/

		if (!empty($topic_id) && !is_wp_error($topic_id)) {

			// Update counts, etc...
			do_action('bbp_edit_topic', $topic_id, $forum_id, $anonymous_data, $topic_author, true/* Is edit */);

			/** Revisions *********************************************************/

			// Revision Reason
			if (!empty($topic_new->bbp_topic_edit_reason)) {
				$topic_edit_reason = esc_attr(strip_tags($topic_new->bbp_topic_edit_reason));
			}

			// Update revision log
			if (!empty($topic_new->bbp_log_topic_edit) && ("1" === $topic_new->bbp_log_topic_edit)) {
				$revision_id = wp_save_post_revision($topic_id);
				if (!empty($revision_id)) {
					bbp_update_topic_revision_log(array(
						'topic_id' => $topic_id,
						'revision_id' => $revision_id,
						'author_id' => bbp_get_current_user_id(),
						'reason' => $topic_edit_reason,
					));
				}
			}

			/** Move Topic ********************************************************/

			// If the new forum id is not equal to the old forum id, run the
			// bbp_move_topic action and pass the topic's forum id as the
			// first arg and topic id as the second to update counts.
			if ($forum_id !== $topic->post_parent) {
				bbp_move_topic_handler($topic_id, $topic->post_parent, $forum_id);
			}

			/** Stickies **********************************************************/

			if (!empty($topic_new->bbp_stick_topic) && in_array($topic_new->bbp_stick_topic, array_keys(bbp_get_topic_types()))) {

				// What's the caps?
				if (current_user_can('moderate')) {

					// What's the haps?
					switch ($topic_new->bbp_stick_topic) {

					// Sticky in forum
					case 'stick':
						bbp_stick_topic($topic_id);
						break;

					// Sticky in all forums
					case 'super':
						bbp_stick_topic($topic_id, true);
						break;

					// Normal
					case 'unstick':
					default:
						bbp_unstick_topic($topic_id);
						break;
					}
				}
			}

			/** Subscription **********************************************************/

			if (isset($topic_new->bbp_topic_subscription)) {

				// What's the caps?
				if (current_user_can('moderate')) {
					$user_id = bbp_get_user_id(0, true, true);
					if ($topic_new->bbp_topic_subscription) {
						$is_subscription = bbp_is_user_subscribed($user_id, $topic_id);
						if (!$is_subscription) {
							bbp_add_user_subscription($user_id, $topic_id);
						}
					} else {
						$is_subscription = bbp_is_user_subscribed($user_id, $topic_id);
						if ($is_subscription) {
							bbp_remove_user_subscription($user_id, $topic_id);
						}
					}
				}
			}

			/** Additional Actions (After Save) ***********************************/

			do_action('bbp_edit_topic_post_extras', $topic_id);

			/**
			 * Rest Hook
			 */
			do_action('boss_rest_bbpress_update_topic', $topic_id, $request);

			$request["id"] = $topic_id;

			return $this->_get_item($topic_id, $request);

			/** Errors ****************************************************************/

		} else {
			$append_error = (is_wp_error($topic_id) && $topic_id->get_error_message()) ? $topic_id->get_error_message() . ' ' : '';
			return new WP_Error('bp_json_topic_error', __('The following problem(s) have been found with your topic:' . $append_error . 'Please try again.', 'buddyboss-app'), array('status' => 400));
		}
	}

	/**
	 * @api {PATCH} /wp-json/appboss/bbpress/v1/topic/merge Merge topic
	 * @apiName MergeTopic
	 * @apiGroup BBPressTopic
	 * @apiVersion 1.0.0
	 * @apiDescription Merge BBPress Topic (Handles the front end merge topic submission)
	 * @apiDeprecated  Merge Topic. Check (#Forum Topics:MergeBBPTopic)
	 * @apiUse apidocForMergeTopicV1
	 * @apiPrivate
	 */
	/**
	 * @param string $action The requested action to compare this function to
	 * @return WP_Error
	 * @since bbPress (r2756)
	 *
	 */
	public function merge($request) {

		// Define local variable(s)
		$source_topic_id = $destination_topic_id = 0;
		$source_topic = $destination_topic = 0;
		$subscribers = $favoriters = $replies = array();

		/** Source Topic **********************************************************/

		// Topic id
		if (empty($request['bbp_topic_id'])) {
			return new WP_Error('bp_json_merge_topic_source_id', __('Topic ID not found.', 'buddyboss-app'), array('status' => 404));
		} else {
			$source_topic_id = (int) $request['bbp_topic_id'];
		}

		// Source topic not found
		if (!$source_topic = bbp_get_topic($source_topic_id)) {
			return new WP_Error('bp_json_merge_topic_source_not_found', __('The topic you want to merge was not found.', 'buddyboss-app'), array('status' => 404));
		}

		// Cannot edit source topic
		if (!current_user_can('edit_topic', $source_topic->ID)) {
			return new WP_Error('bp_json_merge_topic_source_permission', __('You do not have the permissions to edit the source topic.', 'buddyboss-app'), array('status' => 403));
		}

		/** Destination Topic *****************************************************/

		// Topic id
		if (empty($request['bbp_destination_topic'])) {
			return new WP_Error('bp_json_merge_topic_destination_id', __('Destination topic ID not found.', 'buddyboss-app'), array('status' => 404));
		} else {
			$destination_topic_id = (int) $request['bbp_destination_topic'];
		}

		// Destination topic not found
		if (!$destination_topic = bbp_get_topic($destination_topic_id)) {
			return new WP_Error('bp_json_merge_topic_destination_not_found', __('The topic you want to merge to was not found.', 'buddyboss-app'), array('status' => 404));
		}

		// Cannot edit destination topic
		if (!current_user_can('edit_topic', $destination_topic->ID)) {
			return new WP_Error('bp_json_merge_topic_destination_permission', __('You do not have the permissions to edit the destination topic.', 'buddyboss-app'), array('status' => 403));
		}

		/** No Errors *************************************************************/

		// Update counts, etc...
		do_action('bbp_merge_topic', $destination_topic->ID, $source_topic->ID);

		/** Date Check ************************************************************/

		// Check if the destination topic is older than the source topic
		if (strtotime($source_topic->post_date) < strtotime($destination_topic->post_date)) {

			// Set destination topic post_date to 1 second before source topic
			$destination_post_date = date('Y-m-d H:i:s', strtotime($source_topic->post_date) - 1);

			// Update destination topic
			wp_update_post(array(
				'ID' => $destination_topic_id,
				'post_date' => $destination_post_date,
				'post_date_gmt' => get_gmt_from_date($destination_post_date),
			));
		}

		/** Subscriptions *********************************************************/

		if (!empty($request['bbp_topic_subscribers']) && ("1" === $request['bbp_topic_subscribers']) && bbp_is_subscriptions_active()) {

			// Get subscribers from source topic
			$subscribers = bbp_get_topic_subscribers($source_topic->ID);

			// Remove the topic from everybody's subscriptions
			if (!empty($subscribers)) {

				// Loop through each user
				foreach ((array) $subscribers as $subscriber) {

					// Shift the subscriber if told to
					bbp_add_user_subscription($subscriber, $destination_topic->ID);

					// Remove old subscription
					bbp_remove_user_subscription($subscriber, $source_topic->ID);
				}
			}
		}

		/** Favorites *************************************************************/

		if (!empty($request['bbp_topic_favoriters']) && "1" === $request['bbp_topic_favoriters']) {
			// Get favoriters from source topic
			$favoriters = bbp_get_topic_favoriters($source_topic->ID);

			// Remove the topic from everybody's favorites
			if (!empty($favoriters)) {

				// Loop through each user
				foreach ((array) $favoriters as $favoriter) {

					// Shift the favoriter if told to
					bbp_add_user_favorite($favoriter, $destination_topic->ID);

					// Remove old favorite
					bbp_remove_user_favorite($favoriter, $source_topic->ID);
				}
			}
		}

		/** Tags ******************************************************************/

		if (!empty($request['bbp_topic_tags']) && ("1" === $request['bbp_topic_tags'])) {
			// Get the source topic tags
			$source_topic_tags = wp_get_post_terms($source_topic->ID, bbp_get_topic_tag_tax_id(), array('fields' => 'names'));

			// Tags to possibly merge
			if (!empty($source_topic_tags) && !is_wp_error($source_topic_tags)) {

				// Shift the tags if told to
				wp_set_post_terms($destination_topic->ID, $source_topic_tags, bbp_get_topic_tag_tax_id(), true);

				// Delete the tags from the source topic
				wp_delete_object_term_relationships($source_topic->ID, bbp_get_topic_tag_tax_id());
			}
		}

		/** Source Topic **********************************************************/

		// Status
		bbp_open_topic($source_topic->ID);

		// Sticky
		bbp_unstick_topic($source_topic->ID);

		// Get the replies of the source topic
		$replies = (array) get_posts(array(
			'post_parent' => $source_topic->ID,
			'post_type' => bbp_get_reply_post_type(),
			'posts_per_page' => -1,
			'order' => 'ASC',
		));

		// Prepend the source topic to its replies array for processing
		array_unshift($replies, $source_topic);

		if (!empty($replies)) {

			/** Merge Replies *****************************************************/

			// Change the post_parent of each reply to the destination topic id
			foreach ($replies as $reply) {

				// Update the reply
				wp_update_post(array(
					'ID' => $reply->ID,
					'post_title' => sprintf(__('Reply To: %s', 'bbpress'), $destination_topic->post_title),
					'post_name' => false,
					'post_type' => bbp_get_reply_post_type(),
					'post_parent' => $destination_topic->ID,
					'guid' => '',
				));

				// Adjust reply meta values
				bbp_update_reply_topic_id($reply->ID, $destination_topic->ID);
				bbp_update_reply_forum_id($reply->ID, bbp_get_topic_forum_id($destination_topic->ID));

				// Adjust reply to values
				$reply_to = bbp_get_reply_to($reply->ID);
				if (empty($reply_to)) {
					bbp_update_reply_to($reply->ID, $source_topic->ID);
				}

				// Do additional actions per merged reply
				do_action('bbp_merged_topic_reply', $reply->ID, $destination_topic->ID);
			}
		}

		/** Successful Merge ******************************************************/

		// Update topic's last meta data
		bbp_update_topic_last_reply_id($destination_topic->ID);
		bbp_update_topic_last_active_id($destination_topic->ID);
		bbp_update_topic_last_active_time($destination_topic->ID);

		// Send the post parent of the source topic as it has been shifted
		// (possibly to a new forum) so we need to update the counts of the
		// old forum as well as the new one
		do_action('bbp_merged_topic', $destination_topic->ID, $source_topic->ID, $source_topic->post_parent);

		$request["id"] = $destination_topic->ID;

		return $this->_get_item($destination_topic->ID, $request);

	}

	/**
	 * @api {PATCH} /wp-json/appboss/bbpress/v1/topic/split Split topic
	 * @apiName SplitTopic
	 * @apiGroup BBPressTopic
	 * @apiVersion 1.0.0
	 * @apiDescription Split BBPress Topic (Handles the front end split topic submission)
	 * @apiDeprecated  Split Topic. Check (#Forum Topics:SplitBBPTopic)
	 * @apiUse apidocForSplitTopicV1
	 * @apiPrivate
	 */
	/**
	 * @param string $action The requested action to compare this function to
	 * @return void|WP_Error
	 */
	public function split($request) {

		global $wpdb;

		// Prevent debug notices
		$from_reply_id = $destination_topic_id = 0;
		$destination_topic_title = '';
		$destination_topic = $from_reply = $source_topic = '';
		$split_option = false;

		/** Split Reply ***********************************************************/

		if (empty($request['bbp_reply_id'])) {
			return new WP_Error('bp_json_split_topic_reply_id', __('Reply ID to split the topic from was not found.', 'buddyboss-app'), array('status' => 400));
		} else {
			$from_reply_id = (int) $request['bbp_reply_id'];
		}

		$from_reply = bbp_get_reply($from_reply_id);

		// Reply exists
		if (empty($from_reply)) {
			return new WP_Error('bp_json_split_topic_not_found', __('The reply you want to split from was not found.', 'buddyboss-app'), array('status' => 404));
		}

		/** Topic to Split ********************************************************/

		// Get the topic being split
		$source_topic = bbp_get_topic($from_reply->post_parent);

		// No topic
		if (empty($source_topic)) {
			return new WP_Error('bp_json_split_topic_source_not_found', __('The topic you want to split was not found.', 'buddyboss-app'), array('status' => 404));
		}

		// Use cannot edit topic
		if (!current_user_can('edit_topic', $source_topic->ID)) {
			return new WP_Error('bp_json_split_topic_source_permission', __('You do not have the permissions to edit the source topic.', 'buddyboss-app'), array('status' => 403));
		}

		// How to Split
		if (!empty($request['bbp_topic_split_option'])) {
			$split_option = (string) trim($request['bbp_topic_split_option']);
		}

		// Invalid split option
		if (empty($split_option) || !in_array($split_option, array('existing', 'reply'))) {
			return new WP_Error('bp_json_split_topic_option', __('You need to choose a valid split option.', 'buddyboss-app'), array('status' => 400));

			// Valid Split Option
		} else {

			// What kind of split
			switch ($split_option) {

			// Into an existing topic
			case 'existing':

				// Get destination topic id
				if (empty($request['bbp_destination_topic'])) {
					return new WP_Error('bp_json_split_topic_destination_id', __('Destination topic ID not found!', 'buddyboss-app'), array('status' => 404));
				} else {
					$destination_topic_id = (int) $request['bbp_destination_topic'];
				}

				// Get the destination topic
				$destination_topic = bbp_get_topic($destination_topic_id);

				// No destination topic
				if (empty($destination_topic)) {
					return new WP_Error('bp_json_split_topic_destination_not_found', __('The topic you want to split to was not found.', 'buddyboss-app'), array('status' => 404));
				}

				// User cannot edit the destination topic
				if (!current_user_can('edit_topic', $destination_topic->ID)) {
					return new WP_Error('bp_json_split_topic_destination_permission', __('You do not have the permissions to edit the destination topic!', 'buddyboss-app'), array('status' => 403));
				}

				break;

			// Split at reply into a new topic
			case 'reply':
			default:

				// User needs to be able to publish topics
				if (current_user_can('publish_topics')) {

					// Use the new title that was passed
					if (!empty($request['bbp_topic_split_destination_title'])) {
						$destination_topic_title = esc_attr(strip_tags($request['bbp_topic_split_destination_title']));

						// Use the source topic title
					} else {
						$destination_topic_title = $source_topic->post_title;
					}

					// Update the topic
					$destination_topic_id = wp_update_post(array(
						'ID' => $from_reply->ID,
						'post_title' => $destination_topic_title,
						'post_name' => false,
						'post_type' => bbp_get_topic_post_type(),
						'post_parent' => $source_topic->post_parent,
						'menu_order' => 0,
						'guid' => '',
					));
					$destination_topic = bbp_get_topic($destination_topic_id);

					// Make sure the new topic knows its a topic
					bbp_update_topic_topic_id($from_reply->ID);

					// Shouldn't happen
					if (false === $destination_topic_id || is_wp_error($destination_topic_id) || empty($destination_topic)) {
						return new WP_Error('bp_json_split_topic_destination_reply', __('There was a problem converting the reply into the topic. Please try again.', 'buddyboss-app'), array('status' => 500));
					}

					// User cannot publish posts
				} else {
					return new WP_Error('bp_json_split_topic_destination_permission', __('You do not have the permissions to create new topics. The reply could not be converted into a topic.', 'buddyboss-app'), array('status' => 403));
				}

				break;
			}
		}

		// Bail if there are errors
		if (bbp_has_errors()) {
			return;
		}

		/** No Errors - Do the Spit ***********************************************/

		// Update counts, etc...
		do_action('bbp_pre_split_topic', $from_reply->ID, $source_topic->ID, $destination_topic->ID);

		/** Date Check ************************************************************/

		// Check if the destination topic is older than the from reply
		if (strtotime($from_reply->post_date) < strtotime($destination_topic->post_date)) {

			// Set destination topic post_date to 1 second before from reply
			$destination_post_date = date('Y-m-d H:i:s', strtotime($from_reply->post_date) - 1);

			// Update destination topic
			wp_update_post(array(
				'ID' => $destination_topic_id,
				'post_date' => $destination_post_date,
				'post_date_gmt' => get_gmt_from_date($destination_post_date),
			));
		}

		/** Subscriptions *********************************************************/

		// Copy the subscribers
		if (!empty($request['bbp_topic_subscribers']) && "1" === $request['bbp_topic_subscribers'] && bbp_is_subscriptions_active()) {

			// Get the subscribers
			$subscribers = bbp_get_topic_subscribers($source_topic->ID);

			if (!empty($subscribers)) {

				// Add subscribers to new topic
				foreach ((array) $subscribers as $subscriber) {
					bbp_add_user_subscription($subscriber, $destination_topic->ID);
				}
			}
		}

		/** Favorites *************************************************************/

		// Copy the favoriters if told to
		if (!empty($request['bbp_topic_favoriters']) && ("1" === $request['bbp_topic_favoriters'])) {

			// Get the favoriters
			$favoriters = bbp_get_topic_favoriters($source_topic->ID);

			if (!empty($favoriters)) {

				// Add the favoriters to new topic
				foreach ((array) $favoriters as $favoriter) {
					bbp_add_user_favorite($favoriter, $destination_topic->ID);
				}
			}
		}

		/** Tags ******************************************************************/

		// Copy the tags if told to
		if (!empty($request['bbp_topic_tags']) && ("1" === $request['bbp_topic_tags'])) {

			// Get the source topic tags
			$source_topic_tags = wp_get_post_terms($source_topic->ID, bbp_get_topic_tag_tax_id(), array('fields' => 'names'));

			if (!empty($source_topic_tags)) {
				wp_set_post_terms($destination_topic->ID, $source_topic_tags, bbp_get_topic_tag_tax_id(), true);
			}
		}

		/** Split Replies *********************************************************/

		// get_posts() is not used because it doesn't allow us to use '>='
		// comparision without a filter.
		$replies = (array) $wpdb->get_results($wpdb->prepare("SELECT * FROM {$wpdb->posts} WHERE {$wpdb->posts}.post_date >= %s AND {$wpdb->posts}.post_parent = %d AND {$wpdb->posts}.post_type = %s ORDER BY {$wpdb->posts}.post_date ASC", $from_reply->post_date, $source_topic->ID, bbp_get_reply_post_type()));

		// Make sure there are replies to loop through
		if (!empty($replies) && !is_wp_error($replies)) {

			// Calculate starting point for reply positions
			switch ($split_option) {

			// Get topic reply count for existing topic
			case 'existing':
				$reply_position = bbp_get_topic_reply_count($destination_topic->ID);
				break;

			// Account for new lead topic
			case 'reply':
				$reply_position = 1;
				break;
			}

			// Save reply ids
			$reply_ids = array();

			// Change the post_parent of each reply to the destination topic id
			foreach ($replies as $reply) {

				// Bump the reply position each iteration through the loop
				$reply_position++;

				// Update the reply
				wp_update_post(array(
					'ID' => $reply->ID,
					'post_title' => sprintf(__('Reply To: %s', 'bbpress'), $destination_topic->post_title),
					'post_name' => false, // will be automatically generated
					'post_parent' => $destination_topic->ID,
					'menu_order' => $reply_position,
					'guid' => '',
				));

				// Gather reply ids
				$reply_ids[] = $reply->ID;

				// Adjust reply meta values
				bbp_update_reply_topic_id($reply->ID, $destination_topic->ID);
				bbp_update_reply_forum_id($reply->ID, bbp_get_topic_forum_id($destination_topic->ID));

				// Adjust reply to values
				$reply_to = bbp_get_reply_to($reply->ID);

				// Not a reply to a reply that moved over
				if (!in_array($reply_to, $reply_ids)) {
					bbp_update_reply_to($reply->ID, 0);
				}

				// New topic from reply can't be a reply to
				if (($from_reply->ID === $destination_topic->ID) && ($from_reply->ID === $reply_to)) {
					bbp_update_reply_to($reply->ID, 0);
				}

				// Do additional actions per split reply
				do_action('bbp_split_topic_reply', $reply->ID, $destination_topic->ID);
			}

			// Remove reply to from new topic
			if ($from_reply->ID === $destination_topic->ID) {
				delete_post_meta($from_reply->ID, '_bbp_reply_to');
			}

			// Set the last reply ID and freshness
			$last_reply_id = $reply->ID;
			$freshness = $reply->post_date;

			// Set the last reply ID and freshness to the from_reply
		} else {
			$last_reply_id = $from_reply->ID;
			$freshness = $from_reply->post_date;
		}

		// It is a new topic and we need to set some default metas to make
		// the topic display in bbp_has_topics() list
		if ('reply' === $split_option) {
			bbp_update_topic_last_reply_id($destination_topic->ID, $last_reply_id);
			bbp_update_topic_last_active_id($destination_topic->ID, $last_reply_id);
			bbp_update_topic_last_active_time($destination_topic->ID, $freshness);
		}

		// Update source topic ID last active
		bbp_update_topic_last_reply_id($source_topic->ID);
		bbp_update_topic_last_active_id($source_topic->ID);
		bbp_update_topic_last_active_time($source_topic->ID);

		/** Successful Split ******************************************************/

		// Update counts, etc...
		do_action('bbp_post_split_topic', $from_reply->ID, $source_topic->ID, $destination_topic->ID);

		$request["id"] = $destination_topic->ID;

		return $this->_get_item($destination_topic->ID, $request);

	}

    /**
     * @param $request
     * @return true|WP_Error
     * @api {GET | PATCH} /wp-json/appboss/bbpress/v1/dropdown/:id Dropdown topic
     * @apiName DropdownTopic
     * @apiGroup BBPressTopic
     * @apiVersion 1.0.0
     * @apiDescription Get BBPress dropdown; Update BBPress dropdown
     * @apiHeader {String} accessToken Auth token
     * @apiParam {Number} page Current page of the collection
     * @apiParam {Number} [posts_per_page=10] Maximum number of items to be returned in result set. -1 for all
     * @apiParam {Number} bbp_topic_tags Whether topic has tag or not.
     * @apiDeprecated  Siblings of the topic. Check (#Forum Topics:DropdownBBPTopic)
     * @apiPrivate
     */
	public function dropdown($request) {

		$topic_id = (int) $request['id'];

		$is_valid = $this->check_topic($topic_id);
		if (is_wp_error($is_valid)) {
			return $is_valid;
		}

		$parent = bbp_get_topic_forum_id($topic_id);

		$args = array(
			'posts_per_page' => isset($request['posts_per_page']) ? $request['posts_per_page'] : -1,
			'page' => isset($request['page']) ? $request['page'] : 1,
			'post_type' => bbp_get_topic_post_type(),
			'post_parent' => $parent,
			'orderby' => 'menu_order title',
			'order' => 'ASC',
			'exclude' => $topic_id,
			'disable_categories' => true,
			'post_status' => "publish",
		);

		$posts = get_posts($args);

		$data = array();

		foreach ($posts as $post) {
			$data[] = array(
				"id" => $post->ID,
				"title" => array("rendered" => $post->post_title),
				"ID" => $post->ID, // fallback Support.
				"post_title" => $post->post_title, // fallback Support.
			);
		}

		if (!(bbp_has_topics(array('show_stickies' => false, 'post_parent' => $parent, 'post__not_in' => array($topic_id))))) {
			return new WP_Error('bp_json_no_other_topics', __('There are no other topics in this forum.', 'buddyboss-app'), array('status' => 400));
		}

		return rest_ensure_response($data);

	}

	/**
	 * Prepare a single topic for create or update
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return object $topic User object.
	 */
	protected function prepare_topic_for_database($request) {
		$topic = new stdClass;

		if (isset($request['id'])) {
			$topic->bbp_topic_id = $request['id'];
		}
		if (isset($request['parent'])) {
			$topic->bbp_forum_id = $request['parent'];
		}
		if (isset($request['group_id'])) {
			$topic->bbp_group_id = $request['group_id'];
		}
		if (isset($request['title'])) {
			$topic->bbp_topic_title = $request['title'];
		}
		if (isset($request['status'])) {
			$topic->bbp_topic_status = $request['status'];
		}
		if (isset($request['tags'])) {
			$topic->bbp_topic_tags = $request['tags'];
		}
		if (isset($request['type'])) {
			$topic->bbp_stick_topic = $request['type'];
		}
		if (isset($request['content'])) {
			$topic->bbp_topic_content = $request['content'];
		}
		if (isset($request['reason'])) {
			$topic->bbp_topic_edit_reason = $request['reason'];
		}
		if (isset($request['log'])) {
			$topic->bbp_log_topic_edit = $request['log'];
		}
		if (isset($request['subscribe']) && (true === $request['subscribe'])) {
			$topic->bbp_topic_subscription = true;
		} else if (isset($request['subscribe']) && (false === $request['subscribe'])) {
			$topic->bbp_topic_subscription = false;
		}

		/**
		 * Filter topic data before inserting user via REST API
		 *
		 * @param object $topic Topic object.
		 * @param WP_REST_Request $request Request object.
		 */
		return apply_filters('boss_bp_json_pre_insert_topic', $topic, $request);
	}

    /**
     * Check if topic id is valid and if topic exists
     * @param $topic_id
     * @return true|WP_Error
     */
	public function check_topic($topic_id) {
		if (!$topic_id || !is_int($topic_id)) {
			return new WP_Error('bp_json_topic_invalid_id', __('Topic ID is invalid.', 'buddyboss-app'), array('status' => 400));
		}

		// Make sure topic exists
		$topic = bbp_get_topic($topic_id);
		if (empty($topic)) {
			return new WP_Error('bp_json_topic_doesnt_exists', __('Topic doesn\'t exist.', 'buddyboss-app'), array('status' => 404));
		}

		return true;
	}

    /**
     * @param $topic_id
     * @param $request
     * @return WP_REST_Response
     */
	public function _get_item($topic_id, $request) {

		/**
		 * Redirect to topic endpoint.
		 */
		global $wp_rest_server;

		$request_topic = new WP_REST_Request("GET", "/" . $this->namespace_slug . "/bbpress/v1/topic/{$topic_id}");
		$server = rest_get_server();
		$response = $server->dispatch($request_topic);

		return $wp_rest_server->response_to_data(rest_ensure_response($response), isset($request['_embed']) ? 1 : 0);
	}

	/**
	 * Dispatch the request item.
	 * @param $request
	 *
	 * @return mixed
	 */
	protected function dispatch($request) {

		$query_params = array();

		if (isset($request->get_query_params()["_embed"])) {
			$query_params["_embed"] = $request->get_query_params()["_embed"];
		}

		$request->set_query_params($query_params);

		$server = rest_get_server();
		$response = $server->dispatch($request);

		return $response;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_create_collection_params() {
		$params = array();

		$params['id'] = array(
			'description' => __('Id of topic'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
		);

		$params['parent'] = array(
			'description' => __('Parent id of topic'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
		);

		$params['group_id'] = array(
			'description' => __('Id of group'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
		);

		$params['title'] = array(
			'description' => __('Title of topic'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
			'required' => true,
		);

		$params['status'] = array(
			'description' => __('Status of topic'),
			'type' => 'string',
			'sanitize_callback' => 'sanitize_key',
			'type' => 'string',
		);

		$params['tags'] = array(
			'description' => __('Tags name of topic'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['type'] = array(
			'description' => __('Type of topic'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['content'] = array(
			'description' => __('Content of topic'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
			'required' => true,
		);

		$params['reason'] = array(
			'description' => __('Reason of topic edit'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['log'] = array(
			'description' => __('Log of topic edit'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['subscribe'] = array(
			'description' => __('whether user subscribe topic or no'),
			'type' => 'boolean',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

}