<?php
namespace BuddyBossApp\Api\BbPress;
use WP_Error;
use WP_Query;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Server;

// NOTE : Old classname was class.boss_bbpress_api_search_api By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for bbpress.
class SearchApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/bbpress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->hooks();
	}

	public function hooks() {
		add_action('rest_api_init', array($this, "register_routes"), 99);
	}

	public function register_routes() {

		register_rest_route($this->namespace, '/search/', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'search'),
				'args' => $this->get_collection_params(),
				'permission_callback' => '__return_true',
			),
		));
	}

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/bbpress/v1/search Search bbpress post
     * @apiName SearchBBPressPosts
     * @apiGroup BBPress
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDeprecated  search endpoint
     * @apiUse apidocForBBPressPostsV1
     * @apiPrivate
     */
	public function search($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		global $wp_rest_server;

		$args = array();
		$args['author__in'] = $request['author'];
		$args['author__not_in'] = $request['author_exclude'];
		$args['menu_order'] = $request['menu_order'];
		$args['offset'] = $request['offset'];
		$args['order'] = $request['order'];
		$args['orderby'] = $request['orderby'];
		$args['paged'] = $request['page'];
		$args['post__in'] = $request['include'];
		$args['post__not_in'] = $request['exclude'];
		$args['posts_per_page'] = $request['per_page'];
		$args['name'] = $request['slug'];

		if (is_array($request['parent'])) {
			$args['post_parent__in'] = $request['parent'];
		} else {
			$args['post_parent'] = $request['parent'];
		}

		$args['post_parent__not_in'] = $request['parent_exclude'];
		$args['s'] = $request['search'];

		// Force the post_type argument, since it's not a user input variable.
		$args['post_type'] = array(bbp_get_topic_post_type(), bbp_get_reply_post_type());

		/**
		 * Post Status
		 */
		// get all post status for moderator and admin
		if (current_user_can('moderate')) {
			// Default view=all statuses
			$post_statuses = array(
				bbp_get_public_status_id(),
				bbp_get_closed_status_id(),
				bbp_get_spam_status_id(),
				bbp_get_trash_status_id(),
			);

			// Add support for private status
			if (current_user_can('read_private_replies')) {
				$post_statuses[] = bbp_get_private_status_id();
			}

			// Join post statuses together
			$args['post_status'] = implode(',', $post_statuses);
		} else {
			$args['perm'] = 'readable';
		}

		$args['fields'] = 'ids';

		$posts_query = new WP_Query($args);

		$request_topic = new WP_REST_Request("GET", "/" . $this->namespace_slug . "/bbpress/v1/topic");
		$request_topic->set_param('include', $posts_query->posts);
		$server = rest_get_server();
		$response = $server->dispatch($request_topic);

		$topics = $wp_rest_server->response_to_data(rest_ensure_response($response), isset($request['_embed']) ? 1 : 0);

		$request_reply = new WP_REST_Request("GET", "/" . $this->namespace_slug . "/bbpress/v1/reply");
		$request_reply->set_param('include', $posts_query->posts);
		$server = rest_get_server();
		$response = $server->dispatch($request_reply);

		$replies = $wp_rest_server->response_to_data(rest_ensure_response($response), isset($request['_embed']) ? 1 : 0);

		$search_response = array_merge($topics, $replies);

		$ids = wp_list_pluck($search_response, 'id');

		$posts_array = array();
		foreach ($posts_query->posts as $post) {
			$index = array_search($post, $ids);
			$posts_array[] = $search_response[$index];
		}

		$page = (int) @$args['paged'];
		$total_posts = $posts_query->found_posts;

		if ($total_posts < 1) {
			// Out-of-bounds, run the query again without LIMIT for total count
			unset($args['paged']);
			$count_query = new WP_Query();
			$count_query->query($args);
			$total_posts = $count_query->found_posts;
		}

		$max_pages = ceil($total_posts / (int) $args['posts_per_page']);

		$response = rest_ensure_response($posts_array);
		$response->header('X-WP-Total', (int) $total_posts);
		$response->header('X-WP-TotalPages', (int) $max_pages);

		return $response;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		$params['exclude'] = array(
			'description' => __('Ensure result set excludes specific ids.'),
			'type' => 'array',
			'default' => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description' => __('Limit result set to specific ids.'),
			'type' => 'array',
			'default' => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset'] = array(
			'description' => __('Offset the result set by a specific number of items.'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['order'] = array(
			'description' => __('Order sort attribute ascending or descending.'),
			'type' => 'string',
			'default' => 'desc',
			'enum' => array('asc', 'desc'),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['orderby'] = array(
			'description' => __('Sort collection by object attribute.'),
			'type' => 'string',
			'default' => 'date',
			'enum' => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'trending',
				'default',
				'activity',
				'popular',
				'items',
				'parent',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['orderby']['enum'][] = 'menu_order';

		$params['slug'] = array(
			'description' => __('Limit result set to posts with a specific slug.'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['status'] = array(
			'default' => 'publish',
			'description' => __('Limit result set to posts assigned a specific status.'),
			'sanitize_callback' => 'sanitize_key',
			'type' => 'string',
		);

		$params['parent'] = array(
			'description' => __('Limit result set to those of particular parent ids.'),
			'type' => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default' => array(),
		);
		$params['parent_exclude'] = array(
			'description' => __('Limit result set to all items except those of a particular parent id.'),
			'type' => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default' => array(),
		);

		$params['author'] = array(
			'description' => __('Limit result set to those of particular author ids.'),
			'type' => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default' => array(),
		);

		$params['author_exclude'] = array(
			'description' => __('Limit result set to all items except those of a particular author id.'),
			'type' => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default' => array(),
		);

		return $params;
	}
}