<?php
namespace BuddyBossApp\Api\BbPress;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Server;

// NOTE : Old classname was class.boss_bbpress_api_rest_api By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for bbpress.
class RestApi extends WP_REST_Controller {

	protected $namespace_slug = '';
	protected $namespace = '/bbpress/v1';

	public function __construct($slug) {
		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->hooks();
	}

	public function hooks() {
		add_action('rest_api_init', array($this, "init_rest"));
		add_action('rest_api_init', array($this, "register_bbpress_fields"), 99);
		add_action('rest_api_init', array($this, "register_routes"), 99);

		add_filter("rest_prepare_forum", array($this, "prepare_forum_linking"), 99, 3);
		add_filter("rest_prepare_topic", array($this, "prepare_topic_linking"), 99, 3);
		add_filter("rest_prepare_reply", array($this, "prepare_reply_linking"), 99, 3);
		add_filter("rest_avatar_sizes", array($this, "avatar_sizes"));

		add_filter("bbp_exclude_forum_ids", array($this, "allow_member_private_forum"), 999, 3);

		add_filter('bbapp_rest_search_tab', array($this, 'add_search_tabs'));
	}

	public function init_rest() {

		/**
		 * Remove revision details from content
		 */
		remove_filter('bbp_get_reply_content', 'bbp_reply_content_append_revisions', 99, 2);
		remove_filter('bbp_get_topic_content', 'bbp_topic_content_append_revisions', 99, 2);
	}

	public function register_routes() {

		register_rest_route($this->namespace, '/settings/', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_settings'),
				'permission_callback' => '__return_true',
			),
		));
	}

	/**
	 * @api {GET} /wp-json/appboss/bbpress/v1/settings Settings
	 * @apiName BBPressSettings
	 * @apiGroup BBPress
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get all BBPress's Settings
	 * @apiDeprecated  Retrieve settings. Check (#Settings:GetBBSettings)
	 * @apiUse apidocForBBPressSettingsV1
	 * @apiPrivate
	 */
	public function get_settings($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$settings = array(
			'bbp_allow_threaded_replies' => bbp_allow_threaded_replies(),
			'bbp_thread_replies_depth' => bbp_thread_replies_depth(),
			'bbp_allow_search' => bbp_allow_search(),
			'bbp_allow_topic_tags' => bbp_allow_topic_tags(),
			'bbp_is_subscriptions_active' => bbp_is_subscriptions_active(),
			'bbp_is_favorites_active' => bbp_is_favorites_active(),
			'bbp_is_anonymous' => bbp_is_anonymous(),
			'bbp_allow_revisions' => bbp_allow_revisions(),
		);

		return rest_ensure_response($settings);
	}

	/**
	 * Prepare linking embeds for the forum post type
	 *
	 * @since 1.0.0
	 *
	 * @param   $response
	 * @param   $post
	 * @param   $request
	 *
	 * @return void
	 */
	public function prepare_forum_linking($response, $post, $request) {

		// Embed Possible Needed Data.

		if (strpos($request->get_route(), '/' . $this->namespace_slug) !== 0) {
			return $response;
		}

		$authors = array();
		$authors[$post->post_author] = $post->post_author;

		if (isset($response->data["last_reply_id"]) && $response->data["last_reply_id"] != "") {
			$p = get_post($response->data["last_reply_id"]);
			if ($p) {
				$authors[$p->post_author] = $p->post_author;
			}
			$response->add_link('reply', rest_url('/' . $this->namespace_slug . '/bbpress/v1/reply/' . $response->data["last_reply_id"]),
				array('embeddable' => true));
		}

		if (isset($response->data["last_topic_id"]) && $response->data["last_topic_id"] != "") {
			$p = get_post($response->data["last_topic_id"]);
			if ($p) {
				$authors[$p->post_author] = $p->post_author;
			}
			$response->add_link('topic', rest_url('/' . $this->namespace_slug . '/bbpress/v1/topic/' . $response->data["last_topic_id"]),
				array('embeddable' => true));
		}

		if (isset($response->data["last_active_author_id"]) && $response->data["last_active_author_id"] != "") {
			$p = get_userdata($response->data["last_active_author_id"]);
			if ($p) {
				$authors[$p->ID] = $p->ID;
			}
		}

		foreach ($authors as $author) {
			if (empty($author)) {
				continue;
			}
			$response->add_link('author', rest_url('/wp/v2/users/' . $author), array('embeddable' => true));
		}

		return $response;
	}

	/**
	 * Prepare linking embeds for the forum post type
	 *
	 * @since 1.0.0
	 *
	 * @param   $response
	 * @param   $post
	 * @param   $request
	 *
	 * @return void
	 */
	public function prepare_topic_linking($response, $post, $request) {

		// Embed Possible Needed Data.
		if (strpos($request->get_route(), '/' . $this->namespace_slug) !== 0) {
			return $response;
		}

		$authors = array();
		$authors[$post->post_author] = $post->post_author;

		if (isset($response->data["last_reply_id"]) && $response->data["last_reply_id"] != "") {
			$p = get_post($response->data["last_reply_id"]);
			if ($p) {
				$authors[$p->post_author] = $p->post_author;
			}
			$response->add_link('reply', rest_url('/' . $this->namespace_slug . '/bbpress/v1/reply/' . $response->data["last_reply_id"]),
				array('embeddable' => true));
		}

		if (isset($response->data["forum_id"]) && $response->data["forum_id"] != "") {
			$p = get_post($response->data["forum_id"]);
			if ($p) {
				$authors[$p->post_author] = $p->post_author;
			}
			$response->add_link('forum', rest_url('/' . $this->namespace_slug . '/bbpress/v1/forum/' . $response->data["forum_id"]),
				array('embeddable' => true));
		}

		if (isset($response->data["last_active_author_id"]) && $response->data["last_active_author_id"] != "") {
			$p = get_userdata($response->data["last_active_author_id"]);
			if ($p) {
				$authors[$p->ID] = $p->ID;
			}
		}

		foreach ($authors as $author) {
			$response->add_link('author', rest_url('/wp/v2/users/' . $author), array('embeddable' => true));
		}

		return $response;
	}

	/**
	 * Prepare linking embeds for the reply post type
	 *
	 * @since 1.0.0
	 *
	 * @param   $response
	 * @param   $post
	 * @param   $request
	 *
	 * @return void
	 */
	public function prepare_reply_linking($response, $post, $request) {

		// Embed Possible Needed Data .
		if (strpos($request->get_route(), '/' . $this->namespace_slug) !== 0) {
			return $response;
		}

		$authors = array();
		$authors[$post->post_author] = $post->post_author;

		// embed topic data
		if (isset($response->data["parent"]) && $response->data["parent"] != "") {
			$p = get_post($response->data["parent"]);
			if ($p) {
				$authors[$p->post_author] = $p->post_author;
			}
			$response->add_link('topic', rest_url('/' . $this->namespace_slug . '/bbpress/v1/topic/' . $response->data["parent"]),
				array('embeddable' => true));
		}

		foreach ($authors as $author) {
			$response->add_link('author', rest_url('/wp/v2/users/' . $author), array('embeddable' => true));
		}

		return $response;

	}

	/**
	 * Register bbpress response fields
	 *
	 * @since 1.0,0
	 * @return void
	 */
	public function register_bbpress_fields() {
		global $wp;

		if ( ! empty( $wp->request ) && strpos( $wp->request, 'buddyboss/v1' ) !== false ) {
			return;
		}

		register_rest_field('forum',
			'short_content',
			array(
				'get_callback' => array($this, "get_short_content"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'is_group',
			array(
				'get_callback' => array($this, "check_group_forum"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'ancestor_forums',
			array(
				'get_callback' => array($this, "get_forum_ancestors"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'sub_forums',
			array(
				'get_callback' => array($this, "get_sub_forums"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'sub_forums_pages',
			array(
				'get_callback' => array($this, "get_sub_forums_pages"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'total_topic_count',
			array(
				'get_callback' => array($this, "_bbp_total_topic_count"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'total_reply_count',
			array(
				'get_callback' => array($this, "_bbp_total_reply_count"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'last_reply_id',
			array(
				'get_callback' => array($this, "_bbp_last_reply_id"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'last_active_author_id',
			array(
				'get_callback' => array($this, "get_last_active_author_id"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'last_active_time',
			array(
				'get_callback' => array($this, "_bbp_last_active_time"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'is_closed',
			array(
				'get_callback' => array($this, "_bbp_is_closed"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'short_content',
			array(
				'get_callback' => array($this, "get_short_content"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'action_states',
			array(
				'get_callback' => array($this, "_bbp_topic_action_states"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'is_group',
			array(
				'get_callback' => array($this, "check_group_forum_topic"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'action_states',
			array(
				'get_callback' => array($this, "_bbp_forum_action_states"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'topic_tags',
			array(
				'get_callback' => array($this, "_bbp_topic_tags"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'reply_count',
			array(
				'get_callback' => array($this, "_bbp_reply_count"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'voice_count',
			array(
				'get_callback' => array($this, "_bbp_voice_count"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'forum_id',
			array(
				'get_callback' => array($this, "_bbp_forum_id"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'is_topic_anonymous',
			array(
				'get_callback' => array($this, "_bbp_is_topic_anonymous"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'anonymous_author_data',
			array(
				'get_callback' => array($this, "bbp_add_anonymous_author_data"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('reply',
			'is_reply_anonymous',
			array(
				'get_callback' => array($this, "_bbp_is_reply_anonymous"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('reply',
			'anonymous_author_data',
			array(
				'get_callback' => array($this, "bbp_add_reply_anonymous_author_data"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'classes',
			array(
				'get_callback' => array($this, "_bbp_get_topic_classes"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'current_user_permissions',
			array(
				'get_callback' => array($this, "_bbp_get_forum_current_user_permissions"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'current_user_permissions',
			array(
				'get_callback' => array($this, "_bbp_get_topic_current_user_permissions"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'topic_status',
			array(
				'get_callback' => array($this, "_bbp_get_topic_topic_status"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'last_reply_id',
			array(
				'get_callback' => array($this, "_bbp_last_reply_id"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'last_active_author_id',
			array(
				'get_callback' => array($this, "get_last_active_author_id"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('topic',
			'last_active_time',
			array(
				'get_callback' => array($this, "_bbp_last_active_time_topic"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('reply',
			'current_user_permissions',
			array(
				'get_callback' => array($this, "_bbp_get_reply_current_user_permissions"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('forum',
			'last_topic_id',
			array(
				'get_callback' => array($this, "_bbp_last_topic_id"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('reply',
			'short_content',
			array(
				'get_callback' => array($this, "get_short_content"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('reply',
			'action_states',
			array(
				'get_callback' => array($this, "_bbp_reply_action_states"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('reply',
			'reply_to',
			array(
				'get_callback' => array($this, "_bbp_reply_to"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('user',
			'nicename',
			array(
				'get_callback' => array($this, "get_nicename"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('user',
			'bbp_user_profile_url',
			array(
				'get_callback' => array($this, "get_bbp_user_profile_url"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('user',
			'topics_started_count',
			array(
				'get_callback' => array($this, "get_topics_started_count"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('user',
			'replies_created_count',
			array(
				'get_callback' => array($this, "get_replies_created_count"),
				'update_callback' => null,
				'schema' => null,
			)
		);

//		register_rest_field('user',
//			'loggedin_user_permissions',
//			array(
//				'get_callback' => array($this, "get_current_user_permissions"),
//				'update_callback' => null,
//				'schema' => null,
//			)
//		);

		register_rest_field('user',
			'forum_role',
			array(
				'get_callback' => array($this, "get_user_forum_role"),
				'update_callback' => null,
				'schema' => null,
			)
		);

	}

	public function avatar_sizes() {
		return array(24, 48, 96, 120, 360);
	}

	/**
	 * check if forum is group forum
	 *
	 * @param $object
	 * @param $field_name
	 * @param $request
	 *
	 * @return bool
	 */
	public function check_group_forum($object, $field_name, $request) {

		if (function_exists("bbp_is_forum_group_forum") && bbp_is_forum_group_forum($object['id']) && function_exists('groups_get_group')) {
			$group_ids = bbp_get_forum_group_ids($object['id']);
			if (!empty($group_ids)) {
				foreach ($group_ids as $group_id) {
					$group = groups_get_group($group_id);
					return $group;
				}
			}
		}
		return false;
	}

	/**
	 * check if topic's forum is group forum
	 *
	 * @param $object
	 * @param $field_name
	 * @param $request
	 *
	 * @return bool
	 */
	public function check_group_forum_topic($object, $field_name, $request) {
		$forum_id = bbp_get_topic_forum_id($object['id']);
		if (function_exists("bbp_is_forum_group_forum") && bbp_is_forum_group_forum($forum_id) && function_exists('groups_get_group')) {
			$group_ids = bbp_get_forum_group_ids($forum_id);
			if (!empty($group_ids)) {
				foreach ($group_ids as $group_id) {
					$group = groups_get_group($group_id);
					return $group;
				}
			}
		}
		return false;
	}

	public function get_forum_ancestors($object, $field_name, $request) {

		$ancestors_forums = bbp_get_forum_ancestors($object["id"]);

		if (empty($ancestors_forums)) {
			return 0;
		}

		$return = array();

		$i = 0;
		foreach ($ancestors_forums as $ancestor_forum) {
			$i++; // Separator count

			$ancestor_forum = bbp_get_forum($ancestor_forum);

			// Get forum details
			$count = array();
			$permalink = bbp_get_forum_permalink($ancestor_forum->ID);
			$title = bbp_get_forum_title($ancestor_forum->ID);
			$slug = $ancestor_forum->post_name;

			// Show topic count
			$count['topics'] = bbp_get_forum_topic_count($ancestor_forum->ID);
			// Show reply count
			$count['replies'] = bbp_get_forum_reply_count($ancestor_forum->ID);

			$return[] = array(
				"id" => $ancestor_forum->ID,
				"slug" => $slug,
				"link" => $permalink,
				"title" => $title,
				"count" => $count,
				"is_group" => $this->check_group_forum(array('id' => $ancestor_forum->ID), $field_name, $request),
			);

		}

		return $return;

	}

	public function get_sub_forums($object, $field_name, $request) {

		$sub_forums = bbp_forum_get_subforums($object["id"]);

		if (empty($sub_forums)) {
			return 0;
		}

		$return = array();

		$i = 0;
		foreach ($sub_forums as $sub_forum) {
			$i++; // Separator count

			// Get forum details
			$count = array();
			$permalink = bbp_get_forum_permalink($sub_forum->ID);
			$title = bbp_get_forum_title($sub_forum->ID);
			$slug = $sub_forum->post_name;

			// Show topic count
			$count['topics'] = bbp_get_forum_topic_count($sub_forum->ID);
			// Show reply count
			$count['replies'] = bbp_get_forum_reply_count($sub_forum->ID);

			$return[] = array(
				"id" => $sub_forum->ID,
				"slug" => $slug,
				"title" => $title,
				"count" => $count,
				"link" => $permalink,
				"is_group" => $this->check_group_forum(array('id' => $sub_forum->ID), $field_name, $request),
			);

		}

		return $return;

	}

	public function get_sub_forums_pages($object, $field_name, $request) {
		$total_forums = bbp_get_forum_subforum_count($object["id"], true);
		if (!empty($total_forums)) {
			$per_page = get_option('_bbp_forums_per_page', 50);
			$total_page = $total_forums / $per_page;
			return ceil($total_page);
		}
		return 0;
	}

	public function get_current_user_permissions($object, $field_name, $request) {
		return array(
			'can_access_create_topic' => current_user_can('publish_topics'),
			'can_access_create_reply' => current_user_can('publish_replies'),
			'can_assign_topic_tags' => current_user_can('assign_topic_tags'),
			'can_post_unfiltered_html' => current_user_can('unfiltered_html'),
			'can_moderate' => current_user_can('moderate'),
			'can_edit_user' => current_user_can('edit_user', $object["id"]),
		);
	}

	/**
	 * Get user forum role
	 *
	 * @param $object
	 *
	 * @return string
	 */
	public function get_user_forum_role($object) {
		return bbp_get_user_display_role($object['id']);
	}

	public function get_topics_started_count($object, $field_name, $request) {

		global $wpdb;

		$query = $wpdb->prepare("SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = %s AND  post_status NOT IN ('spam','trash','pending') AND post_author = %s", 'topic', (int) $object["id"]);

		$count = (int) $wpdb->get_var($query);

		return $count;

	}

	public function get_replies_created_count($object, $field_name, $request) {

		global $wpdb;

		$query = $wpdb->prepare("SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = %s AND  post_status NOT IN ('spam','trash','pending') AND post_author = %s", 'reply', (int) $object["id"]);
		$count = (int) $wpdb->get_var($query);

		return $count;

	}

	public function get_last_active_author_id($object, $field_name, $request) {

		$last_id = get_post_meta($object["id"], "_bbp_last_active_id", true);

		$post = get_post($last_id);

		if ($post) {
			return $post->post_author;
		}

		return 0;

	}

	public function _bbp_last_active_time($object, $field_name, $request) {
		$last_active = get_post_meta($object["id"], '_bbp_last_active_time', true);
		if (empty($last_active)) {
			$reply_id = bbp_get_forum_last_reply_id($object["id"]);
			if (!empty($reply_id)) {
				$last_active = get_post_field('post_date', $reply_id);
			} else {
				$topic_id = bbp_get_forum_last_topic_id($object["id"]);
				if (!empty($topic_id)) {
					$last_active = bbp_get_topic_last_active_time($topic_id);
				} else {
					$last_active = get_post_field('post_date', $object["id"]);
				}
			}
		}

		return get_gmt_from_date($last_active);
	}

	public function _bbp_last_active_time_topic($object, $field_name, $request) {
		$last_active = get_post_meta($object["id"], '_bbp_last_active_time', true);
		if (empty($last_active)) {
			$reply_id = bbp_get_topic_last_reply_id($object["id"]);
			if (!empty($reply_id)) {
				$last_active = get_post_field('post_date', $reply_id);
			} else {
				$last_active = get_post_field('post_date', $object["id"]);
			}
		}

		return get_gmt_from_date($last_active);
	}

	public function _bbp_is_closed($object, $field_name, $request) {
		return bbp_is_forum_closed($object["id"]);
	}

	public function _can_access_content($form_id, $create = false) {
		if (function_exists('bbp_is_forum_group_forum') && bbp_is_forum_group_forum($form_id)) {
			$group_ids = bbp_get_forum_group_ids($form_id);
			$user_id = get_current_user_id();
			$flag = false;
			if (!empty($group_ids)) {
				foreach ($group_ids as $group_id) {

					if (function_exists('bp_is_active') && bp_is_active('groups') ) {

					    //if we checking access for showing forums, topic and replies then we need to check group is public or not.
					    if ( !$create ) {
                            $group = groups_get_group($group_id);
                            if ($group->status == "public") {
                                $flag = true;
                            }
                        }

                        if (groups_is_user_member($user_id, $group_id) ||
                            groups_is_user_mod($user_id, $group_id) ||
                            groups_is_user_admin($user_id, $group_id) ||
                            bbp_is_user_keymaster($user_id)
                        ) {
                            $flag = true;
                        }
                    }
                }
			}
			return $flag;
		} else {
			if (empty($form_id) || !bbp_user_can_view_forum(array(
				'forum_id' => $form_id,
			))) {
				return false;
			}
			return true;
		}
	}

	public function _bbp_topic_tags($object, $field_name, $request) {

		$topic_id = $object["id"];

		// Topic exists
		if (!empty($topic_id)) {

			// Topic is spammed so display pre-spam terms
			if (bbp_is_topic_spam($topic_id)) {

				// Get pre-spam terms
				$new_terms = get_post_meta($topic_id, '_bbp_spam_topic_tags', true);

				// If terms exist, explode them and compile the return value
				if (empty($new_terms)) {
					$new_terms = '';
				}

				// Topic is not spam so get real terms
			} else {
				$terms = array_filter((array) get_the_terms($topic_id, bbp_get_topic_tag_tax_id()));

				// Loop through them
				foreach ($terms as $term) {
					$new_terms[] = $term->name;
				}
			}

			// Define local variable(s)
		} else {
			$new_terms = '';
		}

		// Set the return value
		$topic_tags = (!empty($new_terms)) ? implode(', ', $new_terms) : '';

		return apply_filters('bbp_get_form_topic_tags', esc_attr($topic_tags));
	}

	public function _bbp_topic_action_states($object, $field_name, $request) {

		$topic_id = (int) $object["id"];
		$user_id = bbp_get_user_id(0, true, true);

		$state = array(
			'subscribed' => '-1',
			'favorited' => '-1',
			'open' => bbp_is_topic_open($topic_id),
			'sticky' => bbp_is_topic_sticky($topic_id),
			'super_sticky' => bbp_is_topic_super_sticky($topic_id),
			'spam' => bbp_is_topic_spam($topic_id),
			'trash' => bbp_is_topic_trash($topic_id),
		);

		if (bbp_is_favorites_active() && current_user_can('edit_user', $user_id)) {
			$state['favorited'] = bbp_is_user_favorite($user_id, $topic_id);
		}
		if (bbp_is_subscriptions_active() && current_user_can('edit_user', $user_id)) {
			$state['subscribed'] = bbp_is_user_subscribed($user_id, $topic_id);
		}

		return $state;

	}

	public function _bbp_forum_action_states($object, $field_name, $request) {

		$forum_id = (int) $object["id"];
		$user_id = bbp_get_user_id(0, true, true);

		$state = array(
			'subscribed' => false,
		);

		if (bbp_is_subscriptions_active() && current_user_can('edit_user', $user_id)) {
			$state['subscribed'] = bbp_is_user_subscribed($user_id, $forum_id);
		}

		return $state;

	}

	public function _bbp_reply_action_states($object, $field_name, $request) {

		$reply_id = (int) $object["id"];
		$user_id = bbp_get_user_id(0, true, true);

		$state = array(
			'spam' => bbp_is_reply_spam($reply_id),
			'trash' => bbp_is_reply_trash($reply_id),
		);

		return $state;

	}

	public function _bbp_reply_count($object, $field_name, $request) {
		return (int) get_post_meta($object["id"], "_bbp_reply_count", true);
	}

	public function _bbp_voice_count($object, $field_name, $request) {
		return (int) get_post_meta($object["id"], "_bbp_voice_count", true);
	}

	public function _bbp_total_topic_count($object, $field_name, $request) {
		return (int) get_post_meta($object["id"], "_bbp_total_topic_count", true);
	}

	public function _bbp_total_reply_count($object, $field_name, $request) {
		return (int) get_post_meta($object["id"], "_bbp_total_reply_count", true);
	}

	public function _bbp_last_topic_id($object, $field_name, $request) {
		return (int) get_post_meta($object["id"], "_bbp_last_topic_id", true);
	}

	public function _bbp_forum_id($object, $field_name, $request) {
		return (int) get_post_meta($object["id"], "_bbp_forum_id", true);
	}

	public function _bbp_is_topic_anonymous($object, $field_name, $request) {
		return bbp_is_topic_anonymous($object["id"]);
	}

	public function bbp_add_anonymous_author_data($object, $field_name, $request) {
		if (!bbp_is_topic_anonymous($object['id'])) {
			return false;
		}
		$data = array(
			'name' => bbp_get_topic_author_display_name($object['id']),
			'email' => bbp_get_topic_author_email($object['id']),
			'website' => bbp_get_topic_author_url($object['id']),
			'avatar' => get_avatar_url(bbp_get_topic_author_email($object['id'])),
		);
		return $data;
	}

	public function _bbp_is_reply_anonymous($object, $field_name, $request) {
		return bbp_is_reply_anonymous($object["id"]);
	}

	public function bbp_add_reply_anonymous_author_data($object, $field_name, $request) {
		if (!bbp_is_reply_anonymous($object['id'])) {
			return false;
		}
		$data = array(
			'name' => bbp_get_reply_author_display_name($object['id']),
			'email' => bbp_get_reply_author_email($object['id']),
			'website' => bbp_get_reply_author_url($object['id']),
			'avatar' => get_avatar_url(bbp_get_reply_author_email($object['id'])),
		);
		return $data;
	}

	public function _bbp_get_topic_classes($object, $field_name, $request) {
		return bbp_get_topic_class($object["id"]);
	}

	public function _bbp_get_topic_topic_status($object, $field_name, $request) {
		return bbp_get_topic_status($object["id"]);
	}

	public function _bbp_get_forum_current_user_permissions($object, $field_name, $request) {
		$forum = bbp_get_forum(bbp_get_forum_id((int) $object["id"]));

		return array(
			"show_topics" => $this->_can_access_content($object["id"]),
			"show_subforum" => $this->_can_access_content($object["id"]),
			"topic" => (!empty($forum)) && bbp_current_user_can_publish_topics() && $this->_can_access_content($object["id"], true),
		);
	}

	public function _bbp_get_topic_current_user_permissions($object, $field_name, $request) {
		$topic = bbp_get_topic(bbp_get_topic_id((int) $object["id"]));
		$form_id = get_post_meta($object["id"], '_bbp_forum_id', true);

		return array(
			"show_replies" => $this->_can_access_content($form_id),
			"edit" => (
				current_user_can('moderate') ||
				(
					!empty($topic) &&
					current_user_can('edit_topic', $topic->ID) &&
					!bbp_past_edit_lock($topic->post_date_gmt)
				)
			),
			"moderate" => !empty($topic) && current_user_can('moderate', $object["id"]),
			"reply" => !empty($topic) && bbp_current_user_can_publish_replies() && $this->_can_access_content($form_id, true),
			"trash" => !empty($topic) && current_user_can('delete_topic', $topic->ID),
		);
	}

	public function _bbp_get_reply_current_user_permissions($object, $field_name, $request) {

		$reply = bbp_get_reply(bbp_get_reply_id((int) $object["id"]));

		return array(
			"edit" => (
				current_user_can('edit_others_replies') ||
				(
					!empty($reply) &&
					current_user_can('edit_reply', $reply->ID) &&
					!bbp_past_edit_lock($reply->post_date_gmt)
				)
			),
			"move" => !empty($reply) && current_user_can('moderate', $reply->ID),
			"split" => !empty($reply) && current_user_can('moderate', $reply->ID),
			"spam" => !empty($reply) && current_user_can('moderate', $reply->ID),
			"trash" => !empty($reply) && current_user_can('delete_reply', $reply->ID),
		);

	}

	public function _bbp_last_reply_id($object, $field_name, $request) {
		return (int) get_post_meta($object["id"], "_bbp_last_reply_id", true);
	}

	public function get_author($object, $field_name, $request) {
		$p = get_post($object["id"]);

		return $p->post_author;
	}

	public function get_nicename($object, $field_name, $request) {
		$p = get_userdata($object["id"]);

		return $p->user_nicename;
	}

	public function get_bbp_user_profile_url($object, $field_name, $request) {
		if (bbp_user_has_profile($object["id"])) {
			return bbp_get_user_profile_url($object["id"]);
		}
		return '';
	}

	public function _bbp_reply_to($object, $field_name, $request) {
		return (int) get_post_meta($object["id"], "_bbp_reply_to", true);
	}

	public function get_short_content($object, $field_name, $request) {
		$p = get_post($object["id"]);

		return wp_trim_excerpt($p->post_content);
	}

	public function allow_member_private_forum($retval, $forum_ids, $type) {
		if ( function_exists( "groups_get_groups" ) && ! is_admin() && ( strpos( $_SERVER['REQUEST_URI'], '/wp-json' ) !== false ) ) {
			$user_groups = groups_get_groups(array(
				'type' => 'alphabetical',
				'per_page' => -1,
				'show_hidden' => true,
				'status' => array('private', 'hidden'),
			));
			foreach ($user_groups['groups'] as $group) {
			    if ( $group->enable_forum ) {
                    $forums = groups_get_groupmeta($group->id, 'forum_id');
                    if (!empty($forums[0])) {
                        if (!$group->__get("is_member")) {
                            $forum_ids[] = $forums[0];
                        } else {
                            if (($key = array_search($forums[0], $forum_ids)) !== false) {
                                unset($forum_ids[$key]);
                            }
                        }
                    }
                }
			}
		}

		if (!empty($forum_ids ) && is_admin()) {

			switch ($type) {
			case 'string':
				$retval = implode(',', $forum_ids);
				break;

			// Use forum_ids array
			case 'array':
				$retval = $forum_ids;
				break;

			// Build a meta_query
			case 'meta_query':
				$retval = array(
					'key' => '_bbp_forum_id',
					'value' => implode(',', $forum_ids),
					'type' => 'numeric',
					'compare' => (1 < count($forum_ids)) ? 'NOT IN' : '!=',
				);
				break;
			}
		}
		return $retval;
	}

	/**
	 * Add a require search tab into app.
	 * @param $tabs
	 * @return array
	 */
	public function add_search_tabs($tabs) {
		$bbp_tabs = array(
			array(
				'id' => 'forums',
				'title' => __('Forums', 'buddyboss-app'),
			),
			array(
				'id' => 'topics',
				'title' => __('Topics', 'buddyboss-app'),
			),
		);
		return array_merge($tabs, $bbp_tabs);
	}
}