<?php
namespace BuddyBossApp\Api\BbPress;
use BuddyBossApp\Api\BbPress\PostController;
use stdClass;
use WP_Error;
use WP_REST_Request;
use WP_REST_Server;

// NOTE : Old classname was class.boss_bbpress_api_reply_rest By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for bbpress.
class ReplyRest extends PostController {

	protected $namespace_slug = '';
	protected $namespace = '/bbpress/v1';

	public function __construct($slug) {

		parent::__construct($slug, bbp_get_reply_post_type());

		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->hooks();
	}

	public function hooks() {

		add_action('rest_api_init', array($this, "register_routes"), 99);
		add_filter('boss_bp_json_pre_insert_reply', array($this, 'set_group'), 10, 2);
	}

	public function register_routes() {

		parent::register_routes();

		register_rest_route($this->namespace, '/reply/action/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'do_action'),
				'permission_callback' => '__return_true',
				'args' => array(
					'spam' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'trash' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'delete' => array(
						'type' => 'integer',
						'enum' => array("1"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/reply', array(
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'create'),
				'permission_callback' => '__return_true',
				'args' => $this->get_create_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/reply/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::DELETABLE,
				'callback' => array($this, 'delete'),
				'permission_callback' => '__return_true',
			),
		));

		register_rest_route($this->namespace, '/reply/edit/', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'update'),
				'permission_callback' => '__return_true',
				'args' => $this->get_create_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/reply/move/', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'move'),
				'permission_callback' => '__return_true',
				'args' => array(
					'bbp_reply_id' => array(
						'description' => __('Id of reply'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
					'bbp_destination_topic' => array(
						'description' => __('Id of destination reply'),
						'type' => 'integer',
						'sanitize_callback' => 'absint',
					),
					'bbp_reply_move_option' => array(
						'description' => __('option of move reply'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'bbp_reply_move_destination_title' => array(
						'description' => __('Title of destination reply'),
						'type' => 'string',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

	}

	public function set_group($reply, $request) {
		global $bp;

		$topic_id = filter_var($reply->bbp_topic_id, FILTER_VALIDATE_INT);
		$forum_id = bbp_get_topic_forum_id($topic_id);

		if ($forum_id && function_exists('bbp_get_forum_group_ids')) {
			$group_ids = bbp_get_forum_group_ids($forum_id);
			$group_id = reset($group_ids);
			if ($group_id) {
				$bp->groups->current_group = groups_get_group($group_id);
			}
		}

		return $reply;
	}

	/**
	 * @api {PATCH} /wp-json/appboss/bbpress/v1/reply/action/:id Reply action
	 * @apiName ReplyAction
	 * @apiGroup BBPressReply
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update action of reply section under BBPress's component
	 * @apiDeprecated  Actions on Reply. Check (#Forum Replies:ActionBBPReply)
	 * @apiUse apidocForReplyActionV1
	 * @apiPrivate
	 */
	public function do_action($request) {

		global $wp_rest_server;

		$reply_id = (int) $request['id'];
		$success = true;

		if (!is_user_logged_in()) {
			return new WP_Error('rest_cannot_view', __("You don't have permission.", 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (!$reply_id || !is_int($reply_id)) {
			return new WP_Error('bp_json_reply_invalid_id', __('Reply ID is invalid.', 'buddyboss-app'), array('status' => 400));
		}

		// Make sure reply exists
		$reply = bbp_get_reply($reply_id);
		if (empty($reply)) {
			return new WP_Error('bp_json_reply_doesnt_exists', __('Reply doesn\'t exist.', 'buddyboss-app'), array('status' => 404));
		}

		// What is the user doing here?
		if ((isset($request['spam']) && !current_user_can('edit_reply', $reply->ID)) ||
			((isset($request['trash']) || isset($request['delete'])) && !current_user_can('delete_reply', $reply->ID))) {
			return new WP_Error('bp_json_reply_no_permission', __('You do not have the permission to do that.', 'buddyboss-app'), array('status' => 403));
		}

		//Spam reply
		if (isset($request['spam']) && $request['spam'] == 1) {
			$is_spam = bbp_is_reply_spam($reply_id);
			if (!$is_spam) {
				$success = bbp_spam_reply($reply_id);
			}
		} elseif (isset($request['spam']) && $request['spam'] == 0) {
			$is_spam = bbp_is_reply_spam($reply_id);
			if ($is_spam) {
				$success = bbp_unspam_reply($reply_id);
			}
		}

		//Trash reply
		if (isset($request['trash']) && $request['trash'] == 1) {
			$success = wp_trash_post($reply_id);
		} else if (isset($request['trash']) && $request['trash'] == 0) {
			$success = wp_untrash_post($reply_id);
		}

		//delete reply
		if (isset($request['delete']) && $request['delete'] == 1) {
			$success = wp_delete_post($reply_id, true);
		}

		// Handle errors
		if (false == $success || is_wp_error($success)) {
			return new WP_Error('bp_json_reply_action_failure', __('Action failed.', 'buddyboss-app'), array('status' => 500));
		}

		/**
		 * Redirect to reply endpoint.
		 */
		$_GET['view'] = 'all';
		$request_reply = new WP_REST_Request("GET", "/" . $this->namespace_slug . "/bbpress/v1/reply/{$reply_id}");
		$server = rest_get_server();
		$response = $server->dispatch($request_reply);

		return $wp_rest_server->response_to_data(rest_ensure_response($response), isset($request['_embed']) ? 1 : 0);
	}

	/**
	 * @api {DELETE} /wp-json/appboss/bbpress/v1/reply/:id Delete reply
	 * @apiName DeleteBBPressReply
	 * @apiGroup BBPressReply
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Delete reply of/for BBPress's component
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {Number} id Reply ID
	 * @apiDeprecated  Trash OR Delete a Reply. Check (#Forum Replies:DeleteBBPReply)
	 * @apiPrivate
	 */
	public function delete($request) {

		$reply_id = (int) $request['id'];

		if (!is_user_logged_in()) {
			return new WP_Error('rest_cannot_view', __("You don't have permission.", 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (!$reply_id || !is_int($reply_id)) {
			return new WP_Error('bp_json_reply_invalid_id', __('Reply ID is invalid.', 'buddyboss-app'), array('status' => 400));
		}

		// Make sure reply exists
		$reply = bbp_get_reply($reply_id);
		if (empty($reply)) {
			return new WP_Error('bp_json_reply_doesnt_exists', __('Reply doesn\'t exist.', 'buddyboss-app'), array('status' => 404));
		}

		// What is the user doing here?
		if (!current_user_can('edit_reply', $reply->ID) || (!current_user_can('delete_reply', $reply->ID))) {
			return new WP_Error('bp_json_reply_no_permission', __('You do not have the permission to do that.', 'buddyboss-app'), array('status' => 403));
		}

		//delete reply
		$success = wp_delete_post($reply_id);

		// Handle errors
		if (false == $success || is_wp_error($success)) {
			return new WP_Error('bp_json_reply_action_failure', __('Action failed.', 'buddyboss-app'), array('status' => 500));
		}

		return true;
	}

	/**
	 * @api {POST} /wp-json/appboss/bbpress/v1/reply Create reply
	 * @apiName CreateBBPressReply
	 * @apiGroup BBPressReply
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Create reply under BBPress's component
	 * @apiDeprecated  Create a reply. Check (#Forum Replies:CreateBBPReply)
	 * @apiUse apidocForBBPressReplyV1
	 * @apiPrivate
	 */
	public function create($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		global $wp_rest_server;

		$reply = $this->prepare_reply_for_database($request);

		// Define local variable(s)
		$topic_id = $forum_id = $reply_author = $anonymous_data = $reply_to = 0;
		$reply_title = $reply_content = $terms = '';

		/** Reply Author **********************************************************/

		// User is anonymous
		if (bbp_is_anonymous()) {

			$bbp_anonymous_name = !empty($request['bbp_anonymous_name']) ? $request['bbp_anonymous_name'] : '';
			$bbp_anonymous_email = !empty($request['bbp_anonymous_email']) ? $request['bbp_anonymous_email'] : '';
			$bbp_anonymous_website = !empty($request['bbp_anonymous_website']) ? $request['bbp_anonymous_website'] : '';

			$a_data = array(
				'bbp_anonymous_name' => $bbp_anonymous_name,
				'bbp_anonymous_email' => $bbp_anonymous_email,
				'bbp_anonymous_website' => $bbp_anonymous_website,
			);

			// Filter anonymous data
			$anonymous_data = bbp_filter_anonymous_post_data($a_data);

			// Anonymous data checks out, so set cookies, etc...
			if (!empty($anonymous_data) && is_array($anonymous_data)) {
				bbp_set_current_anonymous_user_data($anonymous_data);
			}

			// User is logged in
		} else {

			// User cannot create replies
			if (!bbp_current_user_can_publish_replies()) {
				return new WP_Error('bp_json_reply_permissions', __('You do not have permission to reply.', 'buddyboss-app'), array('status' => 403));
			}

			// Reply author is current user
			$reply_author = bbp_get_current_user_id();

		}

		/** Topic ID **************************************************************/

		// Topic id was not passed
		if (empty($reply->bbp_topic_id)) {
			return new WP_Error('bp_json_reply_topic_id', __('Topic ID is missing.', 'buddyboss-app'), array('status' => 400));
			// Topic id is not a number
		} elseif (!is_numeric($reply->bbp_topic_id)) {
			return new WP_Error('bp_json_reply_topic_id', __('Topic ID must be a number.', 'buddyboss-app'), array('status' => 400));

			// Topic id might be valid
		} else {

			// Get the topic id
			$posted_topic_id = intval($reply->bbp_topic_id);

			// Topic id is a negative number
			if (0 > $posted_topic_id) {
				return new WP_Error('bp_json_reply_topic_id', __('Topic ID cannot be a negative number.', 'buddyboss-app'), array('status' => 400));

				// Topic does not exist
			} elseif (!bbp_get_topic($posted_topic_id)) {
				return new WP_Error('bp_json_reply_topic_id', __('Topic does not exist.', 'buddyboss-app'), array('status' => 400));

				// Use the POST'ed topic id
			} else {
				$topic_id = $posted_topic_id;
			}
		}

		/** Forum ID **************************************************************/

		// Try to use the forum id of the topic
		if (!isset($reply->bbp_forum_id) && !empty($topic_id)) {
			$forum_id = bbp_get_topic_forum_id($topic_id);

			// Error check the POST'ed forum id
		} elseif ($reply->bbp_forum_id) {

			// Empty Forum id was passed
			if (empty($reply->bbp_forum_id)) {
				return new WP_Error('bp_json_reply_forum_id', __('Forum ID is missing.', 'buddyboss-app'), array('status' => 400));

				// Forum id is not a number
			} elseif (!is_numeric($reply->bbp_forum_id)) {
				return new WP_Error('bp_json_reply_forum_id', __('Forum ID must be a number.', 'buddyboss-app'), array('status' => 400));

				// Forum id might be valid
			} else {

				// Get the forum id
				$posted_forum_id = intval($reply->bbp_forum_id);

				// Forum id is empty
				if (0 === $posted_forum_id) {
					return new WP_Error('bp_json_topic_forum_id', __('Forum ID is missing.', 'buddyboss-app'), array('status' => 400));

					// Forum id is a negative number
				} elseif (0 > $posted_forum_id) {
					return new WP_Error('bp_json_topic_forum_id', __('Forum ID cannot be a negative number.', 'buddyboss-app'), array('status' => 400));

					// Forum does not exist
				} elseif (!bbp_get_forum($posted_forum_id)) {
					return new WP_Error('bp_json_topic_forum_id', __('Forum does not exist.', 'buddyboss-app'), array('status' => 400));

					// Use the POST'ed forum id
				} else {
					$forum_id = $posted_forum_id;
				}
			}
		}

		// Forum exists
		if (!empty($forum_id)) {

			// Forum is a category
			if (bbp_is_forum_category($forum_id)) {
				return new WP_Error('bp_json_reply_forum_category', __('This forum is a category. No replies can be created in this forum.', 'buddyboss-app'), array('status' => 403));
				// Forum is not a category
			} else {

				// Forum is closed and user cannot access
				if (bbp_is_forum_closed($forum_id) && !current_user_can('edit_forum', $forum_id)) {
					return new WP_Error('bp_json_reply_forum_closed', __('This forum has been closed to new replies.', 'buddyboss-app'), array('status' => 403));
				}

				/**
				 * Added logic for group forum
				 * Current user is part of that group or not.
				 * We need to check manually because bbpress updating that caps only on group forum page and
				 * in API those conditional tag will not work.
				 */
				if( function_exists('bbp_get_forum_group_ids') ) {
					$group_ids = bbp_get_forum_group_ids( $forum_id );
					if ( ! empty( $group_ids ) ) {
						$is_member = false;
						foreach ( $group_ids as $group_id ) {
							if ( groups_is_user_member( get_current_user_id(), $group_id ) ) {
								$is_member = true;
								break;
							}
						}
					}
				}

				// Forum is private and user cannot access
				if (bbp_is_forum_private($forum_id) && (!bbp_is_user_keymaster())) {
					if ((empty($group_ids) && !current_user_can('read_private_forums')) || (!empty($group_ids) && !$is_member)) {
						return new WP_Error('bp_json_reply_forum_private', __('This forum is private and you do not have the capability to read or create new topics in it.', 'buddyboss-app'), array('status' => 403));
					}

					// Forum is hidden and user cannot access
				} elseif (bbp_is_forum_hidden($forum_id) && (!bbp_is_user_keymaster())) {
					if ((empty($group_ids) && !current_user_can('read_hidden_forums')) || (!empty($group_ids) && !$is_member)) {
						return new WP_Error('bbp_new_reply_forum_hidden', __('This forum is hidden and you do not have the capability to read or create new topics in it.', 'buddyboss-app'), array('status' => 403));
					}
				}
			}
		}

		/** Unfiltered HTML *******************************************************/

		// Remove kses filters from title and content for capable users and if the nonce is verified
		if (current_user_can('unfiltered_html')) {
			remove_filter('bbp_new_reply_pre_title', 'wp_filter_kses');
			remove_filter('bbp_new_reply_pre_content', 'bbp_encode_bad', 10);
			remove_filter('bbp_new_reply_pre_content', 'bbp_filter_kses', 30);
		}

		/** Reply Title ***********************************************************/

		if (!empty($reply->bbp_reply_title)) {
			$reply_title = esc_attr(strip_tags($reply->bbp_reply_title));
		}

		// Filter and sanitize
		$reply_title = apply_filters('bbp_new_reply_pre_title', $reply_title);

		/** Reply Content *********************************************************/

		if (!empty($reply->bbp_reply_content)) {
			$reply_content = $reply->bbp_reply_content;
		}

		// Filter and sanitize
		$reply_content = apply_filters('bbp_new_reply_pre_content', $reply_content);

		// No reply content
		if (empty($reply_content)) {
			return new WP_Error('bp_json_reply_content', __('Your reply cannot be empty.', 'buddyboss-app'), array('status' => 400));
		}

		/** Reply Flooding ********************************************************/

		if (!bbp_check_for_flood($anonymous_data, $reply_author)) {
			return new WP_Error('bp_json_reply_flood', __('Slow down; you move too fast.', 'buddyboss-app'), array('status' => 400));
		}

		/** Reply Duplicate *******************************************************/

		if (!bbp_check_for_duplicate(array('post_type' => bbp_get_reply_post_type(),
			'post_author' => $reply_author,
			'post_content' => $reply_content,
			'post_parent' => $topic_id,
			'anonymous_data' => $anonymous_data,
		))) {
			return new WP_Error('bp_json_reply_duplicate', __('Duplicate reply detected; it looks as though you&#8217;ve already said that!', 'buddyboss-app'), array('status' => 400));
		}

		/** Reply Blacklist *******************************************************/

		if (!bbp_check_for_blacklist($anonymous_data, $reply_author, $reply_title, $reply_content)) {
			return new WP_Error('bp_json_reply_blacklist', __('Your reply cannot be created at this time.', 'buddyboss-app'), array('status' => 400));
		}

		/** Reply Status **********************************************************/

		// Maybe put into moderation
		if (!bbp_check_for_moderation($anonymous_data, $reply_author, $reply_title, $reply_content)) {
			$reply_status = bbp_get_pending_status_id();

			// Default
		} else {
			$reply_status = bbp_get_public_status_id();
		}

		/** Reply To **************************************************************/

		// Handle Reply To of the reply; $_REQUEST for non-JS submissions
		if (isset($reply->bbp_reply_to)) {
			$reply_to = bbp_validate_reply_to($reply->bbp_reply_to);
		}

		/** Topic Closed **********************************************************/

		// If topic is closed, moderators can still reply
		if (bbp_is_topic_closed($topic_id) && !current_user_can('moderate')) {
			return new WP_Error('bp_json_reply_blacklist', __('Topic is closed.', 'buddyboss-app'), array('status' => 400));
		}

		/** Topic Tags ************************************************************/

		// Either replace terms
		if (bbp_allow_topic_tags() && current_user_can('assign_topic_tags') && !empty($reply->bbp_topic_tags)) {
			$terms = esc_attr(strip_tags($reply->bbp_topic_tags));

			// ...or remove them.
		} elseif (isset($reply->bbp_topic_tags)) {
			$terms = '';

			// Existing terms
		} else {
			$terms = bbp_get_topic_tag_names($topic_id);
		}

		/** Additional Actions (Before Save) **************************************/

		do_action('bbp_new_reply_pre_extras', $topic_id, $forum_id);

		// Bail if errors
		if (bbp_has_errors()) {
			return;
		}

		/** No Errors *************************************************************/

		// Add the content of the form to $reply_data as an array
		// Just in time manipulation of reply data before being created
		$reply_data = apply_filters('bbp_new_reply_pre_insert', array(
			'post_author' => $reply_author,
			'post_title' => $reply_title,
			'post_content' => $reply_content,
			'post_status' => $reply_status,
			'post_parent' => $topic_id,
			'post_type' => bbp_get_reply_post_type(),
			'comment_status' => 'closed',
			'menu_order' => bbp_get_topic_reply_count($topic_id, false) + 1,
		));

		// Insert reply
		$reply_id = wp_insert_post($reply_data);

		/** No Errors *************************************************************/

		// Check for missing reply_id or error
		if (!empty($reply_id) && !is_wp_error($reply_id)) {

			/** Topic Tags ********************************************************/

			// Just in time manipulation of reply terms before being edited
			$terms = apply_filters('bbp_new_reply_pre_set_terms', $terms, $topic_id, $reply_id);

			// Insert terms
			$terms = wp_set_post_terms($topic_id, $terms, bbp_get_topic_tag_tax_id(), false);

			// Term error
			if (is_wp_error($terms)) {
				return new WP_Error('bp_json_reply_tags', __('There was a problem adding the tags to the topic.', 'buddyboss-app'), array('status' => 400));
			}

			/** Trash Check *******************************************************/

			// If this reply starts as trash, add it to pre_trashed_replies
			// for the topic, so it is properly restored.
			if (bbp_is_topic_trash($topic_id) || ($reply_data['post_status'] === bbp_get_trash_status_id())) {

				// Trash the reply
				wp_trash_post($reply_id);

				// Only add to pre-trashed array if topic is trashed
				if (bbp_is_topic_trash($topic_id)) {

					// Get pre_trashed_replies for topic
					$pre_trashed_replies = get_post_meta($topic_id, '_bbp_pre_trashed_replies', true);

					// Add this reply to the end of the existing replies
					$pre_trashed_replies[] = $reply_id;

					// Update the pre_trashed_reply post meta
					update_post_meta($topic_id, '_bbp_pre_trashed_replies', $pre_trashed_replies);
				}

				/** Spam Check ********************************************************/

				// If reply or topic are spam, officially spam this reply
			} elseif (bbp_is_topic_spam($topic_id) || ($reply_data['post_status'] === bbp_get_spam_status_id())) {
				add_post_meta($reply_id, '_bbp_spam_meta_status', bbp_get_public_status_id());

				// Only add to pre-spammed array if topic is spam
				if (bbp_is_topic_spam($topic_id)) {

					// Get pre_spammed_replies for topic
					$pre_spammed_replies = get_post_meta($topic_id, '_bbp_pre_spammed_replies', true);

					// Add this reply to the end of the existing replies
					$pre_spammed_replies[] = $reply_id;

					// Update the pre_spammed_replies post meta
					update_post_meta($topic_id, '_bbp_pre_spammed_replies', $pre_spammed_replies);
				}
			}

			/** Update counts, etc... *********************************************/

			do_action('bbp_new_reply', $reply_id, $topic_id, $forum_id, $anonymous_data, $reply_author, false, $reply_to);

			/** Additional Actions (After Save) ***********************************/

			do_action('bbp_new_reply_post_extras', $reply_id);

			/**
			 * Handle Subscription.
			 */
			if (bbp_is_subscriptions_active() && !empty($reply_author) && !empty($topic_id)) {
				$subscribed = bbp_is_user_subscribed($reply_author, $topic_id);
				$subscheck = $reply->bbp_topic_subscription;

				// Subscribed and unsubscribing
				if (true === $subscribed && false === $subscheck) {
					bbp_remove_user_subscription($reply_author, $topic_id);
				} elseif (false === $subscribed && true === $subscheck) {
					bbp_add_user_subscription($reply_author, $topic_id);
				}
			}

			/**
			 * Rest Hook
			 */
			do_action('boss_rest_bbpress_create_reply', $reply_id, $request);

			/**
			 * Redirect to topic endpoint.
			 */
			$request_reply = new WP_REST_Request("GET", "/" . $this->namespace_slug . "/bbpress/v1/reply/{$reply_id}");
			$server = rest_get_server();
			$response = $server->dispatch($request_reply);

			return $wp_rest_server->response_to_data(rest_ensure_response($response), isset($request['_embed']) ? 1 : 0);

			/** Errors ****************************************************************/

		} else {
			$append_error = (is_wp_error($reply_id) && $reply_id->get_error_message()) ? $reply_id->get_error_message() . ' ' : '';

			return new WP_Error('bp_json_reply_error', __('The following problem(s) have been found with your reply:' . $append_error, 'buddyboss-app'), array('status' => 400));
		}
	}

	/**
	 * @api {PATCH} /wp-json/appboss/bbpress/v1/reply/edit Update reply
	 * @apiName UpdateBBPressReply
	 * @apiGroup BBPressReply
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update BBPress's Reply
	 * @apiUse apidocForBBPressReplyV1
	 * @apiParam {Number} id Reply ID (also used as reply_id)
	 * @apiDeprecated  Update a reply. Check (#Forum Replies:UpdateBBPReply)
	 * @apiPrivate
	 */
	public function update($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		global $wp_rest_server;

		$reply_new = $this->prepare_reply_for_database($request);

		// Define local variable(s)
		$revisions_removed = false;
		$reply = $reply_id = $reply_author = $topic_id = $forum_id = $anonymous_data = 0;
		$reply_title = $reply_content = $reply_edit_reason = $terms = '';

		/** Reply *****************************************************************/

		// Reply id was not passed
		if (empty($reply_new->bbp_reply_id)) {
			return new WP_Error('bp_json_edit_reply_id', __('Reply ID not found.', 'buddyboss-app'), array('status' => 404));

			// Reply id was passed
		} elseif (is_numeric($reply_new->bbp_reply_id)) {
			$reply_id = (int) $reply_new->bbp_reply_id;
			$reply = bbp_get_reply($reply_id);
		}

		// Reply does not exist
		if (empty($reply)) {
			return new WP_Error('bp_json_reply_not_found', __('The reply you want to edit was not found.', 'buddyboss-app'), array('status' => 404));

			// Reply exists
		} else {

			if (!current_user_can('edit_others_replies')) {

				// User cannot edit or it is past the lock time
				if (empty($reply) || !current_user_can('edit_reply', $reply->ID) || bbp_past_edit_lock($reply->post_date_gmt)) {
					return new WP_Error('bp_json_reply_permission', __('You do not have permission to edit that reply.', 'buddyboss-app'), array('status' => 403));
				}
			}

			// Check users ability to create new reply
			if (!bbp_is_reply_anonymous($reply_id)) {

				// User cannot edit this reply
				if (!current_user_can('edit_reply', $reply_id)) {
					return new WP_Error('bp_json_reply_permission', __('You do not have permission to edit that reply.', 'buddyboss-app'), array('status' => 403));
				}

				// Set reply author
				$reply_author = bbp_get_reply_author_id($reply_id);

				// It is an anonymous post
			} else {

				$bbp_anonymous_name = !empty($request['bbp_anonymous_name']) ? $request['bbp_anonymous_name'] : '';
				$bbp_anonymous_email = !empty($request['bbp_anonymous_email']) ? $request['bbp_anonymous_email'] : '';
				$bbp_anonymous_website = !empty($request['bbp_anonymous_website']) ? $request['bbp_anonymous_website'] : '';

				$a_data = array(
					'bbp_anonymous_name' => $bbp_anonymous_name,
					'bbp_anonymous_email' => $bbp_anonymous_email,
					'bbp_anonymous_website' => $bbp_anonymous_website,
				);

				// Filter anonymous data
				$anonymous_data = bbp_filter_anonymous_post_data($a_data);
			}
		}

		// Remove kses filters from title and content for capable users and if the nonce is verified
		if (current_user_can('unfiltered_html')) {
			remove_filter('bbp_edit_reply_pre_title', 'wp_filter_kses');
			remove_filter('bbp_edit_reply_pre_content', 'bbp_encode_bad', 10);
			remove_filter('bbp_edit_reply_pre_content', 'bbp_filter_kses', 30);
		}

		/** Reply Topic ***********************************************************/

		$topic_id = bbp_get_reply_topic_id($reply_id);

		/** Topic Forum ***********************************************************/

		$forum_id = bbp_get_topic_forum_id($topic_id);

		// Forum exists
		if (!empty($forum_id) && ($forum_id !== bbp_get_reply_forum_id($reply_id))) {

			// Forum is a category
			if (bbp_is_forum_category($forum_id)) {
				return new WP_Error('bp_json_edit_reply_forum_category', __('This forum is a category. No replies can be updated in this forum.', 'buddyboss-app'), array('status' => 403));

				// Forum is not a category
			} else {

				// Forum is closed and user cannot access
				if (bbp_is_forum_closed($forum_id) && !current_user_can('edit_forum', $forum_id)) {
					return new WP_Error('bp_json_edit_reply_forum_closed', __('This forum has been closed to update replies.', 'buddyboss-app'), array('status' => 403));
				}

				/**
				 * Added logic for group forum
				 * Current user is part of that group or not.
				 * We need to check manually because bbpress updating that caps only on group forum page and
				 * in API those conditional tag will not work.
				 */
				$group_ids = bbp_get_forum_group_ids($forum_id);
				if (!empty($group_ids)) {
					$is_member = false;
					foreach ($group_ids as $group_id) {
						if (groups_is_user_member(get_current_user_id(), $group_id)) {
							$is_member = true;
							break;
						}
					}
				}

				// Forum is private and user cannot access
				if (bbp_is_forum_private($forum_id) && (!bbp_is_user_keymaster())) {
					if ((empty($group_ids) && !current_user_can('read_private_forums')) || (!empty($group_ids) && !$is_member)) {
						return new WP_Error('bp_json_edit_reply_forum_private', __('This forum is private and you do not have the capability to read or update new topics in it.', 'buddyboss-app'), array('status' => 400));

					}

					// Forum is hidden and user cannot access
				} elseif (bbp_is_forum_hidden($forum_id) && (!bbp_is_user_keymaster())) {
					if ((empty($group_ids) && !current_user_can('read_hidden_forums')) || (!empty($group_ids) && !$is_member)) {
						return new WP_Error('bp_json_edit_reply_forum_hidden', __('This forum is hidden and you do not have the capability to read or update new topics in it.', 'buddyboss-app'), array('status' => 400));
					}
				}
			}
		}

		/** Reply Title ***********************************************************/

		if (!empty($reply_new->bbp_reply_title)) {
			$reply_title = esc_attr(strip_tags($reply_new->bbp_reply_title));
		}

		// Filter and sanitize
		$reply_title = apply_filters('bbp_edit_reply_pre_title', $reply_title, $reply_id);

		/** Reply Content *********************************************************/

		if (!empty($reply_new->bbp_reply_content)) {
			$reply_content = $reply_new->bbp_reply_content;
		}

		// Filter and sanitize
		$reply_content = apply_filters('bbp_edit_reply_pre_content', $reply_content, $reply_id);

		// No reply content
		if (empty($reply_content)) {
			return new WP_Error('bp_json_edit_reply_content', __('Your reply cannot be empty.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 400));
		}

		/** Reply Blacklist *******************************************************/

		if (!bbp_check_for_blacklist($anonymous_data, $reply_author, $reply_title, $reply_content)) {
			return new WP_Error('bp_reply_blacklist', __('Your reply cannot be edited at this time.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 400));
		}

		/** Reply Status **********************************************************/

		// Maybe put into moderation
		if (!bbp_check_for_moderation($anonymous_data, $reply_author, $reply_title, $reply_content)) {

			// Set post status to pending if public
			if (bbp_get_public_status_id() === $reply->post_status) {
				$reply_status = bbp_get_pending_status_id();
			}

			// Use existing post_status
		} else {
			$reply_status = $reply->post_status;
		}

		/** Reply To **************************************************************/

		// Handle Reply To of the reply; $_REQUEST for non-JS submissions
		if (isset($reply_new->bbp_reply_to)) {
			$reply_to = bbp_validate_reply_to($reply_new->bbp_reply_to);
		}

		/** Topic Tags ************************************************************/

		// Either replace terms
		if (bbp_allow_topic_tags() && current_user_can('assign_topic_tags') && !empty($reply_new->bbp_topic_tags)) {
			$terms = esc_attr(strip_tags($reply_new->bbp_topic_tags));

			// ...or remove them.
		} elseif (isset($reply_new->bbp_topic_tags)) {
			$terms = '';

			// Existing terms
		} else {
			$terms = bbp_get_topic_tag_names($topic_id);
		}

		/** Additional Actions (Before Save) **************************************/

		do_action('bbp_edit_reply_pre_extras', $reply_id);

		/** No Errors *************************************************************/

		// Add the content of the form to $reply_data as an array
		// Just in time manipulation of reply data before being edited
		$reply_data = apply_filters('bbp_edit_reply_pre_insert', array(
			'ID' => $reply_id,
			'post_title' => $reply_title,
			'post_content' => $reply_content,
			'post_status' => $reply_status,
			'post_parent' => $topic_id,
			'post_author' => $reply_author,
			'post_type' => bbp_get_reply_post_type(),
		));

		// Toggle revisions to avoid duplicates
		if (post_type_supports(bbp_get_reply_post_type(), 'revisions')) {
			$revisions_removed = true;
			remove_post_type_support(bbp_get_reply_post_type(), 'revisions');
		}

		// Insert topic
		$reply_id = wp_update_post($reply_data);

		// Toggle revisions back on
		if (true === $revisions_removed) {
			$revisions_removed = false;
			add_post_type_support(bbp_get_reply_post_type(), 'revisions');
		}

		/** Topic Tags ************************************************************/

		// Just in time manipulation of reply terms before being edited
		$terms = apply_filters('bbp_edit_reply_pre_set_terms', $terms, $topic_id, $reply_id);

		// Insert terms
		$terms = wp_set_post_terms($topic_id, $terms, bbp_get_topic_tag_tax_id(), false);

		// Term error
		if (is_wp_error($terms)) {
			return new WP_Error('bp_json_reply_tags', __('There was a problem adding the tags to the topic.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 400));
		}

		/** Revisions *************************************************************/

		// Revision Reason
		if (!empty($reply_new->bbp_reply_edit_reason)) {
			$reply_edit_reason = esc_attr(strip_tags($reply_new->bbp_reply_edit_reason));
		}

		// Update revision log
		if (!empty($reply_new->bbp_log_reply_edit) && ("1" === $reply_new->bbp_log_reply_edit)) {
			$revision_id = wp_save_post_revision($reply_id);
			if (!empty($revision_id)) {
				bbp_update_reply_revision_log(array(
					'reply_id' => $reply_id,
					'revision_id' => $revision_id,
					'author_id' => bbp_get_current_user_id(),
					'reason' => $reply_edit_reason,
				));
			}
		}

		/** No Errors *************************************************************/

		if (!empty($reply_id) && !is_wp_error($reply_id)) {

			// Update counts, etc...
			do_action('bbp_edit_reply', $reply_id, $topic_id, $forum_id, $anonymous_data, $reply_author, true, $reply_to);

			bbp_update_topic($topic_id, $forum_id, $anonymous_data, $reply_author, false);

			/** Additional Actions (After Save) ***********************************/

			do_action('bbp_edit_reply_post_extras', $reply_id);

			/**
			 * Handle Subscription.
			 */
			if (bbp_is_subscriptions_active() && !empty($reply_author) && !empty($topic_id)) {
				$subscribed = bbp_is_user_subscribed($reply_author, $topic_id);
				$subscheck = $reply->bbp_topic_subscription;

				// Subscribed and unsubscribing
				if (true === $subscribed && false === $subscheck) {
					bbp_remove_user_subscription($reply_author, $topic_id);
				} elseif (false === $subscribed && true === $subscheck) {
					bbp_add_user_subscription($reply_author, $topic_id);
				}
			}

			/**
			 * Rest Hook
			 */
			do_action('boss_rest_bbpress_update_reply', $reply_id, $request);

			/**
			 * Redirect to topic endpoint.
			 */
			$request_reply = new WP_REST_Request("GET", "/" . $this->namespace_slug . "/bbpress/v1/reply/{$reply_id}");
			$server = rest_get_server();
			$response = $server->dispatch($request_reply);

			return $wp_rest_server->response_to_data(rest_ensure_response($response), isset($request['_embed']) ? 1 : 0);

			/** Errors ****************************************************************/

		} else {
			$append_error = (is_wp_error($reply_id) && $reply_id->get_error_message()) ? $reply_id->get_error_message() . ' ' : '';

			return new WP_Error('bp_json_reply_error', __('The following problem(s) have been found with your reply:' . $append_error . 'Please try again.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 400));
		}
	}

    /**
     * @param $request
     * @return WP_Error
     * @api {PATCH} /wp-json/appboss/bbpress/v1/reply/move Move reply handler
     * @apiName MoveBBPressReply
     * @apiGroup BBPressReply
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Handles the front end move reply submission, available since bbPress (r4521)
     * @apiDeprecated  Move a Reply. Check (#Forum Replies:MoveBBPReply)
     * @apiUse apidocForMoveBBPressReplyV1
     * @apiPrivate
     */
	public function move($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_cannot_view', __("You don't have permission.", 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		// Prevent debug notices
		$move_reply_id = $destination_topic_id = 0;
		$destination_topic_title = '';
		$destination_topic = $move_reply = $source_topic = '';

		/** Move Reply ***********************************************************/

		if (empty($request['bbp_reply_id'])) {
			return new WP_Error('bp_json_move_reply_reply_id', __('Reply ID to move not found!', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 400));
		} else {
			$move_reply_id = (int) $request['bbp_reply_id'];
		}

		$move_reply = bbp_get_reply($move_reply_id);

		// Reply exists
		if (empty($move_reply)) {
			return new WP_Error('bp_json_mover_reply_not_found', __('The reply you want to move was not found.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 404));
		}

		/** Topic to Move From ***************************************************/

		// Get the reply's current topic
		$source_topic = bbp_get_topic($move_reply->post_parent);

		// No topic
		if (empty($source_topic)) {
			return new WP_Error('bp_json_move_reply_source_not_found', __('The topic you want to move from was not found.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 404));
		}

		// Use cannot edit topic
		if (!current_user_can('edit_topic', $source_topic->ID)) {
			return new WP_Error('bp_json_move_reply_source_permission', __('You do not have the permissions to edit the source topic.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 403));
		}

		// How to move
		if (!empty($request['bbp_reply_move_option'])) {
			$move_option = (string) trim($request['bbp_reply_move_option']);
		}

		// Invalid move option
		if (empty($move_option) || !in_array($move_option, array('existing', 'topic'))) {
			return new WP_Error('bp_json_move_reply_option', __('You need to choose a valid move option.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 400));

			// Valid move option
		} else {

			// What kind of move
			switch ($move_option) {

			// Into an existing topic
			case 'existing':

				// Get destination topic id
				if (empty($request['bbp_destination_topic'])) {
					return new WP_Error('bp_json_move_reply_destination_id', __('Destination topic ID not found!', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 400));
				} else {
					$destination_topic_id = (int) $request['bbp_destination_topic'];
				}

				// Get the destination topic
				$destination_topic = bbp_get_topic($destination_topic_id);

				// No destination topic
				if (empty($destination_topic)) {
					return new WP_Error('bp_json_move_reply_destination_not_found', __('The topic you want to move to was not found!', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 404));
				}

				// User cannot edit the destination topic
				if (!current_user_can('edit_topic', $destination_topic->ID)) {
					return new WP_Error('bp_json_move_reply_destination_permission', __('You do not have the permissions to edit the destination topic!', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 403));
				}

				// Bump the reply position
				$reply_position = bbp_get_topic_reply_count($destination_topic->ID) + 1;

				// Update the reply
				wp_update_post(array(
					'ID' => $move_reply->ID,
					'post_title' => sprintf(__('Reply To: %s', 'bbpress'), $destination_topic->post_title),
					'post_name' => false, // will be automatically generated
					'post_parent' => $destination_topic->ID,
					'menu_order' => $reply_position,
					'guid' => '',
				));

				// Adjust reply meta values
				bbp_update_reply_topic_id($move_reply->ID, $destination_topic->ID);
				bbp_update_reply_forum_id($move_reply->ID, bbp_get_topic_forum_id($destination_topic->ID));

				break;

			// Move reply to a new topic
			case 'topic':
			default:

				// User needs to be able to publish topics
				if (current_user_can('publish_topics')) {

					// Use the new title that was passed
					if (!empty($request['bbp_reply_move_destination_title'])) {
						$destination_topic_title = esc_attr(strip_tags($request['bbp_reply_move_destination_title']));

						// Use the source topic title
					} else {
						$destination_topic_title = $source_topic->post_title;
					}

					// Update the topic
					$destination_topic_id = wp_update_post(array(
						'ID' => $move_reply->ID,
						'post_title' => $destination_topic_title,
						'post_name' => false,
						'post_type' => bbp_get_topic_post_type(),
						'post_parent' => $source_topic->post_parent,
						'guid' => '',
					));
					$destination_topic = bbp_get_topic($destination_topic_id);

					// Make sure the new topic knows its a topic
					bbp_update_topic_topic_id($move_reply->ID);

					// Shouldn't happen
					if (false === $destination_topic_id || is_wp_error($destination_topic_id) || empty($destination_topic)) {
						return new WP_Error('bp_json_move_reply_destination_reply', __('There was a problem converting the reply into the topic. Please try again.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 500));
					}

					// User cannot publish posts
				} else {
					return new WP_Error('bp_json_move_reply_create_destination_permission', __('You do not have the permissions to create new topics. The reply could not be converted into a topic.', boss_bbpress_api_domain($this->namespace_slug)), array('status' => 403));
				}

				break;
			}
		}

		/** No Errors - Clean Up **************************************************/

		// Update counts, etc...
		do_action('bbp_pre_move_reply', $move_reply->ID, $source_topic->ID, $destination_topic->ID);

		/** Date Check ************************************************************/

		// Check if the destination topic is older than the move reply
		if (strtotime($move_reply->post_date) < strtotime($destination_topic->post_date)) {

			// Set destination topic post_date to 1 second before from reply
			$destination_post_date = date('Y-m-d H:i:s', strtotime($move_reply->post_date) - 1);

			// Update destination topic
			wp_update_post(array(
				'ID' => $destination_topic_id,
				'post_date' => $destination_post_date,
				'post_date_gmt' => get_gmt_from_date($destination_post_date),
			));
		}

		// Set the last reply ID and freshness to the move_reply
		$last_reply_id = $move_reply->ID;
		$freshness = $move_reply->post_date;

		// Get the reply to
		$parent = bbp_get_reply_to($move_reply->ID);

		// Fix orphaned children
		$children = get_posts(array(
			'post_type' => bbp_get_reply_post_type(),
			'meta_key' => '_bbp_reply_to',
			'meta_value' => $move_reply->ID,
		));
		foreach ($children as $child) {
			bbp_update_reply_to($child->ID, $parent);
		}

		// Remove reply_to from moved reply
		delete_post_meta($move_reply->ID, '_bbp_reply_to');

		// It is a new topic and we need to set some default metas to make
		// the topic display in bbp_has_topics() list
		if ('topic' === $move_option) {
			bbp_update_topic_last_reply_id($destination_topic->ID, $last_reply_id);
			bbp_update_topic_last_active_id($destination_topic->ID, $last_reply_id);
			bbp_update_topic_last_active_time($destination_topic->ID, $freshness);

			// Otherwise update the existing destination topic
		} else {
			bbp_update_topic_last_reply_id($destination_topic->ID);
			bbp_update_topic_last_active_id($destination_topic->ID);
			bbp_update_topic_last_active_time($destination_topic->ID);
		}

		// Update source topic ID last active
		bbp_update_topic_last_reply_id($source_topic->ID);
		bbp_update_topic_last_active_id($source_topic->ID);
		bbp_update_topic_last_active_time($source_topic->ID);

		/** Successful Move ******************************************************/

		// Update counts, etc...
		do_action('bbp_post_move_reply', $move_reply->ID, $source_topic->ID, $destination_topic->ID);

		/**
		 * Redirect to topic endpoint.
		 */
		global $wp_rest_server;

		$request_topic = new WP_REST_Request("GET", "/" . $this->namespace_slug . "/bbpress/v1/topic/{$destination_topic->ID}");
		$server = rest_get_server();
		$response = $server->dispatch($request_topic);

		return $wp_rest_server->response_to_data(rest_ensure_response($response), isset($request['_embed']) ? 1 : 0);
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param $request
	 *
	 * @return mixed
	 */
	protected function dispatch($request) {

		$query_params = array();

		if (isset($request->get_query_params()["_embed"])) {
			$query_params["_embed"] = $request->get_query_params()["_embed"];
		}

		$request->set_query_params($query_params);

		$server = rest_get_server();
		$response = $server->dispatch($request);

		return $response;
	}

	/**
	 * Prepare a single reply for create or update
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return object $topic User object.
	 */
	protected function prepare_reply_for_database($request) {
		$reply = new stdClass;
		$bbp = bbpress();

		if (isset($request['id'])) {
			$reply->bbp_reply_id = $request['id'];
		} elseif (isset($request['reply_id'])) {
			$reply->bbp_reply_id = $request['reply_id'];
		}

		if (isset($request['topic_id'])) {
			$reply->bbp_topic_id = $request['topic_id'];
			$bbp->topic_query->in_the_loop = true;
			$bbp->topic_query->post = get_post($request['topic_id']);
		}
		if (isset($request['forum_id'])) {
			$reply->bbp_forum_id = $request['forum_id'];
		}
		if (isset($request['group_id'])) {
			$reply->bbp_group_id = $request['group_id'];
		}
		if (isset($request['title'])) {
			$reply->bbp_reply_title = $request['title'];
		}
		if (isset($request['content'])) {
			$reply->bbp_reply_content = $request['content'];
		}
		if (isset($request['reply_to'])) {
			$reply->bbp_reply_to = $request['reply_to'];
		}
		if (isset($request['tags'])) {
			$reply->bbp_topic_tags = $request['tags'];
		}
		if (isset($request['type'])) {
			$reply->bbp_stick_topic = $request['type'];
		}
		if (isset($request['reason'])) {
			$reply->bbp_reply_edit_reason = $request['reason'];
		}

		if (isset($request['subscribe']) && (true === $request['subscribe'])) {
			$reply->bbp_topic_subscription = true;
		} else if (isset($request['subscribe']) && (false === $request['subscribe'])) {
			$reply->bbp_topic_subscription = false;
		}

		if (isset($reply->bbp_topic_subscription)) {
			$_POST['bbp_topic_subscription'] = ($reply->bbp_topic_subscription) ? 'bbp_subscribe' : '';
		}

		if (isset($request['log'])) {
			$reply->bbp_log_reply_edit = $request['log'];
		}

		/**
		 * Filter topic data before inserting user via REST API
		 *
		 * @param object $topic Topic object.
		 * @param WP_REST_Request $request Request object.
		 */
		return apply_filters('boss_bp_json_pre_insert_reply', $reply, $request);
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_create_collection_params() {
		$params = array();

		$params['forum_id'] = array(
			'description' => __('Forum id of reply'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
			'required' => false,
		);

		$params['topic_id'] = array(
			'description' => __('Topic id of reply'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
			'required' => true,
		);

		$params['reply_to'] = array(
			'description' => __('Parent id of reply'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
		);

		$params['title'] = array(
			'description' => __('Title of reply'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['tags'] = array(
			'description' => __('Tags name of reply'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['type'] = array(
			'description' => __('Type of reply'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['content'] = array(
			'description' => __('Content of reply'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
			'required' => true,
		);

		$params['reason'] = array(
			'description' => __('Reason of reply edit'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['log'] = array(
			'description' => __('Log of reply edit'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['subscribe'] = array(
			'description' => __('whether user subscribe reply or no.'),
			'type' => 'boolean',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}
}