<?php
namespace BuddyBossApp\Api\BbPress;
use BuddyBossApp\Api\BbPress\ThreadReply;
use WP_Error;
use WP_Query;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

// NOTE : Old classname was class.boss_bbpress_api_post_controller. By Ketan, Oct-2019
/**
 * Adopted from wp rest Class WP_REST_Posts_Controller
 * Provides endpoints to forum/reply/topic cpt.
 */
class PostController extends \WP_REST_Controller {

	protected $post_type;

	public function __construct($slug, $post_type) {
		$this->post_type = $post_type;
		$obj = get_post_type_object($post_type);
		$this->rest_base = !empty($obj->rest_base) ? $obj->rest_base : $obj->name;
	}

	/**
	 * Register the routes for the objects of the controller.
	 */
	public function register_routes() {

		register_rest_route($this->namespace, '/' . $this->rest_base, array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_items'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
				'args' => $this->get_collection_params(),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_item'),
				'permission_callback' => array($this, 'get_item_permissions_check'),
				'args' => array(
					'context' => $this->get_context_param(array('default' => 'view')),
				),
			),
			// @TODO: Check this method later..
			// array(
			// 	'methods'  => WP_REST_Server::DELETABLE,
			// 	'callback' => array( $this, 'delete_item' ),
			// 	'permission_callback' => array( $this, 'delete_item_permissions_check' ),
			// 	'args'     => array(
			// 		'force'    => array(
			// 			'default'      => false,
			// 			'description'  => __( 'Whether to bypass trash and force deletion.' ),
			// 		),
			// 	),
			// ),
			'schema' => array($this, 'get_public_item_schema'),
		));
	}

	/**
	 * Check if a given request has access to read /posts.
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|boolean
	 */
	public function get_items_permissions_check($request) {

		$post_type = get_post_type_object($this->post_type);

		if ('edit' === $request['context'] && !current_user_can($post_type->cap->edit_posts)) {
			return new WP_Error('rest_forbidden_context', __('Sorry, you are not allowed to edit these posts in this post type'), array('status' => rest_authorization_required_code()));
		}

		return true;
	}

	/**
	 * @api {GET} /wp-json/appboss/bbpress/v1/forum Forums
	 * @apiName GetBBPressForums
	 * @apiGroup BBPressForum
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get all forum of/form BBPress component. Must have bbpress enabled
	 * @apiDeprecated  Retrieve forums. Check (#Forums:GetBBPForums)
	 * @apiUse apidocAdditionalForBBpressPostsV1
	 * @apiPrivate
	 */
	/**
	 * @api {GET} /wp-json/appboss/bbpress/v1/topic Topics
	 * @apiName GetBBPressTopics
	 * @apiGroup BBPressTopic
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get all topic of/form BBPress component. Must have bbpress enabled
	 * @apiDeprecated  Retrieve topics. Check (#Forum Topics:GetBBPTopics)
	 * @apiUse apidocAdditionalForBBpressPostsV1
	 * @apiPrivate
	 */
    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/bbpress/v1/reply Replys
     * @apiName GetBBPressReplys
     * @apiGroup BBPressReply
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get all reply of/form BBPress component. Must have bbpress enabled
     * @apiDeprecated  Retrieve Replies. Check (#Forum Replies:GetBBPReplies)
     * @apiUse apidocAdditionalForBBpressPostsV1
     * @apiPrivate
     */
	public function get_items($request) {
		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		global $wpdb;

		$args = array();
		$args['author__in'] = $request['author'];
		$args['author__not_in'] = $request['author_exclude'];
		$args['menu_order'] = $request['menu_order'];
		$args['offset'] = $request['offset'];
		$args['order'] = $request['order'];
		$args['orderby'] = $request['orderby'];
		$args['paged'] = $request['page'];
		$args['post__in'] = $request['include'];
		$args['post__not_in'] = $request['exclude'];
		$args['posts_per_page'] = $request['per_page'];
		$args['name'] = $request['slug'];

		if (is_array($request['parent'])) {
			$args['post_parent__in'] = (array) $request['parent'];
		} else {
			$args['post_parent'] = $request['parent'];
		}

		$args['post_parent__not_in'] = $request['parent_exclude'];
		$args['filter_fav'] = $request['filter_fav'];
		$args['filter_subscribe'] = $request['filter_subscribe'];
		$args['s'] = $request['search'];
		$args['trending'] = $request['trending'];
		$args['hide_group_items'] = $request['hide_group_items'];
		$args['show_myforums'] = $request['myforums'];
		$args['hide_restricted_forums'] = $request['hide_restricted_forums'];

		$args['date_query'] = array();
		// Set before into date query. Date query must be specified as an array of an array.
		if (isset($request['before'])) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into date query. Date query must be specified as an array of an array.
		if (isset($request['after'])) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		if (is_array($request['filter'])) {
			$args = array_merge($args, $request['filter']);
			unset($args['filter']);
		}

		// Force the post_type argument, since it's not a user input variable.
		$args['post_type'] = $this->post_type;

		if (isset($args['filter_fav']) && ($this->post_type == "topic" || $this->post_type == "reply")) {

			$args['filter_fav'] = (int) $args['filter_fav'];

			$fav_topics = get_user_meta($args['filter_fav'], "{$wpdb->prefix}_bbp_favorites", true);

			$fav_topics = @explode(",", $fav_topics);

			$args["post__in"] = $fav_topics;

		} elseif (isset($args['filter_subscribe']) && ($this->post_type == "topic" || $this->post_type == "reply")) {

			$args['filter_subscribe'] = (int) $args['filter_subscribe'];

			$fav_topics = get_user_meta($args['filter_subscribe'], "{$wpdb->prefix}_bbp_subscriptions", true);

			$fav_topics = @explode(",", $fav_topics);

			$args["post__in"] = $fav_topics;

		}

		$args = apply_filters("boss_rest_{$this->post_type}_query", $args, $request);
		$query_args = $this->prepare_items_query($args, $request);

		if (!empty($request['term_ids']) && $this->post_type == bbp_get_topic_post_type()) {
			$query_args['tax_query'][] = array(
				'taxonomy' => bbp_get_topic_tag_tax_id(),
				'field' => 'term_id',
				'terms' => $request['term_ids'],
				'include_children' => false,
			);
		}

		//@NOTE: We don't need this currently.

		// $taxonomies = wp_list_filter( get_object_taxonomies( $this->post_type, 'objects' ), array( 'show_in_rest' => true ) );
		// foreach ( $taxonomies as $taxonomy ) {
		// 	$base = ! empty( $taxonomy->rest_base ) ? $taxonomy->rest_base : $taxonomy->name;

		// 	if ( ! empty( $request[ $base ] ) ) {
		// 		$query_args['tax_query'][] = array(
		// 			'taxonomy'         => $taxonomy->name,
		// 			'field'            => 'term_id',
		// 			'terms'            => $request[ $base ],
		// 			'include_children' => false,
		// 		);
		// 	}
		// }

		/**
		 * Advance Filters
		 */

		$advance_filter = $request['advance_filter'];

		if ($advance_filter == "no_topics") {

			$query_args['meta_query'][] = array(
				'key' => '_bbp_total_topic_count',
				'value' => '0',
				'compare' => '=',
			);

		} else if ($advance_filter == "no_replies") {

			$query_args['meta_query'][] = array(
				'key' => '_bbp_reply_count',
				'value' => '0',
				'compare' => '=',
			);

		}

		//Hide group items setting
		if (!empty($args['hide_group_items']) && $args['hide_group_items'] == '1' && function_exists('bp_is_active') && bp_is_active('groups') && ($this->post_type == "topic" || $this->post_type == "reply")) {
			//Hide group forums topic and reply
			$forums_query = new WP_Query(array(
				'post_type' => 'forum',
				'posts_per_page' => -1,
				'post_status' => array('publish', 'auto-draft', 'draft', 'private', 'hidden', 'trash'),
				'meta_key' => '_bbp_group_ids', // change to how "event date" is stored
				'meta_compare' => 'EXISTS',
				'fields' => 'ids',
			));

			if (!empty($forums_query->posts)) {
				$query_args['meta_query'][] = array(
					'key' => '_bbp_forum_id',
					'value' => $forums_query->posts,
					'compare' => 'NOT IN',
				);
			}
		}

		//Hide Not accesible forums
		if (!empty($args['hide_restricted_forums']) && ($args['hide_restricted_forums'] == '1' || $args['hide_restricted_forums'] == true) && function_exists('bp_is_active') && bp_is_active('groups') && ($this->post_type == "forum")) {
			if (is_user_logged_in()) {
				//Hide a group forums which is not belong to current user
				$loggedin_user_id = get_current_user_id();
				$db_query = "SELECT id FROM {$wpdb->prefix}bp_groups WHERE enable_forum = 1 AND (status = 'private' OR status = 'hidden' ) AND id NOT IN ( select group_id from {$wpdb->prefix}bp_groups_members where user_id = $loggedin_user_id AND is_confirmed = '1' )";
				$group_ids = $wpdb->get_col($db_query);

				/* Get List of Forums for non-accessible Groups. */
				$forum_ids = array();
				if (!empty($group_ids)) {
					foreach ($group_ids as $single_group_id) {
						$forum_id = groups_get_groupmeta($single_group_id, 'forum_id', true);
						if (is_array($forum_id)) {
							$forum_ids[] = $forum_id[0];
						}
					}
				}

				if (isset($query_args['post__not_in'])) {
					if (!is_array($query_args['post__not_in'])) {
						$query_args['post__not_in'] = explode(',', $query_args['post__not_in']);
					}
					$forum_ids = array_merge($query_args['post__not_in'], $forum_ids);
				}

				$query_args['post__not_in'] = $forum_ids;
			}
		}

		if (!empty($args['show_myforums']) && ($args['show_myforums'] == '1' || $args['show_myforums'] == true) && function_exists('bp_is_active') && bp_is_active('groups') && ($this->post_type == "forum")) {
// Get forums to exclude
			$forum_ids = bbp_exclude_forum_ids('array');

			if (isset($query_args['post__not_in'])) {
				if (!is_array($query_args['post__not_in'])) {
					$query_args['post__not_in'] = explode(',', $query_args['post__not_in']);
				}
				$forum_ids = array_merge($query_args['post__not_in'], $forum_ids);
			}

			$query_args['post__not_in'] = $forum_ids;
		}

		/**
		 * Post Status
		 */
		// get all post status for moderator and admin
		if ($this->post_type == "reply" || $this->post_type == "topic") {

			if (bbp_get_view_all()) {
				// Default view=all statuses
				$post_statuses = array(
					bbp_get_public_status_id(),
					bbp_get_closed_status_id(),
					bbp_get_spam_status_id(),
					bbp_get_trash_status_id(),
				);

				// Add support for private status
				if (current_user_can('read_private_replies')) {
					$post_statuses[] = bbp_get_private_status_id();
				}

				// Join post statuses together
				$query_args['post_status'] = implode(',', $post_statuses);
			} else {
				$query_args['perm'] = 'readable';
			}

		} else {
			$query_args['post_status'] = bbp_get_public_status_id();
		}

		/**
		 * Special OrderBy
		 */

		// Popular only for forums
		if ($this->post_type == "forum" && $args['orderby'] == "popular") {
			$query_args['orderby'] = "meta_value_num";
			$query_args['meta_key'] = "_bbp_total_topic_count";
		}
		// Popular only for topics
		if ($this->post_type == "topic" && $args['orderby'] == "popular") {
			$query_args['orderby'] = "meta_value_num";
			$query_args['meta_key'] = "_bbp_reply_count";
		}

		// activity only for forum
		if ($this->post_type == "forum" && $args['orderby'] == "activity") {
			$query_args['orderby'] = "meta_value";
			$query_args['meta_key'] = "_bbp_last_active_time";
		}

		// activity only for topics
		if ($this->post_type == "topic" && $args['orderby'] == "activity") {
			$query_args['orderby'] = "meta_value";
			$query_args['meta_key'] = "_bbp_last_active_time";
		}

		// Trending  only for forums
		if ($this->post_type == "forum" && $args['orderby'] == "trending") {
			$query_args['orderby'] = "meta_value_num";
			$query_args['meta_key'] = "_bbp_total_reply_count";
		}

		if ($this->post_type == "forum" && $args['orderby'] == "default") {
			$query_args['orderby'] = "menu_order title";
		}

		if ($this->post_type == "forum" && $args['orderby'] == "items") {
			$query_args['orderby'] = "meta_value_num";
			$query_args['meta_key'] = "_bbp_total_topic_count";
		}

		// Trending  only for topics
		if ($this->post_type == "topic" && $args['orderby'] == "trending") {
			$query_args['orderby'] = "meta_value_num";
			$query_args['meta_key'] = "_bbp_voice_count";
		}

		if ($this->post_type == "topic" && $args['orderby'] == "items") {
			$query_args['orderby'] = "meta_value_num";
			$query_args['meta_key'] = "_bbp_total_reply_count";
		}

		if ($args["orderby"] == "trending") {

			$trending_after = "";

			switch ($args["trending"]) {
			case 'day':
				$trending_after = "last day";
				break;
			case 'week':
				$trending_after = "this week";
				break;
			case 'month':
				$trending_after = "first day of this month";
				break;
			case 'year':
				$trending_after = 'first day of January ' . date('Y');
				break;
			}

			// Trending Filter
			$query_args['meta_query'][] = array(
				'key' => '_bbp_last_active_time',
				'value' => date("Y-m-d 00:00:00", strtotime($trending_after)),
				'compare' => '>',
			);
		}

		if ($this->post_type == "forum" && 'relevance' == $args['orderby']) {
			$query_args['orderby'] = 'menu_order title';
			$query_args['order'] = 'asc';
		}

		$reply_jump = (int) @$request['reply_jump']; // Jump to the page where reply exists.

		// Bbpress Replies Threads

		if ($this->post_type == "reply" && bbp_thread_replies() && empty($query_args['s'])) {

			// Get replies to loop through in $_replies
			require_once dirname( __FILE__ ) . "/ThreadReply.php";
			$reply_args = $query_args;
			$reply_args["paged"] = 1;
			$reply_args["posts_per_page"] = -1;
			$posts_query = new WP_Query($reply_args);

			// Thread Replies
			$r = array(
				'walker' => null,
				'max_depth' => bbp_thread_replies_depth(),
				'callback' => null,
				'end_callback' => null,
				'page' => $query_args['paged'],
				'per_page' => $query_args['posts_per_page'],
			);

			/**
			 * Add post to into all replies.
			 */
			foreach ($posts_query->posts as $k => $v) {
				$posts_query->posts[$k]->reply_to = (int) get_post_meta($v->ID, "_bbp_reply_to", true);
			}

			global $boss_bbpress_thread_reply;
			$boss_bbpress_thread_reply = array();
			$walker = new ThreadReply();

			//Default Walker
			$walker->paged_walk($posts_query->posts, $r['max_depth'], $r['page'], $r['per_page'], $r);

			$total_parent_replies = $walker->get_number_of_root_elements($posts_query->posts);

			// If Reply Jump Specified.
			if (!empty($reply_jump)) {

				/**
				 * Loop Over All Pages to Catch The Page ID Where Reply Exists.
				 */

				$r_page_page = max(ceil($total_parent_replies / $r['per_page']), 1);

				for ($i = $r_page_page; $i > 0; $i--) {

					$boss_bbpress_thread_reply = array();
					$walker->paged_walk($posts_query->posts, $r['max_depth'], $i, $r['per_page'], $r);
					$found = false;
					foreach ($boss_bbpress_thread_reply as $reply) {
						if ($reply->ID == $reply_jump) {
							$query_args['paged'] = $i; //override the paged query.
							$found = true;
							break;
						}
					}

					if ($found) {
						break;
					}

				}

			}

			$depth_replies = $boss_bbpress_thread_reply;

			$posts_query->posts = $depth_replies;

			$posts_query->found_posts = $total_parent_replies; // for page count.

		} else {

			$posts_query = new WP_Query($query_args);

			if ($this->post_type == "reply" && !empty($reply_jump)) {
				// This is Expensive Query Block. but needed.

				$r_page_page = max(ceil($posts_query->found_posts / $query_args['posts_per_page']), 1);

				$found = false;

				for ($i = $r_page_page; $i > 0; $i--) {
					// do reverce check.

					$query_args['paged'] = $i;
					$posts_query = new WP_Query($query_args);

					foreach ($posts_query->posts as $post) {
						if ($post->ID == $reply_jump) {
							$query_args['paged'] = $i; //override the paged query.
							$found = true;
							break;
						}
					}

					if ($found) {
						break;
					}

				}

			}

		}

		$posts = array();

		/** Stickies **************************************************************/

		// Put sticky posts at the top of the posts array
		if (isset($request['show_stickies']) && $request['show_stickies'] != "" && $query_args['paged'] <= 1 && $this->post_type == "topic") {

			// Get super stickies and stickies in this forum
			$stickies = bbp_get_super_stickies();

			// Get stickies for current forum
			if (!empty($query_args['post_parent__in'][0])) {
				$stickies = array_merge($stickies, bbp_get_stickies($query_args['post_parent__in'][0]));
			}

			// Remove any duplicate stickies
			$stickies = array_unique($stickies);

			// We have stickies
			if (is_array($stickies) && !empty($stickies)) {

				// Start the offset at -1 so first sticky is at correct 0 offset
				$sticky_offset = -1;

				// Loop over topics and relocate stickies to the front.
				foreach ($stickies as $sticky_index => $sticky_ID) {

					// Get the post offset from the posts array
					$post_offsets = wp_filter_object_list($posts_query->posts, array('ID' => $sticky_ID), 'OR', 'ID');

					// Continue if no post offsets
					if (empty($post_offsets)) {
						continue;
					}

					// Loop over posts in current query and splice them into position
					foreach (array_keys($post_offsets) as $post_offset) {
						$sticky_offset++;

						$sticky = $posts_query->posts[$post_offset];

						// Remove sticky from current position
						array_splice($posts_query->posts, $post_offset, 1);

						// Move to front, after other stickies
						array_splice($posts_query->posts, $sticky_offset, 0, array($sticky));

						// Cleanup
						unset($stickies[$sticky_index]);
						unset($sticky);
					}

					// Cleanup
					unset($post_offsets);
				}

				// Cleanup
				unset($sticky_offset);

				// If any posts have been excluded specifically, Ignore those that are sticky.
				if (!empty($stickies) && !empty($query_args['post__not_in'])) {
					$stickies = array_diff($stickies, $query_args['post__not_in']);
				}

				// Fetch sticky posts that weren't in the query results
				if (!empty($stickies)) {

					// Query to use in get_posts to get sticky posts
					$sticky_query = array(
						'post_type' => bbp_get_topic_post_type(),
						'post_parent__in' => (!empty($query_args['post_parent__in'])) ? $query_args['post_parent__in'] : array(),
						'meta_key' => '_bbp_last_active_time',
						'orderby' => 'meta_value',
						'order' => 'DESC',
						'post__in' => $stickies,
					);

					if (!empty($request['term_ids']) && $this->post_type == bbp_get_topic_post_type()) {
						$sticky_query['tax_query'][] = array(
							'taxonomy' => bbp_get_topic_tag_tax_id(),
							'field' => 'term_id',
							'terms' => $request['term_ids'],
							'include_children' => false,
						);
					}

					// Cleanup
					unset($stickies);

					// Conditionally exclude private/hidden forum ID's
					$exclude_forum_ids = bbp_exclude_forum_ids('array');
					if (!empty($exclude_forum_ids)) {
						$sticky_query['post_parent__not_in'] = $exclude_forum_ids;
					}

					// What are the default allowed statuses (based on user caps)
					if (current_user_can('moderate')) {
						$sticky_query['post_status'] = $query_args['post_status'];

						// Lean on the 'perm' query var value of 'readable' to provide statuses
					} else {
						$sticky_query['post_status'] = $query_args['perm'];
					}

					// Get all stickies
					$sticky_posts = new WP_Query($sticky_query);

					if (!empty($sticky_posts)) {
						$sticky_posts = $sticky_posts->posts;

						// Get a count of the visible stickies
						$sticky_count = count($sticky_posts);

						// Merge the stickies topics with the query topics .
						$posts_query->posts = array_merge($sticky_posts, $posts_query->posts);

						// Adjust loop and counts for new sticky positions
						$posts_query->found_posts = (int) $posts_query->found_posts + (int) $sticky_count;
						$posts_query->post_count = (int) $posts_query->post_count + (int) $sticky_count;

						// Cleanup
						unset($sticky_posts);
					}
				}
			}
		}

		/**
		 * ===================  Prepare the responses  ===================
		 */
		foreach ($posts_query->posts as $post) {
			if (!$this->check_read_permission($post)) {
				continue;
			}

			$data = $this->prepare_item_for_response($post, $request);
			$posts[] = $this->prepare_response_for_collection($data);
		}

		$page = (int) @$query_args['paged'];
		$total_posts = $posts_query->found_posts;

		if ($total_posts < 1) {
			// Out-of-bounds, run the query again without LIMIT for total count
			unset($query_args['paged']);
			$count_query = new WP_Query();
			$count_query->query($query_args);
			$total_posts = $count_query->found_posts;
		}

		$max_pages = ceil($total_posts / (int) $query_args['posts_per_page']);

		$response = rest_ensure_response($posts);
		$response->header('X-WP-Page', (int) $page);
		$response->header('X-WP-Total', (int) $total_posts);
		$response->header('X-WP-TotalPages', (int) $max_pages);

		$request_params = $request->get_query_params();
		if (!empty($request_params['filter'])) {
			// Normalize the pagination params.
			unset($request_params['filter']['posts_per_page']);
			unset($request_params['filter']['paged']);
		}
		$base = add_query_arg($request_params, rest_url(sprintf('/%s/%s', $this->namespace, $this->rest_base)));

		if ($page > 1) {
			$prev_page = $page - 1;
			if ($prev_page > $max_pages) {
				$prev_page = $max_pages;
			}
			$prev_link = add_query_arg('page', $prev_page, $base);
			$response->link_header('prev', $prev_link);
		}
		if ($max_pages > $page) {
			$next_page = $page + 1;
			$next_link = add_query_arg('page', $next_page, $base);
			$response->link_header('next', $next_link);
		}

		return $response;
	}

	/**
	 * Check if a given request has access to read a post.
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|boolean
	 */
	public function get_item_permissions_check($request) {

		$post = get_post((int) $request['id']);

		if ('edit' === $request['context'] && $post && !$this->check_update_permission($post)) {
			return new WP_Error('rest_forbidden_context', __('Sorry, you are not allowed to edit this post'), array('status' => rest_authorization_required_code()));
		}

		if ($post) {
			return $this->check_read_permission($post);
		}

		return true;
	}

	/**
	 * @api {GET} /wp-json/appboss/bbpress/v1/forum/:id Forum
	 * @apiName GetBBPressForum
	 * @apiGroup BBPressForum
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get single forum (post_type=forum). Must have bbpress enabled
	 * @apiDeprecated  Retrieve a single forum. Check (#Forums:GetBBPForum)
	 * @apiUse apidocAdditionalForBBpressForumV1
	 * @apiPrivate
	 */
	/**
	 * @api {GET} /wp-json/appboss/bbpress/v1/reply/:id Reply
	 * @apiName GetBBPressReply
	 * @apiGroup BBPressReply
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get single reply (post_type=reply). Must have bbpress enabled
	 * @apiDeprecated  Retrieve a single topic. Check (#Forum Topics:GetBBPTopic)
	 * @apiUse apidocAdditionalForBBpressReplyV1
	 * @apiPrivate
	 */
	/**
	 * @api {GET} /wp-json/appboss/bbpress/v1/topic/:id Topic
	 * @apiName GetBBPressTopic
	 * @apiGroup BBPressTopic
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get single topic (post_type=topic). Must have bbpress enabled
	 * @apiDeprecated  Retrieve a single reply. Check (#Forum Replies:GetBBPReply)
	 * @apiUse apidocAdditionalForBBpressTopicV1
	 * @apiPrivate
	 */
	public function get_item($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$id = (int) $request['id'];
		$post = get_post($id);

		if (empty($id) || empty($post->ID) || $this->post_type !== $post->post_type) {
			return new WP_Error('rest_post_invalid_id', __('Invalid post ID.'), array('status' => 404));
		}

		if ($this->post_type == bbp_get_reply_post_type() && bbp_thread_replies()) {

			// Get replies to loop through in $_replies

			require_once dirname( __FILE__ ) . "/ThreadReply.php";

			$posts_query = new WP_Query();
			$reply_args = array('post_type' => $this->post_type, 'post_parent' => $post->post_parent);
			$reply_args["paged"] = 1;
			$reply_args["posts_per_page"] = -1;

			// get all post status for moderator and admin
			if (bbp_get_view_all()) {
				// Default view=all statuses
				$post_statuses = array(
					bbp_get_public_status_id(),
					bbp_get_closed_status_id(),
					bbp_get_spam_status_id(),
					bbp_get_trash_status_id(),
				);

				// Add support for private status
				if (current_user_can('read_private_replies')) {
					$post_statuses[] = bbp_get_private_status_id();
				}

				// Join post statuses together
				$reply_args['post_status'] = implode(',', $post_statuses);
			} else {
				$reply_args['perm'] = 'readable';
			}

			$reply_result = $posts_query->query($reply_args);

			// Thread Replies
			$r = array(
				'walker' => null,
				'max_depth' => bbp_thread_replies_depth(),
				'callback' => null,
				'end_callback' => null,
				'page' => $reply_args["paged"],
				'per_page' => $reply_args["posts_per_page"],
			);

			foreach ($reply_result as $k => $v) {
				$reply_result[$k]->reply_to = (int) get_post_meta($v->ID, "_bbp_reply_to", true);
			}

			global $boss_bbpress_thread_reply;
			$boss_bbpress_thread_reply = array();
			$walker = new ThreadReply();
			$walker->paged_walk($reply_result, $r['max_depth'], $r['page'], $r['per_page'], $r);

			$depth_replies = $boss_bbpress_thread_reply;

			if (isset($depth_replies[$post->ID])) {
				$post->depth = $depth_replies[$post->ID]->depth;
			}
		}

		$data = $this->prepare_item_for_response($post, $request);
		$response = rest_ensure_response($data);

		if (is_post_type_viewable(get_post_type_object($post->post_type))) {
			$response->link_header('alternate', get_permalink($id), array('type' => 'text/html'));
		}

		return $response;
	}

	/**
	 * Check if a given request has access to create a post.
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|boolean
	 */
	public function create_item_permissions_check($request) {

		$post_type = get_post_type_object($this->post_type);

		if (!empty($request['password']) && !current_user_can($post_type->cap->publish_posts)) {
			return new WP_Error('rest_cannot_publish', __('Sorry, you are not allowed to create password protected posts in this post type'), array('status' => rest_authorization_required_code()));
		}

		if (!empty($request['author']) && get_current_user_id() !== $request['author'] && !current_user_can($post_type->cap->edit_others_posts)) {
			return new WP_Error('rest_cannot_edit_others', __('You are not allowed to create posts as this user.'), array('status' => rest_authorization_required_code()));
		}

		if (!empty($request['sticky']) && !current_user_can($post_type->cap->edit_others_posts)) {
			return new WP_Error('rest_cannot_assign_sticky', __('You do not have permission to make posts sticky.'), array('status' => rest_authorization_required_code()));
		}

		if (!current_user_can($post_type->cap->create_posts)) {
			return new WP_Error('rest_cannot_create', __('Sorry, you are not allowed to create new posts.'), array('status' => rest_authorization_required_code()));
		}

		return true;
	}

	// /**
	//  * Check if a given request has access to delete a post.
	//  *
	//  * @param  WP_REST_Request $request Full details about the request.
	//  * @return bool|WP_Error
	//  */
	// public function delete_item_permissions_check( $request ) {

	// 	$post = get_post( $request['id'] );

	// 	if ( $post && ! $this->check_delete_permission( $post ) ) {
	// 		return new WP_Error( 'rest_cannot_delete', __( 'Sorry, you are not allowed to delete posts.' ), array( 'status' => rest_authorization_required_code() ) );
	// 	}

	// 	return true;
	// }

	/**
	 * Delete a single post.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	// public function delete_item( $request ) {
	// 	$id = (int) $request['id'];
	// 	$force = (bool) $request['force'];

	// 	$post = get_post( $id );

	// 	if ( empty( $id ) || empty( $post->ID ) || $this->post_type !== $post->post_type ) {
	// 		return new WP_Error( 'rest_post_invalid_id', __( 'Invalid post ID.' ), array( 'status' => 404 ) );
	// 	}

	// 	$supports_trash = ( EMPTY_TRASH_DAYS > 0 );
	// 	if ( $post->post_type === 'attachment' ) {
	// 		$supports_trash = $supports_trash && MEDIA_TRASH;
	// 	}

	// 	/**
	// 	 * Filter whether a post is trashable.
	// 	 *
	// 	 * Return false to disable trash support for the post.
	// 	 *
	// 	 * @param boolean $supports_trash Whether the post type support trashing.
	// 	 * @param WP_Post $post           The Post object being considered for trashing support.
	// 	 */
	// 	$supports_trash = apply_filters( "rest_{$this->post_type}_trashable", $supports_trash, $post );

	// 	if ( ! $this->check_delete_permission( $post ) ) {
	// 		return new WP_Error( 'rest_user_cannot_delete_post', __( 'Sorry, you are not allowed to delete this post.' ), array( 'status' => rest_authorization_required_code() ) );
	// 	}

	// 	$request->set_param( 'context', 'edit' );
	// 	$response = $this->prepare_item_for_response( $post, $request );

	// 	// If we're forcing, then delete permanently.
	// 	if ( $force ) {
	// 		$result = wp_delete_post( $id, true );
	// 	} else {
	// 		// If we don't support trashing for this type, error out.
	// 		if ( ! $supports_trash ) {
	// 			return new WP_Error( 'rest_trash_not_supported', __( 'The post does not support trashing.' ), array( 'status' => 501 ) );
	// 		}

	// 		// Otherwise, only trash if we haven't already.
	// 		if ( 'trash' === $post->post_status ) {
	// 			return new WP_Error( 'rest_already_trashed', __( 'The post has already been deleted.' ), array( 'status' => 410 ) );
	// 		}

	// 		// (Note that internally this falls through to `wp_delete_post` if
	// 		// the trash is disabled.)
	// 		$result = wp_trash_post( $id );
	// 	}

	// 	if ( ! $result ) {
	// 		return new WP_Error( 'rest_cannot_delete', __( 'The post cannot be deleted.' ), array( 'status' => 500 ) );
	// 	}

	// 	/**
	// 	 * Fires after a single post is deleted or trashed via the REST API.
	// 	 *
	// 	 * @param object           $post     The deleted or trashed post.
	// 	 * @param WP_REST_Response $response The response data.
	// 	 * @param WP_REST_Request  $request  The request sent to the API.
	// 	 */
	// 	do_action( "rest_delete_{$this->post_type}", $post, $response, $request );

	// 	return $response;
	// }

	/**
	 * Determine the allowed query_vars for a get_items() response and
	 * prepare for WP_Query.
	 *
	 * @param array $prepared_args
	 * @param WP_REST_Request $request
	 *
	 * @return array          $query_args
	 */
	protected function prepare_items_query($prepared_args = array(), $request = null) {

		$valid_vars = array_flip($this->get_allowed_query_vars());
		$query_args = array();
		foreach ($valid_vars as $var => $index) {
			if (isset($prepared_args[$var])) {
				/**
				 * Filter the query_vars used in `get_items` for the constructed query.
				 *
				 * The dynamic portion of the hook name, $var, refers to the query_var key.
				 *
				 * @param mixed $prepared_args [ $var ] The query_var value.
				 *
				 */
				$query_args[$var] = apply_filters("rest_query_var-{$var}", $prepared_args[$var]);
			}
		}

		if ('post' !== $this->post_type || !isset($query_args['ignore_sticky_posts'])) {
			$query_args['ignore_sticky_posts'] = true;
		}

		if ('include' === $query_args['orderby']) {
			$query_args['orderby'] = 'post__in';
		}

		return $query_args;
	}

	/**
	 * Get all the WP Query vars that are allowed for the API request.
	 *
	 * @return array
	 */
	protected function get_allowed_query_vars() {
		global $wp;

		/**
		 * Filter the publicly allowed query vars.
		 *
		 * Allows adjusting of the default query vars that are made public.
		 *
		 * @param array  Array of allowed WP_Query query vars.
		 */
		$valid_vars = apply_filters('query_vars', $wp->public_query_vars);

		$post_type_obj = get_post_type_object($this->post_type);
		if (current_user_can($post_type_obj->cap->edit_posts)) {
			/**
			 * Filter the allowed 'private' query vars for authorized users.
			 *
			 * If the user has the `edit_posts` capability, we also allow use of
			 * private query parameters, which are only undesirable on the
			 * frontend, but are safe for use in query strings.
			 *
			 * To disable anyway, use
			 * `add_filter( 'rest_private_query_vars', '__return_empty_array' );`
			 *
			 * @param array $private_query_vars Array of allowed query vars for authorized users.
			 * }
			 */
			$private = apply_filters('rest_private_query_vars', $wp->private_query_vars);
			$valid_vars = array_merge($valid_vars, $private);
		}
		// Define our own in addition to WP's normal vars.
		$rest_valid = array(
			'author__in',
			'author__not_in',
			'ignore_sticky_posts',
			'menu_order',
			'offset',
			'post__in',
			'post__not_in',
			'post_parent',
			'post_parent__in',
			'post_parent__not_in',
			'posts_per_page',
			'date_query',
		);
		$valid_vars = array_merge($valid_vars, $rest_valid);

		/**
		 * Filter allowed query vars for the REST API.
		 *
		 * This filter allows you to add or remove query vars from the final allowed
		 * list for all requests, including unauthenticated ones. To alter the
		 * vars for editors only, {@see rest_private_query_vars}.
		 *
		 * @param array {
		 *    Array of allowed WP_Query query vars.
		 *
		 * @param string $allowed_query_var The query var to allow.
		 * }
		 */
		$valid_vars = apply_filters('rest_query_vars', $valid_vars);

		return $valid_vars;
	}

	/**
	 * Check the post excerpt and prepare it for single post output.
	 *
	 * @param string $excerpt
	 *
	 * @return string|null $excerpt
	 */
	protected function prepare_excerpt_response($excerpt) {
		if (post_password_required()) {
			return __('There is no excerpt because this is a protected post.');
		}

		/** This filter is documented in wp-includes/post-template.php */
		$excerpt = apply_filters('the_excerpt', apply_filters('get_the_excerpt', $excerpt));

		if (empty($excerpt)) {
			return '';
		}

		return $excerpt;
	}

	/**
	 * Check the post_date_gmt or modified_gmt and prepare any post or
	 * modified date for single post output.
	 *
	 * @param string $date_gmt
	 * @param string|null $date
	 *
	 * @return string|null ISO8601/RFC3339 formatted datetime.
	 */
	protected function prepare_date_response($date_gmt, $date = null) {
		// Use the date if passed.
		if (isset($date)) {
			return mysql_to_rfc3339($date);
		}

		// Return null if $date_gmt is empty/zeros.
		if ('0000-00-00 00:00:00' === $date_gmt) {
			return null;
		}

		// Return the formatted datetime.
		return mysql_to_rfc3339($date_gmt);
	}

	protected function prepare_password_response($password) {
		if (!empty($password)) {
			/**
			 * Fake the correct cookie to fool post_password_required().
			 * Without this, get_the_content() will give a password form.
			 */
			require_once ABSPATH . 'wp-includes/class-phpass.php';
			$hasher = new \PasswordHash(8, true);
			$value = $hasher->HashPassword($password);
			$_COOKIE['wp-postpass_' . COOKIEHASH] = wp_slash($value);
		}

		return $password;
	}

	/**
	 * Determine validity and normalize provided status param.
	 *
	 * @param string $post_status
	 * @param object $post_type
	 *
	 * @return WP_Error|string $post_status
	 */
	protected function handle_status_param($post_status, $post_type) {

		switch ($post_status) {
		case 'draft':
		case 'pending':
			break;
		case 'private':
			if (!current_user_can($post_type->cap->publish_posts)) {
				return new WP_Error('rest_cannot_publish', __('Sorry, you are not allowed to create private posts in this post type'), array('status' => rest_authorization_required_code()));
			}
			break;
		case 'publish':
		case 'future':
			if (!current_user_can($post_type->cap->publish_posts)) {
				return new WP_Error('rest_cannot_publish', __('Sorry, you are not allowed to publish posts in this post type'), array('status' => rest_authorization_required_code()));
			}
			break;
		default:
			if (!get_post_status_object($post_status)) {
				$post_status = 'draft';
			}
			break;
		}

		return $post_status;
	}

	/**
	 * Determine the featured media based on a request param.
	 *
	 * @param int $featured_media
	 * @param int $post_id
	 * @return bool|WP_Error
	 */
	protected function handle_featured_media($featured_media, $post_id) {

		$featured_media = (int) $featured_media;
		if ($featured_media) {
			$result = set_post_thumbnail($post_id, $featured_media);
			if ($result) {
				return true;
			} else {
				return new WP_Error('rest_invalid_featured_media', __('Invalid featured media ID.'), array('status' => 400));
			}
		} else {
			return delete_post_thumbnail($post_id);
		}

	}

	/**
	 * Check if a given post type should be viewed or managed.
	 *
	 * @param object|string $post_type
	 *
	 * @return boolean Is post type allowed?
	 */
	/*protected function check_is_post_type_allowed( $post_type ) {
		if ( ! is_object( $post_type ) ) {
			$post_type = get_post_type_object( $post_type );
		}

		if ( ! empty( $post_type ) && ! empty( $post_type->show_in_rest ) ) {
			return true;
		}

		return false;
	}*/

	/**
	 * Check if we can read a post.
	 *
	 * Correctly handles posts with the inherit status.
	 *
	 * @param object $post Post object.
	 *
	 * @return boolean Can we read it?
	 */
	public function check_read_permission($post) {
		if (!empty($post->post_password) && !$this->check_update_permission($post)) {
			return false;
		}

		$post_type = get_post_type_object($post->post_type);
		/*if ( ! $this->check_is_post_type_allowed( $post_type ) ) {
			return false;
		}*/

		// Can we read the post?
		if ('publish' === $post->post_status || current_user_can($post_type->cap->read_post, $post->ID)) {
			return true;
		}

		$post_status_obj = get_post_status_object($post->post_status);
		if ($post_status_obj && $post_status_obj->public) {
			return true;
		}

		// Can we read the parent if we're inheriting?
		if ('inherit' === $post->post_status && $post->post_parent > 0) {
			$parent = get_post($post->post_parent);

			return $this->check_read_permission($parent);
		}

		// If we don't have a parent, but the status is set to inherit, assume
		// it's published (as per get_post_status()).
		if ('inherit' === $post->post_status) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if a post can be edited.
	 *
	 * @param object $post Post object.
	 * @return bool Whether the post can be edited.
	 */
	public function check_update_permission($post) {
		$post_type = get_post_type_object($post->post_type);

		return current_user_can($post_type->cap->edit_post, $post->ID);
	}

	/**
	 * Check if we can delete a post.
	 *
	 * @param object $post Post object.
	 *
	 * @return boolean Can we delete it?
	 */
	// protected function check_delete_permission( $post ) {
	// 	$post_type = get_post_type_object( $post->post_type );

	// 	if ( ! $this->check_is_post_type_allowed( $post_type ) ) {
	// 		return false;
	// 	}

	// 	return current_user_can( $post_type->cap->delete_post, $post->ID );
	// }

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post $post Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response($post, $request) {
		$GLOBALS['post'] = $post;
		setup_postdata($post);

		// Base fields for every post.
		$data = array(
			'id' => $post->ID,
			'date' => $this->prepare_date_response($post->post_date_gmt, $post->post_date),
			'date_gmt' => $this->prepare_date_response($post->post_date_gmt),
			'guid' => array(
				/** This filter is documented in wp-includes/post-template.php */
				'rendered' => apply_filters('get_the_guid', $post->guid),
				'raw' => $post->guid,
			),
			'modified' => $this->prepare_date_response($post->post_modified_gmt, $post->post_modified),
			'modified_gmt' => $this->prepare_date_response($post->post_modified_gmt),
			'password' => $post->post_password,
			'slug' => $post->post_name,
			'status' => $post->post_status,
			'type' => $post->post_type == bbp_get_forum_post_type() ? bbp_get_forum_type($post->ID) : $post->post_type,
			'link' => get_permalink($post->ID),
			'depth' => $post->depth,
		);

		if ($post->post_type == bbp_get_reply_post_type()) {
			$data['link'] = bbp_get_reply_url($post->ID);
		}

		$schema = $this->get_item_schema();

		if (!empty($schema['properties']['title'])) {
			$data['title'] = array(
				'raw' => $post->post_title,
				'rendered' => get_the_title($post->ID),
			);
		}

		if (!empty($schema['properties']['content'])) {

			if (!empty($post->post_password)) {
				$this->prepare_password_response($post->post_password);
			}

			if ($post->post_type == bbp_get_reply_post_type()) {
				$content = bbp_get_reply_content($post->ID);
			} else if ($post->post_type == bbp_get_topic_post_type()) {
				$content = bbp_get_topic_content($post->ID);
			} else {
				$content = $post->post_content;
			}
			$content = apply_filters('the_content', $content);

			$data['content'] = array(
				'raw' => $post->post_content,
				/** This filter is documented in wp-includes/post-template.php */
				'rendered' => $content,
			);

			// Don't leave our cookie lying around: https://github.com/WP-API/WP-API/issues/1055.
			if (!empty($post->post_password)) {
				$_COOKIE['wp-postpass_' . COOKIEHASH] = '';
			}
		}

		if (!empty($schema['properties']['excerpt'])) {
			$data['excerpt'] = array(
				'raw' => $post->post_excerpt,
				'rendered' => $this->prepare_excerpt_response($post->post_excerpt),
			);
		}

		if (!empty($schema['properties']['author'])) {
			$data['author'] = (int) $post->post_author;
		}

		if (!empty($schema['properties']['featured_media'])) {
			$data['featured_media'] = (int) get_post_thumbnail_id($post->ID);
		}

		if (!empty($schema['properties']['parent'])) {
			$data['parent'] = (int) $post->post_parent;
		}

		if (!empty($schema['properties']['menu_order'])) {
			$data['menu_order'] = (int) $post->menu_order;
		}

		if (!empty($schema['properties']['comment_status'])) {
			$data['comment_status'] = $post->comment_status;
		}

		if (!empty($schema['properties']['ping_status'])) {
			$data['ping_status'] = $post->ping_status;
		}

		if (!empty($schema['properties']['sticky'])) {
			$data['sticky'] = is_sticky($post->ID);
		}

		if (!empty($schema['properties']['template'])) {
			if ($template = get_page_template_slug($post->ID)) {
				$data['template'] = $template;
			} else {
				$data['template'] = '';
			}
		}

		if (!empty($schema['properties']['format'])) {
			$data['format'] = get_post_format($post->ID);
			// Fill in blank post format.
			if (empty($data['format'])) {
				$data['format'] = 'standard';
			}
		}

		$taxonomies = wp_list_filter(get_object_taxonomies($this->post_type, 'objects'), array('show_in_rest' => true));
		foreach ($taxonomies as $taxonomy) {
			$base = !empty($taxonomy->rest_base) ? $taxonomy->rest_base : $taxonomy->name;
			$terms = get_the_terms($post, $taxonomy->name);
			$data[$base] = $terms ? wp_list_pluck($terms, 'term_id') : array();
		}

		$context = !empty($request['context']) ? $request['context'] : 'view';
		$data = $this->add_additional_fields_to_object($data, $request);
		$data = $this->filter_response_by_context($data, $context);

		// Wrap the data in a response object.
		$response = rest_ensure_response($data);

		$response->add_links($this->prepare_links($post));

		/**
		 * Filter the post data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
		 * prepared for the response.
		 *
		 * @param WP_REST_Response $response The response object.
		 * @param WP_Post $post Post object.
		 * @param WP_REST_Request $request Request object.
		 */
		return apply_filters("rest_prepare_{$this->post_type}", $response, $post, $request);
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param WP_Post $post Post object.
	 *
	 * @return array Links for the given post.
	 */
	protected function prepare_links($post) {
		$base = sprintf('/%s/%s', $this->namespace, $this->rest_base);

		// Entity meta
		$links = array(
			'self' => array(
				'href' => rest_url(trailingslashit($base) . $post->ID),
			),
			'collection' => array(
				'href' => rest_url($base),
			),
			'about' => array(
				'href' => rest_url('/wp/v2/types/' . $this->post_type),
			),
		);

		if ((in_array($post->post_type, array(
			'post',
			'page',
		)) || post_type_supports($post->post_type, 'author'))
			&& !empty($post->post_author)) {
			$links['author'] = array(
				'href' => rest_url('/wp/v2/users/' . $post->post_author),
				'embeddable' => true,
			);
		};

		if (in_array($post->post_type, array(
			'post',
			'page',
		)) || post_type_supports($post->post_type, 'comments')) {
			$replies_url = rest_url('/wp/v2/comments');
			$replies_url = add_query_arg('post', $post->ID, $replies_url);
			$links['replies'] = array(
				'href' => $replies_url,
				'embeddable' => true,
			);
		}

		if (in_array($post->post_type, array(
			'post',
			'page',
		)) || post_type_supports($post->post_type, 'revisions')) {
			$links['version-history'] = array(
				'href' => rest_url(trailingslashit($base) . $post->ID . '/revisions'),
			);
		}
		$post_type_obj = get_post_type_object($post->post_type);
		if ($post_type_obj->hierarchical && !empty($post->post_parent)) {
			$links['up'] = array(
				'href' => rest_url(trailingslashit($base) . (int) $post->post_parent),
				'embeddable' => true,
			);
		}

		// If we have a featured media, add that.
		if ($featured_media = get_post_thumbnail_id($post->ID)) {
			$image_url = rest_url('wp/v2/media/' . $featured_media);
			$links['https://api.w.org/featuredmedia'] = array(
				'href' => $image_url,
				'embeddable' => true,
			);
		}
		if (!in_array($post->post_type, array('attachment', 'nav_menu_item', 'revision'))) {
			$attachments_url = rest_url('wp/v2/media');
			$attachments_url = add_query_arg('parent', $post->ID, $attachments_url);
			$links['https://api.w.org/attachment'] = array(
				'href' => $attachments_url,
			);
		}

		$taxonomies = get_object_taxonomies($post->post_type);
		if (!empty($taxonomies)) {
			$links['https://api.w.org/term'] = array();

			foreach ($taxonomies as $tax) {
				$taxonomy_obj = get_taxonomy($tax);
				// Skip taxonomies that are not public.
				if (empty($taxonomy_obj->show_in_rest)) {
					continue;
				}

				$tax_base = !empty($taxonomy_obj->rest_base) ? $taxonomy_obj->rest_base : $tax;
				$terms_url = add_query_arg(
					'post',
					$post->ID,
					rest_url('wp/v2/' . $tax_base)
				);

				$links['https://api.w.org/term'][] = array(
					'href' => $terms_url,
					'taxonomy' => $tax,
					'embeddable' => true,
				);
			}
		}

		return $links;
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {

		$schema = array(
			'$schema' => 'http://json-schema.org/draft-04/schema#',
			'title' => $this->post_type,
			'type' => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'date' => array(
					'description' => __("The date the object was published, in the site's timezone."),
					'type' => 'string',
					'format' => 'date-time',
					'context' => array('view', 'edit', 'embed'),
				),
				'date_gmt' => array(
					'description' => __('The date the object was published, as GMT.'),
					'type' => 'string',
					'format' => 'date-time',
					'context' => array('view', 'edit', 'embed'),
				),
				'guid' => array(
					'description' => __('The globally unique identifier for the object.'),
					'type' => 'object',
					'context' => array('view', 'edit'),
					'readonly' => true,
					'properties' => array(
						'raw' => array(
							'description' => __('GUID for the object, as it exists in the database.'),
							'type' => 'string',
							'context' => array('edit'),
						),
						'rendered' => array(
							'description' => __('GUID for the object, transformed for display.'),
							'type' => 'string',
							'context' => array('view', 'edit'),
						),
					),
				),
				'id' => array(
					'description' => __('Unique identifier for the object.'),
					'type' => 'integer',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'link' => array(
					'description' => __('URL to the object.'),
					'type' => 'string',
					'format' => 'uri',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'modified' => array(
					'description' => __("The date the object was last modified, in the site's timezone."),
					'type' => 'string',
					'format' => 'date-time',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'modified_gmt' => array(
					'description' => __('The date the object was last modified, as GMT.'),
					'type' => 'string',
					'format' => 'date-time',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
				'password' => array(
					'description' => __('A password to protect access to the post.'),
					'type' => 'string',
					'context' => array('edit'),
				),
				'slug' => array(
					'description' => __('An alphanumeric identifier for the object unique to its type.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'type' => array(
					'description' => __('Type of Post for the object.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
					'readonly' => true,
				),
			),
		);

		$post_type_obj = get_post_type_object($this->post_type);
		if ($post_type_obj->hierarchical) {
			$schema['properties']['parent'] = array(
				'description' => __('The id for the parent of the object.'),
				'type' => 'integer',
				'context' => array('view', 'edit'),
			);
		}

		$schema['properties']['title'] = array(
			'description' => __('The title for the object.'),
			'type' => 'object',
			'context' => array('view', 'edit', 'embed'),
			'properties' => array(
				'raw' => array(
					'description' => __('Title for the object, as it exists in the database.'),
					'type' => 'string',
					'context' => array('edit'),
				),
				'rendered' => array(
					'description' => __('HTML title for the object, transformed for display.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
			),
		);

		$schema['properties']['content'] = array(
			'description' => __('The content for the object.'),
			'type' => 'object',
			'context' => array('view', 'edit', 'embed'),
			'properties' => array(
				'raw' => array(
					'description' => __('Content for the object, as it exists in the database.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
				'rendered' => array(
					'description' => __('HTML content for the object, transformed for display.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
			),
		);

		$schema['properties']['author'] = array(
			'description' => __('The id for the author of the object.'),
			'type' => 'integer',
			'context' => array('view', 'edit', 'embed'),
		);

		$schema['properties']['featured_media'] = array(
			'description' => __('The id of the featured media for the object.'),
			'type' => 'integer',
			'context' => array('view', 'edit'),
		);

		$schema['properties']['title'] = array(
			'description' => __('The title for the object.'),
			'type' => 'object',
			'context' => array('view', 'edit', 'embed'),
			'properties' => array(
				'raw' => array(
					'description' => __('Title for the object, as it exists in the database.'),
					'type' => 'string',
					'context' => array('view', 'edit'),
				),
				'rendered' => array(
					'description' => __('HTML title for the object, transformed for display.'),
					'type' => 'string',
					'context' => array('view', 'edit', 'embed'),
				),
			),
		);

		$schema['properties']['menu_order'] = array(
			'description' => __('The order of the object in relation to other object of its type.'),
			'type' => 'integer',
			'context' => array('view', 'edit'),
		);

		$schema['properties']['sticky'] = array(
			'description' => __('Whether or not the object should be treated as sticky.'),
			'type' => 'boolean',
			'context' => array('view', 'edit'),
		);

		$schema['properties']['parent'] = array(
			'description' => __('The id for the parent of the object.'),
			'type' => 'integer',
			'context' => array('view', 'edit'),
		);

		$taxonomies = wp_list_filter(get_object_taxonomies($this->post_type, 'objects'), array('show_in_rest' => true));
		foreach ($taxonomies as $taxonomy) {
			$base = !empty($taxonomy->rest_base) ? $taxonomy->rest_base : $taxonomy->name;
			$schema['properties'][$base] = array(
				'description' => sprintf(__('The terms assigned to the object in the %s taxonomy.'), $taxonomy->name),
				'type' => 'array',
				'context' => array('view', 'edit'),
			);
		}

		return $this->add_additional_fields_schema($schema);
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		$params['after'] = array(
			'description' => __('Limit response to resources published after a given ISO8601 compliant date.'),
			'type' => 'string',
			'format' => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description' => __('Limit response to resources published before a given ISO8601 compliant date.'),
			'type' => 'string',
			'format' => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['exclude'] = array(
			'description' => __('Ensure result set excludes specific ids.'),
			'type' => 'array',
			'default' => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description' => __('Limit result set to specific ids.'),
			'type' => 'array',
			'default' => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset'] = array(
			'description' => __('Offset the result set by a specific number of items.'),
			'type' => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['order'] = array(
			'description' => __('Order sort attribute ascending or descending.'),
			'type' => 'string',
			'default' => 'desc',
			'enum' => array('asc', 'desc'),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['orderby'] = array(
			'description' => __('Sort collection by object attribute.'),
			'type' => 'string',
			'default' => 'date',
			'enum' => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'trending',
				'default',
				'activity',
				'popular',
				'items',
				'parent',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['trending'] = array(
			'description' => __('Sort collection by object trending attribute.'),
			'type' => 'string',
			'default' => 'year',
			'enum' => array('day', 'week', 'month', 'year'),
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['orderby']['enum'][] = 'menu_order';

		$params['slug'] = array(
			'description' => __('Limit result set to posts with a specific slug.'),
			'type' => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['status'] = array(
			'default' => 'publish',
			'description' => __('Limit result set to posts assigned a specific status.'),
			'sanitize_callback' => 'sanitize_key',
			'type' => 'string',
			'validate_callback' => array($this, 'validate_user_can_query_private_statuses'),
		);

		if ($this->post_type == "forum") {

			$params['advance_filter'] = array(
				'description' => __('Limit result and show accordingly to given advance filter'),
				'default' => 'none',
				'type' => 'string',
				"enum" => array("none", "no_topics", "no_replies"),
			);
			$params['myforums'] = array(
				'description' => __('Limit response to resources which is current user forms.'),
				'type' => 'boolean',
				'validate_callback' => 'rest_validate_request_arg',
			);
		}

		if ($this->post_type == "topic") {

			$params['term_ids'] = array(
				'description' => __('taxonomy query on term topic'),
				'default' => array(),
				'type' => 'array',
				'sanitize_callback' => 'wp_parse_id_list',
			);

		}

		if ($this->post_type == "topic" || $this->post_type == "reply") {
			$params['filter_fav'] = array(
				'description' => __('Limit result and show accordingly to user favourite.'),
				'type' => 'integer',
				'sanitize_callback' => 'absint',
				'validate_callback' => 'rest_validate_request_arg',
			);
			$params['filter_subscribe'] = array(
				'description' => __('Limit result and show accordingly to user subscribe.'),
				'type' => 'integer',
				'sanitize_callback' => 'absint',
				'validate_callback' => 'rest_validate_request_arg',
			);
			$params['hide_group_items'] = array(
				'description' => __('Limit result and show only non group items.'),
				'type' => 'integer',
				'sanitize_callback' => 'absint',
				'validate_callback' => 'rest_validate_request_arg',
			);
		}

		$post_type_obj = get_post_type_object($this->post_type);
		if ($post_type_obj->hierarchical || 'attachment' === $this->post_type) {
			$params['parent'] = array(
				'description' => __('Limit result set to those of particular parent ids.'),
				'type' => 'array',
				'sanitize_callback' => 'wp_parse_id_list',
				'default' => array(),
			);
			$params['parent_exclude'] = array(
				'description' => __('Limit result set to all items except those of a particular parent ids.'),
				'type' => 'array',
				'sanitize_callback' => 'wp_parse_id_list',
				'default' => array(),
			);
		}

		$params['author'] = array(
			'description' => __('Limit result set to those of particular author ids.'),
			'type' => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default' => array(),
		);

		$params['author_exclude'] = array(
			'description' => __('Limit result set to all items except those of a particular author ids.'),
			'type' => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'default' => array(),
		);

		return $params;
	}

	/**
	 * Validate whether the user can query private statuses
	 *
	 * @param  mixed $value
	 * @param  WP_REST_Request $request
	 * @param  string $parameter
	 *
	 * @return WP_Error|boolean
	 */
	public function validate_user_can_query_private_statuses($value, $request, $parameter) {
		if ('publish' === $value) {
			return true;
		}
		$post_type_obj = get_post_type_object($this->post_type);
		if (current_user_can($post_type_obj->cap->edit_posts)) {
			return true;
		}

		return new WP_Error('rest_forbidden_status', __('Status is forbidden'), array('status' => rest_authorization_required_code()));
	}

}