<?php
namespace BuddyBossApp\Api\BbPress;
use BuddyBossApp\Api\BbPress\PostController;
use WP_Error;
use WP_REST_Request;
use WP_REST_Server;

// NOTE : Old classname was class.boss_bbpress_api_forum_rest. By Ketan, Oct-2019
// (v1 Standard) Contain functionality for required additional rest api endpoints for bbpress.

class ForumRest extends PostController {

	protected $namespace_slug = '';
	protected $namespace = '/bbpress/v1';

	public function __construct($slug) {
		parent::__construct($slug, bbp_get_forum_post_type());

		/** Nothing here */
		$this->namespace_slug = $slug;
		$this->namespace = $this->namespace_slug . $this->namespace;
		$this->hooks();
	}

	public function hooks() {

		add_action('rest_api_init', array($this, "register_routes"), 99);

	}

	public function register_routes() {

		parent::register_routes();

		register_rest_route($this->namespace, '/forum/action/(?P<id>\d+)', array(
			array(
				'methods' => WP_REST_Server::EDITABLE,
				'callback' => array($this, 'do_action'),
				'permission_callback' => '__return_true',
				'args' => array(
					'subscribe' => array(
						'type' => 'integer',
						'enum' => array("1", "0"),
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

		register_rest_route($this->namespace, '/forum/dropdown/', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'dropdown'),
				'permission_callback' => '__return_true',
				'args' => array(
					'page' => array(
						'description' => __('Current page of the collection.'),
						'type' => 'integer',
						'default' => 1,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
						'minimum' => 1,
					),
					'per_page' => array(
						'description' => __('Maximum number of items to be returned in result set.'),
						'type' => 'integer',
						'minimum' => 1,
						'maximum' => 100,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
					'orderby' => array(
						'description' => __('Sort collection by forums attribute.'),
						'type' => 'string',
						'default' => 'date',
						'enum' => array(
							'date',
							'name',
							'id',
							'slug',
							'title',
							'menu_order',
						),
						'validate_callback' => 'rest_validate_request_arg',
					),
					'order' => array(
						'description' => __('Order sort attribute ascending or descending of forums.'),
						'type' => 'string',
						'default' => 'desc',
						'enum' => array('asc', 'desc'),
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
		));

		register_rest_route($this->namespace, '/forum/(?P<id>\d+)/sub_forums', array(
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_sub_forums'),
				'permission_callback' => '__return_true',
				'args' => array(
					'page' => array(
						'description' => __('Current page of the collection.'),
						'type' => 'integer',
						'default' => 1,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
						'minimum' => 1,
					),
					'per_page' => array(
						'description' => __('Maximum number of items to be returned in result set.'),
						'type' => 'integer',
						'default' => get_option('_bbp_forums_per_page', 50),
						'minimum' => 1,
						'maximum' => 100,
						'sanitize_callback' => 'absint',
						'validate_callback' => 'rest_validate_request_arg',
					),
				),
			),
			'schema' => array($this, 'get_public_item_schema'),
		));

	}

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/appboss/bbpress/v1/forum/dropdown Forum dropdown
     * @apiName GetForumDropdown
     * @apiGroup BBPressForum
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Return all items but only title and id.
     * @apiDeprecated  Retrieve forums dropdown list. Check (#Forums:GetBBPForums)
     * @apiUse apidocForForumDropdownV1
     * @apiPrivate
     */
	public function dropdown($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		// query argument add or remove using this fitler.
		$args = apply_filters( 'bbapp_bbpress_forum_dropdown_args', array(
			'posts_per_page' => isset($request['per_page']) ? $request['per_page'] : -1,
			'page' => isset($request['page']) ? $request['page'] : 1,
			'post_type' => bbp_get_forum_post_type(),
			'post_status' => "publish",
			'fields' => 'ids',
			'orderby' => $request['orderby'],
			'order' => $request['order'],
		) );


		if (function_exists('bp_is_active') && bp_is_active('groups')) {
			// Get forums to exclude
			$forum_ids = bbp_exclude_forum_ids('array');
			$args['post__not_in'] = $forum_ids;
		}

		$posts = get_posts($args);

		$data = array();

		foreach ($posts as $post_id) {
			$data[] = array(
				"id" => $post_id,
				"title" => array("rendered" => get_the_title($post_id)),
				"ID" => $post_id, // fallback Support.
				"post_title" => get_the_title($post_id), // fallback Support.
			);
		}

		return rest_ensure_response($data);

	}

	/**
	 * @api {PATCH} /wp-json/appboss/bbpress/v1/forum/action/:id Forum action
	 * @apiName ForumAction
	 * @apiGroup BBPressForum
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Perform different action on forum
	 * @apiDeprecated  Subscribe/Unsubscribe forum for the user. Check (#Forums:GetBBPForumSubscribe)
	 * @apiUse apidocForForumActionV1
	 * @apiPrivate
	 */
	public function do_action($request) {

		$forum_id = (int) $request['id'];
		$success = true;

		if (!is_user_logged_in()) {
			return new WP_Error('rest_cannot_view', __("You don't have permission.", 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		if (isset($request['subscribe']) && !bbp_is_subscriptions_active()) {
			return new WP_Error('bp_json_forum_action_disabled', __('This action is disabled.', 'buddyboss-app'), array('status' => rest_authorization_required_code()));
		}

		// Get required data
		$user_id = bbp_get_user_id(0, true, true);

		// Check for empty forum
		if (empty($forum_id)) {
			return new WP_Error('bp_json_subscription_forum_id', __('No forum found. Which forum are you subscribing/unsubscribing to?', 'buddyboss-app'), array('status' => 400));

			// Check current user's ability to edit the user
		} elseif (!current_user_can('edit_user', $user_id)) {
			return new WP_Error('bp_json_subscription_permissions', __('You don\'t have the permission to edit favorites of that user!', 'buddyboss-app'), array('status' => 403));
		}

		/** No errors *************************************************************/

		// Subscribe user to forum
		if (isset($_REQUEST['subscribe']) && $request['subscribe'] == 1) {
			$is_subscription = bbp_is_user_subscribed($user_id, $forum_id);
			if (!$is_subscription) {

				$success = bbp_add_user_subscription($user_id, $forum_id);

				// Do additional subscriptions actions
				do_action('bbp_subscriptions_handler', $success, $user_id, $forum_id, 'subscribe');
			}
			//$failure = true === $is_open ? __( 'There was a problem unsubscribing from that forum!', boss_bbpress_api_domain( $this->namespace_slug ) ) : __( 'There was a problem subscribing to that forum!', boss_bbpress_api_domain( $this->namespace_slug ) );
		} else if (isset($_REQUEST['subscribe']) && $request['subscribe'] == 0) {
			$is_subscription = bbp_is_user_subscribed($user_id, $forum_id);
			if ($is_subscription) {
				$success = bbp_remove_user_subscription($user_id, $forum_id);

				// Do additional subscriptions actions
				do_action('bbp_subscriptions_handler', $success, $user_id, $forum_id, 'unsubscribe');
			}
			//$failure = true === $is_open ? __( 'There was a problem unsubscribing from that forum!', boss_bbpress_api_domain( $this->namespace_slug ) ) : __( 'There was a problem subscribing to that forum!', boss_bbpress_api_domain( $this->namespace_slug ) );
		}

		// Handle errors
		if (false == $success || is_wp_error($success)) {
			return new WP_Error('bp_json_forum_action_failure', __('Action failed.', 'buddyboss-app'), array('status' => 500));
		}

		return $this->_get_item($forum_id, $request);

	}

	/**
	 * @api {GET} /wp-json/appboss/bbpress/v1/forum/:id/sub_forums Forum sub-forum
	 * @apiName GetSubForum
	 * @apiGroup BBPressForum
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Get sub-forums of particular forum
	 * @apiDeprecated  Retrieve sub-forums. Check (#Forums:GetBBPForums)
	 * @apiUse apidocGetSubForumsV1
	 * @apiPrivate
	 */
	public function get_sub_forums($request) {

		if (!is_user_logged_in()) {
			return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array('status' => rest_authorization_required_code() ));
		}

		$forum_id = (int) $request['id'];

		$sub_forums = bbp_forum_get_subforums(array(
			'post_parent' => $forum_id,
			'paged' => isset($request['page']) ? $request['page'] : 1,
			'posts_per_page' => isset($request['per_page']) ? $request['per_page'] : -1,
		));

		if (empty($sub_forums)) {
			return array();
		}

		$return = array();

		$i = 0;
		foreach ($sub_forums as $sub_forum) {
			$i++; // Separator count

			// Get forum details
			$count = array();
			$permalink = bbp_get_forum_permalink($sub_forum->ID);
			$title = bbp_get_forum_title($sub_forum->ID);
			$slug = $sub_forum->post_name;

			// Show topic count
			$count['topics'] = bbp_get_forum_topic_count($sub_forum->ID);
			// Show reply count
			$count['replies'] = bbp_get_forum_reply_count($sub_forum->ID);

			$return[] = array(
				"id" => $sub_forum->ID,
				"slug" => $slug,
				"title" => $title,
				"count" => $count,
				"link" => $permalink,
				"is_group" => $this->check_group_forum(array('id' => $sub_forum->ID), 'is_group', $request),
			);

		}

		return $return;
	}

	/**
	 * check if forum is group forum
	 *
	 * @param $object
	 * @param $field_name
	 * @param $request
	 *
	 * @return bool
	 */
	public function check_group_forum($object, $field_name, $request) {

		if (function_exists("bbp_is_forum_group_forum") && bbp_is_forum_group_forum($object['id']) && function_exists('groups_get_group')) {
			$group_ids = bbp_get_forum_group_ids($object['id']);
			if (!empty($group_ids)) {
				foreach ($group_ids as $group_id) {
					$group = groups_get_group($group_id);
					return $group;
				}
			}
		}
		return false;
	}

	public function _get_item($forum_id, $request) {

		/**
		 * Redirect to topic endpoint.
		 */
		global $wp_rest_server;

		$request_forum = new WP_REST_Request("GET", "/" . $this->namespace_slug . "/bbpress/v1/forum/{$forum_id}");

		$response = $this->dispatch($request_forum);

		return $wp_rest_server->response_to_data(rest_ensure_response($response), isset($request['_embed']) ? 1 : 0);
	}

	public function _get_items($request) {

		/**
		 * Redirect to topic endpoint.
		 */
		$request = new WP_REST_Request("GET", "/" . $this->namespace_slug . "/bbpress/v1/forum");

		$response = $this->dispatch($request);

		return rest_ensure_response($response);
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param $request
	 *
	 * @return mixed
	 */
	protected function dispatch($request) {

		$query_params = array();

		if (isset($request->get_query_params()["_embed"])) {
			$query_params["_embed"] = $request->get_query_params()["_embed"];
		}

		$request->set_query_params($query_params);

		$server = rest_get_server();
		$response = $server->dispatch($request);

		return $response;
	}

}