<?php

namespace BuddyBossApp\Migration\Auth;

use BuddyBossApp\ManageApp;
use BuddyBossApp\RestErrors;
use \WP_REST_Controller as WP_REST_Controller;
use \WP_Error as WP_Error;

// Contain functionality for required additional rest api endpoints for Authentication.
class Restv2 extends WP_REST_Controller {


	protected $namespace = 'appboss/auth/v2';
	private static $instance;

	public function __construct() {

	}

	/**
	 * Get class instance
	 *
	 * @return Restv2
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'registerRoutes' ), 99 );
	}

	/**
	 * Register API Routes.
	 */
	public function registerRoutes() {
		register_rest_route(
			$this->namespace,
			'/jwt/login',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'rest_login' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'username'     => array(
						'type'              => 'string',
						'required'          => true,
						'description'       => __( 'Username of user wants to authenticate, Email is also valid.' ),
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_user( $param );
						},
					),
					'password'     => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'Password of user wants to authenticate.' ),
					),
					'device_token' => array(
						'type'        => 'string',
						'required'    => false,
						'description' => __( 'Firebase app device token.' ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/access-token',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'rest_access_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'refresh_token' => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'User Refresh Token.' ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/logout',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'rest_logout' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'refresh_token' => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'User Refresh Token.' ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/validate',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'validate_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'token' => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'Require access token or refresh token.' ),
					),
				),
			)
		);

	}

	/**
	 * @param $request
	 * @return WP_Error
     * @apiPrivate
	 * @api {POST} /wp-json/appboss/auth/v2/jwt/login Request token
	 * @apiPrivate true
	 * @apiName RequestToken
	 * @apiGroup Auth
	 * @apiVersion 2.0.0
	 * @apiPermission Public
	 * @apiDescription Get token on authentication
	 * @apiUse apidocForRequestTokenV2
	 */
	public function rest_login( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v2/jwt/login' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );

	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @apiPrivate
	 * @api {POST} /wp-json/appboss/auth/v2/jwt/access-token Regenerate token
	 * @apiName RegenerateAccessToken
	 * @apiGroup Auth
	 * @apiVersion 2.0.0
	 * @apiPermission Public
	 * @apiDescription Regenerate access token
	 * @apiParam device_token Firebase app device token.
	 * @apiParam refresh_token User Refresh Token.
	 */
	public function rest_access_token( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v2/jwt/access-token' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 * @return array|bool|WP_Error
	 * @apiPrivate
	 * @api {POST} /wp-json/appboss/auth/v2/jwt/logout Revoke refresh token
	 * @apiName RevokeRefreshToken
	 * @apiGroup Auth
	 * @apiVersion 2.0.0
	 * @apiPermission Public
	 * @apiDescription Revoke token //TODO : Review permission of this endpoint
	 * @apiParam {String} refresh_token User Refresh Token.
	 */
	public function rest_logout( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v2/jwt/logout' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 * @return WP_Error
	 * @apiPrivate
	 * @api {POST} /wp-json/appboss/auth/v2/jwt/validate Validate token
	 * @apiName ValidateToken
	 * @apiGroup Auth
	 * @apiVersion 2.0.0
	 * @apiPermission Public
	 * @apiDescription Validate if token is valid or not
	 * @apiParam {String} token Access token
	 */
	public function validate_token( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v2/jwt/validate' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );
		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}
}
