<?php

namespace BuddyBossApp\Migration\Auth;

use \WP_REST_Controller as WP_REST_Controller;
use \WP_Error as WP_Error;
use \WP_REST_Request as WP_REST_Request;

// Contain functionality for required additional rest api endpoints for Authentication.
class Rest extends WP_REST_Controller {

	protected $namespace = 'appboss/auth/v1';
	private static $instance;

	public function __construct() {
	}

	/**
	 * Get class instance
	 *
	 * @return Rest
	 */
	public static function instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'registerRoutes' ), 99 );
	}

	/**
	 * Register Quick & Lite Rest Endpoints
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function registerRoutes() {

		register_rest_route(
			$this->namespace,
			'/register',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'user_register_response_v1' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'username'    => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_user( $param );
						},
					),
					'email'       => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return is_email( $param );
						},
					),
					'description' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/forgot',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'user_forgot_pass_v1' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'email' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return is_email( $param );
						},
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/token',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'generate_jwt_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'username' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_user( $param );
						},
					),
					'email'    => array(),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/token',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'delete_jwt_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'accesstoken' => array(),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/token/validate',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'validate_jwt_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(),
			)
		);

		register_rest_route(
			$this->namespace,
			'/verify',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'verify_account' ),
				'permission_callback' => '__return_true',
				'args'                => array(),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/switch',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'app_switch_user' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'app_id'         => array(
						'required'    => true,
						'type'        => 'string',
						'description' => __( 'App ID' ),
					),
					'action'         => array(
						'required'    => true,
						'type'        => 'string',
						'description' => __( 'Switch user action' ),
						'enum'        => array( 'switch', 'restore' ),
					),
					'switch_user_id' => array(
						'required'    => false,
						'type'        => 'string',
						'description' => __( 'User id' ),
					),
				),
			)
		);

	}

	/**
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {POST} /wp-json/appboss/auth/v1/register Register account
	 * @apiName        RegisterAccount
	 * @apiGroup       Auth
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription This endpoint allow user to register a account on WordPress, note on multisite registration are only allowed on main network site.
	 * @apiParam {String} username Username
	 * @apiParam {String} password Password
	 * @apiParam {String} email Email-address
	 * @apiParam {String} [first_name=null] First name
	 * @apiParam {String} [last_name=null] Last name
	 * @apiParam {String} [display_name=null] Display name
	 * @apiParam {String} [description=null] This is user description same like WordPress core has.
	 */
	public function user_register_response_v1( WP_REST_Request $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v1/register' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 *
	 * @return array
     * @apiPrivate
	 * @api            {POST} /wp-json/appboss/auth/v1/forgot Reset password
	 * @apiName        ResetPassword
	 * @apiGroup       Auth
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Reset password
	 * @apiParam {String} user_login User login
	 */
	public function user_forgot_pass_v1( $request ) {

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v1/forgot' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {POST} /wp-json/appboss/auth/v1/jwt/token Request token
	 * @apiName        RequestToken
	 * @apiGroup       Auth
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Get token on authentication
	 * @apiUse         apidocForRequestTokenV1
	 */
	public function generate_jwt_token( $request ) {

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v1/jwt/token' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );

	}

	/**
	 * @param $request
	 *
	 * @return array|bool|WP_Error
     * @apiPrivate
	 * @api            {DELETE} /wp-json/appboss/auth/v1/jwt/token Delete token
	 * @apiName        DeleteToken
	 * @apiGroup       Auth
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription //TODO : Review permission of this endpoint
	 * @apiParam {String} accesstoken Token
	 */
	public function delete_jwt_token( $request ) {

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'DELETE', '/buddyboss-app/auth/v1/jwt/token' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {POST} /wp-json/appboss/auth/v1/jwt/token/validate Validate token
	 * @apiName        ValidateToken
	 * @apiGroup       Auth
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Validate existing token. Can be invalid or expired.
	 * @apiHeader {String} accessToken Auth token
	 */
	public function validate_jwt_token( $request ) {

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v1/jwt/token/validate' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );

	}

	/**
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api            {POST} /wp-json/appboss/auth/v1/verify Verify account
	 * @apiName        VerifyAccount
	 * @apiGroup       Auth
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Verify if the account is valid based on associated/attached email
	 * @apiParam {String} email Valid email of user
	 * @apiParam {Number} code Activation code(_bbapp_activation_code) sent as email
	 */
	public function verify_account( WP_REST_Request $request ) {

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v1/verify' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );

	}

	/**
	 * Switch user rest callback function.
	 *
	 * @param $request
	 *
	 * @return array|bool|mixed|WP_Error|\WP_HTTP_Response|\WP_REST_Response
     * @apiPrivate
	 * @api            {POST} /wp-json/appboss/auth/v1/jwt/switch Request token
	 * @apiName        RequestToken
	 * @apiGroup       Auth
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Get token on authentication     *
	 */
	public function app_switch_user( $request ) {

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/auth/v1/jwt/switch' );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );

	}

	/**
	 * Dispatch the request item.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}

}
