<?php
/**
 * Menu Manager class for handling database operations for app menus.
 *
 * @since   2.4.10
 * @package BuddyBossApp\Menus
 */

namespace BuddyBossApp\Menus;

// Exit if accessed directly.
use BuddyBossApp\Admin\Menus\Menus;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MenuManager class.
 *
 * @since 2.4.10
 */
class MenuManager {

	/**
	 * Class instance.
	 *
	 * @since 2.4.10
	 * @var MenuManager $instance
	 */
	private static $instance;

	/**
	 * Database tables.
	 *
	 * @since 2.4.10
	 * @var array
	 */
	private $tables;


	/**
	 * Current language code.
	 *
	 * @since 2.4.10
	 * @var string
	 */
	public $current_lang_code;

	/**
	 * Get the instance of the class.
	 *
	 * @since 2.4.10
	 *
	 * @return MenuManager
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.4.10
	 */
	public function __construct() {
		global $wpdb;

		$this->tables = array(
			'menus'      => $wpdb->prefix . 'bbapp_menus',
			'menu_items' => $wpdb->prefix . 'bbapp_menu_items',
		);

		$raw_lang_code           = ! empty( $_GET['lang'] ) ? sanitize_text_field( wp_unslash( $_GET['lang'] ) ) : bbapp_get_app_language_header(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		$this->current_lang_code = bbapp_wp_locale_to_app_locale( $raw_lang_code );
	}

	/**
	 * Insert a new menu.
	 *
	 * @param array $menu_data Menu data.
	 *
	 * @since 2.4.10
	 *
	 * @return int|false The number of rows inserted, or false on error.
	 */
	public function insert_menu( $menu_data ) {
		global $wpdb;

		// Validate required fields.
		if ( empty( $menu_data['menu_type'] ) || empty( $menu_data['menu_name'] ) ) {
			return false;
		}

		// Prepare default data.
		$current_time = current_time( 'mysql' );
		$site_id      = get_current_blog_id();

		// Create default array with keys in the same order as format specifiers.
		$defaults_args = array(
			'site_id'       => $site_id,
			'menu_type'     => '',
			'menu_name'     => '',
			'login_state'   => 0,
			'access_groups' => array(),
			'language_code' => $this->current_lang_code,
			'priority'      => 0,
			'data'          => array(),
			'created_at'    => $current_time,
			'updated_at'    => $current_time,
		);

		$data = wp_parse_args( $menu_data, $defaults_args );

		// Handle serialized data if necessary.
		if ( isset( $data['access_groups'] ) && is_array( $data['access_groups'] ) ) {
			$data['access_groups'] = maybe_serialize( $data['access_groups'] );
		}

		if ( isset( $data['data'] ) && is_array( $data['data'] ) ) {
			$data['data'] = maybe_serialize( $data['data'] );
		}

		// Insert into database.
		$result = $wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$this->tables['menus'],
			$data,
			array(
				'%d',     // site_id.
				'%s',     // menu_type.
				'%s',     // menu_name.
				'%d',     // login_state.
				'%s',     // access_groups.
				'%s',     // language_code.
				'%d',     // priority.
				'%s',     // data.
				'%s',     // created_at.
				'%s',     // updated_at.
			)
		);

		if ( $result ) {
			return $wpdb->insert_id;
		}

		return false;
	}

	/**
	 * Update an existing menu.
	 *
	 * @param int   $menu_id   Menu ID.
	 * @param array $menu_data Menu data.
	 *
	 * @since 2.4.10
	 *
	 * @return int|false The number of rows updated, or false on error.
	 */
	public function update_menu( $menu_id, $menu_data ) {
		global $wpdb;

		if ( empty( $menu_id ) ) {
			return false;
		}

		// Make sure menu exists.
		$menu = $this->get_menu( $menu_id );
		if ( ! $menu ) {
			return false;
		}

		// Handle data field specially - merge with existing data rather than replace.
		if ( isset( $menu_data['data'] ) && is_array( $menu_data['data'] ) ) {
			$existing_data = ! empty( $menu->data ) ? $menu->data : array();

			// If existing_data is not already an array, make it one.
			if ( ! is_array( $existing_data ) ) {
				$existing_data = array();
			}

			// Merge existing data with new data, with new data taking precedence.
			$menu_data['data'] = array_merge( $existing_data, $menu_data['data'] );
		}

		// Handle serialized data if necessary.
		if ( isset( $menu_data['access_groups'] ) && is_array( $menu_data['access_groups'] ) ) {
			$menu_data['access_groups'] = maybe_serialize( $menu_data['access_groups'] );
		}

		if ( isset( $menu_data['data'] ) && is_array( $menu_data['data'] ) ) {
			$menu_data['data'] = maybe_serialize( $menu_data['data'] );
		}

		// Always update the updated_at timestamp.
		$menu_data['updated_at'] = current_time( 'mysql' );

		// Update the menu.
		return $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$this->tables['menus'],
			$menu_data,
			array( 'id' => $menu_id ),
			null,
			array( '%d' )
		);
	}

	/**
	 * Delete a menu.
	 *
	 * @param int $menu_id Menu ID.
	 *
	 * @since 2.4.10
	 *
	 * @return int|false The number of rows deleted, or false on error.
	 */
	public function delete_menu( $menu_id ) {
		global $wpdb;

		if ( empty( $menu_id ) ) {
			return false;
		}

		// First delete all menu items associated with this menu.
		$this->delete_menu_items( $menu_id );

		// Then delete the menu.
		$result = $wpdb->delete( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$this->tables['menus'],
			array( 'id' => $menu_id ),
			array( '%d' )
		);

		return $result;
	}

	/**
	 * Get a single menu by ID.
	 *
	 * @param int   $menu_id   Menu ID.
	 * @param array $args      {
	 *                         Optional. Arguments to retrieve menu.
	 *
	 * @type bool   $has_items Whether to include menu items.
	 *                         }
	 *
	 * @since 2.4.10
	 *
	 * @return object|null Menu object or null if not found.
	 */
	public function get_menu( $menu_id, $args = array() ) {
		global $wpdb;

		$default_args = array(
			'has_items' => false,
		);

		$args = wp_parse_args( $args, $default_args );

		if ( empty( $menu_id ) ) {
			return null;
		}

		$menu = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT * FROM {$this->tables['menus']} WHERE id = %d", // phpcs:ignore
				$menu_id
			)
		);

		if ( ! $menu ) {
			return null;
		}

		// Unserialize data.
		if ( ! empty( $menu->access_groups ) ) {
			$menu->access_groups = maybe_unserialize( $menu->access_groups );
		}

		if ( ! empty( $menu->data ) ) {
			$menu->data = maybe_unserialize( $menu->data );
		}

		if ( ! empty( $args['has_items'] ) ) {
			$menu_items_args = array(
				'menu_id'   => $menu->id,
				'menu_type' => $menu->menu_type,
			);
			$menu_items_args = array_merge( $menu_items_args, $args['has_items'] );
			$menu->items     = $this->get_menu_items( $menu_items_args, $menu );
		}

		return $menu;
	}

	/**
	 * Get menus with filters.
	 *
	 * @param array $args            {
	 *                               Optional. Arguments to retrieve menus.
	 *
	 * @type string $menu_type       Type of menu (tab_bar, header_bar, more_menu).
	 * @type int    $site_id         Site ID for multisite.
	 * @type bool   $login_state     Login state (1 for logged in, 0 for logged out).
	 * @type string $language_code   Language code.
	 * @type string $order_by         Order by field.
	 * @type string $order           Order (ASC or DESC).
	 * @type int    $per_page        Number of items per page.
	 * @type int    $page            Page number.
	 * @type bool   $count           Whether to return menu with count of menus.
	 * @type bool   $has_items       Whether to return menus if there are items.
	 * @type bool   $has_items_count Whether to include a count of items with each menu.
	 *                               }
	 * @since 2.4.10
	 *
	 * @return array|int Array of menu objects or count.
	 */
	public function get_menus( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'menu_type'       => '',
			'site_id'         => get_current_blog_id(),
			'login_state'     => '',
			'language_code'   => $this->current_lang_code,
			'order_by'        => 'id',
			'order'           => 'ASC',
			'per_page'        => - 1,
			'page'            => 1,
			'count'           => false,
			'has_items'       => false,
			'has_items_count' => false,
			'default'         => false,
			'logout_menu'     => false,
		);

		$args = wp_parse_args( $args, $defaults );

		// Build query.
		$where  = 'WHERE 1=1'; // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$values = array();

		if ( ! empty( $args['menu_type'] ) ) {
			$where    .= ' AND menu_type = %s'; // phpcs:ignore
			$values[] = $args['menu_type'];
		}

		if ( ! empty( $args['site_id'] ) ) {
			$where    .= ' AND site_id = %d'; // phpcs:ignore
			$values[] = $args['site_id'];
		}

		if ( '' !== $args['login_state'] ) {
			$where    .= ' AND login_state = %d'; // phpcs:ignore
			$values[] = (int) $args['login_state'];
		}

		if ( ! empty( $args['language_code'] ) ) {
			$where    .= ' AND language_code = %s'; // phpcs:ignore
			$values[] = $args['language_code'];
		}

		if ( ! bbapp_is_active( 'access_controls' ) ) {
			$where    .= ' AND access_groups = %s'; // phpcs:ignore
			$values[] = maybe_serialize( array() );
		}

		// Handle ordering.
		$order_by = sanitize_sql_orderby( "{$args['order_by']} {$args['order']}" );
		if ( ! $order_by ) {
			$order_by = 'id ASC';
		}

		// Handle pagination.
		$limit = '';
		if ( $args['per_page'] > 0 ) {
			$offset   = ( $args['page'] - 1 ) * $args['per_page'];
			$limit    = 'LIMIT %d, %d';
			$values[] = $offset;
			$values[] = $args['per_page'];
		}

		// Prepare the full query.
		$query = $wpdb->prepare( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT * FROM {$this->tables['menus']} $where ORDER BY {$order_by} {$limit}",  // phpcs:ignore
			$values
		);

		$menus = $wpdb->get_results( $query ); // phpcs:ignore

		// Unserialize data.
		foreach ( $menus as &$menu ) {
			if ( ! empty( $menu->access_groups ) ) {
				$menu->access_groups = maybe_unserialize( $menu->access_groups );
			}

			if ( ! empty( $menu->data ) ) {
				$menu->data = maybe_unserialize( $menu->data );
			}
		}

		if ( $args['has_items_count'] ) {
			foreach ( $menus as &$menu ) {
				$menu->items_count = $this->get_menu_items_count( $menu->id );
			}
		}

		if ( $args['has_items'] ) {
			foreach ( $menus as &$menu ) {
				$menu_items_args = array( 'menu_id' => $menu->id );
				$menu_items_args = array_merge( $menu_items_args, $args['has_items'] );
				$menu->items     = $this->get_menu_items( $menu_items_args );
			}
		}

		$return_data          = array();
		$return_data['menus'] = $menus;

		if ( $args['count'] ) {
			$return_data['count'] = $this->get_menus_count( $args );
		}

		return $return_data;
	}


	/**
	 * Get count of menus based on filters.
	 *
	 * @param array $args          {
	 *                             Optional. Arguments to filter menus.
	 *
	 * @type string $menu_type     Type of menu (tabbar, headerbar, more).
	 * @type int    $site_id       Site ID for multisite.
	 * @type bool   $login_state   Login state (1 for logged in, 0 for logged out).
	 * @type string $language_code Language code.
	 *                             }
	 * @since 2.4.10
	 *
	 * @return int Number of menus found.
	 */
	public function get_menus_count( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'menu_type'     => '',
			'site_id'       => get_current_blog_id(),
			'login_state'   => '',
			'language_code' => $this->current_lang_code,
		);

		$args = wp_parse_args( $args, $defaults );

		// Build query for counting menus.
		$where  = 'WHERE 1=1';
		$values = array();

		if ( ! empty( $args['menu_type'] ) ) {
			$where    .= ' AND menu_type = %s'; // phpcs:ignore
			$values[] = $args['menu_type'];
		}

		if ( ! empty( $args['site_id'] ) ) {
			$where    .= ' AND site_id = %d'; // phpcs:ignore
			$values[] = $args['site_id'];
		}

		if ( '' !== $args['login_state'] ) {
			$where    .= ' AND login_state = %d'; // phpcs:ignore
			$values[] = (int) $args['login_state'];
		}

		if ( ! empty( $args['language_code'] ) ) {
			$where    .= ' AND language_code = %s'; // phpcs:ignore
			$values[] = $args['language_code'];
		}

		if ( ! bbapp_is_active( 'access_controls' ) ) {
			$where    .= ' AND access_groups = %s'; // phpcs:ignore
			$values[] = maybe_serialize( array() );
		}

		$query = $wpdb->prepare(
			"SELECT COUNT(*) FROM {$this->tables['menus']} {$where}",  // phpcs:ignore
			$values
		);

		return (int) $wpdb->get_var( $query ); // phpcs:ignore
	}

	/**
	 * Insert a new menu item.
	 *
	 * @param array $item_data Menu item data.
	 *
	 * @since 2.4.10
	 *
	 * @return int|false The number of rows inserted, or false on error.
	 */
	public function insert_menu_item( $item_data ) {
		global $wpdb;

		// Validate required fields before proceeding.
		if ( empty( $item_data['menu_id'] ) || empty( $item_data['label'] ) || empty( $item_data['item_type'] ) ) {
			return false;
		}

		// Prepare default data for the item.
		$current_time = current_time( 'mysql' );

		$default_data = array(
			'item_id'    => 0,
			'item_slug'  => isset( $item_data['object'] ) ? $item_data['object'] : $item_data['item_type'],
			'parent_id'  => 0,
			'menu_order' => 0,
			'created_at' => $current_time,
			'updated_at' => $current_time,
		);

		$item_data = wp_parse_args( $item_data, $default_data );

		// Create data array in the exact order needed for the insert statement to work.
		$data = array(
			'menu_id'        => $item_data['menu_id'],
			'label'          => $item_data['label'],
			'item_id'        => $item_data['item_id'],
			'item_type'      => $item_data['item_type'],
			'item_slug'      => $item_data['item_slug'],
			'item_icon_data' => isset( $item_data['item_icon_data'] ) ? $item_data['item_icon_data'] : null,
			'item_data'      => isset( $item_data['item_data'] ) ? $item_data['item_data'] : array(),
			'item_link'      => isset( $item_data['item_link'] ) ? $item_data['item_link'] : null,
			'parent_id'      => isset( $item_data['parent_id'] ) ? $item_data['parent_id'] : 0,
			'menu_order'     => isset( $item_data['menu_order'] ) ? $item_data['menu_order'] : 0,
			'created_at'     => $current_time,
			'updated_at'     => $current_time,
		);

		// Handle serialized data for icon data if necessary.
		if ( isset( $data['item_icon_data'] ) && is_array( $data['item_icon_data'] ) ) {
			$data['item_icon_data'] = maybe_serialize( $data['item_icon_data'] );
		}

		// Insert into database and check for success.
		$result = $wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$this->tables['menu_items'],
			$data,
			array(
				'%d',     // menu_id.
				'%s',     // label.
				'%d',     // item_id.
				'%s',     // item_type.
				'%s',     // item_slug.
				'%s',     // item_icon_data.
				'%s',     // item_data.
				'%s',     // item_link.
				'%d',     // parent_id.
				'%d',     // menu_order.
				'%s',     // created_at.
				'%s',     // updated_at.
			)
		);

		if ( $result ) {
			return $wpdb->insert_id;
		}

		return false;
	}

	/**
	 * Update an existing menu item.
	 *
	 * @param int   $item_id   Menu item ID.
	 * @param array $item_data Menu item data.
	 *
	 * @since 2.4.10
	 *
	 * @return int|false The number of rows updated, or false on error.
	 */
	public function update_menu_item( $item_id, $item_data ) {
		global $wpdb;

		if ( empty( $item_id ) ) {
			return false;
		}

		// Make sure menu item exists before proceeding.
		$item = $this->get_menu_item( $item_id );
		if ( ! $item ) {
			return $this->insert_menu_item( $item_data );
		}

		// Handle serialized data for icon data if necessary.
		if ( isset( $item_data['item_icon_data'] ) && is_array( $item_data['item_icon_data'] ) ) {
			$item_data['item_icon_data'] = maybe_serialize( $item_data['item_icon_data'] );
		}

		// Handle serialized data for icon data if necessary.
		if ( isset( $item_data['item_data'] ) && is_array( $item_data['item_data'] ) ) {
			$item_data['item_data'] = maybe_serialize( $item_data['item_data'] );
		}

		// Always update the updated_at timestamp to the current time.
		$item_data['updated_at'] = current_time( 'mysql' );

		// Update the menu item in the database.
		$result = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$this->tables['menu_items'],
			$item_data,
			array( 'id' => $item_id ),
			null,
			array( '%d' )
		);

		return $result;
	}

	/**
	 * Delete a menu item.
	 *
	 * @param int $item_id Menu item ID.
	 *
	 * @since 2.4.10
	 *
	 * @return int|false The number of rows deleted, or false on error.
	 */
	public function delete_menu_item( $item_id ) {
		global $wpdb;

		if ( empty( $item_id ) ) {
			return false;
		}

		// First get the menu item to check if it's a section or not.
		$item = $this->get_menu_item( $item_id );
		if ( ! $item ) {
			return false;
		}

		// If this is a section, delete all its items first.
		if ( 'section' === $item['type'] ) {
			$this->delete_section_items( $item_id );
		}

		// Then delete the menu item itself.
		$result = $wpdb->delete( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$this->tables['menu_items'],
			array( 'id' => $item_id ),
			array( '%d' )
		);

		return $result;
	}

	/**
	 * Delete all items within a section.
	 *
	 * @param int $section_id Section menu item ID.
	 *
	 * @since 2.4.10
	 *
	 * @return int|false The number of rows deleted, or false on error.
	 */
	public function delete_section_items( $section_id ) {
		global $wpdb;

		if ( empty( $section_id ) ) {
			return false;
		}

		$result = $wpdb->delete( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$this->tables['menu_items'],
			array( 'parent_id' => $section_id ),
			array( '%d' )
		);

		return $result;
	}

	/**
	 * Get a single menu item by ID.
	 *
	 * @param int $item_id Menu item ID.
	 *
	 * @since 2.4.10
	 *
	 * @return array|null Menu item array or null if not found.
	 */
	public function get_menu_item( $item_id ) {
		global $wpdb;

		if ( empty( $item_id ) ) {
			return null;
		}

		$item = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT * FROM {$this->tables['menu_items']} WHERE id = %d", // phpcs:ignore
				$item_id
			)
		);

		if ( ! $item ) {
			return null;
		}

		// Unserialize icon data if it exists.
		if ( ! empty( $item->item_icon_data ) ) {
			$item->item_icon_data = maybe_unserialize( $item->item_icon_data );
		}

		// Always convert to array format.
		return $this->prepare_menu_items( $item );
	}

	/**
	 * Get menu items with optional filtering.
	 *
	 * @param array             $args                  {
	 *                                                 Optional. Arguments to filter menu items.
	 *
	 * @type int|string $menu_id               Menu ID to retrieve items for.
	 * @type string     $item_type             Item type to filter by.
	 * @type int        $parent_id             Parent ID to filter by.
	 * @type string     $menu_type             Menu type (tabbar, headerbar, more) for filters.
	 * @type bool       $check_more_menu_items Whether to check and filter out 'more' type items if the default more menu has no items.
	 * @type bool       $apply_deeplink        Whether to apply deeplink paths
	 * @type string     $fields                Fields to select from the database
	 * @type int|bool   $has_more_menu_items   MEnu id or retrieve.
	 *                                         }
	 * @param null|array|object $menu_obj                 Menu object to filter by.
	 *
	 * @since 2.4.10
	 *
	 * @return array|int Array of menu items in array format.
	 */
	public function get_menu_items( $args = array(), $menu_obj = null ) {
		global $wpdb;
		$no_default_more = false;

		$defaults = array(
			'menu_id'             => 0,
			'item_type'           => '',
			'item_slug'           => '',
			'parent_id'           => 0, // null means don't filter by parent_id.
			'menu_type'           => '',
			'apply_deeplink'      => true, // Whether to apply deeplink paths.
			'fields'              => '*', // Fields to select from the database.
			'has_more_menu_items' => false, // Menu ID to check for 'more' type items.
			'count_only'          => false,
		);

		$args = wp_parse_args( $args, $defaults );

		// Start building the query.
		$query = "SELECT {$args['fields']} FROM {$this->tables['menu_items']} WHERE 1=1";

		// Initialize values array for prepare.
		$values = array();

		// Add menu_id filter if provided.
		if ( ! empty( $args['menu_id'] ) ) {
			$query    .= ' AND menu_id = %s'; // phpcs:ignore
			$values[] = $args['menu_id'];
		}

		// Add item_type filter if provided.
		if ( ! empty( $args['item_type'] ) ) {
			$query    .= ' AND item_type = %s'; // phpcs:ignore
			$values[] = $args['item_type'];
		}

		// Add item_slug filter if provided.
		if ( ! empty( $args['item_slug'] ) ) {
			$query    .= ' AND item_slug = %s'; // phpcs:ignore
			$values[] = $args['item_slug'];
		}

		// Add parent_id filter if provided and not null.
		if ( null !== $args['parent_id'] ) {
			$query    .= ' AND parent_id = %d'; // phpcs:ignore
			$values[] = $args['parent_id'];
		}

		// Check if we need to filter out 'more' type items.
		if ( ! empty( $args['has_more_menu_items'] ) && 'tabbar' === $args['menu_type'] ) {
			// Case 1: Check if there are no more menu items available in the default more menu.
			$more_menu_id = $args['has_more_menu_items'];

			if ( true === $args['has_more_menu_items'] ) {
				$more_menu_id = $this->get_default_menu_id( 'more' );
			}

			// Check if more menu has any items.
			$more_has_items = $this->has_any_menu_items( $more_menu_id, 'more' );

			// Case 2: Check if there's a related headerbar with "more menu position" set to headerbar.
			$menu_id            = null !== $menu_obj ? $menu_obj : $args['menu_id'];
			$headerbar_id       = $this->get_headerbar_menu_from_tabbar_id( $menu_id );
			$more_menu_position = $headerbar_id ? $this->get_more_menu_position( $headerbar_id ) : 'tabbar';

			// If more menu has no items, filter out 'more' type items from results.
			if ( ! $more_has_items || 'headerbar' === $more_menu_position ) {
				$query           .= ' AND item_type != %s'; // phpcs:ignore
				$values[]        = 'more';
				$no_default_more = true;
			}
		}

		// Add order by clause.
		$query .= ' ORDER BY menu_order ASC';

		// Execute the query.
		if ( ! empty( $values ) ) {
			$items = $wpdb->get_results( $wpdb->prepare( $query, $values ) ); // phpcs:ignore
		} else {
			$items = $wpdb->get_results( $query ); // phpcs:ignore
		}

		$prepared_items = array();

		// Check if we're requesting specific fields (not '*') and if items are not empty.
		$is_specific_fields = '*' !== $args['fields'] && ! empty( $items );

		// If requesting specific fields like 'id', return simple array with those fields only.
		if ( $is_specific_fields && false === strpos( $args['fields'], ',' ) && ! preg_match( '/\s+as\s+/i', $args['fields'] ) ) {
			// Handle single field request (e.g. 'id' only).
			$field = trim( $args['fields'] );
			foreach ( $items as $item ) {
				if ( property_exists( $item, $field ) ) {
					$value = $item->$field;
					
					// Cast numeric fields to appropriate types.
					if ( in_array( $field, array( 'id', 'menu_id', 'item_id', 'parent_id', 'menu_order', 'site_id', 'login_state', 'priority' ), true ) ) {
						$value = (int) $value;
					}
					
					$prepared_items[] = $value;
				}
			}

			return $prepared_items;
		}

		// Process the results.
		if ( ! empty( $items ) ) {
			// If requesting count only, return the count without further processing of items.
			if ( true === $args['count_only'] ) {
				return count( $items );
			}

			// If requesting specific fields, return those directly without further processing.
			if ( $is_specific_fields ) {
				return $items;
			}

			// Otherwise continue with full item preparation
			// Unserialize icon data and convert to arrays if necessary.
			foreach ( $items as $key => $item ) {
				if ( ! empty( $item->item_icon_data ) ) {
					$item->item_icon_data = maybe_unserialize( $item->item_icon_data );
				}

				if ( ! empty( $item->item_data ) ) {
					$item->item_data = maybe_unserialize( $item->item_data );
				}

				// Always convert to array format for consistency.
				$prepared_items[ $key ] = $this->prepare_menu_items( $item );
			}
			// Apply filters if menu_type is provided and not empty.
			if ( ! empty( $args['menu_type'] ) ) { // phpcs:ignore
				// Filter items based on dependencies for the given menu_type if class exists.
				$filter_callback = function ( $status, $menu ) use ( $args ) {
					if ( class_exists( '\BuddyBossApp\Admin\Menus\Menus' ) ) {
						return \BuddyBossApp\Admin\Menus\Menus::instance()->filter_app_menu_by_dependency( $status, $menu );
					}

					return $status;
				};

				foreach ( $prepared_items as $key => $item ) {

					$filter_status = apply_filters( 'bbapp_filter_menu_item_' . $args['menu_type'], true, $item );
					$filter_status = $filter_callback( $filter_status, $item );

					if ( ! $filter_status ) {
						unset( $prepared_items[ $key ] );
					}
				}
			}

			// Apply deeplink paths if requested and class exists.
			if ( $args['apply_deeplink'] ) {
				/**
				 * Get all menus.
				 *
				 * @param array $menus All menus.
				 *
				 * @since 1.5.2.1
				 */
				$all_menus = apply_filters( 'bbapp_get_all_menus', array() );

				// Apply deeplink path to each menu item if class exists.
				foreach ( $prepared_items as $key => $menu_item ) {
					if ( class_exists( '\BuddyBossApp\Admin\Menus\Menus' ) ) {
						$admin_menus            = \BuddyBossApp\Admin\Menus\Menus::instance();
						$prepared_items[ $key ] = $admin_menus->bbapp_deeplink_path( $menu_item, $all_menus );
					}
				}
			}

			// Filter out items based on menu_type if class exists.
			$prepared_items = Menus::instance()->exclude_menu_from_menu_list( $prepared_items, $args['menu_type'] );
		}

		if ( in_array( $args['menu_type'], array( 'tabbar', 'headerbar' ), true ) ) {
			if ( false === $no_default_more ) {
				$menu_data = null !== $menu_obj ? $menu_obj : $args['menu_id'];
				if ( ! is_object( $menu_data ) ) {
					$menu_data = $this->get_menu( $menu_data );
				}
				$prepared_items = $this->load_default_more( $prepared_items, $menu_data );
			}

			if ( 'headerbar' === $args['menu_type'] && bbapp_is_buddyboss_platform_enabled() ) {
				$prepared_items = $this->load_default_headerbar_action( $prepared_items );
			}
		}

		return $prepared_items;
	}

	/**
	 * Check if a menu has any items.
	 *
	 * @param int $menu_id Menu ID.
	 *
	 * @since 2.4.10
	 *
	 * @return bool True if the menu has items, false otherwise.
	 */
	public function has_menu_items_by_menu_id( $menu_id ) {
		global $wpdb;

		if ( empty( $menu_id ) ) {
			return false;
		}

		// Check if the menu has any items.
		$query = $wpdb->prepare(
			"SELECT COUNT(*) FROM {$this->tables['menu_items']} WHERE menu_id = %d", // phpcs:ignore
			$menu_id
		);

		return (int) $wpdb->get_var( $query ); // phpcs:ignore
	}

	/**
	 * Prepare menu items by converting objects to the expected array format.
	 *
	 * @param object|array $item Menu item object or array.
	 *
	 * @since 2.4.10
	 *
	 * @return array Prepared menu item array.
	 */
	public function prepare_menu_items( $item ) {
		if ( empty( $item ) ) {
			return array();
		}

		// If already an array, return it while ensuring all expected keys exist and are set.
		if ( is_array( $item ) ) {
			// Make sure 'data' exists and is an array to avoid errors.
			if ( ! isset( $item['data'] ) || ! is_array( $item['data'] ) ) {
				$item['data'] = array();
			}

			// Add required keys if not present and set default values.
			if ( ! isset( $item['type'] ) && isset( $item['item_type'] ) ) {
				$item['type'] = $item['item_type'];
			}

			if ( ! isset( $item['object'] ) ) {
				if ( isset( $item['item_slug'] ) ) {
					$item['object'] = $item['item_slug'];
				} elseif ( isset( $item['item_type'] ) ) {
					$item['object'] = $item['item_type'];
				}
			}

			if ( ! isset( $item['original'] ) && isset( $item['label'] ) ) {
				$item['original'] = $item['label'];
			}

			return $item;
		}

		// Convert object to array if not already an array.
		$menu_item = array(
			'id'         => isset( $item->id ) ? $item->id : 0,
			'menu_id'    => isset( $item->menu_id ) ? $item->menu_id : 0,
			'label'      => isset( $item->label ) ? $item->label : '',
			'item_id'    => isset( $item->item_id ) ? $item->item_id : 0,
			'item_type'  => isset( $item->item_type ) ? $item->item_type : '',
			'item_slug'  => isset( $item->item_slug ) ? $item->item_slug : '',
			'type'       => isset( $item->item_type ) ? $item->item_type : '',
			'object'     => isset( $item->item_slug ) ? $item->item_slug : ( isset( $item->item_type ) ? $item->item_type : '' ),
			'parent_id'  => isset( $item->parent_id ) ? $item->parent_id : 0,
			'menu_order' => isset( $item->menu_order ) ? $item->menu_order : 0,
			'original'   => isset( $item->label ) ? $item->label : '',
			'data'       => array(
				'id'            => isset( $item->item_id ) ? $item->item_id : 0,
				'link'          => isset( $item->item_link ) ? $item->item_link : '',
				'parent'        => isset( $item->parent_id ) ? $item->parent_id : 0,
				'open_external' => false,
				'deeplink_path' => '',
			),
		);

		if ( ! empty( $item->item_data ) ) {
			$menu_item['data']['open_external'] = ! empty( $item->item_data['open_external'] ) ? $item->item_data['open_external'] : false;
			$menu_item['data']['deeplink_path'] = isset( $item->item_data['deeplink_path'] ) ? $item->item_data['deeplink_path'] : '';
		}

		// Convert icon data if present and not empty.
		if ( ! empty( $item->item_icon_data ) ) {
			$menu_item['icon'] = $item->item_icon_data;
		} else {
			// Default icon values to prevent errors in IconPicker if not set.
			$menu_item['icon'] = array(
				'id'         => '',
				'type'       => 'buddyboss',
				'style'      => 'lined',
				'box_style'  => '',
				'color'      => 'default',
				'fill_color' => true,
				'extra'      => array(
					'uniq_id' => uniqid(),
					'img_url' => '',
				),
			);
		}

		// Specific handling for section type items.
		if ( 'section' === $item->item_type ) {
			$menu_item['section'] = $this->get_menu_items_with_childs( $item->id );
		}

		// Specific handling for headerbar-action type items.
		if ( 'headerbar' === $item->item_type ) {
			$menu_item['child_items'] = $this->get_menu_items_with_childs( $item->id );
		}

		return $menu_item;
	}

	/**
	 * Get menu items with their child items.
	 *
	 * @param int $parent_id Parent ID.
	 *
	 * @since 2.4.10
	 *
	 * @return array Array of menu items.
	 */
	public function get_menu_items_with_childs( $parent_id ) {
		$menu_items = $this->get_menu_items( array( 'parent_id' => $parent_id ) );

		if ( empty( $menu_items ) ) {
			return array();
		}

		return $menu_items;
	}

	/**
	 * Get menu by type and login state.
	 *
	 * @param string $menu_type   Menu type.
	 * @param int    $login_state Login state (1 for logged in, 0 for logged out).
	 * @param string $lang_code   Language code.
	 *
	 * @return object|null Menu object or null if not found.
	 *
	 * @todo : check for delete
	 */
	public function get_menu_by_type( $menu_type, $login_state = 1, $lang_code = '' ) {
		global $wpdb;

		if ( empty( $menu_type ) ) {
			return null;
		}

		if ( empty( $lang_code ) ) {
			$lang_code = $this->current_lang_code;
		}

		$menu = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT * FROM {$this->tables['menus']} 
				WHERE menu_type = %s 
				AND login_state = %d 
				AND language_code = %s 
				AND site_id = %d 
				ORDER BY id DESC LIMIT 1",
				$menu_type,
				$login_state,
				$lang_code,
				get_current_blog_id()
			)
		);

		if ( ! $menu ) {
			return null;
		}

		// Unserialize data if it exists.
		if ( ! empty( $menu->access_groups ) ) {
			$menu->access_groups = maybe_unserialize( $menu->access_groups );
		}

		if ( ! empty( $menu->data ) ) {
			$menu->data = maybe_unserialize( $menu->data );
		}

		return $menu;
	}

	/**
	 * Check if there are any more menu items.
	 *
	 * @param string $menu_type Menu type (tabbar, headerbar, more).
	 *
	 * @since 2.4.10
	 *
	 * @return bool True if there are more menu items, false otherwise.
	 */
	public function has_more_menu_bar( $menu_type ) {
		$all_menus = $this->get_menus(
			array(
				'menu_type' => $menu_type,
				'site_id'   => get_current_blog_id(),
			)
		);

		return ! empty( $all_menus['menus'] );
	}

	/**
	 * Get the default menu ID based on menu type and login state.
	 *
	 * @param string $menu_type   Menu type (tabbar, headerbar, more).
	 * @param int    $login_state Login state (1 for logged in, 0 for logged out).
	 * @param string $lang_code   Language code.
	 *
	 * @since 2.4.10
	 * @return int Menu ID or 0 if not found.
	 */
	public function get_default_menu_id( $menu_type, $login_state = 1, $lang_code = '' ) {
		global $wpdb;

		if ( empty( $menu_type ) ) {
			return 0;
		}

		if ( empty( $lang_code ) ) {
			$lang_code = $this->current_lang_code;
		}

		if ( 0 === $login_state ) {
			$menu_name = 'Logged-out';
		} else {
			$menu_name = 'Default';
		}

		$query = "SELECT id FROM {$this->tables['menus']}  WHERE menu_type = %s AND menu_name = %s AND language_code = %s AND site_id = %d";

		$values = array( $menu_type, $menu_name, $lang_code, get_current_blog_id() );

		if ( 0 === $login_state ) {
			$query    .= ' AND login_state = %d'; // phpcs:ignore
			$values[] = $login_state;
		}

		$query .= ' ORDER BY id DESC LIMIT 1';

		$menu_id = $wpdb->get_var( $wpdb->prepare( $query, $values ) ); // phpcs:ignore

		if ( null === $menu_id && str_contains( $_SERVER['REQUEST_URI'], '/wp-json' ) && 'en' !== $lang_code ) { // phpcs:ignore
			$menu_id = $this->get_default_menu_id( $menu_type, $login_state, 'en' );
		}

		return ! empty( $menu_id ) ? $menu_id : 0;
	}

	/**
	 * Get the headerbar menu ID associated with a given tabbar menu ID.
	 *
	 * @param int|object $menu Tabbar menu ID.
	 *
	 * @since 2.4.10
	 * @return int Headerbar menu ID or 0 if not found.
	 */
	public function get_headerbar_menu_from_tabbar_id( $menu ) {
		if ( empty( $menu ) ) {
			return 0;
		}

		if ( ! is_object( $menu ) ) {
			$menu = $this->get_menu( $menu );
		}

		$headerbar_id = ! empty( $menu->data['related_headerbar_id'] ) ? $menu->data['related_headerbar_id'] : 0;

		return $headerbar_id;
	}

	/**
	 * Get the tabbar menu ID associated with a given headerbar menu ID.
	 *
	 * @param int|object $menu Menu ID or object.
	 *
	 * @since 2.4.10
	 * @return int Tabbar menu ID or 0 if not found.
	 */
	public function get_tabbar_menu_from_headerbar_id( $menu ) {
		if ( empty( $menu ) ) {
			return 0;
		}

		if ( ! is_object( $menu ) ) {
			$menu = $this->get_menu( $menu );
		}

		$tabbar_id = ! empty( $menu->data['related_tabbar_id'] ) ? $menu->data['related_tabbar_id'] : 0;

		return $tabbar_id;
	}

	/**
	 * Get the position of the more menu.
	 *
	 * @param int|object $menu Menu ID or object.
	 *
	 * @since 2.4.10
	 * @return string Position of the more menu (tabbar or headerbar).
	 */
	public function get_more_menu_position( $menu ) {
		if ( empty( $menu ) ) {
			return 'tabbar';
		}

		if ( ! is_object( $menu ) ) {
			$menu = $this->get_menu( $menu );
		}

		return ! empty( $menu->data['more_menu_position'] ) ? $menu->data['more_menu_position'] : 'tabbar';
	}

	/**
	 * Get the number of menu items for a given menu ID.
	 *
	 * @param int $menu_id Menu ID.
	 *
	 * @since 2.4.10
	 * @return int Number of menu items.
	 */
	public function get_menu_items_count( $menu_id ) {
		if ( empty( $menu_id ) ) {
			return 0;
		}
		$menu             = $this->get_menu( $menu_id );
		$menu_type        = $menu ? $menu->menu_type : '';
		$menu_items_count = $this->get_menu_items(
			array(
				'menu_id'             => $menu_id,
				'menu_type'           => $menu_type,
				'has_more_menu_items' => true,
				'count_only'          => true,
			)
		);

		return $menu_items_count;
	}

	/**
	 * Get the position of the title in the headerbar.
	 *
	 * @param int $menu_id Menu ID.
	 *
	 * @since 2.4.10
	 * @return string Position of the title (left, center, or right).
	 */
	public function get_title_position( $menu_id ) {
		if ( empty( $menu_id ) ) {
			return 'left';
		}

		$menu = $this->get_menu( $menu_id );

		return ! empty( $menu->data['title_position'] ) ? $menu->data['title_position'] : 'left';
	}

	/**
	 * Get the status of the post input field in the menu.
	 *
	 * @param int $menu_id Menu ID.
	 *
	 * @since 2.4.10
	 * @return bool True if post input field is enabled, false otherwise.
	 */
	public function get_post_input_field_status( $menu_id ) {
		if ( empty( $menu_id ) ) {
			return false;
		}

		$menu = $this->get_menu( $menu_id );

		return ! empty( $menu->data['post_input_field'] ) ? $menu->data['post_input_field'] : false;
	}

	/**
	 * Check if a tabbar item exists.
	 *
	 * @since 2.4.10
	 *
	 * @return bool True if tabbar item exists, false otherwise.
	 */
	public function is_tabbar_item_exist() {
		$menu_id = $this->get_default_menu_id( 'tabbar' );
		if ( empty( $menu_id ) ) {
			return false;
		}

		$menu_items = $this->has_menu_items_by_menu_id( $menu_id );

		return ! empty( $menu_items );
	}

	/**
	 * Check if any menu items exist for a given menu ID.
	 *
	 * @param int    $menu_id   Menu ID.
	 * @param string $menu_type Menu type (tabbar, headerbar, more).
	 *
	 * @since 2.4.10
	 * @return bool True if any menu items exist, false otherwise.
	 */
	public function has_any_menu_items( $menu_id = 0, $menu_type = '' ) {
		if ( empty( $menu_id ) ) {
			$menu_id = $this->get_default_menu_id( $menu_type );
		}

		global $wpdb;
		$menu_items = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$this->tables['menu_items']} WHERE menu_id = %d", $menu_id ) ); // phpcs:ignore

		return $menu_items > 0;
	}

	/**
	 * Get the logged-out menu ID for a given menu type.
	 *
	 * @param string $menu_type Menu type (tabbar, headerbar, more).
	 *
	 * @since 2.4.10
	 * @return int Logged-out menu ID or 0 if not found.
	 */
	public function get_logout_menu_id( $menu_type ) {
		if ( empty( $menu_type ) ) {
			return 0;
		}

		$menu = $this->get_menu_by_type( $menu_type, 0 );

		return $menu ? $menu->id : 0;
	}

	/**
	 * Get the menu name by menu ID.
	 *
	 * @param int $menu_id Menu ID.
	 *
	 * @since 2.4.10
	 * @return string Menu name or empty string if not found.
	 */
	public function get_menu_name_by_menu_id( $menu_id ) {
		global $wpdb;

		if ( empty( $menu_id ) ) {
			return '';
		}

		$menu = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT menu_name FROM {$this->tables['menus']} WHERE id = %d", // phpcs:ignore
				$menu_id
			)
		);

		return $menu ? $menu : '';
	}

	/**
	 * Check if the menu is the default logged-out menu.
	 *
	 * @param int    $nav_id    Menu ID.
	 * @param string $menu_type Menu type (tabbar, headerbar, more).
	 * @param object $nav_menu  Menu object.
	 *
	 * @since 2.4.10
	 * @return bool True if it's the default logged-out menu, false otherwise.
	 */
	public function is_default_logged_out( $nav_id, $menu_type, $nav_menu ) {
		$default_menu_id = $this->get_default_menu_id( $menu_type );

		if ( $default_menu_id === $nav_id ) {
			return true;
		}

		if ( ! empty( $nav_menu ) && '0' === $nav_menu->login_state ) {
			return true;
		}

		return false;
	}

	/**
	 * Create a new menu with improved functionality.
	 *
	 * @param string $menu_type Menu type (tabbar, more, headerbar).
	 * @param array  $menu_data Menu data.
	 *
	 * @since 2.4.10
	 * @return int|false Menu ID or false if creation failed.
	 */
	public function create_menu( $menu_type, $menu_data = array() ) {
		// Prepare default data for the menu.
		$default_data = array(
			'menu_name'     => isset( $menu_data['menu_name'] ) ? $menu_data['menu_name'] : '',
			'login_state'   => isset( $menu_data['login_state'] ) && ( 'logged-out' === $menu_data['login_state'] || 0 === $menu_data['login_state'] ) ? 0 : 1,
			'access_groups' => isset( $menu_data['access_groups'] ) ? $menu_data['access_groups'] : array(),
			'language_code' => isset( $menu_data['language_code'] ) ? $menu_data['language_code'] : $this->current_lang_code,
			'data'          => array(),
		);

		// Set priority based on menu name and login state if not explicitly provided in $menu_data.
		if ( isset( $menu_data['priority'] ) ) {
			$default_data['priority'] = intval( $menu_data['priority'] );
		} else {
			$default_data['priority'] = $this->get_next_menu_priority(
				$menu_type,
				$default_data['menu_name'],
				$default_data['login_state']
			);
		}

		// Prepare data for database insertion.
		$db_data = array(
			'menu_type'     => $menu_type,
			'menu_name'     => $default_data['menu_name'],
			'login_state'   => $default_data['login_state'],
			'access_groups' => $default_data['access_groups'],
			'language_code' => $default_data['language_code'],
			'priority'      => $default_data['priority'],
			'site_id'       => get_current_blog_id(),
		);

		if ( 'headerbar' === $menu_type ) {
			$default_data['data']['title_position']     = isset( $menu_data['title_position'] ) ? $menu_data['title_position'] : 'left';
			$default_data['data']['more_menu_position'] = isset( $menu_data['more_menu_position'] ) ? $menu_data['more_menu_position'] : 'tabbar';
			$default_data['data']['related_tabbar_id']  = isset( $menu_data['tabbar_id'] ) ? $menu_data['tabbar_id'] : '';
		}

		// Set the data field to the serialized data.
		$db_data['data'] = $default_data['data'];

		// Insert the menu into the database.
		$menu_id = $this->insert_menu( $db_data );

		// If this is a tabbar menu, automatically create a corresponding headerbar menu if needed.
		if ( $menu_id && 'tabbar' === $menu_type ) {
			// Create header bar using create_linked_headerbar function if it doesn't exist.
			$headerbar_menu_data = array(
				'title_position'     => 'left',
				'more_menu_position' => 'tabbar',
				'priority'           => $default_data['priority'], // Pass the same priority to headerbar menu.
			);

			$headerbar_id = $this->create_linked_headerbar( $menu_id, $headerbar_menu_data );
		}

		return $menu_id;
	}

	/**
	 * Add a menu item to a menu.
	 *
	 * @param int    $menu_id   Menu ID.
	 * @param string $label     Menu item label.
	 * @param string $item_type Item type.
	 * @param array  $options   Additional options.
	 *
	 * @since 2.4.10
	 * @return int|false Menu item ID or false if creation failed.
	 */
	public function add_menu_item( $menu_id, $label, $item_type, $options = array() ) {
		$item_data = array(
			'menu_id'   => $menu_id,
			'label'     => $label,
			'item_type' => $item_type,
		);

		if ( isset( $options['item_id'] ) ) {
			$item_data['item_id'] = $options['item_id'];
		}

		if ( isset( $options['item_link'] ) ) {
			$item_data['item_link'] = $options['item_link'];
		}

		if ( isset( $options['parent_id'] ) ) {
			$item_data['parent_id'] = $options['parent_id'];
		}

		if ( isset( $options['menu_order'] ) ) {
			$item_data['menu_order'] = $options['menu_order'];
		}

		if ( isset( $options['item_icon_data'] ) ) {
			$item_data['item_icon_data'] = $options['item_icon_data'];
		}

		if ( isset( $options['object'] ) ) {
			$item_data['item_slug'] = $options['object'];
		}

		return $this->insert_menu_item( $item_data );
	}


	/**
	 * Add a section to a menu.
	 *
	 * @param int    $menu_id Menu ID.
	 * @param string $label   Section label.
	 * @param array  $options Additional options.
	 *
	 * @since 2.4.10
	 * @return int|false Section ID or false if creation failed.
	 */
	public function add_section( $menu_id, $label, $options = array() ) {
		return $this->add_menu_item( $menu_id, $label, 'section', $options );
	}

	/**
	 * Check if a menu is a default menu.
	 *
	 * @param int|string $menu_id   Menu ID to check.
	 * @param string     $menu_type Menu type (optional, for more specific check).
	 *
	 * @since 2.4.10
	 *
	 * @return bool True if this is a default menu, false otherwise.
	 */
	public function is_default_menu( $menu_id, $menu_type = '' ) {
		// Check if it's a numeric default menu ID or a string.
		$default_menu_ids = array();

		// Get specific menu type default ID if provided or all default IDs.
		if ( ! empty( $menu_type ) ) {
			$default_menu_ids[] = $this->get_default_menu_id( $menu_type, 1 ); // Logged in.
			$default_menu_ids[] = $this->get_default_menu_id( $menu_type, 0 ); // Logged out.
		} else {
			// Get all default menu IDs for tabbar, more, and headerbar.
			$menu_types = array( 'tabbar', 'more', 'headerbar' );
			foreach ( $menu_types as $type ) {
				$default_menu_ids[] = $this->get_default_menu_id( $type, 1 ); // Logged in.
				$default_menu_ids[] = $this->get_default_menu_id( $type, 0 ); // Logged out.
			}
		}

		return in_array( $menu_id, $default_menu_ids, true );
	}

	/**
	 * Get menu with matching access group ID.
	 *
	 * @param int    $access_group_id Access group ID to search for.
	 * @param string $menu_type       Optional. Menu type to filter by.
	 *
	 * @since 2.4.10
	 * @return object|null Menu object if found, or null if no match.
	 */
	public function get_menu_by_access_group( $access_group_id, $menu_type = '', $lang_code = '' ) {
		global $wpdb;

		if ( empty( $access_group_id ) ) {
			return null;
		}

		// Base query to get menus for the current site.
		$query  = "SELECT * FROM {$this->tables['menus']} WHERE site_id = %d";
		$values = array( get_current_blog_id() );

		// Add menu_type filter if provided.
		if ( ! empty( $menu_type ) ) {
			$query    .= ' AND menu_type = %s'; // phpcs:ignore
			$values[] = $menu_type;
		}

		if ( ! empty( $lang_code ) ) {
			$query    .= ' AND language_code = %s';
			$values[] = $lang_code;
		}

		// Complete the query with access_groups filter.
		$menus = $wpdb->get_results( $wpdb->prepare( $query, $values ) ); // phpcs:ignore

		if ( empty( $menus ) ) {
			return null;
		}

		// Check each menu's access_groups for the access_group_id provided.
		foreach ( $menus as $menu ) {
			// Unserialize access_groups data for checking.
			$access_groups = maybe_unserialize( $menu->access_groups );

			// Check if access_group_id exists in the array of access groups.
			if ( is_array( $access_groups ) && in_array( $access_group_id, $access_groups, true ) ) {
				// Unserialize all data fields before returning the menu.
				$menu->access_groups = $access_groups;

				if ( ! empty( $menu->data ) ) {
					$menu->data = maybe_unserialize( $menu->data );
				}

				return $menu;
			}
		}

		return null;
	}

	/**
	 * Create a linked header bar menu for an existing tab bar.
	 *
	 * @param int   $tabbar_id       Tab bar menu ID.
	 * @param array $additional_data Additional data for the header bar.
	 *
	 * @since 2.4.10
	 * @return int|false Header bar menu ID or false if creation failed.
	 */
	public function create_linked_headerbar( $tabbar_id, $additional_data = array() ) {
		// Make sure the tab bar exists and is of the correct type.
		$tabbar_menu = $this->get_menu( $tabbar_id );
		if ( ! $tabbar_menu || 'tabbar' !== $tabbar_menu->menu_type ) {
			return false;
		}

		// Check if this tab bar already has a linked header bar.
		if ( ! empty( $tabbar_menu->data['related_headerbar_id'] ) ) {
			$existing_headerbar = $this->get_menu( $tabbar_menu->data['related_headerbar_id'] );
			if ( $existing_headerbar ) {
				// Return the existing header bar ID if it exists.
				return $existing_headerbar->id;
			}
		}

		// Create header bar menu data.
		$headerbar_data = array(
			'menu_type'     => 'headerbar',
			'menu_name'     => $tabbar_menu->menu_name,
			'login_state'   => $tabbar_menu->login_state,
			'access_groups' => $tabbar_menu->access_groups,
			'language_code' => isset( $tabbar_menu->language_code ) ? $tabbar_menu->language_code : $this->current_lang_code,
			'priority'      => isset( $additional_data['priority'] ) ? intval( $additional_data['priority'] ) : ( isset( $tabbar_menu->priority ) ? $tabbar_menu->priority : 0 ),
			'site_id'       => $tabbar_menu->site_id,
			'data'          => array(
				'title_position'     => isset( $additional_data['title_position'] ) ? $additional_data['title_position'] : 'left',
				'more_menu_position' => isset( $additional_data['more_menu_position'] ) ? $additional_data['more_menu_position'] : 'tabbar',
				'related_tabbar_id'  => $tabbar_id,
			),
		);

		// Merge any additional data for the data field if provided.
		if ( isset( $additional_data['data'] ) && is_array( $additional_data['data'] ) ) {
			$headerbar_data['data'] = array_merge( $headerbar_data['data'], $additional_data['data'] );
		}

		// Insert the header bar menu into the database.
		$headerbar_id = $this->insert_menu( $headerbar_data );

		if ( $headerbar_id ) {
			// Update the tab bar with the header bar ID if the insertion was successful.
			$this->update_menu(
				$tabbar_id,
				array(
					'data' => array(
						'related_headerbar_id' => $headerbar_id,
					),
				)
			);
		}

		return $headerbar_id;
	}

	/**
	 * Add "more" menu item to the headerbar if it has items.
	 *
	 * @param array|object      $items    Menu items.
	 * @param null|object|array $menu_obj Menu object or array.
	 *
	 * @since 2.4.10
	 *
	 * @return array Updated menu items.
	 */
	public function load_default_more( $items, $menu_obj = null ) {

		// Get the more menu to check if it has items.
		$more_menu_id   = $this->get_default_menu_id( 'more' );
		$more_has_items = $this->has_any_menu_items( $more_menu_id, 'more' );

		if ( 'headerbar' === $menu_obj->menu_type ) {
			if ( 'tabbar' === $menu_obj->data['more_menu_position'] ) {
				$more_has_items = false;
			}
		}

		// Only add "more" item to headerbar if it has actual items.
		if ( $more_has_items ) {
			// Check if "more" menu item already exists in this headerbar menu.
			$more_item_exists = false;
			foreach ( $items as $item ) {
				if ( isset( $item['item_slug'] ) && 'more' === $item['item_slug'] ) {
					$more_item_exists = true;
					break;
				}
			}

			// If the "more" menu item doesn't exist, add it at the end of the menu.
			if ( ! $more_item_exists ) {
				// Create a "more" menu item with a unique ID.
				$items[] = $this->get_default_more_menu();
			}
		}

		return $items;
	}

	/**
	 * Get the default "more" menu item.
	 *
	 * @since 2.4.10
	 *
	 * @return array Default "more" menu item.
	 */
	public function get_default_more_menu() {
		$more_item = array(
			'id'        => uniqid( 'more_' ),
			'item_id'   => 0,
			'item_type' => 'core',
			'item_slug' => 'more',
			'object'    => 'more',
			'type'      => 'core',
			'label'     => __( 'More', 'buddyboss-app' ),
			'icon_type' => get_option( 'more_menu_icon_type', 'avatar' ), // ADD THIS LINE.
			'data'      => array(
				'id'     => 'more',
				'parent' => 0,
			),
			'icon'      => array(),
		);

		// Add icon if needed for the "more" menu item.
		if ( class_exists( '\BuddyBossApp\Common\IconPicker' ) ) {
			$more_item['icon'] = array(
				'id'         => 'grid-large',
				'type'       => 'buddyboss',
				'style'      => 'lined',
				'box_style'  => '',
				'color'      => 'default',
				'fill_color' => true,
				'extra'      => array(
					'uniq_id' => 'more_menu',
					'label'   => __( 'More Menu', 'buddyboss-app' ),
					'img_url' => \BuddyBossApp\Common\IconPicker::instance()->bbapp_get_icon_url( 'ellipsis-h', 'outline', 'none' ),
				),
			);
		}

		return $more_item;
	}

	/**
	 * Add "headerbar" item to the headerbar action menu.
	 *
	 * @param array $items Menu items.
	 *
	 * @since 2.4.10
	 *
	 * @return array Updated menu items.
	 */
	public function load_default_headerbar_action( $items ) {
		// Check if "headerbar" item already exists in this headerbar action menu.
		$headerbar_action_item_exists = false;
		foreach ( $items as $item ) {
			if ( isset( $item['item_slug'] ) && 'headerbar-action' === $item['item_slug'] ) {
				$headerbar_action_item_exists = true;
				break;
			}
		}

		// If the "headerbar" item doesn't exist, add it at the end of the menu.
		if ( ! $headerbar_action_item_exists ) {
			// Create a "headerbar" menu item with a unique ID.
			$headerbar_action_item = $this->headerbar_action_menu_default_items();
			if ( ! empty( $items ) ) {
				array_unshift( $items, $headerbar_action_item );
			} else {
				array_push( $items, $headerbar_action_item );
			}
		}

		return $items;
	}

	/**
	 * Get default items for the headerbar action menu.
	 *
	 * @since 2.4.10
	 *
	 * @return array Array of default items.
	 */
	public function headerbar_action_menu_default_items() {
		$default_items = array(
			'original'    => __( 'Action', 'buddyboss-app' ),
			'id'          => uniqid(),
			'item_slug'   => 'headerbar-action',
			'item_type'   => 'headerbar',
			'type'        => 'headerbar',
			'object'      => 'headerbar-action',
			'label'       => __( 'Action', 'buddyboss-app' ),
			'data'        => array(
				'id'     => 'headerbar',
				'parent' => 0,
			),
			'icon'        => array(),
			'child_items' => array(
				'headerbar-action-create-post' => array(
					'id'        => uniqid(),
					'item_type' => 'headerbar-action-menu',
					'item_slug' => 'headerbar-action-create-post',
					'type'      => 'headerbar-action-menu',
					'object'    => 'headerbar-action-create-post',
					'icon'      => array(
						'id'         => 'pencil',
						'type'       => 'buddyboss',
						'style'      => 'filled',
						'box_style'  => 'round',
						'color'      => 'default',
						'fill_color' => true,
						'extra'      => array(
							'uniq_id'   => uniqid(),
							'label'     => __( 'Create Post Button', 'buddyboss-app' ),
							'item_slug' => 'headerbar-action-create-post',
							'img_url'   => \BuddyBossApp\Common\IconPicker::instance()->bbapp_get_icon_url( 'pencil', 'filled', 'round' ),
						),
					),
				),
				'headerbar-action-add-new'     => array(
					'id'        => uniqid(),
					'item_type' => 'headerbar-action-menu',
					'item_slug' => 'headerbar-action-add-new',
					'type'      => 'headerbar-action-menu',
					'object'    => 'headerbar-action-add-new',
					'icon'      => array(
						'id'         => 'plus',
						'type'       => 'buddyboss',
						'style'      => 'filled',
						'box_style'  => 'round',
						'color'      => 'default',
						'fill_color' => true,
						'extra'      => array(
							'uniq_id'   => uniqid(),
							'label'     => __( 'Add New Button', 'buddyboss-app' ),
							'item_slug' => 'headerbar-action-add-new',
							'img_url'   => \BuddyBossApp\Common\IconPicker::instance()->bbapp_get_icon_url( 'plus', 'filled', 'round' ),

						),
					),
				),
				'headerbar-action-new-message' => array(
					'id'        => uniqid(),
					'item_type' => 'headerbar-action-menu',
					'item_slug' => 'headerbar-action-new-message',
					'type'      => 'headerbar-action-menu',
					'object'    => 'headerbar-action-new-message',
					'icon'      => array(
						'id'         => 'envelope',
						'type'       => 'buddyboss',
						'style'      => 'filled',
						'box_style'  => 'round',
						'color'      => 'default',
						'fill_color' => true,
						'extra'      => array(
							'uniq_id'   => uniqid(),
							'label'     => __( 'New Message Button', 'buddyboss-app' ),
							'item_slug' => 'headerbar-action-new-message',
							'img_url'   => \BuddyBossApp\Common\IconPicker::instance()->bbapp_get_icon_url( 'envelope', 'filled', 'round' ),
						),
					),
				),
				'headerbar-action-option'      => array(
					'id'        => uniqid(),
					'item_type' => 'headerbar-action-menu',
					'item_slug' => 'headerbar-action-option',
					'type'      => 'headerbar-action-menu',
					'object'    => 'headerbar-action-option',
					'icon'      => array(
						'id'         => 'ellipsis-h',
						'type'       => 'buddyboss',
						'style'      => 'filled',
						'box_style'  => 'round',
						'color'      => 'default',
						'fill_color' => true,
						'extra'      => array(
							'uniq_id'   => uniqid(),
							'item_slug' => 'headerbar-action-option',
							'label'     => __( 'Option Button', 'buddyboss-app' ),
							'img_url'   => \BuddyBossApp\Common\IconPicker::instance()->bbapp_get_icon_url( 'ellipsis-h', 'filled', 'round' ),
						),
					),
				),
			),
		);

		return $default_items;
	}

	/**
	 * Check if a menu name matches the default menu name pattern.
	 *
	 * @param string $menu_name The menu name to check.
	 *
	 * @since 2.4.10
	 * @return bool True if it's a default menu name, false otherwise.
	 */
	public function is_default_menu_name( $menu_name ) {
		$default_patterns = array(
			'Default',
			'Logged-out',
		);

		foreach ( $default_patterns as $pattern ) {
			if ( strcasecmp( $menu_name, $pattern ) === 0 ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Get only custom menus (excluding default and logged-out menus).
	 *
	 * @param string $menu_type   Menu type (tabbar, headerbar, more_menu).
	 * @param string $login_state Optional. Login state (1 for logged in, 0 for logged out).
	 *
	 * @since 2.4.10
	 * @return array Array of custom menu objects.
	 */
	public function get_custom_menus( $menu_type, $login_state = '1' ) {
		$all_menus = $this->get_menus(
			array(
				'menu_type'   => $menu_type,
				'login_state' => $login_state,
			)
		);

		$custom_menus = array();

		if ( ! empty( $all_menus['menus'] ) ) {
			foreach ( $all_menus['menus'] as $menu ) {
				// Skip default menus based on menu name patterns.
				if ( is_object( $menu ) && isset( $menu->menu_name ) && ! $this->is_default_menu_name( $menu->menu_name ) ) {
					$custom_menus[] = $menu;
				}
			}
		}

		return $custom_menus;
	}

	/**
	 * Check if any custom menu exists for the given menu type.
	 *
	 * @param string $menu_type Menu type (tabbar, headerbar, more_menu).
	 *
	 * @since 2.4.10
	 * @return bool True if custom menu exists, false otherwise.
	 */
	public function is_custom_menu_exists( $menu_type ) {
		$custom_menus = $this->get_custom_menus( $menu_type, '1' );

		return ! empty( $custom_menus );
	}

	/**
	 * Get the tutorial link for a specific menu type.
	 *
	 * @param string $menu_type Menu type (tabbar, headerbar, more).
	 *
	 * @since 2.4.10
	 * @return string Tutorial link URL.
	 */
	public function get_menu_tutorial_link( $menu_type ) {
		$menu_tutorial_link = admin_url( 'admin.php?page=bbapp-help&article=125672' );

		if ( 'headerbar' === $menu_type ) {
			$menu_tutorial_link = admin_url( 'admin.php?page=bbapp-help&article=127649' );
		}

		if ( 'tabbar' === $menu_type || 'more' === $menu_type ) {
			$menu_tutorial_link = admin_url( 'admin.php?page=bbapp-help&article=121776' );
		}

		return $menu_tutorial_link;
	}

	/**
	 * Check if default menus exist for the given menu type and login state.
	 *
	 * @param string $menu_type     Menu type (tabbar, headerbar, more).
	 * @param int    $login_state   Login state (1 for logged in, 0 for logged out).
	 * @param string $language_code Language code (optional).
	 *
	 * @since 2.4.10
	 * @return bool True if default menu exists, false otherwise.
	 */
	public function default_menu_exists( $menu_type, $login_state = 1, $language_code = 'en' ) {
		$menu_id = $this->get_default_menu_id( $menu_type, $login_state, $language_code );

		return ! empty( $menu_id );
	}

	/**
	 * Delete all menu items for a specific menu.
	 *
	 * @param int $menu_id Menu ID.
	 *
	 * @since 2.4.10
	 *
	 * @return bool True on success, false on error.
	 */
	public function delete_menu_items( $menu_id ) {
		global $wpdb;

		if ( empty( $menu_id ) ) {
			return false;
		}

		$table = $this->tables['menu_items'];

		$result = $wpdb->delete( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$table,
			array( 'menu_id' => $menu_id ),
			array( '%d' )
		);

		return false !== $result;
	}

	/**
	 * Check if a setting menu exists.
	 *
	 * @param int $menu_id Menu ID.
	 *
	 * @since 2.4.10
	 *
	 * @return bool True if setting menu exists, false otherwise.
	 */
	public function setting_menu_exists( $menu_id ) {
		if ( empty( $menu_id ) ) {
			return false;
		}

		$menu_items = $this->get_menu_items( $menu_id );

		if ( empty( $menu_items ) ) {
			return false;
		}

		foreach ( $menu_items as $item ) {
			if ( 'settings' === $item['item_slug'] ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if more menu can be added.
	 *
	 * @since 2.4.10
	 *
	 * @return bool True if more menu can be added, false otherwise.
	 */
	public function can_more_menu_add() {
		$can_more_enable = false;

		$all_menus = $this->get_menus(
			array(
				'menu_type' => 'more',
			)
		);

		if ( ! empty( $all_menus['menus'] ) ) {
			foreach ( $all_menus['menus'] as $nav_menu ) {
				if ( $this->has_menu_items_by_menu_id( $nav_menu->id ) ) {
					$can_more_enable = true;
				}
			}
		}

		return $can_more_enable;
	}

	/**
	 * Update menu priorities in bulk
	 *
	 * @param string $menu_type Menu type (tabbar, headerbar, more).
	 * @param array  $criteria  Array of menu priority data with menu_id and priority pairs.
	 *
	 * @since 2.4.10
	 * @return boolean True if any priorities were updated, false otherwise
	 */
	public function update_menu_priority( $menu_type, $criteria ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_menus';
		$updated    = false;

		// Validate input data.
		if ( empty( $menu_type ) || empty( $criteria ) || ! is_array( $criteria ) ) {
			return false;
		}

		// Update each menu with its new priority if provided.
		foreach ( $criteria as $menu_data ) {
			if ( ! isset( $menu_data['menu_id'] ) || ! isset( $menu_data['priority'] ) ) {
				continue;
			}

			$menu_id  = intval( $menu_data['menu_id'] );
			$priority = intval( $menu_data['priority'] );

			// Update the menu priority in the database.
			$result = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
				$table_name,
				array( 'priority' => $priority ),
				array( 'id' => $menu_id ),
				array( '%d' ),
				array( '%d' )
			);

			if ( false !== $result ) {
				$updated = true;
			}
		}

		// If we're updating priorities for a tabbar menu, also update any linked headerbar menus with the same priority.
		if ( $updated && 'tabbar' === $menu_type ) {
			foreach ( $criteria as $menu_data ) {
				if ( ! isset( $menu_data['menu_id'] ) || ! isset( $menu_data['priority'] ) ) {
					continue;
				}

				$tabbar_id = intval( $menu_data['menu_id'] );
				$priority  = intval( $menu_data['priority'] );

				// Get the linked headerbar for this tabbar menu if it exists.
				$headerbar_id = $this->get_headerbar_menu_from_tabbar_id( $tabbar_id );

				if ( $headerbar_id ) {
					// Update the headerbar menu with the same priority as the tabbar menu.
					$wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
						$table_name,
						array( 'priority' => $priority ),
						array( 'id' => $headerbar_id ),
						array( '%d' ),
						array( '%d' )
					);
				}
			}
		}

		return $updated;
	}

	/**
	 * Get the next available priority for a custom menu.
	 *
	 * This assigns priorities incrementally (0, 1, 2...) to custom menus
	 * while preserving special priorities for Default (9999) and Logged-out (9998) menus.
	 *
	 * @param string $menu_type   Menu type (tabbar, headerbar, more).
	 * @param string $menu_name   Menu name.
	 * @param int    $login_state Login state (1 for logged in, 0 for logged out).
	 *
	 * @since 2.4.10
	 * @return int Next available priority.
	 */
	public function get_next_menu_priority( $menu_type, $menu_name, $login_state = 1 ) {
		global $wpdb;

		// Default menu has highest priority (9999).
		if ( 'Default' === $menu_name ) {
			return 9999;
		}

		// Logged-out menu has second highest priority (9998).
		if ( 'Logged-out' === $menu_name || '0' === $login_state ) {
			return 9998;
		}

		// For custom menus, find the highest existing priority below 9998.
		$query = $wpdb->prepare(
			"SELECT MAX(priority) FROM {$this->tables['menus']} 
			WHERE menu_type = %s AND priority < 9998 AND site_id = %d",
			$menu_type,
			get_current_blog_id()
		);

		$max_priority = $wpdb->get_var( $query ); // phpcs:ignore

		// Start with priority 0 if no custom menus exist yet or if the max priority is null.
		if ( null === $max_priority ) {
			return 0;
		}

		// Otherwise, increment the highest priority by 1 to get the next available priority.
		return intval( $max_priority ) + 1;
	}

	/**
	 * Get menu item by slug for a specific menu
	 *
	 * @param int    $menu_id   Menu ID to check.
	 * @param string $item_slug Item slug to look for.
	 *
	 * @since 2.4.10
	 * @return int|false Menu item ID if found, false otherwise.
	 */
	public function get_menu_item_id_by_slug( $menu_id, $item_slug ) {
		global $wpdb;

		if ( empty( $menu_id ) || empty( $item_slug ) ) {
			return false;
		}

		$item_id = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT id FROM {$this->tables['menu_items']} WHERE menu_id = %d AND item_slug = %s LIMIT 1", // phpcs:ignore
				$menu_id,
				$item_slug
			)
		);

		return $item_id ? absint( $item_id ) : false;
	}

	/**
	 * Get active languages with menu items count for the currently selected menu name.
	 *
	 * @param array  $active_languages Array of active languages.
	 * @param string $menu_type        Menu type (tabbar, headerbar, more).
	 * @param int    $current_menu_id  Current menu ID to match by name.
	 *
	 * @since 2.4.10
	 * @return array Active languages with menu items count.
	 */
	public function get_active_languages_with_menu_items_count( $active_languages, $menu_type, $current_menu_id = 0 ) {
		global $wpdb;

		// If we have a current menu ID, get its name to find equivalent menus in other languages.
		$current_menu_name = '';
		if ( ! empty( $current_menu_id ) ) {
			$current_menu = $this->get_menu( $current_menu_id );
			if ( $current_menu && isset( $current_menu->menu_name ) ) {
				$current_menu_name = $current_menu->menu_name;
			}
		}

		foreach ( $active_languages as $lang_code => $lang_name ) {
			$menu_id = 0;

			// If we have a current menu name, find the menu with the same name in this language and type.
			if ( ! empty( $current_menu_name ) ) {
				// Get the equivalent menu ID for this language and type.
				$menu_id = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
					$wpdb->prepare(
						"SELECT id FROM {$this->tables['menus']} 
					WHERE language_code = %s 
					AND menu_type = %s 
					AND site_id = %d
					AND menu_name = %s
					LIMIT 1",
						$lang_code,
						$menu_type,
						get_current_blog_id(),
						$current_menu_name
					)
				);
			}

			if ( empty( $menu_id ) ) {
				// If no current menu is specified or couldn't find one with the same name, use the default menu ID.
				$menu_id = $this->get_default_menu_id( $menu_type, 1, $lang_code );
			}

			$items_count = 0;

			// If menu exists in the database, count its items using the existing function to get the count.
			if ( ! empty( $menu_id ) ) {
				// Get menu object to check default items for headerbar and more menus.
				$menu_obj = $this->get_menu( $menu_id );

				// Get actual database item count - only parent items for headerbar and more menus to avoid duplicates.
				if ( in_array( $menu_type, array( 'headerbar' ), true ) ) {
					$db_items_count = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
						$wpdb->prepare(
							"SELECT COUNT(*) FROM {$this->tables['menu_items']} WHERE menu_id = %d AND parent_id = 0", // phpcs:ignore
							$menu_id
						)
					);
				} else {
					$db_items_count = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
						$wpdb->prepare(
							"SELECT COUNT(*) FROM {$this->tables['menu_items']} WHERE menu_id = %d", // phpcs:ignore
							$menu_id
						)
					);
				}

				// If there are no database items, check for default items for headerbar and more menus.
				if ( empty( $db_items_count ) && ! empty( $menu_obj ) ) {
					// For headerbar, we always add the action menu (only count as 1 parent item).
					if ( 'headerbar' === $menu_type ) {
						$items_count = 1; // Only count the parent action menu, not its children.
					} elseif ( 'tabbar' === $menu_type ) {
						// Only add the "more" item to the count if the corresponding more menu has items in it.
						$more_menu_id = $this->get_default_menu_id( 'more', 1, $lang_code );
						if ( $more_menu_id ) {
							// Check if more menu has any items in it.
							$more_has_items = $this->has_any_menu_items( $more_menu_id, 'more' );
							$items_count    = $more_has_items ? 1 : 0; // Add "more" item only if it has content in the more menu.
						} else {
							$items_count = 0; // No default more menu exists for this language.
						}
					} elseif ( 'more' === $menu_type ) {
						$items_count = 0;
					}
				} elseif ( in_array( $menu_type, array( 'headerbar', 'more' ), true ) ) {
					$items_count = $db_items_count;
				} else {
					// For other menu types, use the regular item count function to get the total count.
					$items_count = $this->get_menu_items_count( $menu_id );
				}
			}

			// Make sure we always store a numeric value.
			$active_languages[ $lang_code ]['items_count'] = (int) $items_count;
		}

		return $active_languages;
	}
}
