<?php
/**
 * Holds abstract class functionality for menus types.
 *
 * @package BuddyBossApp\Menus\Types
 */

namespace BuddyBossApp\Menus\Types;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class CoreAbstract
 *
 * @package BuddyBossApp\Menus\Types
 */
abstract class CoreAbstract {

	/**
	 * Class instance.
	 *
	 * @var bool
	 */
	private static $instances = array();

	/**
	 * Integration Menu Type.
	 *
	 * @var bool
	 */
	public $screen_group = 'custom_screen';

	/**
	 * Menu deeplink base.
	 *
	 * @var string
	 */
	public $deeplink_base = 'screen';

	/**
	 * Menu external link.
	 *
	 * @var bool $external_link
	 */
	public $external_link = false;

	/**
	 * Register metabox for screen.
	 *
	 * @var bool
	 */
	private $register_metabox = array();

	/**
	 * Menu metabox callback.
	 *
	 * @var bool
	 */
	private $menu_metabox_callback = array();

	/**
	 * Menu result callback.
	 *
	 * @var bool
	 */
	private $menu_result_callback = array();

	/**
	 * Menus.
	 *
	 * @var array
	 */
	public $menus = array();

	/**
	 * CoreAbstract constructor.
	 */
	public function __construct() {
		/** Nothing here */
		add_filter( 'bbapp_register_menu_metaboxes', array( $this, 'register_menu_metabox' ), 10 );
		add_filter( 'bbapp_get_menu_icon', array( $this, 'menu_icon' ), 10, 3 );
		add_action( 'bbapp_menu_metabox_render', array( $this, 'metabox_render' ), 10, 3 );
		add_filter( 'bbapp_get_menu_result', array( $this, 'get_menu_result' ), 10, 3 );
		add_filter( 'bbapp_get_all_menus', array( $this, 'get_all_menus' ), 10, 1 );
		add_filter( 'bbapp_menu_disallow_external_link', array( $this, 'disallow_screen_external_link' ), 11, 1 );
	}

	/**
	 * Class instance called.
	 *
	 * @return mixed
	 */
	public static function instance() {
		$class = get_called_class();
		if ( ! isset( self::$instances[ $class ] ) ) {
			self::$instances[ $class ] = new $class();
			self::$instances[ $class ]->app_menu_setup();
		}

		return self::$instances[ $class ];
	}

	/**
	 * Setup method.
	 */
	public function app_menu_setup() {
		if ( method_exists( $this, 'setup' ) ) {
			$this->setup();
		}
	}

	/**
	 * Register App menu metabox.
	 *
	 * @param string $screen_group          Type of menu for metabox.
	 * @param string $title                 Title of metabox.
	 * @param array  $args                  Argument for metabox.
	 *                                      array(
	 *                                      $callback       callable    Callback function.
	 *                                      $desc           string      Description of metabox.
	 *                                      $is_hide        bool        Metabox menu list hide show.
	 *                                      $deeplink_base  string      Argument for menu deeplink.
	 *                                      $screen_group   string      Type of menu for metabox.
	 *                                      ).
	 * @param bool   $logged_in             Whether user is loggedin.
	 */
	public function register_screen_group( $screen_group, $title, $args = array(), $logged_in = false ) {

		// If post type then we don't need to pass default callback args.
		$this->screen_group = $screen_group;
		if ( is_array( $args ) ) {
			$args_default = array(
				'callback'      => '',
				'desc'          => '',
				'is_hide'       => false,
				'deeplink_base' => $this->deeplink_base,
				'screen_group'  => $this->screen_group,
				'external_link' => false,
			);

			$args = wp_parse_args( $args, $args_default );
		}
		$callback = $args['callback'];
		unset( $args['callback'] ); // Unset not necessary callback argument.

		$this->register_metabox[ $this->screen_group ] = array(
			'title'     => $title,
			'callback'  => $callback,
			'args'      => isset( $args[0] ) ? $args[0] : $args, // $args[0] for app page, wordpress pages.
			'logged_in' => $logged_in,
		);

		$external_link       = ! empty( $args['external_link'] ) ? $args['external_link'] : false;
		$this->external_link = ( isset( $args[0] ) && ! empty( $args[0]->external_link ) ) ? $args[0]->external_link : $external_link;
	}

	/**
	 * Register Metabox.
	 *
	 * @param array $register_metaboxes Metabox list.
	 *
	 * @return array|mixed
	 */
	public function register_menu_metabox( $register_metaboxes ) {
		if ( ! empty( $this->register_metabox ) ) {
			$register_metaboxes = array_merge( $register_metaboxes, $this->register_metabox );
		}

		return $register_metaboxes;
	}

	/**
	 * Register menu.
	 *
	 * @param string       $name                  Menu title (eg. setting,profile/view).
	 * @param string       $label                 Menu label.
	 * @param string|array $default_icon          Name of the icon.
	 * @param array        $settings              Default settings.
	 *                                            array('icon_monochrome_checkbox' => 'yes',
	 *                                            'monochrome_option'        => 'default|custom',
	 *                                            'icon_monochrome_color'    => '#fff').
	 * @param null         $menu_metabox_callback Menu metabox.
	 * @param null         $menu_result_callback  Menu callback.
	 * @param bool         $logged_in             Whether user is loggedin.
	 */
	public function register_screen( $name, $label, $default_icon, $settings = array(), $menu_metabox_callback = null, $menu_result_callback = null, $logged_in = false ) {

		$defaults = array(
			'id'   => uniqid(),
			'data' => array(),
		);

		// Default setting added for manage deeplinking slug.
		$default_setting = array( 'deeplink_slug' => $name );
		$settings        = wp_parse_args( $settings, $default_setting );
		$deeplink_slug   = $settings['deeplink_slug'];
		unset( $settings['deeplink_slug'] ); // Unset not necessary deeplink_slug argument.

		if ( is_string( $default_icon ) ) {

			// Catch the legacy icon in string type.
			if ( strpos( $default_icon, 'bbapp/' ) !== false ) {
				$icon = array(
					'type' => 'legacy',
					'id'   => explode( '/', $default_icon )[1],
				);
			} elseif ( strpos( $default_icon, 'custom/' ) !== false ) {
				$icon = array(
					'type' => 'custom',
					'id'   => explode( '/', $default_icon )[1],
				);
			} else {
				$icon = array(
					'type' => 'buddyboss',
					'id'   => $default_icon,

				);
			}

			if ( ! empty( $settings['icon_monochrome_checkbox'] ) ) {
				$settings['fill_color'] = 'yes' === $settings['icon_monochrome_checkbox'];
			}

			$icon['color']      = 'default';
			$icon['fill_color'] = ! empty( $settings['fill_color'] ) ? $settings['fill_color'] : true;

		} else {
			$icon = $default_icon;

			if ( ! empty( $settings['icon_monochrome_checkbox'] ) ) {
				$settings['fill_color'] = 'yes' === $settings['icon_monochrome_checkbox'];
			}
			if ( ! isset( $icon['color'] ) ) {
				$icon['color'] = 'default';
			}
			if ( ! isset( $icon['fill_color'] ) ) {
				$icon['fill_color'] = ! empty( $settings['fill_color'] ) ? $settings['fill_color'] : true;
			}
		}

		$args = array(
			'object'    => $name,
			'slug'      => $deeplink_slug,
			'label'     => $label,
			'original'  => $label,
			'group'     => $this->screen_group,
			'icon'      => $icon,
			'data'      => array(
				'id'     => $name,
				'parent' => $name,
			),
			'logged_in' => $logged_in,
		);

		$args         = wp_parse_args( $args, $defaults );
		$args['type'] = $args['group'];

		$this->menus[ $this->screen_group ][ $name ] = $args;

		$this->menu_metabox_callback[ $this->screen_group ][ $name ] = $menu_metabox_callback;
		$this->menu_result_callback[ $this->screen_group ][ $name ]  = $menu_result_callback;
	}

	/**
	 * Returns the registered menu list.
	 *
	 * @return array|mixed
	 */
	public function get_menus() {

		if ( ! isset( $this->menus[ $this->screen_group ] ) || ! is_array( $this->menus[ $this->screen_group ] ) ) {
			$this->menus[ $this->screen_group ] = array();
		}

		return $this->menus[ $this->screen_group ];
	}

	/**
	 * Get all menus.
	 *
	 * @param array $all_menus Array of all menus.
	 *
	 * @return array
	 */
	public function get_all_menus( $all_menus ) {
		// Same type with different class support added.
		if ( isset( $all_menus[ $this->screen_group ] ) && isset( $this->menus[ $this->screen_group ] ) ) {
			$all_menus[ $this->screen_group ] = array_merge( $all_menus[ $this->screen_group ], $this->menus[ $this->screen_group ] );

			return $all_menus;
		}

		return array_merge( $all_menus, $this->menus );
	}

	/**
	 * Update default menu icon.
	 *
	 * @param array  $icon_defaults Icon data.
	 * @param string $item_name     Item name.
	 * @param string $type          Item type.
	 *
	 * @return mixed|string[]
	 */
	public function menu_icon( $icon_defaults, $item_name, $type ) {
		if ( isset( $this->menus[ $this->screen_group ][ $item_name ] ) && $this->menus[ $this->screen_group ][ $item_name ]['object'] === $item_name && $this->menus[ $this->screen_group ][ $item_name ]['type'] === $type ) {
			$icon_defaults = $this->menus[ $this->screen_group ][ $item_name ]['icon'];
		}

		return $icon_defaults;
	}

	/**
	 * Add metabox for menu.
	 *
	 * @param string       $menu_id Menu unique id.
	 * @param array|string $menu    Menu.
	 */
	public static function metabox( $menu_id, $menu ) {
	}

	/**
	 * Result for menu response.
	 *
	 * @param array|string $menu    Menu.
	 * @param array|object $request request data.
	 *
	 * @return mixed
	 */
	public static function get_results( $menu, $request ) {
		unset( $menu['settings'] );

		return $menu;
	}

	/**
	 * Metabox callback
	 *
	 * @param string $menu_id   Menu unique id.
	 * @param array  $menu      Menu.
	 * @param string $menu_name Menu object name like. post_type,activity,etc.
	 */
	public function metabox_render( $menu_id, $menu, $menu_name ) {
		if ( isset( $this->menus[ $this->screen_group ][ $menu_name ] ) && $menu_name === $this->menus[ $this->screen_group ][ $menu_name ]['object'] ) {
			if ( $this->screen_group === $menu['item_type'] ) {
				if ( ! empty( $this->menu_metabox_callback[ $this->screen_group ][ $menu_name ] ) ) {
					call_user_func_array(
						$this->menu_metabox_callback[ $this->screen_group ][ $menu_name ],
						array(
							$menu_id,
							$menu,
						)
					);
				} elseif ( method_exists( get_parent_class( $this ), 'metabox' ) ) {
					static::metabox( $menu_id, $menu );
				}
			}
		}
	}

	/**
	 * Metabox callback
	 *
	 * @param array        $menu      Menu.
	 * @param array|object $request   request data.
	 * @param string       $menu_name Menu object name like. post_type,activity,etc.
	 *
	 * @return mixed
	 */
	public function get_menu_result( $menu, $request, $menu_name ) {
		if ( ! empty( $menu ) && isset( $this->menus[ $this->screen_group ][ $menu_name ] ) && $menu_name === $this->menus[ $this->screen_group ][ $menu_name ]['object'] ) {
			if ( $this->screen_group === $menu['type'] ) {
				if ( ! empty( $this->menu_result_callback[ $this->screen_group ][ $menu_name ] ) ) {
					$menu = call_user_func_array(
						$this->menu_result_callback[ $this->screen_group ][ $menu_name ],
						array(
							$menu,
							$request,
						)
					);
				} elseif ( method_exists( get_parent_class( $this ), 'get_results' ) ) {
					return static::get_results( $menu, $request );
				}
			}
		}

		return $menu;
	}

	/**
	 * Remove disallow custom screen for extranal link.
	 *
	 * @param array $disallow_screen Disallow screen.
	 *
	 * @since 1.7.0
	 *
	 * @return mixed
	 */
	public function disallow_screen_external_link( $disallow_screen ) {
		if ( ! empty( $this->register_metabox ) ) {
			if ( true !== (bool) $this->external_link ) {
				$disallow_screen[] = $this->screen_group;
			}
		}

		return array_unique( $disallow_screen );
	}


	/*********** Abstract methods **********/
	/**
	 * Do all register and initiate from this function when writing extend class.
	 *
	 * @return mixed
	 */
	abstract public function setup();
}
