<?php
/**
 * Holds menu access controls functionality.
 *
 * @package BuddyBossApp\Menus
 */

namespace BuddyBossApp\Menus;

use BuddyBossApp\AccessControls\AccessRule;
use BuddyBossApp\AccessControls\Core\Settings\AppPages;
use BuddyBossApp\AccessControls\Core\Settings\Pages;
use BuddyBossApp\AccessControls\Helpers;
use BuddyBossApp\Api\AccessControls\Main;
use BuddyBossApp\Helpers\Custom_Multiselect_Dropdown;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Access Control Support for App Menus.
 * Class AccessControls
 *
 * @package BuddyBossApp\Menus
 */
class AccessControls {

	/**
	 * Class instance.
	 *
	 * @since 1.5.2.1
	 * @var object
	 */
	private static $instance;

	/**
	 * Using Singleton, see instance()
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return AccessControls
	 * @since 1.5.2.1
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Init method.
	 *
	 * @return void
	 */
	public function init() {
		add_filter( 'bb_access_controls_group_list_used_by', array( $this, 'group_used_by_menus' ), 10, 2 );
		add_filter( 'bb_access_controls_menu_data', array( $this, 'get_menu_bb_access' ), 10, 2 );
		add_filter( 'bb_access_controls_support_for_core_screens', array( $this, 'support_for_core_screens' ), 10, 1 );

		Custom_Multiselect_Dropdown::instance()->register_results_callback(
			'access_groups',
			array(
				'results_callback' => array(
					$this,
					'bbapp_is_group_exists_in_menu',
				),
			)
		);
	}

	/**
	 * Function to show menu count on group list page.
	 *
	 * @param array $used_by list of used by item.
	 * @param array $group   group array data.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function group_used_by_menus( $used_by, $group ) {
		if ( isset( $group['menus'] ) ) {
			$menus_count = count( $group['menus'] );
			if ( ! empty( $menus_count ) ) {
				$url_base = bbapp_get_admin_url( "admin.php?page=bbapp-access-controls&setting=access-groups&action=edit&tab=menus&id={$group['id']}" );
				/* translators: %s: Menu count */
				$menu_count_html = sprintf( _n( '%s Menu', '%s Menus', number_format_i18n( $menus_count ), 'buddyboss-app' ), number_format_i18n( $menus_count ) );
				$used_by['menu'] = sprintf( '<a href="%1$s"><strong>%2$s</strong></a>', $url_base, esc_html( $menu_count_html ) );
			}
		}

		return $used_by;
	}

	/**
	 * Get bb access field on rest.
	 *
	 * @param array $bb_access Access rule format for rest.
	 * @param array $menu      The menu object.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed
	 */
	public function get_menu_bb_access( $bb_access, $menu ) {
		$rule_data = array();
		/**
		 * Filters the supported type for menus.
		 *
		 * @param array $supported_type Supported post types.
		 *
		 * @since 1.5.2.1
		 */
		$supported_type = apply_filters( 'bb_access_controls_menu_supported_types', array( 'post_type', 'core' ) );

		if ( ! empty( $menu['type'] ) && ! in_array( $menu['type'], $supported_type, true ) ) {
			return $bb_access;
		}

		/**
		 * Filters the supported core screen for menus.
		 *
		 * @param array $menu Menu item.
		 *
		 * @since 1.8.30
		 */
		$menu = apply_filters( 'bb_access_controls_support_for_core_screens', $menu );

		if ( ! empty( $menu['object'] ) && in_array( $menu['object'], array( 'app_page', 'page' ), true ) ) {
			$rule_data = AccessRule::instance()->get_access_rule( $menu['data']['id'], $menu['object'] );
			$rule_data = ! empty( $rule_data['rule_data'] ) ? $rule_data['rule_data'] : array();
		}

		// Check if stored access group conditions are still exist in code.
		if ( ! empty( $rule_data ) ) {
			$bb_access['can_access']       = bb_access_controls_user_can_access_rule( $rule_data );
			$bb_access['restrict_message'] = ( true !== $bb_access['can_access'] && isset( $rule_data['restricted_message'] ) ) ? bb_access_controls_get_restrict_message( $rule_data['restricted_message'] ) : null;

			return $bb_access;
		}

		return $bb_access;
	}

	/**
	 * Access control support for core screens.
	 *
	 * @param array $menu Menu.
	 *
	 * @since 1.8.30
	 *
	 * @return mixed
	 */
	public function support_for_core_screens( $menu ) {
		// Check if menu is not empty.
		if ( empty( $menu ) ) {
			return $menu;
		}

		// Check if menu is not core screen.
		if ( ! empty( $menu['type'] ) && 'core' !== $menu['type'] ) {
			return $menu;
		}

		// Blog screen identify.
		if ( 'blog' === $menu['object'] && get_option( 'show_on_front' ) === 'page' ) {
			$blog_page          = get_post( get_option( 'page_for_posts' ) );
			$menu['data']['id'] = $blog_page->ID;
			$menu['object']     = $blog_page->post_type;

			return $menu;
		}

		// BuddyPress core screen identify.
		if ( function_exists( 'bp_core_get_directory_pages' ) ) {
			$bp_core_pages = bp_core_get_directory_pages();

			// BuddyBoss Platform core screen identify.
			if ( property_exists( $bp_core_pages, $menu['object'] ) ) {
				// Update menu screen response.
				$menu['data']['id'] = $bp_core_pages->{$menu['object']}->id;
				$menu['object']     = 'page';

				return $menu;
			}
		}

		// Get page id from object url.
		$page_id = Main::instance()->get_page_id_from_url( $menu['object'] );
		if ( $page_id ) {
			// Update menu screen response.
			$menu['data']['id'] = $page_id;
			$menu['object']     = 'page';
		}

		return $menu;
	}

	/**
	 * Function to check group is exists in menu.
	 *
	 * @param string $search Search term.
	 * @param int    $page   Page number.
	 * @param int    $limit  Number if items to be fetched.
	 * @param array  $extras Extra parameters.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function bbapp_is_group_exists_in_menu( $search = '', $page = 1, $limit = 20, $extras = array() ) {
		$nonce             = ( ! empty( $extras['nonce'] ) ) ? $extras['nonce'] : '';
		$result            = array();
		$result['message'] = '';
		$result['html']    = '';

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp-get-dropdown' ) ) {
			$result['message'] = __( 'Security check failed.', 'buddyboss-app' );
			wp_send_json_error( $result );
		}

		$term_type = ( ! empty( $_POST['term'] ) && ! empty( $_POST['term']['_type'] ) ) ? sanitize_text_field( wp_unslash( $_POST['term']['_type'] ) ) : '';

		if ( 'query' === $term_type ) {
			Helpers::instance()->get_access_group_list( $search, $page, $limit, $extras );
			wp_die();
		}

		$extras       = ( ! empty( $_POST['extras'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['extras'] ) ) : array(); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$selection_id = ( ! empty( $extras['value'] ) ) ? $extras['value'] : '';
		$menu_type    = ( ! empty( $extras['menu_type'] ) ) ? $extras['menu_type'] : '';
		$menu_id      = ( ! empty( $extras['menu_id'] ) ) ? trim( $extras['menu_id'] ) : 0;
		$lang_code    = ( ! empty( $extras['lang'] ) ) ? trim( $extras['lang'] ) : 0;
		$_nav_item    = MenuManager::instance()->get_menu_by_access_group( $selection_id, $menu_type, $lang_code );

		if ( ! empty( $_nav_item ) && $_nav_item->id !== $menu_id && $_nav_item->language_code === $lang_code ) {
			$menu_url = add_query_arg(
				array(
					'page'      => 'bbapp-menus',
					'menu-type' => $menu_type,
					'menu'      => $_nav_item->id,
					'lang'      => $lang_code,
				),
				admin_url( 'admin.php' )
			);

			$menu_label     = isset( $_nav_item->menu_name ) ? $_nav_item->menu_name : 'Menu #' . $_nav_item->id;
			$result['html'] = sprintf( __( "This group is already assigned to <a href='%1\$s' target='_blank'>%2\$s</a>", 'buddyboss-app' ), esc_url( $menu_url ), esc_html( $menu_label ) );
		}
		$result['result'] = $_nav_item;
		wp_send_json_success( $result );
	}

}
