<?php
/**
 * Holds main class functionality for menus.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

// Exit if accessed directly.
use BuddyBossApp\Common\IconPicker;
use BuddyBossApp\Menus\MenuManager;
use WP_REST_Request;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MEnu class.
 */
class Menu {

	/**
	 * Class instance.
	 *
	 * @var Menu $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return Menu Instance of the class.
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * AppMenu constructor.
	 */
	public function __construct() {
	}

	/**
	 * Load all hooks and action.
	 */
	public function load() {
		if ( bbapp_is_active( 'access_controls' ) ) {
			\BuddyBossApp\Menus\AccessControls::instance();
		}
	}

	/**
	 * Create database tables on plugin activation.
	 */
	public function on_activation() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		// Create bbapp_menus table for menus.
		$table_name = $wpdb->prefix . 'bbapp_menus';
		$sql        = "CREATE TABLE IF NOT EXISTS $table_name (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			site_id bigint(20) NOT NULL,
			menu_type varchar(255) NOT NULL,
			menu_name varchar(255) NOT NULL,
			login_state tinyint(1) NOT NULL DEFAULT 0,
			access_groups longtext,
			language_code varchar(255) NOT NULL DEFAULT 'en',
			priority int(11) NOT NULL DEFAULT 0,
			data longtext,
			created_at datetime NOT NULL,
			updated_at datetime NOT NULL,
			PRIMARY KEY  (id),
			KEY site_id (site_id),
			KEY menu_type (menu_type),
			KEY menu_name (menu_name),
			KEY language_code (language_code),
			KEY priority (priority),
			KEY created_at (created_at),
			KEY updated_at (updated_at)
		) $charset_collate;";
		dbDelta( $sql );

		// Create bbapp_menu_items table for menu items.
		$table_name = $wpdb->prefix . 'bbapp_menu_items';
		$sql        = "CREATE TABLE IF NOT EXISTS $table_name (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			menu_id bigint(20) NOT NULL,
			label varchar(255) NOT NULL,
			item_id bigint(20) NOT NULL DEFAULT 0,
			item_slug varchar(255) NOT NULL,
			item_type varchar(255) NOT NULL,
			item_icon_data longtext,
			item_data longtext,
			item_link varchar(255),
			parent_id bigint(20) NOT NULL DEFAULT 0,
			menu_order bigint(20) NOT NULL DEFAULT 0,
			created_at datetime NOT NULL,
			updated_at datetime NOT NULL,
			PRIMARY KEY  (id),
			KEY menu_id (menu_id),
			KEY created_at (created_at),
			KEY updated_at (updated_at)
		) $charset_collate;";
		dbDelta( $sql );
	}

	/**
	 * Get current menu.
	 *
	 * @param string $default_setting Default value for the current menu.
	 *
	 * @return mixed|string
	 */
	public function get_current_menu( $default_setting = 'tabbar' ) {
		return ( ! empty( $_REQUEST['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['setting'] ) ) : $default_setting; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	}

	/**
	 * Rest Endpoint Function for Returning App Menu
	 *
	 * @param WP_REST_Request $request        Request parameters.
	 * @param bool            $tabbar_menu_id Tabbar menu id.
	 * @param bool            $more_menu_id   More menu id.
	 *
	 * @return mixed
	 */
	public function rest_response( $request, $tabbar_menu_id = false, $more_menu_id = false ) {
		$allow_more = isset( $request['allow_more'] ) && ! empty( $request['allow_more'] ) ? (bool) $request['allow_more'] : false;

		if ( ! $tabbar_menu_id ) {
			$tabbar_menu_id = $this->bbapp_get_user_nav_menu_id();
		}

		$headerbar_menu_id  = MenuManager::instance()->get_headerbar_menu_from_tabbar_id( $tabbar_menu_id );
		$more_menu_position = \BuddyBossApp\Menus\MenuManager::instance()->get_more_menu_position( $headerbar_menu_id );
		if ( ! $more_menu_id ) {
			$more_menu_id = $this->bbapp_get_user_nav_menu_id( 'more' );
		}

		$app_tabbar_menu    = MenuManager::instance()->get_menu_items(
			array(
				'menu_id'             => $tabbar_menu_id,
				'menu_type'           => 'tabbar',
				'has_more_menu_items' => true,
			)
		);
		$app_more_menu      = MenuManager::instance()->get_menu_items(
			array(
				'menu_id'   => $more_menu_id,
				'menu_type' => 'more',
			)
		);
		$app_headerbar_menu = MenuManager::instance()->get_menu_items(
			array(
				'menu_id'             => $headerbar_menu_id,
				'menu_type'           => 'headerbar',
				'has_more_menu_items' => true,
			)
		);

		if ( ! $app_tabbar_menu && ! $app_more_menu && ! $app_headerbar_menu ) {
			$app_menu = array(
				'tabbar'                 => array(),
				'more'                   => array(),
				'headerbar'              => array(),
				'headerbar-action-items' => array(),
			);

			return rest_ensure_response( $app_menu );
		}

		$app_tabbar_screen      = array();
		$app_more_screen        = array();
		$app_headerbar_screen   = array();
		$headerbar_action_items = array();

		if ( isset( $app_tabbar_menu ) ) {
			// Separate regular items and "More" items for proper ordering and limits.
			$regular_tabbar_items = array();
			$more_tabbar_item     = null;

			// First pass: separate regular items from "More" items.
			foreach ( $app_tabbar_menu as $k => $tabbar_menu ) {
				if ( bbapp_is_menu_locked_page( $tabbar_menu ) ) {
					continue;
				}

				if ( false === $allow_more && 'more' === $tabbar_menu['object'] ) {
					continue;
				}

				// Skip "More" items if position doesn't match current menu type (tabbar).
				if ( 'more' === $tabbar_menu['object'] && 'tabbar' !== $more_menu_position ) {
					continue;
				}

				if ( 'more' === $tabbar_menu['object'] ) {
					// Handle "More" item icon type.
					if ( ! empty( $tabbar_menu['icon_type'] ) ) {
						$icon_type = $tabbar_menu['icon_type'];
					}

					if ( empty( $icon_type ) && ! empty( $tabbar_menu['icon']['extra'] ) && is_array( $tabbar_menu['icon']['extra'] ) && isset( $tabbar_menu['icon']['extra']['icon_type'] ) ) {
						$icon_type = $tabbar_menu['icon']['extra']['icon_type'];
					}

					if ( empty( $icon_type ) ) {
						$icon_type = get_option( 'more_menu_icon_type', '' );
					}

					if ( empty( $icon_type ) ) {
						$icon_type = 'avatar';
					}

					$tabbar_menu['icon_type'] = $icon_type;
					$more_tabbar_item         = array(
						'key'  => $k,
						'data' => $tabbar_menu,
					);
				} else {
					// Regular tabbar item.
					$regular_tabbar_items[ $k ] = $tabbar_menu;
				}
			}

			// Enforce 5-item limit based on more menu position.
			if ( 'tabbar' === $more_menu_position && $more_tabbar_item ) {
				// Case 1: 4 regular + 1 more = 5 total.
				$max_regular_items = 4;
			} else {
				// Case 2: 5 regular items (no more).
				$max_regular_items = 5;
			}

			// Limit regular items to the calculated maximum.
			if ( count( $regular_tabbar_items ) > $max_regular_items ) {
				$regular_tabbar_items = array_slice( $regular_tabbar_items, 0, $max_regular_items, true );
			}

			// Process regular items first.
			foreach ( $regular_tabbar_items as $k => $tabbar_menu ) {
				$app_tabbar_screen[ $k ] = $this->get_user_app_menu_filter_menu_data( $tabbar_menu, $request, $k, 'tabbar' );
			}

			// Process "More" item last (if it should be included).
			if ( $more_tabbar_item ) {
				$k                       = $more_tabbar_item['key'];
				$tabbar_menu             = $more_tabbar_item['data'];
				$app_tabbar_screen[ $k ] = $this->get_user_app_menu_filter_menu_data( $tabbar_menu, $request, $k, 'tabbar' );
			}
		}

		if ( isset( $app_more_menu ) ) {
			// More icon menu.
			foreach ( $app_more_menu as $k => $more_menu ) {
				if ( bbapp_is_menu_locked_page( $more_menu ) ) {
					continue;
				}

				if ( 'section' === $more_menu['type'] ) {
					unset( $more_menu['original'] );
					$app_more_screen[ $k ] = $more_menu;
					// make sure section is array format.
					if ( ! is_array( $app_more_menu[ $k ]['section'] ) ) {
						$app_more_screen[ $k ]['section'] = array();
					}

					if ( ! empty( $more_menu['section'] ) && is_array( $more_menu['section'] ) ) {
						$app_more_screen_section = array();
						foreach ( $more_menu['section'] as $j => $section_menu ) {
							if ( bbapp_is_menu_locked_page( $section_menu ) ) {
								continue;
							}

							$app_more_screen_section[ $j ] = $this->get_user_app_menu_filter_menu_data( $section_menu, $request, $j, 'more' );
						}
						// NOTE : Need to do this else =>  2: { }.
						$app_more_screen[ $k ]['section'] = array_values( $app_more_screen_section );
					}
				} else {
					$app_more_screen[ $k ] = $this->get_user_app_menu_filter_menu_data( $more_menu, $request, $k, 'more' );
				}
			}
		}

		if ( isset( $app_headerbar_menu ) ) {
			$menu_order         = 1;
			$is_platform_active = bbapp_is_buddyboss_platform_enabled();

			// Determine limits based on platform status and more menu position.
			$max_total_menus = 3;
			$reserved_slots  = 0;

			// Reserve slot for action menu if platform is active.
			if ( $is_platform_active ) {
				++$reserved_slots;
			}

			// Reserve slot for more menu if position is headerbar.
			if ( 'headerbar' === $more_menu_position ) {
				++$reserved_slots;
			}

			// Calculate available slots for regular menus.
			$available_regular_slots = $max_total_menus - $reserved_slots;
			$regular_menu_count      = 0;

			// Headerbar menu.
			foreach ( $app_headerbar_menu as $k => $headerbar_menu ) {
				if ( bbapp_is_menu_locked_page( $headerbar_menu ) ) {
					continue;
				}

				if ( ! empty( $headerbar_menu['type'] ) && 'headerbar' === $headerbar_menu['type'] ) {
					$headerbar_action_item_icons = array();

					if ( ! empty( $headerbar_menu['child_items'] ) ) {
						foreach ( $headerbar_menu['child_items'] as $key => $icon ) {
							$icon['object']                                    = 'headerbar';
							$headerbar_action_item_icons[ $icon['item_slug'] ] = $this->get_user_app_menu_filter_menu_data( $icon, $request, $key, 'headerbar' );
						}

						$headerbar_menu['icons'] = $headerbar_action_item_icons;
					}

					$headerbar_action_items = $this->get_user_app_menu_filter_menu_data( $headerbar_menu, $request, $k, 'headerbar' );

					$headerbar_action_items['order'] = $menu_order;
					unset( $headerbar_action_items['child_items'] );

					++$menu_order;
					continue;
				}

				// Handle "More" menu items.
				if ( 'more' === $headerbar_menu['object'] ) {
					if ( 'headerbar' !== $more_menu_position ) {
						continue;
					}
					// Handle "More" item icon type.
					if ( ! empty( $headerbar_menu['icon_type'] ) ) {
						$icon_type = $headerbar_menu['icon_type'];
					}

					if ( empty( $icon_type ) && ! empty( $headerbar_menu['icon']['extra'] ) && is_array( $headerbar_menu['icon']['extra'] ) && isset( $headerbar_menu['icon']['extra']['icon_type'] ) ) {
						$icon_type = $headerbar_menu['icon']['extra']['icon_type'];
					}

					if ( empty( $icon_type ) ) {
						$icon_type = get_option( 'more_menu_icon_type', '' );
					}

					if ( empty( $icon_type ) ) {
						$icon_type = 'avatar';
					}

					$headerbar_menu['icon_type'] = $icon_type;

				} else {
					// For regular menu items, check if we've reached the limit.
					if ( $regular_menu_count >= $available_regular_slots ) {
						continue; // Skip extra regular menus.
					}
					++$regular_menu_count;
				}

				$app_headerbar_screen[ $k ]          = $this->get_user_app_menu_filter_menu_data( $headerbar_menu, $request, $k, 'headerbar' );
				$app_headerbar_screen[ $k ]['order'] = $menu_order;
				++$menu_order;
			}
		}

		$app_menu = array(
			'tabbar'                 => array_values( $app_tabbar_screen ),
			'more'                   => array_values( $app_more_screen ),
			'headerbar'              => array_values( $app_headerbar_screen ),
			'headerbar-action-items' => $headerbar_action_items,
		);

		unset( $app_tabbar_screen, $app_more_screen, $app_headerbar_screen, $headerbar_action_items, $app_tabbar_menu, $app_more_menu, $app_headerbar_menu, $headerbar_menu_id, $app_settings );

		return rest_ensure_response( $app_menu );
	}

	/**
	 * Get the menu based on user permissions if the access control module is enabled. else it will just return
	 * logged in or logged out menu based on user authentications state.
	 *
	 * @param string $menu_type Type of menu.
	 *
	 * @since 1.5.2.1
	 * @return bool|int|string
	 */
	public function bbapp_get_user_nav_menu_id( $menu_type = 'tabbar' ) {
		$main_menu_id = MenuManager::instance()->get_default_menu_id( $menu_type );

		// Get logged out user menu id.
		if ( ! is_user_logged_in() ) {
			$logged_out_menu_id = MenuManager::instance()->get_logout_menu_id( $menu_type );
			if ( ! empty( $logged_out_menu_id ) ) {
				$main_menu_id = $logged_out_menu_id;
			}
		} else {
			// Logged in menu.
			$user_id   = get_current_user_id();
			$all_menus = MenuManager::instance()->get_menus(
				array(
					'menu_type'   => $menu_type,
					'order_by'    => 'priority',
					'order'       => 'ASC',
					'login_state' => '1',
				)
			);

			// Initialize to a high value so that any valid priority will be lower than this.
			$lowest_priority  = PHP_INT_MAX;
			$priority_menu_id = 0;

			if ( ! empty( $all_menus['menus'] ) ) {
				foreach ( $all_menus['menus'] as $menu ) {
					$has_access = false;

					// Check if user has access to the menu based on access groups if the access control module is enabled.
					if ( ! empty( $menu->access_groups ) ) {
						foreach ( $menu->access_groups as $access_group_id ) {
							// Check user has group access.
							if ( function_exists( 'bb_access_controls_user_has_group' ) && bb_access_controls_user_has_group( $user_id, $access_group_id ) ) {
								$has_access = true;
								break;
							}
						}
					}

					// If user has access and menu has lower priority value (higher precedence) than current lowest, update the menu id.
					if ( $has_access && isset( $menu->priority ) && $menu->priority < $lowest_priority ) {
						$lowest_priority  = $menu->priority;
						$priority_menu_id = $menu->id;
					}
				}

				// Set the menu with lowest priority value (highest precedence) if found and user has access.
				if ( $priority_menu_id > 0 ) {
					$main_menu_id = $priority_menu_id;
				}
			}
		}

		return $main_menu_id;
	}

	/**
	 * Filter menu data.
	 *
	 * @param array                   $menu        Array of menu.
	 * @param array| \WP_REST_Request $request     Array of request parameters.
	 * @param string                  $index       Menu index.
	 * @param string                  $menu_screen Menu screen name.
	 *
	 * @return null|mixed
	 */
	public function get_user_app_menu_filter_menu_data( $menu, $request, $index, $menu_screen ) {
		unset( $menu['original'] );

		/**
		 * Get menu result.
		 */
		$menu = apply_filters( 'bbapp_get_menu_result', $menu, $request, $menu['object'] );

		$menu['data']['open_external'] = ( isset( $menu['data']['open_external'] ) && 'yes' === $menu['data']['open_external'] ) ? true : false;

		$menu['icon'] = IconPicker::instance()->get_icon_for_api( $menu, $menu_screen, true );

		// Menu custom link deeplink data show.
		if ( bbapp_is_rest() && isset( $menu['data']['link'] ) && bbapp_is_valid_url( $menu['data']['link'] ) ) {
			$in_menu_object = array( 'custom' );

			if ( bbapp_is_lite_live_app() ) {
				$in_menu_object[] = 'page';
			}

			if ( in_array( $menu['object'], $in_menu_object, true ) ) {
				$menu['deeplink_data'] = $this->get_deeplink_data( $menu['data']['link'], $request );
			}
		}

		return apply_filters( 'bbapp_menu_filter_menu_data', $menu );
	}

	/**
	 * Get deeplink data.
	 *
	 * @param string $link    Link.
	 * @param array  $request Request.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function get_deeplink_data( $link = '', $request = array() ) {
		/**
		 * Redirect to topic endpoint.
		 */
		global $wp_rest_server;
		$menu_deeplink_request = new WP_REST_Request( 'POST', '/buddyboss-app/core/v1/url-details' );
		$menu_deeplink_request->set_param( 'url', $link );
		$menu_deeplink_request->set_param( '_embed', true );
		$server   = rest_get_server();
		$response = $server->dispatch( $menu_deeplink_request );

		$status               = $response->get_status();
		$menu_deeplink_result = array();
		if ( 200 === $status ) {
			$menu_deeplink_result = $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
		}

		return $menu_deeplink_result;
	}
}
