<?php
/**
 * Holds app management functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

// Exit if accessed directly.
use BuddyBossApp\AppStores\Android;
use BuddyBossApp\AppStores\Apple;
use WP_Error;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * App manage class.
 */
class ManageApp {

	/**
	 * App information.
	 *
	 * @var $app_info
	 */
	public static $app_info;

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of class
	 *
	 * @return ManageApp
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * ManageApp constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters and hooks here.
	 */
	public function load() {
		add_action( 'bbapp_clear_license_data', array( $this, 'clear_license_data' ) );
	}

	/**
	 * Get apps raw data from db
	 *
	 * @return array|mixed|void
	 */
	public function get_apps_data() {
		$data = bbapp_get_network_option( 'bbapps' );

		if ( ! is_array( $data ) ) {
			$data = array();
		}

		return $data;
	}

	/**
	 * Set app raw data to db.
	 *
	 * @param array $data Data to be updated.
	 *
	 * @return bool
	 */
	public function update_apps_data( $data ) {
		global $apps_data_cache;

		if ( ! is_array( $data ) ) {
			$data = array();
		}

		$apps_data_cache = null;

		return bbapp_set_network_option( 'bbapps', $data );
	}

	/**
	 * Return the App ID.
	 *
	 * @return bool|mixed
	 */
	public function get_app_id() {
		$app = $this->get_app();

		if ( $app ) {
			return isset( $app['bbapp_app_id'] ) ? $app['bbapp_app_id'] : false;
		}

		return false;
	}

	/**
	 * Returns the App Key.
	 *
	 * @return bool|mixed
	 */
	public function get_app_key() {
		$app = $this->get_app();

		if ( $app ) {
			return $app['bbapp_app_key'];
		}

		return false;
	}

	/**
	 * Returns the App Key.
	 *
	 * @return bool|mixed
	 */
	public function get_auth_app_key() {
		$app = $this->get_app();

		if ( $app && isset( $app['bbapp_app_key'] ) ) {
			return sha1( $app['bbapp_app_key'] );
		}

		return false;
	}

	/**
	 * Returns the app information of given app id.
	 * for reference
	 * app_id is the reference id of app stored in WordPress
	 * bbapp_app_id is the ID of app at bbapp.
	 *
	 * @return array
	 */
	public function get_app() {
		global $apps_data_cache; // helps to cache.

		if ( empty( $apps_data_cache ) ) {
			$apps_data_cache = $this->get_apps_data();
		}

		return $apps_data_cache;
	}

	/**
	 * Tells weather the app is connected as dev or not.
	 *
	 * @return bool
	 */
	public function is_current_primary_site() {
		$app = $this->get_app();

		if ( ! $app ) {
			return false;
		}

		if ( isset( $app['bbapp_site_type'] ) && 'primary' === $app['bbapp_site_type'] ) {
			return true;
		}

		return false;
	}

	/**
	 * Tells weather the app is connected as dev or not.
	 *
	 * @return bool
	 */
	public function is_current_secondary_site() {
		$app = $this->get_app();

		if ( ! $app ) {
			return false;
		}

		if ( isset( $app['bbapp_site_type'] ) && 'primary' !== $app['bbapp_site_type'] ) {
			return true;
		}

		return false;
	}

	/**
	 * Quick way to get app setting.
	 *
	 * @param string $setting_name Setting name.
	 * @param bool   $network If needs to get network settings.
	 *
	 * @return bool|mixed
	 */
	public function get_app_setting( $setting_name, $network = false ) {
		$settings = $this->get_app_settings( $network );

		if ( isset( $settings[ $setting_name ] ) ) {
			return $settings[ $setting_name ];
		} else {
			return false;
		}
	}

	/**
	 * Returns the settings of app.
	 *
	 * @param boolean $network If needs to get network settings.
	 *
	 * @return array
	 */
	public function get_app_settings( $network = false ) {
		return $this->get_settings( $network );
	}

	/**
	 * Updates settings of app.
	 *
	 * @param string|array $settings Settings to be updated.
	 *
	 * @return array|mixed
	 */
	public function update_app_settings( $settings ) {
		return $this->update_settings( $settings );
	}

	/**
	 * Get asettings.
	 *
	 * @param false $network If needs to get network settings.
	 *
	 * @return bool|mixed
	 */
	public function get_settings( $network = false ) {
		if ( true === $network && bbapp()->is_network_activated() ) {
			$apps_settings = get_blog_option( get_current_network_id(), 'bbapp_settings' );
		} else {
			$apps_settings = get_option( 'bbapp_settings' );
		}

		if ( ! is_array( $apps_settings ) ) {
			$apps_settings = array();
		}

		return $apps_settings;
	}

	/**
	 * Update App Settings.
	 *
	 * @param  array $settings Settings to be updated.
	 *
	 * @return mixed
	 */
	public function update_settings( $settings ) {
		if ( ! is_array( $settings ) ) {
			$settings = array();
		}

		update_option( 'bbapp_settings', $settings );

		return $settings;
	}


	/**
	 * Helper function to purge app sealed cache.
	 *
	 * @return bool
	 */
	public function purge_app_sealed_cache() {
		$cache_keys    = array();
		$cache_keys [] = '_bbapp_app_signed_details';
		$cache_keys [] = bbapp_get_app_info_transient_key();

		foreach ( $cache_keys as $cache_key ) {
			delete_site_transient( $cache_key );
		}

		return true;
	}

	/**
	 * Purge license data.
	 *
	 * @since 2.2.80
	 */
	public function purge_license_data() {
		AppSettings::instance()->delete_setting_value( 'app_license_type' );
		delete_site_transient( 'bbapp_license_type' );
		AppSettings::instance()->delete_setting_value( 'license_data_token' );
		AppSettings::instance()->delete_setting_value( 'trigger_build_notice' );
	}

	/**
	 * Clear license data.
	 *
	 * @since 2.3.00
	 */
	public function clear_license_data() {
		if ( $this->is_app_center_connected() || $this->is_app_center_connected( 'secondary' ) ) {
			return;
		}

		$this->purge_license_data();
		AppSettings::instance()->delete_setting_value( 'trigger_site_disconnect_notice' );
	}

	/**
	 * Fetch the app information from BuddyBossApp server.
	 *
	 * @return array|bool|mixed|void
	 */
	public function get_app_sign_details() {
		if ( isset( $GLOBALS['get_app_signed_details_runtime_cache'] ) ) {
			return $GLOBALS['get_app_signed_details_runtime_cache'];
		}

		$bbapp_app_id = self::instance()->get_app_id();

		if ( empty( $bbapp_app_id ) ) {
			return array();
		}

		$cache_key = '_bbapp_app_signed_details';
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached ) {
			$GLOBALS['get_app_signed_details_runtime_cache'] = $cached;

			return $cached;
		}

		$api_url    = ClientCommon::instance()->get_center_api_url( 'v2' );
		$do_request = bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => array(
					'action'    => 'app_signature',
					'bbapp_id'  => $bbapp_app_id,
					'bbapp_key' => self::instance()->get_auth_app_key(),
				),
			)
		);

		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			$GLOBALS['get_app_signed_details_runtime_cache'] = false;

			return false;
		}

		$body     = isset( $do_request['body'] ) ? $do_request['body'] : '';
		$response = json_decode( $body, true );

		if ( ! isset( $response['data'] ) || ! isset( $response['sign'] ) ) {
			$GLOBALS['get_app_signed_details_runtime_cache'] = false;

			set_site_transient( $cache_key, $response, 5 * 60 ); // cache it for 5 min.  // don't ping app center too often.

			return false;
		} else {
			set_site_transient( $cache_key, $response, 20 * DAY_IN_SECONDS ); // cache it for 20 days.
		}

		$GLOBALS['get_app_signed_details_runtime_cache'] = $response;

		return $response;
	}

	/**
	 * Connects the app with ab app id.
	 *
	 * @param string $bbapp_app_id    BB App ID.
	 * @param string $bbapp_app_key   BB App key.
	 * @param string $bbapp_site_type BB App Site Url. primary or secondary.
	 *
	 * @return bool|WP_Error
	 */
	public function connect_app( $bbapp_app_id, $bbapp_app_key, $bbapp_site_type = 'primary' ) {
		$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/connect-site' );

		if ( empty( $bbapp_app_id ) || empty( $bbapp_app_key ) ) {
			return new WP_Error( 'value_missing', __( 'Please provide your App ID & Key.', 'buddyboss-app' ) );
		}

		$url  = $this->get_home_url();
		$body = array(
			'bbapp_id'  => $bbapp_app_id,
			'bbapp_key' => sha1( $bbapp_app_key ),
			'site_url'  => $url,
			'site_type' => $bbapp_site_type,
		);

		if ( bbapp_is_bedrock_enabled() ) {
			$body['admin_url'] = $this->get_site_url();
		}

		$this->purge_app_sealed_cache();

		$response = bbapp_remote_post(
			$api_url,
			array(
				'timeout'  => 15,
				'blocking' => true,
				'cookies'  => array(),
				'body'     => $body,
			)
		);

		if ( is_wp_error( $response ) ) {
			return new WP_Error( 'cannot_connect', __( 'Unable to connect to the BuddyBoss server.', 'buddyboss-app' ) );
		} else {
			$body     = isset( $response['body'] ) ? $response['body'] : '';
			$response = json_decode( $body );

			if ( is_array( $response ) || is_object( $response ) ) {
				$response = (array) $response;

				if ( isset( $response['success'] ) && true === $response['success'] ) {
					// update & save data.
					$apps_data                    = $this->get_apps_data();
					$apps_data['bbapp_app_key']   = $bbapp_app_key;
					$apps_data['bbapp_app_id']    = $bbapp_app_id;
					$apps_data['bbapp_site_type'] = $bbapp_site_type;
					$apps_data['user_app_id']     = ! empty( $response['user_app_id'] ) ? $response['user_app_id'] : '';
					$apps_data['verified']        = true;
					$apps_data['token']           = ! empty( $response['token'] ) ? $response['token'] : '';
					$apps_data['site_url']        = ! empty( $response['site_url'] ) ? $response['site_url'] : '';

					$this->update_apps_data( $apps_data );
					$this->update_license_data(
						array(
							'license_type'      => ! empty( $response['license_type'] ) ? $response['license_type'] : 'full',
							'license_transient' => ! empty( $response['license_type'] ) ? $response['license_type'] : 'full',
							'license_token'     => ! empty( $response['license_data'] ) ? $response['license_data'] : '',
						)
					);

					// Clear scheduled the single event.
					if ( wp_next_scheduled( 'bbapp_clear_license_data' ) ) {
						wp_clear_scheduled_hook( 'bbapp_clear_license_data' );
						AppSettings::instance()->delete_setting_value( 'trigger_site_disconnect_notice' );
					}

					// Load the app info so it will update the license type.
					ManageApp::instance()->verify_license_data();

					/**
					 * Action after connect app.
					 *
					 * @peram array $apps_data App data.
					 *
					 * @since 1.6.7
					 */
					do_action( 'bbapp_connect_app', $apps_data );

					return $apps_data;
				} else {
					if ( 'invalid_app' === $response['code'] ) {
						return new WP_Error( 'invalid_app', __( 'The key you entered is invalid.', 'buddyboss-app' ) );
					} elseif ( in_array(
						$response['code'],
						array(
							'primary_not_possible',
							'secondary_not_possible',
						),
						true
					) ) {
						return new WP_Error( $response['code'], $response['message'] );
					} else {
						/* translators: %s is the response code */
						$message = sprintf( __( 'We were not able to verify at the moment. Please contact BuddyBoss support with error code \'%s\'.', 'buddyboss-app' ), $response['code'] );

						return new WP_Error( 'invalid_err', $message );
					}
				}
			} else {
				return new WP_Error( 'cannot_connect', __( 'We were not able to verify the information added. Please contact BuddyBoss support.', 'buddyboss-app' ) );
			}
		}

		return false;
	}

	/**
	 * Unlink the app from WordPress if it's connected.
	 *
	 * @param string $bbapp_app_id    BB App ID.
	 * @param string $bbapp_app_key   BB App Key.
	 * @param string $bbapp_site_type BB App Site type, primary or secondary.
	 * @param string $bbapp_site_url  BB Site Url.
	 *
	 * @return bool|WP_Error
	 */
	public function disconnect_app( $bbapp_app_id = '', $bbapp_app_key = '', $bbapp_site_type = '', $bbapp_site_url = '' ) {
		$app = $this->get_app();

		if ( empty( $app ) ) {
			return false;
		}

		if ( ! $app['verified'] ) {
			return false;
		}

		if ( empty( $bbapp_app_id ) ) {
			$bbapp_app_id = isset( $app['bbapp_app_id'] ) ? $app['bbapp_app_id'] : '';
		}

		if ( empty( $bbapp_app_key ) ) {
			$bbapp_app_key = isset( $app['bbapp_app_key'] ) ? $app['bbapp_app_key'] : '';
		}

		if ( empty( $bbapp_site_type ) ) {
			$bbapp_site_type = isset( $app['bbapp_site_type'] ) ? $app['bbapp_site_type'] : 'primary';
		}

		$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/delete-site' );

		if ( empty( $bbapp_app_id ) || empty( $bbapp_app_key ) ) {
			return new WP_Error( 'value_missing', __( 'Please provide your App ID & Key.', 'buddyboss-app' ) );
		}

		$this->purge_app_sealed_cache();

		$url = ! empty( $bbapp_site_url ) ? $bbapp_site_url : $this->get_home_url();
		$body_args = array(
			'bbapp_id'  => $bbapp_app_id,
			'bbapp_key' => self::get_auth_app_key(),
			'site_url'  => $url,
			'site_type' => $bbapp_site_type,
		);

		if ( 'secondary' === $bbapp_site_type && ( ! empty( $app['bbapp_site_type'] ) && 'secondary' !== $app['bbapp_site_type'] ) ) {
			$body_args['disconnect_client_site'] = '1';
		}
		$args = array(
			'timeout'  => 15,
			'blocking' => true,
			'cookies'  => array(),
			'body'     => $body_args,
		);

		$response = bbapp_remote_post( $api_url, $args );

		if ( is_wp_error( $response ) ) {
			return new WP_Error( 'cannot_connect', __( 'Unable to connect to the BuddyBoss server.', 'buddyboss-app' ) );
		} else {
			$response = wp_remote_retrieve_body( $response );
			$response = json_decode( $response, true );

			if ( is_array( $response ) && ! empty( $response ) ) {
				if ( ( isset( $response['success'] ) && true === $response['success'] ) || ( isset( $response['code'] ) && in_array(
					$response['code'],
					array(
						'site_not_found',
						'invalid_app',
					),
					true
				) ) ) {
					if ( 'primary' === $bbapp_site_type || ( ! empty( $app['bbapp_site_type'] ) && 'secondary' === $app['bbapp_site_type'] ) ) {
						$apps_data                    = $this->get_apps_data();
						$apps_data['bbapp_app_key']   = '';
						$apps_data['bbapp_site_type'] = '';
						$apps_data['user_app_id']     = '';
						$apps_data['verified']        = false;
						$apps_data['token']           = '';
						$apps_data['site_url']        = '';

						$this->update_apps_data( $apps_data );
					}

					bbapp_delete_transients( bbapp_get_app_info_transient_key() );

					// Schedule the single event in next 48 hours.
					if ( ! wp_next_scheduled( 'bbapp_clear_license_data' ) ) {
						wp_schedule_single_event( time() + ( 48 * HOUR_IN_SECONDS ), 'bbapp_clear_license_data' );
						AppSettings::instance()->update_setting_value( 'trigger_site_disconnect_notice', true );
					}

					/**
					 * Action after disconnect app.
					 *
					 * @peram array $apps_data App data.
					 *
					 * @since 1.6.7
					 */
					do_action( 'bbapp_disconnect_app', $apps_data );

					return true;
				}

				bbapp_delete_transients( bbapp_get_app_info_transient_key() );
			} else {
				return new WP_Error( 'cannot_connect', __( 'We were not able to verify the information added. Please contact BuddyBoss support.', 'buddyboss-app' ) );
			}
		}

		return false;
	}

	/**
	 * Get app info.
	 *
	 * @param bool $override_cache If needs to orver-ride cache.
	 *
	 * @return array|WP_Error
	 */
	public function get_app_info( $override_cache = false ) {
		if ( ! empty( self::$app_info ) && true !== $override_cache ) {
			return self::$app_info;
		}

		$cache_key = bbapp_get_app_info_transient_key();
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			self::$app_info = $cached;

			return $cached;
		}

		$app_info = $this->get_app_info_request();

		set_site_transient( $cache_key, $app_info, 10 * 60 ); // Set the cache for 10min.

		return $app_info;
	}

	/**
	 * Update license data.
	 *
	 * @param bool $override_cache If needs to override cache.
	 *
	 * @return void
	 * @since 2.2.80
	 */
	public function verify_license_data( $override_cache = false ) {
		$app_info = $this->get_app_info( $override_cache );

		// Save the license type.
		$bbapp_license_type = get_site_transient( 'bbapp_license_type' );

		if (
			(
				false === $bbapp_license_type ||
				true === $override_cache
			) &&
			! empty( $app_info ) &&
			! is_wp_error( $app_info ) &&
			! empty( $app_info['app_license_type'] )
		) {
			set_site_transient( 'bbapp_license_type', $app_info['app_license_type'], 60 * 60 * 8 );
		}
	}

	/**
	 * Update license data.
	 *
	 * @param array $license_data License data.
	 *
	 * @return void
	 * @since 2.2.80
	 */
	public function update_license_data( $license_data ) {
		if ( ! empty( $license_data['license_type'] ) ) {
			AppSettings::instance()->update_setting_value( 'app_license_type', $license_data['license_type'] );
		}

		if ( ! empty( $license_data['license_transient'] ) ) {
			set_site_transient( 'bbapp_license_type', $license_data['license_transient'], 60 * 60 * 8 );
		} elseif ( false === $license_data['license_transient'] ) {
			delete_site_transient( 'bbapp_license_type' );
		}

		if ( ! empty( $license_data['license_token'] ) ) {
			AppSettings::instance()->update_setting_value( 'license_data_token', $license_data['license_token'] );
		}
	}

	/**
	 * App information request.
	 *
	 * @since 1.7.72
	 * @return array|WP_Error
	 */
	public function get_app_info_request() {
		$app = $this->get_app();

		if ( empty( $app ) ) {
			return new WP_Error( 'app_not_found', __( 'There is no such app.', 'buddyboss-app' ) );
		}

		if ( empty( $app['bbapp_app_id'] ) || empty( $app['bbapp_app_key'] ) ) {
			return new WP_Error( 'value_missing', __( 'Please provide your App ID & Key in the app.', 'buddyboss-app' ) );
		}

		$bbapp_app_id = $app['bbapp_app_id'];
		$api_url      = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-app-info' );
		$response     = bbapp_remote_get(
			$api_url,
			array(
				'timeout'  => 15,
				'blocking' => true,
				'cookies'  => array(),
				'body'     => array(
					'bbapp_id'         => $bbapp_app_id,
					'bbapp_key'        => $this->get_auth_app_key(),
					'purge_cache'      => true,
					'app_license_info' => true,
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			return new WP_Error( 'cannot_connect', __( 'Unable to connect to the BuddyBoss server.', 'buddyboss-app' ) );
		} else {
			$body     = wp_remote_retrieve_body( $response );
			$response = json_decode( $body );

			if ( ! is_array( $response ) && ! is_object( $response ) ) {

				return new WP_Error( 'cannot_connect', __( 'We were not able to verify the information added. Please contact BuddyBoss support.', 'buddyboss-app' ) );
			}

			self::$app_info = (array) $response;

			return self::$app_info;
		}
	}

	/**
	 * App is disabled.
	 *
	 * @return integer
	 */
	public function is_app_disabled() {
		$app_info = $this->get_app_info();

		if ( ! is_wp_error( $app_info ) && isset( $app_info['code'] ) && 'app_is_disabled' === $app_info['code'] ) {
			return true;
		}

		return false;
	}

	/**
	 * Verify the AppCenter App Connection.
	 *
	 * Enhanced with:
	 * - Conditional execution (only if connected)
	 * - Cache-based change detection
	 * - Smart time-based throttling
	 * - Early bailout for efficiency
	 *
	 * @since 2.10.0
	 *
	 * @return bool
	 */
	public function verify_app() {
		// Early bailout: Skip if not connected to App Center.
		if ( ! $this->is_app_center_connected() && ! $this->is_app_center_connected( 'secondary' ) ) {
			return false;
		}

		$app = $this->get_app();

		// Early bailout: Skip if app not properly configured.
		if ( empty( $app ) || ! isset( $app['verified'] ) || true !== $app['verified'] ) {
			return false;
		}

		if ( empty( $app['bbapp_app_id'] ) || empty( $app['bbapp_app_key'] ) || empty( $app['bbapp_site_type'] ) ) {
			return false;
		}

		// Get current state.
		$bb_app_id    = $app['bbapp_app_id'];
		$bb_app_key   = $app['bbapp_app_key'];
		$bb_site_type = $app['bbapp_site_type'];
		$current_url  = $this->get_home_url();

		// Get last verified state from cache.
		$cache_key        = 'bbapp_last_verified_app_state';
		$last_verified    = get_site_transient( $cache_key );
		$last_verify_time = isset( $last_verified['timestamp'] ) ? $last_verified['timestamp'] : 0;
		$last_url         = isset( $last_verified['url'] ) ? $last_verified['url'] : null;
		$last_site_type   = isset( $last_verified['site_type'] ) ? $last_verified['site_type'] : null;

		// Check if state has changed.
		$state_changed = (
			$last_url !== $current_url ||
			$last_site_type !== $bb_site_type
		);

		// Smart Throttling - Force verification every 2 hours even if no change.
		$force_verify_interval  = 2 * HOUR_IN_SECONDS; // 2 hours.
		$time_since_last_verify = time() - $last_verify_time;
		$should_force_verify    = ( $time_since_last_verify >= $force_verify_interval );

		// Decide whether to make API call.
		if ( ! $state_changed && ! $should_force_verify ) {
			// No change and not time for forced verification - skip API call.
			return false;
		}

		// Make the API call (state changed OR time for periodic verification).
		$app_info = $this->get_app_info( true );

		if ( ! is_wp_error( $app_info ) ) {
			$url = $current_url;

			if ( 'primary' === $bb_site_type && ! empty( $app_info['primary_site']->url ) && ! $this->compare_site_url( $url, $app_info['primary_site']->url ) ) {
				$this->disconnect_app( $bb_app_id, $bb_app_key, $bb_site_type );

				// Clear cache on disconnect.
				delete_site_transient( $cache_key );
			} elseif ( 'secondary' === $bb_site_type && ! empty( $app_info['secondary_sites'] ) ) {
				$secondary_sites      = $app_info['secondary_sites'];
				$secondary_sites_list = array();

				foreach ( $secondary_sites as $secondary_site ) {
					$secondary_sites_list[] = preg_replace( '#^[^:/.]*[:/]+#i', '', trailingslashit( trim( $secondary_site->url ) ) );
				}

				if ( ! in_array( trailingslashit( trim( preg_replace( '#^[^:/.]*[:/]+#i', '', trailingslashit( trim( $url ) ) ) ) ), $secondary_sites_list, true ) ) {
					$this->disconnect_app( $bb_app_id, $bb_app_key, $bb_site_type );

					// Clear cache on disconnect.
					delete_site_transient( $cache_key );
				} else {
					// Verification passed - update cache.
					$new_cache = array(
						'url'       => $current_url,
						'site_type' => $bb_site_type,
						'timestamp' => time(),
					);

					// Cache for 24 hours (cron will still check hourly, but API call is controlled).
					set_site_transient( $cache_key, $new_cache, 24 * HOUR_IN_SECONDS );
				}
			} else {
				// Primary verification passed - update cache.
				$new_cache = array(
					'url'       => $current_url,
					'site_type' => $bb_site_type,
					'timestamp' => time(),
				);
				// Cache for 24 hours (cron will still check hourly, but API call is controlled).
				set_site_transient( $cache_key, $new_cache, 24 * HOUR_IN_SECONDS );
			}

			return true;
		}

		return false;
	}

	/**
	 * Compare urls.
	 *
	 * @param string $url1 URL to compare.
	 * @param string $url2 URL to compare.
	 *
	 * @return bool
	 */
	public function compare_site_url( $url1, $url2 ) {
		$url1 = preg_replace( '#^[^:/.]*[:/]+#i', '', trailingslashit( trim( $url1 ) ) );
		$url2 = preg_replace( '#^[^:/.]*[:/]+#i', '', trailingslashit( trim( $url2 ) ) );

		if ( ! empty( $url1 ) && ! empty( $url2 ) && $url1 === $url2 ) {
			return true;
		}

		return false;
	}

	/**
	 * Tells if App Center is in Maintenance.
	 *
	 * @param bool $override_cache If needs to override cache.
	 *
	 * @return bool | WP_Error
	 */
	public function is_appcenter_under_maintenance( $override_cache = false ) {
		if ( ! ManageApp::instance()->is_app_center_connected() && ! ManageApp::instance()->is_app_center_connected( 'secondary' ) ) {
			return new WP_Error( 'app_not_connected', __( 'The app is not connected to the app center.', 'buddyboss-app' ) );
		}

		$override_cache = true;
		$retval         = false;
		$cache_key      = 'bbapp_is_under_maintenance';
		$cached         = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$retval = $cached;
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-site-status' );
			$request = bbapp_remote_get( $api_url );
			$body    = ( is_array( $request ) && isset( $request['body'] ) ? json_decode( $request['body'], true ) : array() );

			if ( isset( $body['code'] ) && 'maintenance_mode_enabled' === $body['code'] ) {
				$retval = true;

				set_site_transient( $cache_key, $retval, 10 * 60 ); // cache it for 10 min.
			} elseif ( is_wp_error( $request ) || ! isset( $body['success'] ) || true !== (bool) $body['success'] ) {
				$retval = new WP_Error(
					'could_not_connect_to_appcenter',
					__( 'Couldn\'t contact to app center, please refresh the page.', 'buddyboss-app' ),
					array( 'status' => 400 )
				);
			}
		}

		return $retval;
	}

	/**
	 * Tells if App Center is in Maintenance.
	 *
	 * @param bool $override_cache If needs to over-ride cache.
	 *
	 * @return bool
	 */
	public function get_appcenter_publisher_email_address( $override_cache = false ) {
		$override_cache = true;
		$cache_key      = 'bbapp_publisher_email_address';
		$cached         = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$retval = $cached;
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/support-email-address' );
			$request = bbapp_remote_get( $api_url );
			$body    = ( is_array( $request ) && isset( $request['body'] ) ? json_decode( $request['body'], true ) : array() );

			if ( isset( $body['support_email_address'] ) && ! empty( $body['support_email_address'] ) ) {
				$retval = $body['support_email_address'];

				set_site_transient( $cache_key, $retval, 10 * 60 ); // cache it for 10 min.
			} else {
				$retval = 'apphelp@buddyboss.com';
			}
		}

		return $retval;
	}

	/**
	 * Get Primary site URL.
	 */
	public function get_primary_site_url() {
		$app_info = $this->get_app_info();

		if ( is_wp_error( $app_info ) ) {
			return '';
		}

		if ( ! is_wp_error( $app_info ) ) {
			return ( isset( $app_info['primary_site']->url ) ? $app_info['primary_site']->url : '' );
		}

		return '';
	}

	/**
	 * Get Primary site admin URL.
	 * 
	 * @since 2.3.20
	 * 
	 * @return string
	 */
	public function get_primary_site_admin_url() {
		$app_info = $this->get_app_info();

		if ( is_wp_error( $app_info ) ) {
			return '';
		}

		return ( isset( $app_info['primary_site']->admin_url ) ? $app_info['primary_site']->admin_url : $this->get_primary_site_url() );

	}

	/**
	 * Get Secondary sites URL.
	 */
	public function get_secondary_sites_url() {
		$app_info = $this->get_app_info();
		$sites    = array();

		if ( is_wp_error( $app_info ) ) {
			return $sites;
		}

		if ( ! is_wp_error( $app_info ) && isset( $app_info['secondary_sites'] ) && is_array( $app_info['secondary_sites'] ) ) {
			foreach ( $app_info['secondary_sites'] as $site ) {
				$sites[] = ( isset( $site->url ) ? $site->url : '' );
			}
		}

		return $sites;
	}

	/**
	 * Validates the requests coming from BuddyBoss Server
	 *
	 * @param bool $token App token.
	 *
	 * @return bool
	 */
	public function validate_app_id_request( $token = false ) {
		$info = self::instance()->get_app();

		if ( sha1( $info['bbapp_app_key'] . $info['bbapp_app_id'] ) === $token ) {
			return true;
		}

		return false;
	}

	/**
	 * Get site url.
	 *
	 * @return string|void
	 */
	public function get_site_url() {
		return bbapp()->is_network_activated() ? network_site_url() : site_url();
	}

	/**
	 * Get home url.
	 *
	 * @since 2.3.20
	 *
	 * @return string|void
	 */
	public function get_home_url() {
		return bbapp()->is_network_activated() ? network_home_url() : home_url();
	}

	/**
	 * Check and verify Google/Apple connectivity.
	 *
	 * Syncing via this function should only sync store cached values only.
	 * Please check verify_apps_connection function for real connection flow.
	 *
	 * Enhanced with:
	 * - Conditional execution (only if connected)
	 * - Cache-based change detection
	 * - Smart time-based throttling
	 * - Early bailout for efficiency
	 *
	 * @since 2.10.0
	 *
	 * @return bool
	 */
	public function sync_store_connectivity_with_appcenter() {
		// Conditional Execution - Only run if connected to App Center.
		if ( ! $this->is_app_center_connected() && ! $this->is_app_center_connected( 'secondary' ) ) {
			return false; // Not connected, skip sync.
		}

		// Get current connection statuses.
		$android_connection     = Android::instance()->is_google_connected();
		$apple_connection       = Apple::instance()->is_connected();
		$current_android_status = ( ! is_wp_error( $android_connection ) && $android_connection );
		$current_apple_status   = ( ! is_wp_error( $apple_connection ) && $apple_connection );

		// Get last synced statuses from cache.
		$cache_key      = 'app_last_synced_store_connectivity';
		$last_synced    = get_site_transient( $cache_key );
		$last_sync_time = isset( $last_synced['timestamp'] ) ? $last_synced['timestamp'] : 0;
		$last_android   = isset( $last_synced['android'] ) ? $last_synced['android'] : null;
		$last_apple     = isset( $last_synced['apple'] ) ? $last_synced['apple'] : null;

		// Check if status has changed.
		$status_changed = (
			$last_android !== $current_android_status ||
			$last_apple !== $current_apple_status
		);

		// Smart Throttling - Force sync every 2 hours even if no change.
		$force_sync_interval  = 2 * HOUR_IN_SECONDS; // 2 hours.
		$time_since_last_sync = time() - $last_sync_time;
		$should_force_sync    = ( $time_since_last_sync >= $force_sync_interval );

		// Decide whether to make API call.
		if ( ! $status_changed && ! $should_force_sync ) {
			// No change and not time for forced sync - skip API call.
			return false;
		}

		// Make the API call (status changed OR time for periodic sync).
		$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'update-app-info' );

		$response = bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => array(
					'bbapp_is_google_connected' => $current_android_status,
					'bbapp_is_apple_connected'  => $current_apple_status,
					'bbapp_id'                  => $this->get_app_id(),
					'bbapp_key'                 => $this->get_auth_app_key(),
				),
			)
		);

		// If successful, update cache.
		if ( ! is_wp_error( $response ) ) {
			$new_cache = array(
				'android'   => $current_android_status,
				'apple'     => $current_apple_status,
				'timestamp' => time(),
			);

			// Cache for 24 hours (cron will still check every 30 min, but API call is controlled).
			set_site_transient( $cache_key, $new_cache, 24 * HOUR_IN_SECONDS );

			return true;
		}

		return false;
	}

	/**
	 * Is App connected on app center.
	 *
	 * @param string $as Site as.
	 *
	 * @return bool
	 * @since 1.4.3
	 */
	public function is_app_center_connected( $as = 'primary' ) {
		$app = $this->get_app();

		if ( ! empty( $app ) && isset( $app['verified'] ) && true === $app['verified'] && ! empty( $app['bbapp_app_id'] ) && ! empty( $app['bbapp_app_key'] ) && ! empty( $app['bbapp_site_type'] ) ) {
			$bb_site_type = $app['bbapp_site_type'];
			$app_info     = $this->get_app_info();

			if ( ! is_wp_error( $app_info ) ) {
				$url = $this->get_home_url();

				if ( 'primary' === $bb_site_type && ! empty( $app_info['primary_site']->url ) && $this->compare_site_url( $url, $app_info['primary_site']->url ) ) {
					return ( 'primary' === $as ) ? true : false;
				} elseif ( 'secondary' === $bb_site_type && ! empty( $app_info['secondary_sites'] ) ) {
					$secondary_sites      = $app_info['secondary_sites'];
					$secondary_sites_list = array();

					foreach ( $secondary_sites as $secondary_site ) {
						$secondary_sites_list[] = preg_replace( '#^[^:/.]*[:/]+#i', '', trailingslashit( trim( $secondary_site->url ) ) );
					}

					if ( in_array( trailingslashit( trim( preg_replace( '#^[^:/.]*[:/]+#i', '', trailingslashit( trim( $url ) ) ) ) ), $secondary_sites_list, true ) ) {
						return ( 'secondary' === $as ) ? true : false;
					}
				}
			}
		}

		return false;
	}

	/**
	 * Create AppCenter app logs.
	 *
	 * @param string $development_type Two development type (ios or android).
	 * @param string $type             type of action.
	 * @param string $message          message of log.
	 *
	 * @since 1.4.3
	 *@return bool|WP_Error
	 */
	public function bbapp_appcenter_app_log( $development_type, $type, $message ) {
		$api_url       = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/add-app-log' );
		$bbapp_app_id  = self::instance()->get_app_id();
		$bbapp_app_key = self::instance()->get_auth_app_key();
		if ( empty( $bbapp_app_id ) || empty( $bbapp_app_key ) ) {
			return new WP_Error( 'value_missing', __( 'Please provide your App ID & Key.', 'buddyboss-app' ) );
		}

		$body     = array(
			'bbapp_id'         => $bbapp_app_id,
			'bbapp_key'        => $bbapp_app_key,
			'development_type' => $development_type,
			'type'             => $type,
			'message'          => $message,
		);
		$response = bbapp_remote_post(
			$api_url,
			array(
				'timeout'  => 15,
				'blocking' => true,
				'cookies'  => array(),
				'body'     => $body,
			)
		);

		if ( is_wp_error( $response ) ) {
			return new WP_Error( 'cannot_create_log', __( 'Unable to create log to the BuddyBoss server.', 'buddyboss-app' ) );
		} else {
			$body     = isset( $response['body'] ) ? $response['body'] : '';
			$response = json_decode( $body );

			if ( is_array( $response ) || is_object( $response ) ) {
				$response = (array) $response;

				if ( isset( $response['upp_log_id'] ) ) {
					return true;
				}
			} else {
				return new WP_Error( 'cannot_create_log_to_server', __( 'Unable to create log to the BuddyBoss server.', 'buddyboss-app' ) );
			}
		}

		return false;
	}

	/**
	 * Disconnect development access.
	 *
	 * @since 1.4.4
	 */
	public function disconnect_app_development_access() {
		$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'set-app-dev-access' );

		return bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => array(
					'repo_url'     => '',
					'provider'     => '',
					'access_token' => '',
					'bbapp_id'     => $this->get_app_id(),
					'bbapp_key'    => $this->get_auth_app_key(),
				),
			)
		);
	}
}
