<?php

declare (strict_types=1);
namespace BuddyBossAppScoping\Readdle\AppStoreServerAPI;

use Exception;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Exception\HTTPRequestAborted;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Exception\HTTPRequestFailed;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Exception\WrongEnvironmentException;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Exception\InvalidImplementationException;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Exception\MalformedResponseException;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Exception\UnimplementedContentTypeException;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\AbstractRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\ExtendSubscriptionRenewalDateRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\GetAllSubscriptionStatusesRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\GetNotificationHistoryRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\GetRefundHistoryRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\GetStatusOfSubscriptionRenewalDateExtensionsRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\GetTestNotificationStatusRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\GetTransactionHistoryRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\GetTransactionHistoryRequestV2;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\GetTransactionInfoRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\LookUpOrderIdRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\MassExtendSubscriptionRenewalDateRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\RequestTestNotificationRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Request\SendConsumptionInformationRequest;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestBody\AbstractRequestBody;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestBody\ConsumptionRequestBody;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestBody\ExtendRenewalDateRequestBody;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestBody\MassExtendRenewalDateRequestBody;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestBody\NotificationHistoryRequestBody;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestQueryParams\AbstractRequestQueryParams;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestQueryParams\GetAllSubscriptionStatusesQueryParams;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestQueryParams\GetNotificationHistoryQueryParams;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestQueryParams\GetRefundHistoryQueryParams;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\RequestQueryParams\GetTransactionHistoryQueryParams;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\AbstractResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\CheckTestNotificationResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\ExtendRenewalDateResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\MassExtendRenewalDateResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\MassExtendRenewalDateStatusResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\NotificationHistoryResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\OrderLookupResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\RefundHistoryResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\SendTestNotificationResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\HistoryResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\StatusResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\TransactionInfoResponse;
use function call_user_func;
use function in_array;
use function is_subclass_of;
final class AppStoreServerAPI implements AppStoreServerAPIInterface
{
    const PRODUCTION_BASE_URL = 'https://api.storekit.itunes.apple.com/inApps';
    const SANDBOX_BASE_URL = 'https://api.storekit-sandbox.itunes.apple.com/inApps';
    private string $environment;
    private Payload $payload;
    private Key $key;
    /**
     * @throws WrongEnvironmentException
     */
    public function __construct(string $environment, string $issuerId, string $bundleId, string $keyId, string $key)
    {
        if (!in_array($environment, [Environment::PRODUCTION, Environment::SANDBOX])) {
            throw new WrongEnvironmentException($environment);
        }
        $this->environment = $environment;
        $this->payload = new Payload($issuerId, $bundleId);
        $this->key = new Key($keyId, $key);
    }
    public function getTransactionHistory(string $transactionId, array $queryParams = []) : HistoryResponse
    {
        /**
         * @var HistoryResponse $response
         */
        $response = $this->performRequest(GetTransactionHistoryRequest::class, HistoryResponse::class, ['transactionId' => $transactionId], new GetTransactionHistoryQueryParams($queryParams));
        return $response;
    }
    public function getTransactionHistoryV2(string $transactionId, array $queryParams = []) : HistoryResponse
    {
        /**
         * @var HistoryResponse $response
         */
        $response = $this->performRequest(GetTransactionHistoryRequestV2::class, HistoryResponse::class, ['transactionId' => $transactionId], new GetTransactionHistoryQueryParams($queryParams));
        return $response;
    }
    public function getTransactionInfo(string $transactionId) : TransactionInfoResponse
    {
        /**
         * @var TransactionInfoResponse $response
         */
        $response = $this->performRequest(GetTransactionInfoRequest::class, TransactionInfoResponse::class, ['transactionId' => $transactionId]);
        return $response;
    }
    public function getAllSubscriptionStatuses(string $transactionId, array $queryParams = []) : StatusResponse
    {
        /**
         * @var StatusResponse $response
         */
        $response = $this->performRequest(GetAllSubscriptionStatusesRequest::class, StatusResponse::class, ['transactionId' => $transactionId], new GetAllSubscriptionStatusesQueryParams($queryParams));
        return $response;
    }
    public function sendConsumptionInformation(string $transactionId, array $requestBody) : void
    {
        $this->performRequest(SendConsumptionInformationRequest::class, null, ['transactionId' => $transactionId], null, new ConsumptionRequestBody($requestBody));
    }
    public function lookUpOrderId(string $orderId) : OrderLookupResponse
    {
        /**
         * @var OrderLookupResponse $response
         */
        $response = $this->performRequest(LookUpOrderIdRequest::class, OrderLookupResponse::class, ['orderId' => $orderId]);
        return $response;
    }
    public function getRefundHistory(string $transactionId) : RefundHistoryResponse
    {
        /**
         * @var RefundHistoryResponse $response
         */
        $response = $this->performRequest(GetRefundHistoryRequest::class, RefundHistoryResponse::class, ['transactionId' => $transactionId], new GetRefundHistoryQueryParams());
        return $response;
    }
    /**
     * @inheritdoc
     * @throws Exception
     */
    public function extendSubscriptionRenewalDate(string $originalTransactionId, array $requestBody) : ExtendRenewalDateResponse
    {
        /**
         * @var ExtendRenewalDateResponse $response
         */
        $response = $this->performRequest(ExtendSubscriptionRenewalDateRequest::class, ExtendRenewalDateResponse::class, ['originalTransactionId' => $originalTransactionId], null, new ExtendRenewalDateRequestBody($requestBody));
        return $response;
    }
    /**
     * @inheritdoc
     * @throws Exception
     */
    public function massExtendSubscriptionRenewalDate(array $requestBody) : MassExtendRenewalDateResponse
    {
        /**
         * @var MassExtendRenewalDateResponse $response
         */
        $response = $this->performRequest(MassExtendSubscriptionRenewalDateRequest::class, MassExtendRenewalDateResponse::class, [], null, new MassExtendRenewalDateRequestBody($requestBody));
        return $response;
    }
    public function getStatusOfSubscriptionRenewalDateExtensionsRequest(string $productId, string $requestIdentifier) : MassExtendRenewalDateStatusResponse
    {
        /**
         * @var MassExtendRenewalDateStatusResponse $response
         */
        $response = $this->performRequest(GetStatusOfSubscriptionRenewalDateExtensionsRequest::class, MassExtendRenewalDateStatusResponse::class, ['productId' => $productId, 'requestIdentifier' => $requestIdentifier]);
        return $response;
    }
    public function getNotificationHistory(array $requestBody) : NotificationHistoryResponse
    {
        /**
         * @var NotificationHistoryResponse $response
         */
        $response = $this->performRequest(GetNotificationHistoryRequest::class, NotificationHistoryResponse::class, [], new GetNotificationHistoryQueryParams(), new NotificationHistoryRequestBody($requestBody));
        return $response;
    }
    public function requestTestNotification() : SendTestNotificationResponse
    {
        /**
         * @var SendTestNotificationResponse $response
         */
        $response = $this->performRequest(RequestTestNotificationRequest::class, SendTestNotificationResponse::class);
        return $response;
    }
    public function getTestNotificationStatus(string $testNotificationToken) : CheckTestNotificationResponse
    {
        /**
         * @var CheckTestNotificationResponse $response
         */
        $response = $this->performRequest(GetTestNotificationStatusRequest::class, CheckTestNotificationResponse::class, ['testNotificationToken' => $testNotificationToken]);
        return $response;
    }
    private function createRequest(string $requestClass, ?AbstractRequestQueryParams $queryParams, ?AbstractRequestBody $body) : AbstractRequest
    {
        /** @var AbstractRequest $request */
        $request = new $requestClass($this->key, $this->payload, $queryParams, $body);
        $request->setURLVars(['baseUrl' => $this->getBaseURL()]);
        return $request;
    }
    private function getBaseURL() : string
    {
        return $this->environment === Environment::PRODUCTION ? self::PRODUCTION_BASE_URL : self::SANDBOX_BASE_URL;
    }
    /**
     * @param array<string, mixed> $requestUrlVars
     *
     * @throws HTTPRequestAborted
     * @throws HTTPRequestFailed
     * @throws InvalidImplementationException
     * @throws MalformedResponseException
     * @throws UnimplementedContentTypeException
     */
    private function performRequest(string $requestClass, ?string $responseClass, array $requestUrlVars = [], ?AbstractRequestQueryParams $requestQueryParams = null, ?AbstractRequestBody $requestBody = null) : ?AbstractResponse
    {
        if (!is_subclass_of($requestClass, AbstractRequest::class) || !empty($responseClass) && !is_subclass_of($responseClass, AbstractResponse::class)) {
            throw new InvalidImplementationException($requestClass, $responseClass);
        }
        $request = $this->createRequest($requestClass, $requestQueryParams, $requestBody);
        if (!empty($requestUrlVars)) {
            $request->setURLVars($requestUrlVars);
        }
        $responseText = HTTPRequest::performRequest($request);
        if (empty($responseClass)) {
            return null;
        }
        return call_user_func([$responseClass, 'createFromString'], $responseText, $request);
    }
}
