<?php

/**
 * Google Play Developer API wrapper for PHP 7.4 compatibility.
 *
 * @since   2.9.1
 * @package BuddyBossApp\Library\Composer
 */
namespace BuddyBossAppScoping\BuddyBossApp\Library\Composer;

/**
 * Google Play Developer API wrapper for PHP 7.4 compatibility.
 * Handles the new monetization API endpoints via direct HTTP calls.
 *
 * @since 2.9.1
 */
class GooglePlayAPI
{
    /**
     * Class instance.
     *
     * @var GooglePlayAPI|null
     */
    private static $instance;
    /**
     * Google API access token.
     *
     * @var string
     */
    private $access_token;
    /**
     * Android package name.
     *
     * @var string
     */
    private $package_name;
    /**
     * Get the instance of the class.
     *
     * @return GooglePlayAPI
     */
    public static function instance()
    {
        if (!isset(self::$instance)) {
            $class = __CLASS__;
            self::$instance = new $class();
        }
        return self::$instance;
    }
    /**
     * Initialize with Google Client for authentication.
     *
     * @param \Google_Client $google_client The authenticated Google Client instance.
     * @param string         $package_name  The Android package name.
     *
     * @since 2.9.1
     * @throws \Exception If access token cannot be obtained.
     */
    public function init($google_client, $package_name)
    {
        $this->package_name = $package_name;
        // Get access token from authenticated Google Client.
        $token_data = $google_client->fetchAccessTokenWithAssertion();
        if (isset($token_data['access_token'])) {
            $this->access_token = $token_data['access_token'];
        } else {
            throw new \Exception('Failed to obtain access token from Google Client');
        }
    }
    /**
     * Make HTTP request to Google Play API.
     *
     * @param string $endpoint The API endpoint path.
     * @param array  $params   Optional query parameters.
     *
     * @since 2.9.1
     * @return array The decoded JSON response.
     * @throws \Exception If access token is not available, request error occurs, or invalid response.
     */
    private function make_request($endpoint, $params = array())
    {
        if (empty($this->access_token)) {
            throw new \Exception('Access token not available');
        }
        $base_url = 'https://androidpublisher.googleapis.com/androidpublisher/v3/applications/' . $this->package_name;
        $url = $base_url . $endpoint;
        if (!empty($params)) {
            $url .= '?' . \http_build_query($params);
        }
        $request_args = array('timeout' => 30, 'headers' => array('Authorization' => 'Bearer ' . $this->access_token, 'Content-Type' => 'application/json', 'Accept' => 'application/json'));
        $response = bbapp_remote_get($url, $request_args);
        if (is_wp_error($response)) {
            throw new \Exception('Request Error: ' . esc_html($response->get_error_message()));
        }
        $http_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        if (200 !== $http_code) {
            $error_data = \json_decode($body, \true);
            $error_message = $error_data['error']['message'] ?? 'HTTP Error: ' . $http_code;
            throw new \Exception(esc_html($error_message) . ' (HTTP ' . (int) $http_code . ')');
        }
        $data = \json_decode($body, \true);
        if (\JSON_ERROR_NONE !== \json_last_error()) {
            throw new \Exception('Invalid JSON response: ' . esc_html(\json_last_error_msg()));
        }
        return $data;
    }
    /**
     * Get in-app products (one-time purchases).
     *
     * @since 2.9.1
     * @return array Array of formatted in-app products.
     */
    public function get_inapp_products()
    {
        try {
            // Use the current working Google Play Developer API v3 endpoint for listing in-app products.
            $response = $this->make_request('/oneTimeProducts');
            // Handle the oneTimeProducts response format.
            if (isset($response['oneTimeProducts']) && !empty($response['oneTimeProducts'])) {
                return $this->format_inapp_products($response['oneTimeProducts']);
            }
            return array();
        } catch (\Exception $e) {
            if (\defined('BuddyBossAppScoping\\WP_DEBUG') && WP_DEBUG) {
                \error_log('GooglePlayAPI: Failed to fetch in-app products: ' . $e->getMessage());
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            }
            return array();
        }
    }
    /**
     * Get subscription products.
     *
     * @since 2.9.1
     * @return array Array of formatted subscription products.
     */
    public function get_subscriptions()
    {
        try {
            // Use the current working Google Play Developer API v3 endpoint.
            // URL: applications/{packageName}/subscriptions (for monetization.subscriptions).
            $response = $this->make_request('/subscriptions');
            // Handle different response formats from monetization API.
            if (isset($response['subscriptions'])) {
                return $this->format_subscriptions($response['subscriptions']);
            } elseif (isset($response['subscription'])) {
                return $this->format_subscriptions($response['subscription']);
            } elseif (isset($response['results'])) {
                return $this->format_subscriptions($response['results']);
            } elseif (\is_array($response) && !empty($response)) {
                return $this->format_subscriptions($response);
            }
            return array();
        } catch (\Exception $e) {
            if (\defined('BuddyBossAppScoping\\WP_DEBUG') && WP_DEBUG) {
                \error_log('GooglePlayAPI: Failed to fetch subscriptions: ' . $e->getMessage());
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            }
            return array();
        }
    }
    /**
     * Get all products (in-app products and subscriptions combined).
     *
     * @since 2.9.1
     * @return array Array of all formatted products.
     */
    public function get_all_products()
    {
        $products = array();
        // Get in-app products.
        $inapp_products = $this->get_inapp_products();
        if (!empty($inapp_products)) {
            $products = \array_merge($products, $inapp_products);
        }
        // Get subscriptions.
        $subscriptions = $this->get_subscriptions();
        if (!empty($subscriptions)) {
            $products = \array_merge($products, $subscriptions);
        }
        return $products;
    }
    /**
     * Format in-app products response.
     *
     * @param array $products Raw product data from Google Play API.
     *
     * @since 2.9.1
     * @return array Formatted product array.
     */
    private function format_inapp_products($products)
    {
        $formatted = array();
        foreach ($products as $product) {
            $title = '';
            $status = 'active';
            $product_id = $product['productId'] ?? '';
            // Extract title from listings (en-US or first available).
            if (!empty($product['listings'])) {
                if (isset($product['listings']['en-US']['title'])) {
                    $title = $product['listings']['en-US']['title'];
                } else {
                    $first_listing = \reset($product['listings']);
                    $title = $first_listing['title'] ?? '';
                }
            }
            // Extract status from purchaseOptions.
            if (!empty($product['purchaseOptions'])) {
                $first_option = \reset($product['purchaseOptions']);
                if (isset($first_option['state'])) {
                    $status = \strtolower($first_option['state']);
                }
            }
            $formatted[] = array('id' => $product_id, 'name' => !empty($title) ? $title . ' (' . $product_id . ')' : $product_id, 'type' => 'managedUser', 'status' => $status);
        }
        return $formatted;
    }
    /**
     * Format subscriptions response.
     *
     * @param array $subscriptions Raw subscription data from Google Play API.
     *
     * @since 2.9.1
     * @return array Formatted subscription array.
     */
    private function format_subscriptions($subscriptions)
    {
        $formatted = array();
        foreach ($subscriptions as $subscription) {
            $title = '';
            $status = 'inactive';
            // Extract title from listings.
            if (!empty($subscription['listings'])) {
                $first_listing = \reset($subscription['listings']);
                $title = $first_listing['title'] ?? '';
            }
            // Extract status.
            if (\is_array($subscription['basePlans']) && !empty($subscription['basePlans'])) {
                foreach ($subscription['basePlans'] as $base_plan) {
                    if (isset($base_plan['state']) && 'active' === \strtolower($base_plan['state'])) {
                        $status = 'active';
                        break;
                    }
                }
            }
            $formatted[] = array('id' => $subscription['sku'] ?? $subscription['productId'] ?? '', 'name' => $title . ' (' . ($subscription['sku'] ?? $subscription['productId'] ?? '') . ')', 'type' => 'subscription', 'status' => $status);
        }
        return $formatted;
    }
}
