<?php

/**
 * This class provides functionalities to interact with Apple's App Store Server API.
 *
 * @package BuddyBossApp\Library\Composer
 */
namespace BuddyBossAppScoping\BuddyBossApp\Library\Composer;

use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Environment;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Exception\AppStoreServerAPIException;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Exception\MalformedJWTException;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Exception\WrongEnvironmentException;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Response\TransactionInfoResponse;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\TransactionInfo;
use BuddyBossAppScoping\Readdle\AppStoreServerAPI\Util\JWT;
use BuddyBossAppScoping\WP_Error;
/**
 * Class AppStoreServerAPI
 *
 * Provides helper methods to interact with the App Store Server API for managing transactions, subscriptions, and environments.
 */
class AppStoreServerAPI
{
    /**
     * Class instance.
     *
     * @var $instance
     */
    private static $instance;
    /**
     * Get the instance of the class.
     *
     * @since 2.3.91
     * @return AppStoreServerAPI
     */
    public static function instance()
    {
        if (!isset(self::$instance)) {
            $class = __CLASS__;
            self::$instance = new $class();
        }
        return self::$instance;
    }
    /**
     * Initializes and creates an instance of the AppStoreServerAPI class.
     *
     * @param string $environment The environment to use (e.g., production or sandbox).
     * @param string $issuer_id The unique identifier issued by Apple for the App Store Connect API key.
     * @param string $release_bundle_id The bundle identifier of your app.
     * @param string $key_id The key identifier of the App Store Connect API key.
     * @param string $secret_content The private key content associated with the API key, used for authentication.
     *
     * @since 2.3.91
     * @return \Readdle\AppStoreServerAPI\AppStoreServerAPI An instance of the AppStoreServerAPI class.
     */
    public function AppStoreServerAPI($environment, $issuer_id, $release_bundle_id, $key_id, $secret_content)
    {
        try {
            $api_client = new \BuddyBossAppScoping\Readdle\AppStoreServerAPI\AppStoreServerAPI($environment, $issuer_id, $release_bundle_id, $key_id, $secret_content);
        } catch (WrongEnvironmentException $e) {
            $api_client = new WP_Error('error_api_client_connection', __('Failed to fetch the API client: ', 'buddyboss-app') . $e->getMessage());
        }
        return $api_client;
    }
    /**
     * Determines whether the receipt token is valid for production or sandbox.
     *
     * @param string $iap_receipt_token Receipt token to validate.
     *
     * @since 2.3.91
     * @return WP_Error | bool
     */
    public function isProduction($iap_receipt_token)
    {
        $is_production = \false;
        try {
            $decoded_transaction_info = TransactionInfo::createFromRawTransactionInfo(JWT::parse($iap_receipt_token));
            if ('Production' === $decoded_transaction_info->getEnvironment()) {
                $is_production = \true;
            }
        } catch (MalformedJWTException $e) {
            return new WP_Error('error_iap_validation', __('Error while validating order, could be invalid receipt token.', 'buddyboss-app') . $e->getMessage());
        }
        return $is_production;
    }
    /**
     * Determines the appropriate environment based on the provided flag.
     *
     * @param bool $is_production Indicates whether the production environment should be used. True for production, false for sandbox.
     *
     * @since 2.3.91
     * @return string The environment constant, either production or sandbox.
     */
    public function get_environment($is_production)
    {
        return $is_production ? Environment::PRODUCTION : Environment::SANDBOX;
    }
}
