<?php
/**
 * Holds jobs related functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\Tools\Logger;

/**
 * Class Jobs
 * Class helps to execute jobs on WordPress.
 */
class Jobs {
	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Job table name.
	 *
	 * @var string $table
	 */
	private $table = 'bbapp_queue';

	/**
	 * Max items per batch.
	 *
	 * @var int $max_item_per_batch
	 */
	protected $max_item_per_batch = 0;

	/**
	 * Max memory allowed.
	 *
	 * @var string $max_memory_limit
	 */
	protected $max_memory_limit = '2000M';

	/**
	 * Minimum memory.
	 *
	 * @var string
	 */
	protected $min_memory_limit = '256M';

	/**
	 * Execution start time.
	 *
	 * @var int $exec_start_time
	 */
	protected $exec_start_time = 0;

	/**
	 * Max execution time.
	 *
	 * @var int $max_execution_time
	 */
	protected $max_execution_time = 20; // 20 sec.

	/**
	 * Jobs constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of class.
	 *
	 * @return Jobs
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load the class required code.
	 */
	public function load() {
		/**
		 * Filter to use to change max item per batch.
		 *
		 * @param int $per_batch Per batch.
		 *
		 * @since 1.7.7
		 */
		$this->max_item_per_batch = (int) apply_filters( 'bbapp_max_item_per_batch', 20 );

		// Prefix Table Name.
		$this->table = bbapp_get_network_table( $this->table );

		add_action( 'wp_ajax_bbapp_queue', array( $this, 'run_process' ) );
		add_action( 'wp_ajax_nopriv_bbapp_queue', array( $this, 'run_process' ) );
		add_action( 'bbapp_every_5min', array( $this, 'rescue_the_runner' ) );
	}

	/**
	 * Start background job.
	 */
	public function rescue_the_runner() {
		$this->start(); // starts if it's requires.
	}

	/**
	 * Add Queue Entry on Database.
	 *
	 * @param string $type     Job type.
	 * @param array  $data     Job data.
	 * @param int    $priority Priority.
	 *
	 * @return bool
	 */
	public function add( $type, $data = array(), $priority = 5 ) {
		global $wpdb;

		if ( empty( $type ) || empty( $priority ) ) {
			return false;
		}

		$insert = $wpdb->insert( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$this->table,
			array(
				'created'  => current_time( 'mysql', 1 ),
				'modified' => current_time( 'mysql', 1 ),
				'type'     => $type,
				'data'     => maybe_serialize( $data ),
				'blog_id'  => get_current_blog_id(),
				'priority' => $priority,
			)
		);

		return $insert;
	}

	/**
	 * Delete queue item from database.
	 *
	 * @param int $id Id to delete.
	 *
	 * @return bool
	 */
	public function delete( $id ) {
		global $wpdb;

		if ( empty( $id ) ) {
			return false;
		}

		return $wpdb->delete( $this->table, array( 'id' => $id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	}

	/**
	 * Return the count of how many items are in queue for type.
	 *
	 * @param string $type Job type.
	 *
	 * @return int
	 */
	public function has_process_items( $type ) {
		global $wpdb;

		$count = $wpdb->get_var( $wpdb->prepare( "SELECT count(type) FROM {$this->table} WHERE type = %s AND blog_id = %s", $type, get_current_blog_id() ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		return (int) $count;
	}

	/**
	 * Get job types.
	 *
	 * @return array
	 */
	private function get_types() {
		global $wpdb;

		return $wpdb->get_col( $wpdb->prepare( "SELECT type FROM {$this->table} WHERE blog_id = %d", get_current_blog_id() ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Logs information.
	 *
	 * @param string $text Log text.
	 */
	public function log( $text ) {
		Logger::instance()->add(
			'bbapp_queue',
			$text
		);
	}

	/**
	 * Check if process running.
	 *
	 * @param string $job_type job type.
	 *
	 * @return bool
	 */
	public function determine_process_running( $job_type ) {
		$process = get_site_transient( 'bbapp_queue_process_lk_' . $job_type );
		$ret_val = false;

		if ( $process ) {
			$ret_val = true;
		}

		return $ret_val;
	}

	/**
	 * Lock process.
	 *
	 * @param string $job_type Job type.
	 *
	 * @return bool
	 */
	public function lock_process( $job_type ) {
		$this->exec_start_time = time();
		$lock_limit            = 60 * 2; // 2 min

		return set_site_transient( 'bbapp_queue_process_lk_' . $job_type, microtime(), $lock_limit );
	}

	/**
	 * Unlock process.
	 *
	 * @param string $job_type Job type.
	 *
	 * @return bool
	 */
	public function unlock_process( $job_type ) {
		return delete_site_transient( 'bbapp_queue_process_lk_' . $job_type );
	}

	/**
	 * Return the items to be process per batch
	 *
	 * @param string $job_type job type.
	 *
	 * @return array|null|object|false
	 */
	public function get_processing_items( $job_type ) {
		global $wpdb;

		if ( empty( $job_type ) ) {
			return false;
		}

		return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$this->table} WHERE type = %s AND blog_id = %s ORDER BY `created`, `priority` LIMIT %d", $job_type, get_current_blog_id(), $this->max_item_per_batch ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Process tasks
	 */
	public function run_process() {
		$job_type  = ( ! empty( $_GET['type'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['type'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$parent_id = ( ! empty( $_POST['parent_id'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['parent_id'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing

		if ( empty( $job_type ) ) {
			return;
		}

		$time_start = microtime( true );

		global $bbapp_queue_debug;

		if ( ! is_array( $bbapp_queue_debug ) ) {
			$bbapp_queue_debug = array();
		}

		session_write_close();

		register_shutdown_function( array( $this, 'shutdown' ) );

		$this->lock_process( $job_type );
		$this->log( __( 'BuddyBoss App queue batch has been started.', 'buddyboss-app' ) );

		$process                = $this->get_processing_items( $job_type );
		$task_count             = 0;
		$did_completed_smoothly = true;
		$is_finished            = false;

		if ( empty( $process ) ) {
			$is_finished = true;

			$this->log( __( 'All items have been finished closing batch process.', 'buddyboss-app' ) );
			$this->unlock_process( $job_type );
		}

		if ( ! empty( $process ) && ! $is_finished ) {
			foreach ( $process as $task ) {
				/**
				 * Action to use before running task.
				 *
				 * @param object $task      Task object.
				 * @param string $parent_id Parent id.
				 *
				 * @since 2.0.80
				 */
				do_action( 'bbapp_task_started', $task, $parent_id );

				$bbapp_queue_debug['running_task'] = $task;

				$this->delete( $task->id ); // delete task if it's not finished or anything processor is responsible for adding new if not finished.

				/**
				 * Action to use before running task.
				 *
				 * @param object $task Task object.
				 *
				 * @since 2.0.80
				 */
				do_action( "bbapp_queue_task_{$task->type}", $task );

				++$task_count;

				$bbapp_queue_debug['running_task']->done = true;

				if ( $this->php_limits_reached() ) {
					$did_completed_smoothly = false;
					/* translators: Task count. */
					$this->log( sprintf( __( 'Maximum PHP limit reached. Stopping batch after running %s tasks...', 'buddyboss-app' ), $task_count ) );
				}

				/**
				 * Action to use after running task.
				 *
				 * @param object $task      Task object.
				 * @param string $parent_id Parent id.
				 *
				 * @since 2.0.80
				 */
				do_action( 'bbapp_task_finished', $task, $parent_id, $did_completed_smoothly );

				if ( ! $did_completed_smoothly ) {
					break;
				}
			}
		}

		if ( $did_completed_smoothly ) {
			/* translators: Task count. */
			$this->log( sprintf( __( 'Finished %s tasks smoothly on this batch.', 'buddyboss-app' ), $task_count ) );
		} else {
			$this->log( __( 'batch was not finished smoothly due to some reasons check above.', 'buddyboss-app' ) );
		}

		$this->unlock_process( $job_type );
		$this->start(); // start another batch.

		$time_end       = microtime( true );
		$execution_time = ( $time_end - $time_start );

		echo '<b>Total Execution Time:</b> ' . esc_html( $execution_time ) . ' Sec';

		/**
		 * Fires when job is completed.
		 *
		 * @param array  $process   Running process.
		 * @param string $parent_id Parent id.
		 *
		 * @since 2.0.80
		 */
		do_action( 'bbapp_job_completed', $process, $parent_id );

		die( 'Success' );
	}

	/**
	 * Checks if we have more resources in php to continue.
	 *
	 * @return bool
	 */
	public function php_limits_reached() {
		$retval = false;

		/**
		 * Check Memory Limits.
		 */
		$memory_limit = ( function_exists( 'ini_get' ) ) ? ini_get( 'memory_limit' ) : $this->min_memory_limit;

		if ( intval( $memory_limit ) === - 1 || ! $memory_limit ) {
			$memory_limit = $this->max_memory_limit;
		}

		// mb to bytes.
		$memory_limit = intval( $memory_limit ) * pow( 1024, 2 );
		// Reduce it to some amount to be safe.
		$memory_limit  *= 0.8;
		$current_memory = memory_get_usage( true );

		if ( $current_memory >= $memory_limit ) {
			$retval = true;
		}

		if ( $retval ) {
			return $retval;
		}

		/**
		 * Check Timeout Limits.
		 */
		$executed_time = $this->exec_start_time + $this->max_execution_time;

		if ( time() >= $executed_time ) {
			$retval = true;
		}

		return $retval;
	}

	/**
	 * Dispatch the queue for processing.
	 */
	public function start() {
		$job_types = $this->get_types();

		if ( ! empty( $job_types ) ) {
			foreach ( $job_types as $job_type ) {
				if ( $this->determine_process_running( $job_type ) ) {
					return false;
				}

				$this->lock_process( $job_type ); // lock so we don't create duplicate process.

				$args = array(
					'timeout'   => apply_filters( 'bbapp_jobs_start_timeout', 0.01 ),
					'blocking'  => false,
					'body'      => array(),
					'cookies'   => $_COOKIE,
					'sslverify' => false,
				);

				/**
				 * Action to use before starting queue.
				 *
				 * @param array  $args     Args.
				 * @param string $job_type Job type.
				 *
				 * @since 2.0.80
				 */
				$args = apply_filters( 'bbapp_job_before_start', $args, $job_type );

				$job_url   = esc_url_raw( admin_url( 'admin-ajax.php' ) . '?action=bbapp_queue&type=' . $job_type );
				$job_start = wp_remote_post( $job_url, $args );

				/**
				 * Action to use after starting queue.
				 *
				 * @param string $job_type Job type.
				 *
				 * @since 2.0.80
				 */
				do_action( 'bbapp_job_after_start', $job_start, $job_type, $args );
			}
		}

		return true;
	}

	/**
	 * Handles any fatal error occurs while running tasks.
	 */
	public function shutdown() {
		global $bbapp_queue_debug;

		$job_type = ( ! empty( $_GET['type'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['type'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $job_type ) ) {
			return;
		}

		$last_error = error_get_last();

		if ( isset( $bbapp_queue_debug['running_task'] ) && ! isset( $bbapp_queue_debug['running_task']->done ) ) {
			$debug_info = array(
				'task'      => $bbapp_queue_debug['running_task'],
				'env_error' => $last_error,
			);
			/* translators: %s: Debug information.  */
			$prepare_log = sprintf( __( "There was an error while running tasks, debug info below \n ---- \n %s", 'buddyboss-app' ), print_r( $debug_info, true ) ); //phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r

			$this->log( $prepare_log );

			if ( E_ERROR === $last_error['type'] ) {
				$this->log( __( 'Restarting queue processes due to fatal error.', 'buddyboss-app' ) );
				$this->unlock_process( $job_type );
				$this->start(); // start another batch.
			}
		}
	}
}
