<?php
/**
 * Holds item meta utility functionality.
 *
 * @package BuddyBossApp\ItemMeta
 */

namespace BuddyBossApp\ItemMeta;

/**
 * ItemMetaUtils class.
 */
trait itemMetaUtils { // phpcs:ignore

	/**
	 * Update or insert the item.
	 *
	 * @param int    $item_id   Item id.
	 * @param string $item_type Item type.
	 * @param array  $meta_data Meta data.
	 *
	 * @return boolean
	 */
	public static function update_or_insert_item( $item_id, $item_type, $meta_data ) {
		global $wpdb;

		$table_name     = $wpdb->prefix . 'bbapp_itemmeta';
		$serialize_data = maybe_serialize( $meta_data );
		$exists         = $wpdb->get_results( $wpdb->prepare( "SELECT count(*) as total FROM {$table_name} WHERE item_id = %d", $item_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( isset( $exists[0]->total ) && 0 === (int) $exists[0]->total ) {
			$status = $wpdb->insert( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$table_name,
				array(
					'item_id'   => $item_id,
					'item_type' => $item_type,
					'meta_data' => $serialize_data,
				),
				array( '%d', '%s', '%s' )
			);
		} else {
			$status = $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$table_name,
				array(
					'item_type' => $item_type,
					'meta_data' => $serialize_data,
				),
				array( 'item_id' => $item_id )
			);
		}

		return $status;
	}

	/**
	 * Set Transient for item under BuddyBossApp prefix
	 *
	 * @param int $item_id Item id.
	 *
	 * @return array
	 */
	public static function set_transient_for_item( $item_id ) {
		$transient_name = ItemMetaManager::prepare_transient_name( $item_id );
		$meta_data      = self::prepare_meta( $item_id );

		// TODO : Verify/Decide right time for caching this information?
		set_transient( $transient_name, $meta_data, 1 * 60 );

		$data = get_transient( $transient_name );

		return $data;
	}


	/**
	 * Prepare Meta
	 *
	 * @param int $item_id Item id.
	 *
	 * @return array
	 */
	public static function prepare_meta( $item_id ) {
		$data          = array();
		$modified_time = get_post_modified_time( 'G', true, $item_id );
		$last_changed  = $modified_time;
		$child_ids     = array();
		$child_updates = array();

		// @todo : Implement below.
		$parent_ids     = array();
		$parent_updates = array();
		$post_type      = get_post_type( $item_id );

		switch ( $post_type ) {
			case 'sfwd-courses':
				if ( function_exists( 'learndash_get_course_lessons_list' ) ) {
					$lessons = learndash_get_course_lessons_list( $item_id, null, array() );

					foreach ( $lessons as $key => $lesson ) {
						$child_ids['lessons'][] = $lesson['post']->ID;
						$modified_time          = get_post_modified_time( 'G', true, $lesson['post']->ID );

						if ( ! empty( $child_updates['lessons'] ) && $child_updates['lessons'][ $lesson['post']->ID ] < $modified_time ) {
							$child_updates['lessons'][ $lesson['post']->ID ] = $modified_time;
						}
					}
				}

				if ( function_exists( 'learndash_course_get_steps_by_type' ) ) {
					$topics = learndash_course_get_steps_by_type( $item_id, 'sfwd-topic' );

					foreach ( $topics as $key => $topic_id ) {
						$child_ids['topics'][] = $topic_id;

						if ( ! empty( $child_updates['topics'] ) && $child_updates['topics'][ $topic_id ] < $modified_time ) {
							$child_updates['topics'][ $topic_id ] = $modified_time;
						}
					}
				}
				break;
		}

		$data['updates']    = array(
			'last_changed'   => $last_changed,
			'child_updated'  => $child_updates,
			'parent_updated' => $parent_updates,
		);
		$data['child_ids']  = $child_ids;
		$data['parent_ids'] = $parent_ids;

		return $data;
	}

	/**
	 * Prepare Last Changed.
	 *
	 * @param int    $item_id         Item id.
	 * @param string $since_timestamp SSince timestamp.
	 *
	 * @return array
	 */
	public static function prepare_last_changed( $item_id, $since_timestamp ) {
		$modified_time        = get_post_modified_time( 'G', true, $item_id );
		$data                 = array();
		$child_updates        = array();
		$parent_updates       = array();
		$data['self_changed'] = $modified_time;
		$post_type            = get_post_type( $item_id );

		switch ( $post_type ) {
			case 'sfwd-courses':
				if ( function_exists( 'learndash_get_course_lessons_list' ) ) {
					$lessons = learndash_get_course_lessons_list( $item_id, null, array() );

					foreach ( $lessons as $key => $lesson ) {
						$lesson_modified_time = get_post_modified_time( 'G', true, $lesson['post']->ID );

						if ( $lesson_modified_time > $since_timestamp ) {
							$child_updates[] = array(
								'id'   => $lesson['post']->ID,
								'type' => 'lesson',
							);
						}
					}
				}

				if ( function_exists( 'learndash_course_get_steps_by_type' ) ) {
					$topics = learndash_course_get_steps_by_type( $item_id, 'sfwd-topic' );

					foreach ( $topics as $key => $topic_id ) {
						$topic_modified_time = get_post_modified_time( 'G', true, $topic_id );

						if ( $topic_modified_time > $since_timestamp ) {
							$child_updates[] = array(
								'id'   => $topic_id,
								'type' => 'topic',
							);
						}
					}
				}
				break;
		}

		$data['child_updated']  = array_values( $child_updates );
		$data['parent_updated'] = array_values( $parent_updates );

		return $data;
	}


	/**
	 * Get Items by Type
	 *
	 * @param string $item_type Item type.
	 *
	 * @return array
	 */
	public static function get_items_by_type( $item_type ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'bbapp_itemmeta';

		return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table_name} WHERE item_type= %s", $item_type ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Prepare Transient Name
	 *
	 * @param int $item_id Item id.
	 *
	 * @return string
	 */
	public static function prepare_transient_name( $item_id ) {
		return bbapp()->transient_key . $item_id . '-meta';
	}
}
