<?php
/**
 * Holds meta api functionality.
 *
 * @package BuddyBossApp\ItemMeta
 */

namespace BuddyBossApp\ItemMeta;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}


use WP_Error;
use WP_REST_Request;
use WP_REST_Server;

/**
 * Meta api class.
 */
class MetaApi {

	/**
	 * Table name.
	 *
	 * @var string $table_name
	 */
	public $table_name = 'bbapp_itemmeta';

	/**
	 * Api namespace.
	 *
	 * @var string $namespace
	 */
	public $namespace = 'buddyboss-app/v1';

	/**
	 * Route slug.
	 *
	 * @var string $route_slug
	 */
	private $route_slug = '/meta';

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * MetaApi constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return MetaApi
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function init() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register routes.
	 */
	public function register_routes() {
		// Item Meta Last changed.
		register_rest_route(
			$this->namespace,
			$this->route_slug . '/last-changed/(?P<item_type>\S+)/',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'rest_item_last_changed' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'item_type' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'Content Type.', 'buddyboss-app' ),
							'enum'        => array( 'course', 'lesson', 'topic' ),
						),
						'item_id'   => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'Item ID', 'buddyboss-app' ),
						),
						'timestamp' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'Timestamp', 'buddyboss-app' ),
						),
					),
				),
			)
		);

    }

    /**
     * @param $request
     * @return WP_Error
     * @api {GET} /wp-json/buddyboss-app/v1/meta/last-changed/:item_type LearnDash item last changed
     * @apiName GetItemMetaLastChanged
     * @apiGroup LD Meta
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Last change information for self and children's
     * @apiUse apidocForItemMetaLastChangedV1
     */
    public function rest_item_last_changed($request)
    {
        $item_type = @$request->get_param('item_type');
        $item_id = @$request->get_param('item_id');
        $since_timestamp = @$request->get_param('timestamp');

        $meta_data = array();
        $last_change = null;

		switch ( $item_type ) {
			case 'course':
				$meta_data = CourseMeta::get_item_last_changed( 'sfwd-courses', $item_id, $since_timestamp );
				break;
		}

		if ( is_wp_error( $meta_data ) ) {
			return new WP_Error( 'error_while_retrieving', __( 'Error encountered while retrieving item by ID.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

        return rest_ensure_response($meta_data);
    }

	/**
	 * Fires on activation.
	 *
	 * @return void
	 */
	public function on_activation() {
		global $wpdb;

		$global_prefix   = bbapp_get_global_db_prefix();
		$charset_collate = $wpdb->get_charset_collate();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$table_name = "{$global_prefix}bbapp_itemmeta";

		$sql = "CREATE TABLE {$table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            item_id bigint(20) NOT NULL,
            item_type varchar(100) NOT NULL,
            meta_data longtext NOT NULL,
            date_created datetime DEFAULT '0000-00-00 00:00:00' NULL,
            date_updated datetime DEFAULT '0000-00-00 00:00:00' NULL
		    ) {$charset_collate}";

		dbDelta( $sql );
	}

	/**
	 * Fires on plugin deactivate.
	 *
	 * @return void
	 */
	public function on_plugin_deactivate() {
	}
}
