<?php
/**
 * Holds item meta main class functionality.
 *
 * @package BuddyBossApp\ItemMeta
 */

namespace BuddyBossApp\ItemMeta;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * ItemMeta abstract class.
 */
abstract class ItemMeta {

	/**
	 * Must be specified/override in subClass.
	 *
	 * @var null $post_type
	 */
	protected $post_type = null;

	/**
	 * ItemMeta constructor.
	 */
	private function __construct() {
		// ... leave empty, see setUp() below.
	}

	/**
	 * This will initialize hooks(actions and filters)
	 *
	 * @return void
	 */
	public function init() {
		// NOTE : Hooks would be invoked for subClass(s) based on $this->postType value.
		add_action( "save_post_{$this->post_type}", array( $this, 'on_save_or_update' ), 10, 3 );
		add_action( "update_post_{$this->post_type}", array( $this, 'on_save_or_update' ), 10, 3 );
	}

	/**
	 * This will produce and store meta by postType into bbapp_itemmeta table
	 *
	 * @return void
	 */
	public function store_meta_once() {
		// Setting arguments for the query.
		$args  = array(
			'numberposts' => - 1, // -1 is for all
			'post_type'   => $this->post_type,
		);
		$posts = get_posts( $args );

		foreach ( $posts as $post ) {
			$meta_data = ItemMetaManager::prepare_meta( $post->ID );
			$status    = ItemMetaManager::update_or_insert_item( $post->ID, $this->post_type, $meta_data );

			if ( $status ) {
				// NOTE : Regenerating transient for this item.
				ItemMetaManager::set_transient_for_item( $post->ID );
			}
		}
	}


	/**
	 * Should be defined/implemented on sub class.
	 * Fires when Item(Eg : Course or Lesson or Topic) is saved/updated
	 *
	 * @param \WP_REST_Response $response API response.
	 * @param \WP_Post          $post     Post data.
	 * @param bool              $update   Save or update.
	 *
	 * @return mixed
	 */
	abstract public function on_save_or_update( $response, $post, $update );

	/**
	 * If Item has child
	 *
	 * @return boolean
	 */
	abstract public function has_child();

	/**
	 * If Item has parent
	 *
	 * @return boolean
	 */
	abstract public function has_parent();
}
